// =============================================================================
//
//      --- kvi_integerselector_qt.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviIntegerSelector"

#include <qobjectlist.h>
#include <qspinbox.h>

#include "kvi_debug.h"
#include "kvi_integerselector_qt.h"
#include "kvi_label.h"

KviIntegerSelector::KviIntegerSelector(
	QWidget *parent, const char *description, void *pNumber, IntSelectorType type,
	int lowerBound, int upperBound, int defValue, bool bEnabled, int fieldWidth,
	const char *name, const char *suffix)
	: QWidget(parent, name)
{
	__range_valid(pNumber);
	m_pNumber    = pNumber;
	m_type       = type;
	m_pLabel     = new KviLabel(_CHAR_2_QSTRING(description), this);
	m_pSpinBox   = new QSpinBox(this);
	m_iLowBound  = lowerBound;
	m_iHighBound = upperBound;
	m_szDefault.setNum(defValue);
	m_pSpinBox->setMinValue(lowerBound);
	m_pSpinBox->setMaxValue(upperBound);
	if( *suffix ) m_pSpinBox->setSuffix(suffix);

	switch( type ) {
		case Int :
			m_pSpinBox->setValue(*((int *) (pNumber)));
			break;
		case UInt :
			m_pSpinBox->setValue(*((unsigned int *) (pNumber)));
			break;
		case UShort:
			m_pSpinBox->setValue(*((unsigned short int *) (pNumber)));
			break;
		default:
			m_pSpinBox->setValue(*((short int *) (pNumber)));
	}
	m_iFieldWidth = (fieldWidth > 0) ? fieldWidth : 100;
	setMinimumWidth(m_iFieldWidth + m_pLabel->sizeHint().width() + 1);
	setMinimumHeight(m_pSpinBox->sizeHint().height());
	setEnabled(bEnabled);
}

KviIntegerSelector::~KviIntegerSelector()
{
	// Nothing here
}

void KviIntegerSelector::setEnabled(bool bEnabled)
{
	m_pLabel->setEnabled(bEnabled);
	m_pSpinBox->setEnabled(bEnabled);
	QWidget::setEnabled(bEnabled);
}

void KviIntegerSelector::commitChanges()
{
	KviStr szT = m_pSpinBox->cleanText();
	bool bOk = false;
	// Adjust the value
	int value = szT.toInt(&bOk);
	if( !bOk ) szT = m_szDefault.ptr();
	if( value < m_iLowBound  ) szT.setNum(m_iLowBound);
	if( value > m_iHighBound ) szT.setNum(m_iHighBound);
	switch( m_type ) {
		case Int:
			(*((int *) m_pNumber)) = szT.toInt(&bOk);
			if( !bOk ) (*((int *) m_pNumber)) = m_szDefault.toInt(&bOk);
			break;
		case UInt:
			(*((unsigned int *) m_pNumber)) = szT.toUInt(&bOk);
			if( !bOk ) (*((unsigned int *) m_pNumber)) = m_szDefault.toUInt(&bOk);
			break;
		case UShort:
			(*((unsigned short *) m_pNumber)) = szT.toUShort(&bOk);
			if( !bOk ) (*((unsigned short *) m_pNumber)) = m_szDefault.toUShort(&bOk);
			break;
		default:
			(*((short *) m_pNumber)) = szT.toShort(&bOk);
			if( !bOk ) (*((short *) m_pNumber)) = m_szDefault.toShort(&bOk);
	}
}

void KviIntegerSelector::commitAll(QWidget *w)
{
	QObjectList *l = w->queryList("KviIntegerSelector");
	if( !l ) return;
	QObjectListIt it(*l);
	while( it.current() ) {
		((KviIntegerSelector *) it.current())->commitChanges();
		++it;
	}
	delete l;
}

void KviIntegerSelector::resizeEvent(QResizeEvent *)
{
	m_pSpinBox->setGeometry(width() - m_iFieldWidth, 0, m_iFieldWidth, height());
	m_pLabel->setGeometry(0, 0, width() - (m_iFieldWidth + 1), height());
}

QSize KviIntegerSelector::sizeHint()
{
	return QSize(m_pLabel->sizeHint().width() + m_iFieldWidth + 1, m_pSpinBox->sizeHint().height());
}

#include "m_kvi_integerselector_qt.moc"
