/**
 * Copyright (C) 2003 Billy Biggs <vektor@dumbterm.net>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Non-linear response compression function from this paper:
 *   G.M. Johnson and M.D. Fairchild.  Rendering HDR Images.
 *   IS&T/SID 11th Color Imaging Conference. 36-41 (2003)
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include "exrinput.h"
#include "exroutput.h"

static double calculate_fl_from_la_ciecam02( double la )
{
    double la5 = la * 5.0;
    double k = 1.0 / (la5 + 1.0);

    /* Calculate k^4. */
    k = k * k;
    k = k * k;

    return (0.2 * k * la5) + (0.1 * (1.0 - k)
                                  * (1.0 - k)
                                  * pow(la5, 1.0 / 3.0));
}

static void xyz_to_srgb( double *r, double *g, double *b,
                         double  x, double  y, double  z )
{
    *r = (  3.2406 * x ) + ( -1.5372 * y ) + ( -0.4986 * z );
    *g = ( -0.9689 * x ) + (  1.8758 * y ) + (  0.0415 * z );
    *b = (  0.0557 * x ) + ( -0.2040 * y ) + (  1.0570 * z );
}

static void srgb_to_xyz( double *x, double *y, double *z,
                         double  r, double  g, double  b )
{
    *x = ( 0.4124 * r ) + ( 0.3576 * g ) + ( 0.1805 * b );
    *y = ( 0.2126 * r ) + ( 0.7152 * g ) + ( 0.0722 * b );
    *z = ( 0.0193 * r ) + ( 0.1192 * g ) + ( 0.9505 * b );
}

static void lms_to_xyz( double *x, double *y, double *z,
                        double l, double m, double s )
{
    *x = (1.85024 * l) - (1.13830 * m) + (0.23843 * s);
    *y = (0.36683 * l) + (0.64388 * m) - (0.01067 * s);
    *z = (1.08885 * s);
}

static void xyz_to_lms( double *l, double *m, double *s,
                        double x, double y, double z )
{
    *l = ( 0.4002 * x) + (0.7075 * y) - (0.0807 * z);
    *m = (-0.2280 * x) + (1.1500 * y) + (0.0612 * z);
    *s = ( 0.9184 * z);
}

int main( int argc, char **argv )
{
    exroutput_t *exrout;
    exrinput_t *exrin;
    exrinput_t *whitein;
    double *scanline;
    int width, height;
    int y;

    if( argc < 4 ) {
        fprintf( stderr, "Usage: exricamtm input.exr blur.exr output.exr\n" );
        return 1;
    }

    exrin = exrinput_new( argv[ 1 ] );
    if( !exrin ) {
        fprintf( stderr, "exricamtm: Cannot open %s.\n", argv[ 1 ] );
        return 1;
    }
    width = exrinput_get_width( exrin );
    height = exrinput_get_height( exrin );
    fprintf( stdout, "exricamtm: %s [%dx%d]\n", argv[ 1 ], width, height );
    scanline = malloc( width * 4 * sizeof( double ) );

    whitein = exrinput_new( argv[ 2 ] );
    if( !whitein ) {
        fprintf( stderr, "exricamtm: Cannot open %s.\n", argv[ 2 ] );
        exrinput_delete( exrin );
        free( scanline );
        return 1;
    }

    if( exrinput_get_width( whitein ) != width ||
        exrinput_get_height( whitein ) != height ) {
        fprintf( stderr, "exricamtm: Whitepoint image (%dx%d) does "
                         "not match input image (%dx%d).\n",
                 exrinput_get_width( whitein ),
                 exrinput_get_height( whitein ), width, height );
        exrinput_delete( whitein );
        exrinput_delete( exrin );
        free( scanline );
        return 1;
    }

    exrout = exroutput_new( argv[ 3 ], width, height );
    if( !exrout ) {
        fprintf( stderr, "exricamtm: Cannot open %s.\n", argv[ 3 ] );
        exrinput_delete( whitein );
        exrinput_delete( exrin );
        free( scanline );
        return 1;
    }

    for( y = 0; y < height; y++ ) {
        double *input = exrinput_get_scanline( exrin, y );
        double *whiteinput = exrinput_get_scanline( whitein, y );
        int x;

        fprintf( stderr, "exricamtm: Correcting scanline %d\r", y );

        for( x = 0; x < width; x++ ) {
            double r, g, b;
            double X, Y, Z;
            double L, M, S;
            double fl;
            double yw;

            r = whiteinput[ (x * 4) + 0 ];
            g = whiteinput[ (x * 4) + 1 ];
            b = whiteinput[ (x * 4) + 2 ];
            srgb_to_xyz( &X, &Y, &Z, r, g, b );
            yw = Y;

            fl = (1.0 / 1.7) * calculate_fl_from_la_ciecam02( yw * 1000.0 );
            if( fl < 0.3 ) fl = 0.3;

            r = 100.0 * input[ (x * 4) + 0 ];
            g = 100.0 * input[ (x * 4) + 1 ];
            b = 100.0 * input[ (x * 4) + 2 ];

            srgb_to_xyz( &X, &Y, &Z, r, g, b );
            xyz_to_lms( &L, &M, &S, X, Y, Z );

            if( L < 0.0 ) L = -pow( -L, fl ); else L = pow( L, fl );
            if( M < 0.0 ) M = -pow( -M, fl ); else M = pow( M, fl );
            if( S < 0.0 ) S = -pow( -S, fl ); else S = pow( S, fl );

            lms_to_xyz( &X, &Y, &Z, L, M, S );
            xyz_to_srgb( &r, &g, &b, X, Y, Z );

            scanline[ (x * 4) + 0 ] = r;
            scanline[ (x * 4) + 1 ] = g;
            scanline[ (x * 4) + 2 ] = b;
            scanline[ (x * 4) + 3 ] = input[ (x * 4) + 3 ];
        }
        exroutput_scanline( exrout, scanline );
    }
    fprintf( stderr, "\nexricamtm: Done.\n" );

    exroutput_delete( exrout );
    exrinput_delete( exrin );
    free( scanline );
    return 0;
}

