/*
 * yank.c
 */

/*
 * yank  -  yet another NoteKeeper
 * Copyright (C) 1999, 2000, 2001 Michael Humann <m.hussmann@home.ins.de>
 *
 * This program is free software; you  can redistribute it and/or modify it
 * under the terms of  the GNU General  Public License as published  by the
 * Free Software Foundation;  either version 2 of  the License, or (at your
 * option) any later version.
 *
 * This program  is  distributed in the  hope  that it will  be useful, but
 * WITHOUT  ANY   WARRANTY;  without   even   the  implied    warranty   of
 * MERCHANTABILITY or    FITNESS FOR A PARTICULAR   PURPOSE.   See  the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the  GNU General Public License along
 * with this program; if not, write to the  Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "yank.h"

#ifdef USE_GCONF
#include <gconf/gconf.h>
#endif /* USE_GCONF */

#include "app.h"
#include "callbacks.h"
#include "fileio.h"
#include "preferences.h"
#include "todolist.h"

#include <glade/glade.h>

/* ------------------------------------------------------ */

GtkWidget        *application;
static char      *geometry = NULL;
static char      *file = NULL;

static struct    poptOption options[] = {
    {
        "geometry",
        'g',
        POPT_ARG_STRING,
        &geometry,
        0,
        N_("Specify the geometry of the main window"),
        N_("GEOMETRY")
    },
    {
        "file",
        'f',
        POPT_ARG_STRING,
        &file,
        0,
        N_("Load notes from this file"),
        N_("FILE")
    },
    { NULL, '\0', 0, NULL, 0, NULL, NULL }
};

/* ------------------------------------------------------ */

int
main(int argc, char* argv[])
{
#ifdef USE_GCONF
    GError	*gc_error;
#endif /* USE_GCONF */
    
    poptContext pctx;
    const char  **args;
    GnomeClient *client;

#ifdef ENABLE_NLS
    bindtextdomain(PACKAGE, GNOMELOCALEDIR);
    textdomain(PACKAGE);
#endif /* ENABLE_NLS */

    /*
     * Initialize the GConf library conditionally.
     */
    
#ifdef USE_GCONF
    gc_error = NULL;
    if (!(gconf_init(argc, argv, &gc_error)))
    {
        if (gc_error != NULL)
        {
            g_warning(_("Error during GConf initialization: %s."),
                      gc_error->message);
        }
        g_clear_error(&gc_error);
    }
#endif /* USE_GCONF */
    
    gnome_init_with_popt_table(PACKAGE, VERSION, argc, argv,
                               options, 0, &pctx);
    glade_gnome_init();
    
#ifdef USE_GCONF
    pref_gconfclient();
#endif /* USE_GCONF */

    /* Argument parsing */
    
    args = poptGetArgs(pctx);
    
    poptFreeContext(pctx);
    
    /* Session Management */
    
    client = gnome_master_client ();
    gtk_signal_connect(GTK_OBJECT(client), "save_yourself",
                       GTK_SIGNAL_FUNC(save_session), argv[0]);
    gtk_signal_connect(GTK_OBJECT(client), "die",
                       GTK_SIGNAL_FUNC(session_die), NULL);
    
    
    /* Main app */
    
    application = yank_app_new(geometry, file);
    
    gtk_widget_show(application);

    show_tool_status(GNOME_APP(application), get_preferences());

    cb_todo_timeout(NULL); /* regularly calls set_todolist_colors() */
    
    gtk_main();
    
#ifdef USE_GCONF
    gtk_object_unref(GTK_OBJECT(pref_gconfclient()));
#endif /* USE_GCONF */

    return (0);
}

/* ------------------------------------------------------ */

static gint
save_session(GnomeClient *client, gint phase, GnomeSaveStyle save_style,
              gint is_shutdown, GnomeInteractStyle interact_style,
              gint is_fast, gpointer client_data)
{
    GtkWindow *rwin;
    gchar     **argv;
    guint     argc;

    rwin = yank_root_win(NULL);
    
    /* allocate 0-filled, so it will be NULL-terminated */
    argv = g_malloc0(sizeof(gchar*)*6);
    argc = 1;
    
    argv[0] = client_data;

    argv[1] = "-g";
    argv[2] = g_strdup_printf("%dx%d+%d+%d", GTK_WIDGET(rwin)->allocation.width,
                              GTK_WIDGET(rwin)->allocation.height,
                              GTK_WIDGET(rwin)->allocation.x,
                              GTK_WIDGET(rwin)->allocation.x);
    argc = 3;
    
    if (get_filename() != NULL)
    {
        argv[3] = "-f";
        argv[4] = g_strdup(get_filename());
        argc += 2;
    }
    
    gnome_client_set_clone_command(client, argc, argv);
    gnome_client_set_restart_command(client, argc, argv);

    g_free(argv[2]);
    g_free(argv[4]);
    
    return (TRUE);
}

/* ------------------------------------------------------ */

static void
session_die(GnomeClient* client, gpointer client_data)
{
#ifdef USE_GCONF
    gtk_object_unref(GTK_OBJECT(pref_gconfclient()));
#endif /* USE_GCONF */
    gtk_main_quit();
}

/* ------------------------------------------------------ */


