/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "singit/singleton.h"

#include "singit_debug.h"

#include <gtk/gtksignal.h>

// Print singleton debug information
// #define DEBUG_SINGLETON 1

enum {
	AFTER_INIT,
	BEFORE_DESTROY,
	LAST_SIGNAL,
};

static GtkObjectClass *parent_class = NULL;
static guint single_signals[LAST_SIGNAL] = { 0 };

static void singit_singleton_class_init (SingitSingletonClass *klass);
static void singit_singleton_init (SingitSingleton *sstat);
static void singit_singleton_destroy(GtkObject *object);

GtkType singit_singleton_get_type (void)
{
	static GtkType singit_singleton_type = 0;

	if (!singit_singleton_type) {

		static const GtkTypeInfo singit_singleton_info =
		{
			(gchar*) "SingitSingleton",
			sizeof (SingitSingleton),
			sizeof (SingitSingletonClass),
			(GtkClassInitFunc) singit_singleton_class_init,
			(GtkObjectInitFunc) singit_singleton_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		singit_singleton_type = gtk_type_unique(GTK_TYPE_OBJECT, &singit_singleton_info);
	}

	return singit_singleton_type;
}

static void singit_singleton_class_init (SingitSingletonClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass*) klass;

	parent_class = gtk_type_class(GTK_TYPE_OBJECT);

	single_signals[AFTER_INIT] =
		gtk_signal_new("after_init",
			GTK_RUN_LAST,
			object_class->type,
			GTK_SIGNAL_OFFSET
				(SingitSingletonClass, after_init),
			gtk_marshal_BOOL__NONE,
			GTK_TYPE_BOOL, 0);

	single_signals[BEFORE_DESTROY] =
		gtk_signal_new("before_destroy",
			GTK_RUN_LAST,
			object_class->type,
			GTK_SIGNAL_OFFSET
				(SingitSingletonClass, before_destroy),
			gtk_marshal_NONE__NONE,
			GTK_TYPE_NONE, 0);

	gtk_object_class_add_signals
		(object_class, single_signals, LAST_SIGNAL);

	klass->after_init = NULL;
	klass->before_destroy = NULL;

	object_class->destroy = singit_singleton_destroy;
}

static void singit_singleton_init(SingitSingleton *single)
{
}

static void singit_singleton_destroy(GtkObject *object)
{
	SingitSingleton *single;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SINGIT_SINGLETON (object));

	single = SINGIT_SINGLETON(object);

	gtk_signal_emit(object, single_signals[BEFORE_DESTROY]);

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy (object);
}

static void singit_singleton_destroyed(gpointer data)
{
#ifdef DEBUG_SINGLETON
	g_print("singit_singleton.c [singit_singleton_destroyed]\n\n");
#endif
	*((gpointer*) data) = NULL;
}

SingitSingleton* singit_singleton_construct
	(SingitSingleton **single, GtkType type, gboolean *init)
{
	g_return_val_if_fail(single != NULL, NULL);
	g_return_val_if_fail(init != NULL, NULL);
	g_return_val_if_fail
		(gtk_type_is_a(type, TYPE_SINGIT_SINGLETON), NULL);

	*init = FALSE;

	if (*single == NULL) {
		*single = gtk_type_new(type);

		if (*single != NULL) {
			GtkObject *object = GTK_OBJECT(*single);
			gtk_object_weakref(object,
				singit_singleton_destroyed, single);

			*init = TRUE;
			gtk_signal_emit(object,
				single_signals[AFTER_INIT], init);

			if (*init == FALSE)
				{ gtk_object_unref(object); }
#ifdef DEBUG_SINGLETON
			else {
				g_print("singit_singleton.c [singit_singleton_construct] - %i (%i)\n",
					object->ref_count, GTK_OBJECT_TYPE(object));
			}
#endif
		}
	}

	return (*single);
}

SingitSingleton* singit_singleton_attach(SingitSingleton *single)
{
#ifdef DEBUG_SINGLETON
	g_print("singit_singleton.c [singit_singleton_attach] : ");
#endif
	if (single != NULL) {

		g_return_val_if_fail (IS_SINGIT_SINGLETON (single), NULL);

		gtk_object_ref(GTK_OBJECT(single));

#ifdef DEBUG_SINGLETON
		g_print("- %i (%i)\n",
			GTK_OBJECT(single)->ref_count,
			GTK_OBJECT_TYPE(single));
#endif
		return single;
	}

#ifdef DEBUG_SINGLETON
	g_print("- Failed\n");
#endif
	return single;
}

void singit_singleton_detach(SingitSingleton **single)
{
	SingitSingleton *_single;

	g_return_if_fail(single != NULL);

	_single = *single;
	if (_single == NULL)
		{ return; }

	g_return_if_fail(IS_SINGIT_SINGLETON(_single));

#ifdef DEBUG_SINGLETON
	g_print("singit_singleton.c [singit_singleton_detach] - %i (%i)\n",
		GTK_OBJECT(_single)->ref_count - 1, GTK_OBJECT_TYPE(_single));
#endif

	gtk_object_unref(GTK_OBJECT(_single));
	_single = NULL;
}

gboolean singit_singleton_initialize(SingitSingleton  *single)
{
	g_return_val_if_fail(single != NULL, FALSE);
	g_return_val_if_fail(IS_SINGIT_SINGLETON(single), FALSE);

	if (GTK_OBJECT_FLOATING(single))
	{
		gtk_object_ref(GTK_OBJECT(single));
		gtk_object_sink(GTK_OBJECT(single));

#ifdef DEBUG_SINGLETON
		g_print("singit_singleton.c [singit_singleton_initialize] - %i (%i)\n",
			GTK_OBJECT(single)->ref_count, GTK_OBJECT_TYPE(single));
#endif
		return TRUE;
	}
	return FALSE;
}
