/* Manage pipelines of partial images.
 * 
 * J.Cupitt, 17/4/93.
 * 1/7/93 JC
 *	- adapted for partial v2
 *	- ANSIfied
 * 6/7/93 JC
 *	- im_setupout() conventions clarified - see autorewind in
 *	  im_iocheck().
 * 20/7/93 JC
 *	- eval callbacks added
 * 7/9/93 JC
 *	- demand hint mechanism added
 * 25/10/93
 *	- asynchronous output mechanisms removed, as no observable speed-up
 * 9/5/94
 *      - new thread stuff added, with a define to turn it off
 * 15/8/94
 *	- start & stop functions can now be NULL for no-op
 * 7/10/94 JC
 *	- evalend callback system added
 * 23/12/94 JC
 *	- IM_ARRAY uses added
 * 22/2/95 JC
 *	- im_fill_copy() added
 *	- im_region_region() uses modified
 * 24/4/95 JC & KM
 *	- im_fill_lines() bug removed
 * 30/8/96 JC
 *	- revised and simplified ... some code shared with im_iterate()
 *	- new im_generate_region() added
 * 2/3/98 JC
 *	- IM_ANY added
 * 20/7/99 JC
 *	- tile geometry made into ints for easy tuning
 * 30/7/99 RP JC
 *	- threads reorganised for POSIX
 * 29/9/99 JC
 *	- threadgroup stuff added
 * 15/4/04
 *	- better how-many-pixels-calculated
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG_IO
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/
#include <vips/intl.h>

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <assert.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /*HAVE_UNISTD_H*/

#include <vips/vips.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Start and stop functions for one image in, input image is first user data.
 */
void *
im_start_one( IMAGE *out, IMAGE *in, void *dummy )
{
	return( im_region_create( in ) );
}

int
im_stop_one( REGION *reg, void *dummy1, void *dummy2 )
{
	im_region_free( reg );

	return( 0 );
}

/* Stop and start functions for many images in. First client is pointer to 
 * null-terminated array of input images.
 */
int
im_stop_many( REGION **ar, void *dummy1, void *dummy2 )
{
	int i;

	for( i = 0; ar[i]; i++ )
		im_region_free( ar[i] );
	im_free( (char *) ar );

	return( 0 );
}

void *
im_start_many( IMAGE *out, IMAGE **in, void *dummy )
{
	int i, n;
	REGION **ar;

	/* How many images?
	 */
	for( n = 0; in[n]; n++ )
		;

	/* Alocate space for region array.
	 */
	if( !(ar = IM_ARRAY( NULL, n + 1, REGION * )) )
		return( NULL );

	/* Create a set of regions.
	 */
	for( i = 0; i < n; i++ )
		if( !(ar[i] = im_region_create( in[i] )) ) {
			im_stop_many( ar, NULL, NULL );
			return( NULL );
		}
	ar[n] = NULL;

	return( ar );
}

/* Convenience function - make a null-terminated array of input images.
 * Use with im_start_many.
 */
IMAGE **
im_allocate_input_array( IMAGE *out, ... )
{
	va_list ap;
	IMAGE **ar;
	IMAGE *im;
	int i, n;

	/* Count input images.
	 */
	va_start( ap, out );
	for( n = 0; (im = va_arg( ap, IMAGE * )); n++ )
		;
	va_end( ap );

	/* Allocate array.
	 */
	if( !(ar = IM_ARRAY( out, n + 1, IMAGE * )) )
		return( NULL );

	/* Fill array.
	 */
	va_start( ap, out );
	for( i = 0; i < n; i++ )
		ar[i] = va_arg( ap, IMAGE * );
	va_end( ap );
	ar[n] = NULL;

	return( ar );
}

/* Output to a memory area. Might be im_setbuf(), im_mmapin()/im_makerw() or
 * im_mmapinrw(). 
 */
static int
eval_to_memory( im_threadgroup_t *tg, REGION *or )
{
	int y, chunk;
	IMAGE *im = or->im;

#ifdef DEBUG_IO
	int ntiles = 0;
        printf( "eval_to_memory: partial image output to memory area\n" );
#endif /*DEBUG_IO*/

	/* Choose a chunk size ... 1/100th of the height of the image, about.
	 * This sets the granularity of user feedback on eval progress, but
	 * does not affect mem requirements etc.
	 */
	chunk = (im->Ysize / 100) + 1;

	/* Loop down the output image, evaling each chunk. 
	 */
	for( y = 0; y < im->Ysize; y += chunk ) {
		Rect pos;

		/* Attach or to this position in image.
		 */
		pos.left = 0;
		pos.top = y;
		pos.width = im->Xsize;
		pos.height = chunk;
		if( im_region_image( or, &pos ) ) 
			return( -1 );

		/* Ask for evaluation of this area.
		 */
		if( im__eval_to_region( tg, or ) ) 
			return( -1 );

		/* Trigger evaluation callbacks for this image.
		 */
		if( im__handle_eval( im, or->valid.width, or->valid.height ) ) 
			return( -1 );

#ifdef DEBUG_IO
		ntiles++;
#endif /*DEBUG_IO*/
	}

#ifdef DEBUG_IO
	printf( "eval_to_memory: success! %d patches written\n", ntiles );
#endif /*DEBUG_IO*/

	return( 0 );
}

/* Eval to file.
 */
static int
eval_to_file( im_threadgroup_t *tg, REGION *or )
{
	IMAGE *im = or->im;
        int y;
 
#ifdef DEBUG_IO
        int nstrips;

        nstrips = 0;
        printf( "eval_to_file: partial image output to file\n" );
#endif /*DEBUG_IO*/

        /* Fill to in steps, write each to the output.
         */
        for( y = 0; y < im->Ysize; y += tg->nlines ) {
		Rect pos;

		/* Attach or to this position in image.
		 */
		pos.left = 0;
		pos.top = y;
		pos.width = im->Xsize;
		pos.height = IM_MIN( tg->nlines, im->Ysize - y );
		if( im_region_local( or, &pos ) ) 
			return( -1 );

		/* Ask for evaluation of this area.
		 */
		if( im__eval_to_region( tg, or ) ) 
			return( -1 );

                /* Write this set of scan lines.
                 */
                if( im__write( im->fd, 
			IM_REGION_ADDR( or, 0, y ), or->bpl * pos.height ) ) 
                        return( -1 ); 

#ifdef DEBUG_IO
                nstrips++;
#endif /*DEBUG_IO*/
        }

#ifdef DEBUG_IO
        printf( "eval_to_file: success! %d strips written\n", strips );
#endif /*DEBUG_IO*/

        return( 0 );
}

/* Attach a generate function to an image.
 */
int
im_generate( IMAGE *im,
        void *(*start_fn)(), int (*gen_fn)(), int (*stop_fn)(),
        void *a, void *b )
{
        int res;
	REGION *or;
	im_threadgroup_t *tg;

	if( im_image_sanity( im ) )
		return( -1 );
	if( im->Xsize <= 0 || im->Ysize <= 0 || im->Bands <= 0 ) {
		im_errormsg( "im_generate: bad dimensions" );
		return( -1 );
	}
 
        /* Look at output type to decide our action.
         */
        switch( im->dtype ) {
        case IM_PARTIAL:
                /* Output to partial image. Just attach functions and return.
                 */
                if( im->generate || im->start || im->stop ) {
                        im_errormsg( "im_generate: func already attached" );
                        return( -1 );
                }

                im->start = start_fn;
                im->generate = gen_fn;
                im->stop = stop_fn;
                im->client1 = a;
                im->client2 = b;
 
#ifdef DEBUG_IO
                printf( "im_generate: attaching partial callbacks\n" );
#endif /*DEBUG_IO*/
 
                break;
 
        case IM_SETBUF:
        case IM_SETBUF_FOREIGN:
        case IM_MMAPINRW:
        case IM_OPENOUT:
                /* Eval now .. sanity check.
                 */
                if( im->generate || im->start || im->stop ) {
                        im_errormsg( "im_generate: func already attached" );
                        return( -1 );
                }

                /* Get output ready.
                 */
                if( im_setupout( im ) )
                        return( -1 );

                /* Attach callbacks.
                 */
                im->start = start_fn;
                im->generate = gen_fn;
                im->stop = stop_fn;
                im->client1 = a;
                im->client2 = b;
 
                /* Evaluate. Two output styles: to memory area (im_setbuf()
                 * or im_mmapinrw()) or to file (im_openout()).
                 */
		if( !(or = im_region_create( im )) )
			return( -1 );
		if( !(tg = im_threadgroup_create( im )) ) {
			im_region_free( or );
			return( -1 );
		}
                if( im->dtype == IM_OPENOUT )
                        res = eval_to_file( tg, or );
                else
                        res = eval_to_memory( tg, or );

                /* Clean up.
                 */
		im_threadgroup_free( tg );
		im_region_free( or );

		/* Evaluation is now complete, with all sequences finished.
		 * Trigger evalend callbacks, then free them to make sure we
		 * don't trigger twice.
		 */
		res |= im__trigger_callbacks( im->evalendfns );
		IM_FREEF( im_slist_free_all, im->evalendfns );
 
                /* Error?
                 */
                if( res )
                        return( -1 );
 
                break;
 
        default:
                /* Not a known output style.
                 */
		im_errormsg( "im_generate: unable to output to a %s image",
			im_dtype2char( im->dtype ) );
                return( -1 );
        }
 
        return( 0 );
}

/* Generate a region of pixels ... with threads! Very like im_prepare(), but
 * threaded and does sub-division. Used by ip for threaded display, for
 * example.
 */
int
im_prepare_thread( im_threadgroup_t *tg, REGION *or, Rect *r )
{
	IMAGE *im = or->im;
	int res;

	if( im_image_sanity( im ) )
		return( -1 );

	switch( im->dtype ) {
	case IM_PARTIAL:
		/* or should have local memory.
		 */
                if( im_region_local( or, r ) )
                        return( -1 );
 
                /* Evaluate into or!
                 */
		res = im__eval_to_region( tg, or );

                /* Error?
                 */
                if( res )
                        return( -1 );
 
		break;

	case IM_OPENIN:
		/* or should be a mmap() window on im.
		 */
		if( im_region_mmap_window( or, r ) )
			return( -1 );

		break;

	case IM_SETBUF:
        case IM_SETBUF_FOREIGN:
	case IM_MMAPIN:
	case IM_MMAPINRW:
		/* Attach to existing buffer.
		 */
		if( im_region_image( or, r ) )
			return( -1 );

		break;

	default:
		im_errormsg( "im_prepare_thread: unable to input from a %s "
			"image", im_dtype2char( im->dtype ) );
		return( -1 );
	}

	return( 0 );
}
