/* ==================================================== ======== ======= *
 *
 *  uuhtml.cpp
 *  Ubit Project [Elc][2002]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2002 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:02] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uuhtml.cpp	ubit:03.06.02"
#include <iostream>
#include <strings.h>
#include <ctype.h> // for isxdigit
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <ubit/xml/uhtmldoc.hpp>
#include <ubit/xml/uhtmltags.hpp>
#include <ubit/xml/uxmlparser.hpp>
using namespace std;

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/* XHTML reader (see class UDoc) */

UDoc* UHtmlDoc::Reader::read(const UStr& _pathname,
                             Callbacks* call, Errors* _err) {
  UHtmlDoc* doc = new UHtmlDoc(_pathname, call);
  Errors errs = doc->readFile(_pathname);

  if (errs.stat <= 0) {
    delete doc;
    if (_err) *_err = errs;
    return null;
  }
  else {
    if (_err) *_err = errs;
    return doc;
  }
}

/* ==================================================== ======== ======= */

UHtmlDoc::UHtmlDoc(const UStr& _pathname, Callbacks* _call)
: UXmlDoc(_pathname, _call) {
  initModels();
}

UDoc::Errors UHtmlDoc::readFile(const UStr& _pathname) {
  // pathnme = _pathname;
  
  Errors errs;
  DOMDocument* dom_doc = parser->parse(_pathname, errs);

  if (!dom_doc) {
    errs.set(UFilestat::CannotOpen);
    return errs;
  }
  
  // createTree calls initModels();
  createTree(_pathname, dom_doc, errs);

  if (errs.stat == 0) errs.stat = UFilestat::Opened;
  else {
    cerr << "XML parse stat: " << errs.stat << endl;
    cerr << errs.messages << endl;
  }
  return errs;
}

UHtmlDoc::~UHtmlDoc() {}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

// Elements
UXmlElemModel UHtmlDoc::elem_models[] = {
  {"body",   UHtml_body::create},
  {"div",    UHtml_div::create},
  {"p",      UHtml_p::create},
  {"pre",    UHtml_pre::create},
  {"blockquote", UHtml_blockquote::create},
  {"br",     UHtml_br::create},

  {"h1",     UHtml_h1::create},
  {"h2",     UHtml_h2::create},
  {"h3",     UHtml_h3::create},
  {"h4",     UHtml_h4::create},
  {"h5",     UHtml_h5::create},
  {"h6",     UHtml_h5::create},

  {"ul",     UHtml_ul::create},
  {"ol",     UHtml_ol::create},
  {"li",     UHtml_li::create},

  {"table",  UHtml_table::create},
  {"tr",     UHtml_tr::create},
  {"td",     UHtml_td::create},

  {"a",      UHtml_a::create},
  {"img",    UHtml_img::create},

  { "span",  UHtml_span::create},
  { "font",  UHtml_font::create},
  { "em",    UHtml_em::create},
  { "b",     UHtml_b::create},
  { "i",     UHtml_i::create},
  { "u",     UHtml_u::create},

  {"head",   UHtml_head::create},
  {"title",  UHtml_title::create},
  {"stitle",  UHtml_stitle::create},
  {"link",   UHtml_link::create},
  {"meta",   UHtml_meta::create},
  {"style",  UHtml_style::create},

  {null, null}
};

// Attributes
UXmlAttrModel UHtmlDoc::attr_models[] = {
  {"class", UHtml_class::create},
  {"name", UHtml_name::create},
  {"type", UHtml_type::create},
  {"target", UHtml_target::create},
  {"src", UHtml_src::create},
  {"href", UHtml_href::create},

  {"align",  UHtml_align::create},
  {"valign", UHtml_valign::create},
  {"color",  UHtml_color::create},
  {"bgcolor",UHtml_bgcolor::create},

  {"colspan",UHtml_colspan::create},
  {"rowspan",UHtml_rowspan::create},
  {"nowrap", UHtml_nowrap::create},

  {"border", UHtml_border::create},
  {"width", UHtml_width::create},
  {"cellspacing", UHtml_cellspacing::create},
  {"cellpadding", UHtml_cellpadding::create},

  {"xmlns", UHtml_xmlns::create},
  {"http-equiv", UHtml_http_equiv::create},  //http-equiv et non _
  {"content", UHtml_content::create},

  {null, null}
};

// values, etc.
static XMLCh *xmlch_left, *xmlch_center, *xmlch_right,
  *xmlch_top, *xmlch_middle, *xmlch_bottom,
  *xmlch_src, *xmlch_href, *xmlch_name, *xmlch_class;

/* ==================================================== [Elc:02] ======= */
/* ==================================================== ======== ======= */

bool UHtmlDoc::is_model_initialized = false;

void UHtmlDoc::initModels() {
  if (is_model_initialized) return;
  is_model_initialized = true;

  for (const UXmlAttrModel* paa = attr_models; paa->cname != null; paa++) {
    paa->dom_name = XMLString::transcode(paa->cname);
  }

  for (const UXmlElemModel* pea = elem_models; pea->cname != null; pea++) {
    pea->dom_name = XMLString::transcode(pea->cname);
  }

  xmlch_left   = XMLString::transcode("left");
  xmlch_center = XMLString::transcode("center");
  xmlch_right  = XMLString::transcode("right");
  xmlch_top    = XMLString::transcode("top");
  xmlch_middle = XMLString::transcode("middle");
  xmlch_bottom = XMLString::transcode("bottom");

  xmlch_src    = XMLString::transcode("src");
  xmlch_href   = XMLString::transcode("href");
  xmlch_name   = XMLString::transcode("name");

  xmlch_class  = XMLString::transcode("class");

  /* inutile
  xmlch_type   = XMLString::transcode("type");
  xmlch_target = XMLString::transcode("target");
  xmlch_content= XMLString::transcode("content");
  xmlch_http_equiv = XMLString::transcode("http-equiv");  // - et non _
  */
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UHtml_head::UHtml_head(DOMElement* e) : UXmlElem(e) {
  show(false);			// this element is not shown
}

UStyle *UHtml_body::style = null;
const UStyle& UHtml_body::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UVbox::makeStyle());
    style->local.padding.set(10, 10);

    style->font = new UFont();
    style->font->setPointSize(16);
 }
  return *style;
}

UStyle *UHtml_div::style = null;
const UStyle& UHtml_div::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UFlowbox::makeStyle());
    style->local.padding.set(1, 6);
 }
  return *style;
}

UStyle *UHtml_p::style = null;
const UStyle& UHtml_p::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UFlowbox::makeStyle());
    style->local.padding.set(1, 6);  // top_bottom, short _left_right
 }
  return *style;
}

UStyle *UHtml_pre::style = null;
const UStyle& UHtml_pre::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UFlowbox::makeStyle());
    style->local.padding.set(8, 6); // top_bottom, short _left_right
    style->font = &UFont::courier;
 }
  return *style;
}

UStyle *UHtml_blockquote::style = null;
const UStyle& UHtml_blockquote::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UFlowbox::makeStyle());
    style->local.padding.set(8, 20); // top_bottom, short _left_right
 }
  return *style;
}

UHtml_br::UHtml_br(DOMElement* e)
  : UStr("\n"), UXmlElem(e) 
{}

// ========================================================================
// ========================================================================

UStyle *UHtml_ul::style = null;
const UStyle& UHtml_ul::getStyle(UContext*) const{
  if (!style) {
    style = new UStyle(UFlowbox::makeStyle());
    style->local.padding.set(1, 1, 0, 30);  // top_bottom, short _left_right
  }
  return *style;
}

UStyle *UHtml_ol::style = null;
const UStyle& UHtml_ol::getStyle(UContext*) const{
  if (!style) {
    style = new UStyle(UFlowbox::makeStyle());
    style->local.padding.set(1, 1, 0, 30);
  }
  return *style;
}

UStyle *UHtml_li::style = null;
const UStyle& UHtml_li::getStyle(UContext*) const{
  if (!style) {
    style = new UStyle(UFlowbox::makeStyle());
    style->local.padding.set(1, 1);
    style->local.content = &ugroup(UPix::rball);
  }
  return *style;
}

// ======================================================================
// ======================================================================

// NB padding:
// set(short _top_bottom, short _left_right);
// set(short _top, short _right, short _bottom, short _left);

UStyle *UHtml_h1::style = null;
const UStyle& UHtml_h1::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UFlowbox::makeStyle());
    
    //style->local.padding.set(12, 6);
    style->local.padding.set(30, 0, 25, 30);

    style->font = new UFont(UFont::bold);
    style->font->setPointSize(34);

    style->fgcolors = UStyle::makeColors(&UColor::orange, &UColor::white);
    //style->bgcolors = UStyle::makeColors(&UColor::teal, &UColor::white);
  }
  return *style;
}

// ======================================================================

UStyle *UHtml_h2::style = null;
const UStyle& UHtml_h2::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UFlowbox::makeStyle());
    
    //style->local.padding.set(7, 6);
    //style->local.padding.set(20, 0, 15, 30);
    style->local.padding.set(15, 0, 10, 30);

    style->font = new UFont();
    style->font->setPointSize(24);

    style->fgcolors = UStyle::makeColors(&UColor::navy, &UColor::white);
    //style->bgcolors = UStyle::makeColors(&UColor::yellow, &UColor::white);
  }
  return *style;
}

// ======================================================================

UStyle *UHtml_h3::style = null;
const UStyle& UHtml_h3::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UFlowbox::makeStyle());

    //style->local.padding.set(5, 6); 
    style->local.padding.set(8, 0, 8, 70);

    style->font = new UFont();
    style->font->setPointSize(20);

    UColor* color = new UColor("#800080");
    style->fgcolors = UStyle::makeColors(color, &UColor::white);
    //style->bgcolors = UStyle::makeColors(&UColor::grey, &UColor::white);

    style->local.content = 
      &ugroup(USymbol::right + "  ");
    //+ USymbol::check + USymbol::radio +USymbol::circle
  }
  return *style;
}

// ======================================================================

UStyle *UHtml_h4::style = null;
const UStyle& UHtml_h4::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UFlowbox::makeStyle());

    //style->local.padding.set(3, 6);
    style->local.padding.set(5, 0, 5, 150);

    style->font = new UFont();
    style->font->setPointSize(18);

    style->fgcolors = UStyle::makeColors(&UColor::blue, &UColor::white);

    style->local.content = &ugroup(UPix::rball + "  ");
  }
  return *style;
}

// ======================================================================

UStyle *UHtml_h5::style = null;
const UStyle& UHtml_h5::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UFlowbox::makeStyle());

    //style->local.padding.set(2, 6); 
    style->local.padding.set(5, 0, 5, 200);

    style->font = new UFont();
    style->font->setPointSize(14);
  }
  return *style;
}

// ======================================================================

UStyle *UHtml_h6::style = null;
const UStyle& UHtml_h6::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UFlowbox::makeStyle());

    //style->local.padding.set(1, 6);
    style->local.padding.set(8, 0, 8, 250);

    style->font = new UFont();
    style->font->setPointSize(14);
  }
  return *style;
}

// ======================================================================
// ======================================================================

UStyle *UHtml_table::style = null;
const UStyle& UHtml_table::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UTable::makeStyle());
    style->font = &UFont::inherit;
    style->local.valign = UValign::BOTTOM;
    style->local.halign = UHalign::LEFT;
  }
  return *style;
}

UHtml_table::UHtml_table(DOMElement* e) : UXmlElem(e) {
  if (e) {
    const XMLCh* class_val = e->getAttribute(xmlch_class);
 
    if (class_val) {
      char* s = XMLString::transcode(class_val);
      if (s && strcmp(s,"stitle")==0) {
	addAttr(UBgcolor::grey);
	addAttr(uwidth(720)); // !!ATT : TAILLE FIXE
      }
      delete[] s; //v2.3.0 XMLString::release(&s);
    }
  }
}

UStyle *UHtml_tr::style = null;
const UStyle& UHtml_tr::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UTrow::makeStyle());
    style->font = &UFont::inherit;
    style->local.hspacing = 0;
    style->local.valign = UValign::BOTTOM;
    style->local.halign = UHalign::LEFT;
  }
  return *style;
}

UStyle *UHtml_td::style = null;
const UStyle& UHtml_td::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UTcell::makeStyle());
    style->font = &UFont::inherit;
    //style->local.padding.set(0,2);
    style->local.valign = UValign::BOTTOM;
    style->local.halign = UHalign::LEFT;
  }
  return *style;
}

UStyle *UHtml_th::style = null;
const UStyle& UHtml_th::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UTcell::makeStyle());
    style->font = &UFont::inherit;
  }
  return *style;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UHtml_a::createChildren(UXmlDoc* doc, UDoc::Errors& errs) {

  UXmlElem::createChildren(doc, errs);  // devrait plutot etre apres?

  if (getDomElement()) {
    const XMLCh* name = getDomElement()->getAttribute(xmlch_name);
    if (name) name_path = doc->makePath(name);

    const XMLCh* href = getDomElement()->getAttribute(xmlch_href);
    if (href) {
      href_path = doc->makePath(href);

      if (href_path) {
        UCall* call = doc->makeCall(href_path);
        if (call) this->addlist(UOn::enter / call
                                + UOn::leave / call
                                + UOn::mpress / call
                                + UOn::mrelease / call
                                + UOn::action / call
                                );
      }
    }
  }
  
  doc->storeLink(this);
}

// ======================================================================
// CF AUSSI: tailles predefinies et rescale !!!

void UHtml_img::createChildren(UXmlDoc* doc, UDoc::Errors& errs) {
  if (getDomElement()) {
    const XMLCh* src = getDomElement()->getAttribute(xmlch_src);
    if (src) {
      src_path = doc->makePath(src);
      if (src_path) this->set(*src_path);  // la lecture du fichier se fera plus tard
    }
  }

  doc->storeObject(this);
}

bool UHtml_img::isLoaded() const {
  return UIma::isLoaded();
}

int UHtml_img::load() {
  return UIma::load();  // reloads file
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UStyle *UHtml_span::style = null;
const UStyle& UHtml_span::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UGroup::makeStyle());
  }
  return *style;
}

UStyle *UHtml_b::style = null;
const UStyle& UHtml_b::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UGroup::makeStyle());
    style->font = &UFont::bold;
 }
  return *style;
}

UStyle *UHtml_i::style = null;
const UStyle& UHtml_i::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UGroup::makeStyle());
    style->font = &UFont::italic;
  }
  return *style;
}

UStyle *UHtml_em::style = null;
const UStyle& UHtml_em::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UGroup::makeStyle());
    style->font = &UFont::fill;
  }
  return *style;
}

UStyle *UHtml_u::style = null;
const UStyle& UHtml_u::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UGroup::makeStyle());
    style->font = &UFont::underline;
  }
  return *style;
}

UStyle *UHtml_font::style = null;
const UStyle& UHtml_font::getStyle(UContext*)  const{
  if (!style) {
    style = new UStyle(UGroup::makeStyle());
  }
  return *style;
}

/*
UBrick* UPFont::create(const DOM_Element& e) {

  char* face = e.getAttribute(DOMString::transcode("face")).transcode();

  UFontFamily* ff = &UFontFamily::any;

  if (!face)
    ;
  else if (strcmp(face, "fixed") == 0)
    ff = &UFontFamily::standard;
  else if (strcmp(face, "courier") == 0)
    ff = &UFontFamily::courier;
  else if (strcmp(face, "helvetica") == 0)
    ff = &UFontFamily::helvetica;
  else if (strcmp(face, "times") == 0)
    ff = &UFontFamily::times;
  else {
    fprintf(stderr, "unknown font face : %s\n", face);
    ff = &UFontFamily::standard;
  }
  delete face;

  short ls;
  char* size = e.getAttribute(DOMString::transcode("size")).transcode();

  if (size) {
    char* endptr;
    ls = strtol(size, &endptr, 10);
    if ((endptr == size) || (ls>UFont::MAX_LSIZE) || (ls<UFont::MIN_LSIZE)) {
      fprintf(stderr, "invalid font size : %s\n", size);
      ls = 0; 
    }
    delete size;
  } 
  else ls = 0;

  //return new UPFont(e, UFont::standard);

  UFont *fonte = new UFont(*ff, 0, ls);
  return new UPFont(e, *fonte);
}

UPFont::UPFont(const DOM_Element& e, const UFont &font) : 
  UFont(font), UXmlElem(e) {}
*/

/* ==================================================== [Elc:02] ======= */
/* ==================================================== ======== ======= */
// Attributes 

UHtml_align::UHtml_align(DOMAttr* attr, UDoc::Errors& e) : UXmlAttr(attr) {
  const XMLCh* name = (attr ? attr->getValue() : null);
  if (name) {  
    if (XMLString::compareString(name, xmlch_left) == 0)
      set(UHalign::left);
    else if (XMLString::compareString(name, xmlch_center) == 0) 
      set(UHalign::center);
    else if (XMLString::compareString(name, xmlch_right) == 0)
      set(UHalign::right);
    else UXmlDoc::setXmlError(e, UXmlDoc::InvalidValue,
		     "bad value for attribute align: ", name);
  }
  else UXmlDoc::setXmlError(e, UXmlDoc::InvalidValue,
		   "null value for attribute align");
}

/* ==================================================== ======== ======= */

UHtml_valign::UHtml_valign(DOMAttr* attr, UDoc::Errors& e) : UXmlAttr(attr) {
  const XMLCh* name = (attr ? attr->getValue() : null);
  if (name) {
    if (XMLString::compareString(name, xmlch_top) == 0)
      set(UValign::top);
    else if (XMLString::compareString(name, xmlch_middle) == 0)
      set(UValign::center);
    else if (XMLString::compareString(name, xmlch_bottom) == 0)
      set(UValign::bottom);
    else UXmlDoc::setXmlError(e, UXmlDoc::InvalidValue,
		     "bad value for attribute valign : ", name);
  }
  else UXmlDoc::setXmlError(e, UXmlDoc::InvalidValue,
		   "null value for attribute valign");
}

/* ==================================================== [Elc:02] ======= */
/* ==================================================== ======== ======= */

static int getIntVal(DOMAttr* attr, bool& stat) {
  stat = false;
  if (!attr) return 0;
  
  const XMLCh* name = attr->getValue();
  if (!name) return 0;

  char* s = XMLString::transcode(name);
  stat = (s && *s);              // chercher char non " "  !!!

  int res = (stat ? atoi(s) : 0); 
  delete[] s;
  //v2.3.0 XMLString::release(&s);

  return res;
}

/* ==================================================== ======== ======= */

UHtml_border::UHtml_border(DOMAttr* attr, UDoc::Errors& e) : UXmlAttr(attr) {
  bool stat;
  val = getIntVal(attr, stat);
  if (!stat) UXmlDoc::setXmlError(e, UXmlDoc::InvalidValue,
			 "null value for attribute border");
}

UHtml_width::UHtml_width(DOMAttr* attr, UDoc::Errors& e) 
  : UXmlAttr(attr) 
{
  bool stat;
  val = getIntVal(attr, stat);
  if (!stat) UXmlDoc::setXmlError(e, UXmlDoc::InvalidValue,
			 "null value for attribute width");
}

UHtml_cellspacing::UHtml_cellspacing(DOMAttr* attr, UDoc::Errors& e)
  : UXmlAttr(attr) 
{
  bool stat;
  val = getIntVal(attr, stat);
  if (!stat) UXmlDoc::setXmlError(e, UXmlDoc::InvalidValue,
			 "null value for attribute cellspacing");
}

UHtml_cellpadding::UHtml_cellpadding(DOMAttr* attr, UDoc::Errors& e)
  : UXmlAttr(attr) 
{
  bool stat;
  val = getIntVal(attr, stat);
  if (!stat) UXmlDoc::setXmlError(e, UXmlDoc::InvalidValue,
			 "null value for attribute cellpadding");
}

UHtml_colspan::UHtml_colspan(DOMAttr* attr, UDoc::Errors& e) 
  : UXmlAttr(attr) 
{
  bool stat;
  val = getIntVal(attr, stat);
  if (!stat) UXmlDoc::setXmlError(e, UXmlDoc::InvalidValue,
			 "null value for attribute colspan");
}

UHtml_rowspan::UHtml_rowspan(DOMAttr* attr, UDoc::Errors& e) 
  : UXmlAttr(attr) 
{
  bool stat;
  val = getIntVal(attr, stat);
  if (!stat) UXmlDoc::setXmlError(e, UXmlDoc::InvalidValue,
			 "null value for attribute rowspan");
}

UHtml_nowrap::UHtml_nowrap(DOMAttr* attr, UDoc::Errors& e) 
  : UXmlAttr(attr) {
  // pas de valeur ..
}

// ======================================================================
// ======================================================================

static UColor* getColorFromAttr(DOMAttr* attr, UDoc::Errors& errs) {
  const XMLCh* name = (attr ? attr->getValue() : null);
  if (name) {
    char* s = XMLString::transcode(name);
    UColor* col = new UColor(s);
    delete[]s;
    //v2.3.0 XMLString::release(&s);
    return col;
  }
  else {
    UXmlDoc::setXmlError(errs, UXmlDoc::InvalidValue,
		"null value for attribute color or bgcolor");
    return &UColor::none;
  }
}

UHtml_bgcolor::UHtml_bgcolor(DOMAttr* attr, UDoc::Errors& e) :
  UBgcolor(*getColorFromAttr(attr, e)), UXmlAttr(attr) {}

UHtml_color::UHtml_color(DOMAttr* attr, UDoc::Errors& e) :
  UColor(*getColorFromAttr(attr, e)), UXmlAttr(attr) {}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */


