/* ==================================================== ======== ======= *
 *
 *  ustr.hpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _ustr_hpp_
#define	_ustr_hpp_
//pragma ident	"@(#)ustr.hpp	ubit:03.06.04"
#include <ubit/uelem.hpp>

/* ==================================================== ======== ======= */
/** Active String class.
 * Notes:
 * - use the onChange() method to add a callback to this object.
 *   This callback will be fired when the object's value changes.
 * - the 'UOn::propChange' callbacks of the object's parents will
 *   also be fired (after the object's callbacks)
 */
class UStr: public UElem {
public:
  static const unsigned int npos = static_cast<unsigned int>(-1);
  static UStr none, newline;
  
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  UStr();
  UStr(const char*);
  UStr(const UStr&);
  UStr(const std::string&);
  UStr(const UIntgBase&);
  UStr(const UFloatBase&);
  /**< constructors
    * - arg is a C, C++ or Ubit string or a Ubit Number.
    * - data is copied internally (that the arg can be freely destroyed)
    */

  friend UStr& ustr()                       {return *new UStr();}
  friend UStr& ustr(const char* _s)         {return *new UStr(_s);}
  friend UStr& ustr(const UStr& _s)         {return *new UStr(_s);}
  friend UStr& ustr(const std::string& _s)  {return *new UStr(_s);}
  ///< creator shortcuts (same as *new UStr()).

  virtual ~UStr();
  ///< destructor.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void clear();
  ///< clears the string's content.

  virtual bool empty() const {return (len == 0);}
  ///< true if content is empty

  int length() const {return len;}
  ///< returns the length of the string.

  friend std::ostream& operator<<(std::ostream&, const UStr&);
  ///< prints string on ostream, for instance: cout << str.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  UStr& operator=(const char* s2)         {set(s2); return *this;}
  UStr& operator=(const UStr& s2)         {set(s2); return *this;}
  UStr& operator=(const std::string& s2)  {set(s2); return *this;}
  UStr& operator=(const UIntgBase& v2)    {set(v2); return *this;}
  UStr& operator=(const UFloatBase& v2)   {set(v2); return *this;}
  ///< copy the value of the argument to this string.

  /* ces operateurs sont ambigus a cause de char* dans le cas: str = 0
     UStr& operator=(int v2)                 {set(v2); return *this;}
     UStr& operator=(double v2)              {set(v2); return *this;}
  */

  virtual void set(const char* s2);
  virtual void set(const UStr& s2);
  virtual void set(const std::string& s2);
  virtual void set(const UIntgBase& v2);
  virtual void set(const UFloatBase& v2);
  virtual void setNum(int v2);
  virtual void setNum(double v2);
  ///< copy the value of the argument to this string.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  bool operator==(const char* s2) const        {return equals(s2);}
  bool operator==(const UStr& s2) const        {return equals(s2);}
  bool operator==(const std::string& s2) const {return equals(s2);}

  virtual bool equals(const char*) const;
  virtual bool equals(const std::string&) const;
  virtual bool equals(const UStr&) const;
  ///< returns true if string contents are the same.

  virtual int compare(const char*) const;
  virtual int compare(const std::string&) const;
  virtual int compare(const UStr&) const;
  
  virtual int compareTo(const char*, bool ignore_case = false) const;
  virtual int compareTo(const std::string&, bool ignore_case = false) const;
  virtual int compareTo(const UStr&, bool ignore_case = false) const;
  /**< compare strings lexicographically.
   * ignores case considerations if second argument is true (checks case
   * by default)
   */
  
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  const char* chars() const {return s;}
  const char* c_str() const {return s;}
  ///< returns a C string that should not be changed nor deleted.

  virtual char charAt(int pos) const;
  virtual char at(int pos) const {return charAt(pos);}
  /**< returns the character at offset 'pos'.
    * - pos = -1 refers to the last char of the string
    * - returns the character or 0 if the position is invalid
    * - does not throw exception (check the returned value)
    */

  virtual char setCharAt(int pos, char c, bool update = true);
  /**< changes the character at offset 'pos'.
    * - pos = -1 refers to the last char of the string
    * - returns the character or 0 if the position is invalid
    * - does not throw exception (check the returned value)
    */
  
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual bool copyTo(UStr& to, int pos = 0,
                      unsigned int nbchars = npos) const;
  virtual bool copyTo(std::string& to, int pos = 0,
                      unsigned int nbchars = npos) const;
  /**< copies a substring of *this string into the argument string.
    * details:
    * - nbchars = npos means "copy to the end of *this string".
    * - the data is appended to the end of the argument string
    *   (if it is not empty)
    */
  
  virtual UStr substring(int from_pos = 0, unsigned int nbchars = npos) const;
  virtual std::string toString(int from_pos = 0, unsigned int nbchars = npos) const;
  /**< returns a substring.
    * details:
    * - nbchars = npos means "copy to the end of *this string".
    */

  virtual UStr split(int pos, bool delete_char_at_pos = false);
  ///< splits the string: removes contents from offset 'pos' and returns the remaining chars.
  
  virtual void trim(bool strip_beginning = true, bool strip_end = true);
  virtual void strip(bool strip_beginning = true, bool strip_end = true);
  ///< removes white spaces, tabs and newlines at the beginning and end of string.
  
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  int find(char);
  int rfind(char);
  int find(const char*);
  int find(const UStr&);
  int find(const std::string&);
  ///< returns -1 if not found.
  
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  friend UStr operator&(const UStr& s1, const UStr& s2);
  friend UStr operator&(const UStr& s1, const char* s2);
  friend UStr operator&(const char* s1, const UStr& s2);
  friend UStr operator&(const UStr& s1, const std::string& s2);
  friend UStr operator&(const std::string& s1, const UStr& s2);
  friend UStr operator&(const UStr&, int);
  friend UStr operator&(const UStr&, float);
  friend UStr operator&(const UStr&, const UIntgBase&);
  friend UStr operator&(const UStr&, const UFloatBase&);
  /**<
    * creates a new string that collates the 2 arguments.
    * Notes:
    * - arguments are not modified
    * - we use & rather than + because the + operator is used for
    *  argument lists.
    */
  
  UStr& operator&=(const UStr& s2)        {append(s2); return *this;}
  UStr& operator&=(const char* s2)        {append(s2); return *this;}
  UStr& operator&=(const std::string& s2) {append(s2); return *this;}
  UStr& operator&=(int v2)                {append(v2); return *this;}
  UStr& operator&=(float v2)              {append(v2); return *this;}
  UStr& operator&=(const UIntgBase& v2)   {append(v2); return *this;}
  UStr& operator&=(const UFloatBase& v2)  {append(v2); return *this;}
  /**<
    * appends argument to this.
   * Notes:
    * - this is modified, the argument is not modified
    * - we use &= rather than += because the + operator is used for 
   *    argument lists.
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void append(char);
  virtual void append(const char*);
  virtual void append(const char*, unsigned int nbchars);
  virtual void append(const UStr&);
  virtual void append(const UStr&, unsigned int nbchars);
  virtual void append(const std::string&);
  virtual void append(const std::string&, unsigned int nbchars);
  virtual void append(int);
  virtual void append(float);
  virtual void append(const UIntgBase&);
  virtual void append(const UFloatBase&);
  ///< appends the string given as an argument to the end of *this string.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual bool insert(int to_pos, char);
  virtual bool insert(int to_pos, const char* from,
                      int from_pos = 0, unsigned int from_nbchars = npos);
  virtual bool insert(int to_pos, const UStr& from,
                      int from_pos = 0, unsigned int from_nbchars = npos);
  virtual bool insert(int to_pos, const std::string& from,
                      int from_pos = 0, unsigned int from_nbchars = npos);
  /**< inserts 'from' string at offset 'to_pos' in *this string.
    * details:
    * - pos = 0 specifies the beginning of the string (BEFORE the first char)
    * - pos = -1 specifies the end of the string (AFTER the last char)
    * - from_nbchars = npos means "to the end of the string"
    */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void remove(int pos, unsigned int nbchars);
  /**< removes 'nbchars' characters from offset 'pos'
    * details:
    * - pos = 0 specifies the first char of the string
    * - pos = -1 specifies the last char of the string
    */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void replace(int pos, unsigned int nbchars, const UStr&);
  virtual void replace(int pos, unsigned int nbchars, const char*);
  virtual void replace(int pos, unsigned int nbchars, const std::string&);
  ///< replaces 'nbchars' characters from offset 'pos' in this string by the argument string.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // file name manipulation

  virtual UStr getFileName(bool with_suffix = true) const;
  ///< returns the file name of a file path (= without the directory); removes the suffix if arg is false.

  virtual UStr getFileSuffix() const;
  ///< returns the suffix of a file path (= the end of the pathname after the dot)

  virtual UStr getFileDir(bool with_final_slash = true) const;
  ///< returns the directory of a file path (= without the file name).

  virtual UStr getFilePath(const UStr& new_suffix) const;
  ///< returns the file path with a new suffix.

  virtual long getFileType() const;
  virtual long getFileType(bool& is_regular_file, bool& is_directory) const;
  ///< returns the Unix file type (0 if the file does not exist)

  virtual bool isFileFound(bool is_regular_file = true) const;
  ///< returns true if file exists (also checks that this is a regular file if arg is true)

  virtual int readFile(const UStr& pathname);
  ///< reads this file into the UStr and returns the loading status (see UFilestat).

  static int readFile(const UStr& filename, std::vector<UStr*>&slist,
                       bool strip_beg_line = false, bool strip_end_line = false);
  ///< reads this file into the UStr vector and returns the loading status (see UFilestat).

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void changed(bool update_now);
  /**< [impl] called when object's content is changed.
    * This function:
    * - updates grahics (if arg is true)
    * - then fires object's UOn::change callbacks
    * - then fires parents' UOn::changeProp callbacks
    * - then fires parents' UOn::changeStr callbacks
    */

  virtual void update();
  ///< updates grahics.
  
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

#ifndef NO_DOC
  UStr(const char*, u_modes bmodes);

  virtual class UStr* strCast() {return this;}
  virtual void syncVals(char*,int);

  virtual void addingTo(class ULink *selflink, UGroup *parent);
  virtual void removingFrom(class ULink *selflink, UGroup *parent);
  ///< NOTE that this function require a specific destructor.

  virtual bool checkFormat(int pos, int newchar);
  virtual bool checkFormat(int pos, const char*);
  ///< checks input conformance to format model (if any).

  virtual bool setImpl(const char*, int len, bool call_callbacks);
  virtual bool insertImpl(int to_pos, char newchar, bool upd);
  virtual bool insertImpl(int to_pos, const char* s2, int from_pos,
                          unsigned int from_len, bool upd);
  virtual bool replaceImpl(int pos, unsigned int nbchars,
                           const char* s2, bool upd);
  
  // prototype for non-warped text (UView)
  virtual void getSize(class UContext*, u_dim *w, u_dim *h) const;

  // prototype for warped text (UFlowView)
  virtual void getSize(class UContext*, u_dim *w, u_dim *h, 
		       u_dim available_width, int offset, 
		       int *sublen, int *change_line) const;

  // prototype for non-warped text (UView)
  virtual void paint(class UWinGraph&, class UContext*, const URegion &r) const;

  // prototype for warped text (UFlowView)
  virtual void paint(class UWinGraph&, class UContext*, const URegion &r,
		     int offset, int cellen) const;
private:
  char* s;
  int len;  
#endif
};

/* === useful functions =============================== ======== ======= */

struct CStr {
  static char *strdup(const char *s1);
  ///< creates a duplicate of 's1' (s1 can be null) NB: this fct is similar to strdup().

  static char *strdupcat(const char *s1, const char *s2);
  ///< creates a duplicate of 's1' + 's2' (arguments can be null).

  static char *strdupcat(const char *s1, char sep, const char *s2);
  ///< creates a duplicate of 's1' + 'sep' + 's2' ('sep' is not added if either 's1' or 's2' is null).

  static char *strext(char *pathname);
  static const char *strext(const char *pathname);
  /**< finds the file suffix in a pathname.
    * - note: the returned value is null or points inside the original string
    *  (its is not duplicated and thus, can't be freed)
    */

  static void strparse(char* path_name, int path_len,
		char*& dir_name, int& dir_len,
                char*& base_name, int& base_len,
		char*& suffix, int& suffix_len);
  /**< decomposes a path name into dir name, base name and suffix components.
    * the returned pointers points inside the path name and should not be freed.
    */
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
