#include "VHDLProcess.hh"
#include "SignalBase.hh"
#include <warped/warped.h>
#include "UniversalReal.hh"

VHDLProcess::VHDLProcess( const string &name, _savant_entity_elab *proc ) : 
  VHDLKernel(name, proc){}

VHDLProcess::~VHDLProcess( ){}

void 
VHDLProcess::vhdlAssertOrReport( bool assertStatement, 
				 bool condition, 
				 const PhysicalType &timeOccurred,
				 const ArrayType &message,
				 const string &severityString, 
				 const string &locatorInfo ){
  if( condition == false ){
    if( assertStatement == true ){
      *wout << "Assertion failiure at ";
    }
    else{
      *wout << "Report at ";
    }
    *wout << "Time: ";
    *wout << timeOccurred;
    *wout << " Report: " << endl
	  << "    ";
    
    *wout << message;
    
    *wout << endl 
	  << "Severity Level: " << severityString 
	  << locatorInfo << endl;
  }
}

void 
VHDLProcess::vhdlAssertOrReport( bool assertStatement, 
				 const RValue &condition, 
				 const PhysicalType &timeOccurred,
				 const ArrayType &message,
				 const RValue &severityLevel, 
				 const string &locatorInfo ) {
  
  bool tempValue = condition.getIntValue();
  int  severity  = severityLevel.getIntValue();
  char *map = Savantseverity_levelType_info.get_imageMap()[severity];
  
  vhdlAssertOrReport( assertStatement, 
		      tempValue,
		      timeOccurred,
		      message,
		      map,
		      locatorInfo);

  if( tempValue == false && severity >= 2 ){
    reportError( "VHDL assertion severity >= ERROR, shutting down", ABORT );
  }
}

void 
VHDLProcess::executeVHDL( VHDLKernel_state & ){
  executeSignalUpdationPhase( scalarSignalsUpdateArray, 
			      compositeResolvedSignalsUpdateArray );
  executeVHDL();
  scalarSignalsUpdateArray.reset();
  compositeResolvedSignalsUpdateArray.reset();
  eventSigs.reset();
}

void
VHDLProcess::executeSignalUpdationPhase( Block &scalarSignals,
					 Block &compositeResolvedSignals ){
  // Call updateSignal() of all the unresolved scalar Signals,
  // Or on elements of composite unresolved signals or
  // on elements of composite resolved signals
  for( int i = 0; i < scalarSignals.getNumberOfElements(); i++ ){
    SignalBase *currentSignal = static_cast<SignalBase*>(scalarSignals.getElement(i));
    updateSignal( currentSignal );
  }

  //Call updateSignal() of all signals that are composite resolved signals
  for(int i = 0; i < compositeResolvedSignals.getNumberOfElements(); i++ ){
    VHDLType *currentSignal = static_cast<VHDLType*>(compositeResolvedSignals.getElement(i));
    updateSignal( currentSignal );
  }
  
    // Update the implicit guard signal if it exists
  if (getGuard() != NULL) {
    updateGuard(getGuard());
  }
}


void
VHDLProcess::updateSignal( SignalBase *signal, 
			   bool initializingSimulation ){
  const VHDLData *oldEffVal = signal->readEffVal();
  const VHDLType *newEffVal = NULL;
  bool eventOccurred = false;
  newEffVal = signal->resolve(this);
  if ( newEffVal == NULL ){
    // This could happen in tha case of signals of register kind where if
    // all the dirvers are disconnected the register value doesn't change
    return;
  }
  if ( !oldEffVal->operator==(newEffVal->getObject()->readVal()) ){  
    // update sensitivity list
    if( signal->isSensitive() ){
      eventSigs.add( signal->getSigId() );
    }
    eventOccurred = true;
  }
  
#ifdef VHDLDBG
  cout << getName() << " updating signal |" << signal->name << 
    "| with drvVal |" << *signal->readDrvVal() << "|, effVal |" <<
    *newEffVal << "|, and oldEffVal |" << *oldEffVal <<
    "| at " << getTimeNow() << endl;
#endif

  if( initializingSimulation == false ){
    // update the implicit attributes iff simulation is in progress.
    // other dont. 
    updateImplicitAttributes(signal, oldEffVal, eventOccurred);
  }
  
  //Update the effective Value of the signal if there is an event
  if(eventOccurred == true) {
    signal->updateEffVal(newEffVal->getObject()->readVal());
  }
}

void
VHDLProcess::updateSignal( VHDLType *signal ){
  const VHDLType *newEffVal = signal->resolve(this);
  
#ifdef VHDLDBG
  cout << getName() << " updating signal  with drvVal(";
  newEffVal->print(cout);
  cout << ") and effVal(";
  newEffVal->print(cout);
  cout << ") and oldEffVal(";
  signal->print(cout);
  cout <<") at " << getTimeNow() << endl;
#endif
  
  updateImplicitAttributes(signal, newEffVal);
  signal->updateEffVal(newEffVal);
}

void
VHDLProcess::assignSignal( VHDLType &dest, 
			   VHDLKernel *srcId, 
			   const VHDLType &src,
			   const PhysicalType &delay, 
			   const PhysicalType &rejTime, 
			   const ArrayInfo &dinfo,
			   const ArrayInfo &sinfo ){  
  ASSERT( srcId != 0 );
  dest.assignSignal( this, 
		     srcId, 
		     src,
		     delay,
		     rejTime,
		     dinfo, 
		     sinfo );
}

void
VHDLProcess::assignSignal( SignalBase &destSignal, 
			   VHDLKernel *srcProcess, 
			   const VHDLData &srcData,
			   const VHDLVTime &delay, 
			   const VHDLVTime &rejTime, 
			   const ArrayInfo &dinfo,
			   const ArrayInfo &sinfo) {
  ASSERT( srcProcess != 0 );
  destSignal.assignSignal(destSignal.getType(), 
			  srcProcess,
			  srcData, 
			  delay,
			  rejTime,
			  dinfo,
			  sinfo);
}

// When this is called, it is assumed to be for a savantBit.
// Anything else will screw up--in particular, chars.
void
VHDLProcess::assignSignal( SignalBase &dest, 
			   VHDLProcess *srcId, 
			   const char src,
			   const VHDLVTime &delay, 
			   const VHDLVTime &rejTime, 
			   const ArrayInfo &dinfo, 
			   const ArrayInfo &sinfo) {
  savantBit b(src);
  dest.assignSignal(dest.getType(), srcId, b, delay, rejTime, dinfo, sinfo);
}

void
VHDLProcess::assignNullTransaction( VHDLType &dest, 
				    VHDLProcess *srcId, 
				    const PhysicalType &delay, 
				    const PhysicalType &rejTime, 
				    const ArrayInfo&,
				    const ArrayInfo &){
  int i = 0;
  int dest_i = 0;
  int dest_lbound = 0, dest_rbound = 0;
  ArrayInfo::ArrayDirn_t dest_dir;
  const ArrayInfo *vec_bounds = NULL;
  
  switch(dest.get_kind()){
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    assignNullTransaction(*((SignalBase*)(*(ScalarType*)&dest).getObject()), srcId, 
		 ((PhysicalType &) delay).getVTime(), 
		 ((PhysicalType &) rejTime).getVTime(), nullInfo, nullInfo);
    break;
  case ARRAY_TYPE:
    vec_bounds  = &(dest.get_bounds(0));
    dest_lbound = vec_bounds->left();
    dest_rbound = vec_bounds->right();
    dest_dir    = vec_bounds->dirn();
    dest_i      = dest_lbound;
    
    for(i = vec_bounds->length(); (i > 0); i--) {
      assignNullTransaction(dest[dest_i], srcId, delay, rejTime, nullInfo,
			    nullInfo);
      dest_i += dest_dir;
    }
    break;
  case RECORD_TYPE:
    for(i = 1; i <= (*(RecordType*)&dest).get_number_of_fields(); i++) {
      assignNullTransaction((*(RecordType*)&dest).get_field(i), srcId,
			    delay, rejTime, nullInfo, nullInfo);
    }
    break;
    
  default:
    cerr << "Signal Assignment statement called for a non Signal Type "<< endl;
    break;
  };
}

void
VHDLProcess::assignNullTransaction( SignalBase &dest, 
				    VHDLProcess *srcId, 
				    const VHDLVTime &delay, 
				    const VHDLVTime &rejTime, 
				    const ArrayInfo &dinfo,
				    const ArrayInfo &sinfo) {

  // The process of posting transactions on signals also involves a phase called
  // marking.  Each scalar signal does its own marking and generates the transactions to
  // be sent out.  The actual events are sent out by the process.
  IntegerType dummySourceValue(ObjectBase::VARIABLE, SavantintegerType_info);
  
  dest.assignSignal( NULL_TRANSACTION, 
		     srcId, 
		     dummySourceValue.readVal(), 
		     delay,
		     rejTime, 
		     dinfo, 
		     sinfo );
}

void
VHDLProcess::disconnectDriver( SignalBase *signal, int processId ){
  VHDLProcess *toDisconnect = dynamic_cast<VHDLProcess *>( listOfProcesses[ processId ] );
  signal->disconnectDriver( toDisconnect );

#ifdef VHDLDBG
  cout << getName() << " Applying a NULL TRANSACTION to signal " << signal->name;
  cout << " at " << getTimeNow() << endl;
#endif

  //Update the signals driver corresponding to the source ID.
  if( signal->isCompositeResolvedSignal() == false ){
    //Add this element into the list of signals that are to be 
    //updated this simulation cycle
    scalarSignalsUpdateArray.addElement(signal);
  }
  else {
    compositeResolvedSignalsUpdateArray.addElement( signal->getParentCompositeType() );
  }
}

void
VHDLProcess::updateDriver( SignalBase *signal,
			   int sigSrc,
			   int senderSigId,
			   const VHDLData *newData,
			   ArrayInfo *sourceInfo,
			   ArrayInfo *destInfo ){
#ifdef VHDLDBG
  cout << getName() << " updating signal " << signal->name << " senderSigId " 
       << senderSigId << " slice " << *destInfo << " with value " << *newData
       << " slice " << *sourceInfo << " at " << getTimeNow() << endl;
#endif

  //Update the signals driver corresponding to the source ID.
  signal->updateDriver( senderSigId, sigSrc, newData, sourceInfo, destInfo );

  
  if(signal->isCompositeResolvedSignal() == false) {
    //updated this simulation cycle
    scalarSignalsUpdateArray.addElement(signal);
  }
  else {
    compositeResolvedSignalsUpdateArray.addElement(signal->getParentCompositeType());
  }
}


//The following function updates the implicit signal attributes
//of the currently updated composite resolved signal

void
VHDLProcess::updateImplicitAttributes( VHDLType *signal,
				      const VHDLType *newEffVal ){
  SignalBase *signalSubElement = NULL;
  VHDLData *newValue = NULL;
  VHDLData *oldEffectiveValue = NULL;
  bool eventOccurred = false;
  int i=0;
  switch(signal->get_kind()){
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    ASSERT(((ScalarType*)signal)->getKind() == ObjectBase::SIGNAL);
    signalSubElement = (SignalBase*)((ScalarType*)signal)->getObject();
    oldEffectiveValue = signalSubElement->readEffVal();
    ASSERT(newEffVal->is_scalar_type() == true);
    newValue = (VHDLData*)&((ScalarType*)newEffVal)->readVal();
    if( !oldEffectiveValue->operator==(*newValue) ){
      eventOccurred = true;
    }
    updateImplicitAttributes(signalSubElement, oldEffectiveValue, eventOccurred);
    break;
  case ARRAY_TYPE: {
    ASSERT(newEffVal->get_kind() == ARRAY_TYPE);
    ArrayType *dest = (ArrayType *) signal;
    ArrayType *src  = (ArrayType *) newEffVal;
    for(i = 0; (i < dest->length(0)); i++) {
      updateImplicitAttributes(&dest->get_element(i), &src->get_element(i));
    }
    break;
  }
  case RECORD_TYPE:
    ASSERT(newEffVal->get_kind() == RECORD_TYPE);
    for(i = 1; i <= (*(RecordType*)signal).get_number_of_fields(); i++) {
      updateImplicitAttributes(&((RecordType*)signal)->get_field(i),&((RecordType*)newEffVal)->get_field(i));
    }
    break;
  default:
    cerr << "ERROR: updateImplicitAttributes(VHDLType*,const VHDLType*)" << endl
	 << " of VHDLType: " << (int) signal->get_kind() << " Called"<< endl;
    abort();
    break;
  };
}

//The following function updates the implicit signal attributes
//of the currently updated function

void
VHDLProcess::updateImplicitAttributes(SignalBase *signal,
				     const VHDLData *oldEffVal,
				     bool eventOccurred) {
  
  for(int i = 0; i < signal->getNumAttributes(); i++) {
    switch(signal->getAttributeList()[i]->attrib) {
    case LAST_EVENT: 
      if(eventOccurred == true) {
	updateAttribute(signal->getAttributeList()[i]);
      }
      break;
    case LAST_VALUE: 
      if(eventOccurred == true) {
	updateAttribute( signal->getAttributeList()[i], *oldEffVal );
      }
      break;
    case LAST_ACTIVE:
      updateAttribute( signal->getAttributeList()[i] );
      break;
    case DRIVING :
      cerr << getProcessName()
	   << ": VHDLProcess::updateSignal : Not yet implemented" << endl;
      break;
    case TRANSACTION: 
      updateAttribute( signal->getAttributeList()[i] );
      break;
    case EVENT: 
    case DRIVING_VALUE:
    case INVALID:
    case ACTIVE:
      break;
    case STABLE:
      if(((SignalBase*) signal->getAttributeList()[i]->value->getObject())->isSensitive() 
	 && eventOccurred ){
	eventSigs.add(((SignalBase*) signal->getAttributeList()[i]->value->getObject())->getSigId());
      }
      break;
    case QUIET:
      if(((SignalBase*) signal->getAttributeList()[i]->value->getObject())->isSensitive() ){
	eventSigs.add(((SignalBase*) signal->getAttributeList()[i]->value->getObject())->getSigId());
      }
      break;
    default:
      cerr << getProcessName()
	   << ": VHDLProcess::updateSignal : (invalid attribute)" << endl;
      break;
    } // switch
  } // for
  updateDefaultAttributes(signal, oldEffVal, eventOccurred);
}

void
VHDLProcess::updateDefaultAttributes( SignalBase *signal,
				      const VHDLData *oldEffVal,
				      bool eventOccurred ){
  if(eventOccurred == true) {
    
    signal->getAllAttributes().last_event = UniversalLongLongInteger(getTimeNow().getMajor());
    signal->getAllAttributes().event = true;
  }

  signal->getAllAttributes().last_active = UniversalLongLongInteger(getTimeNow().getMajor());

  switch(signal->getAllAttributes().sigtype) {
  case VHDLData::UNIVERSAL_BOOLEAN:
  case VHDLData::UNIVERSAL_INTEGER:
    signal->getAllAttributes().last_value.inttype = oldEffVal->getIntValue();
    break;
  case VHDLData::UNIVERSAL_REAL:
    signal->getAllAttributes().last_value.floattype = ((UniversalReal*)oldEffVal)->val;
    break;
  case VHDLData::UNIVERSAL_LONG_LONG_INTEGER:
    signal->getAllAttributes().last_value.physicaltype = ((UniversalLongLongInteger*)oldEffVal)->val;
    break;
  default:
    break;
  }
}

void
VHDLProcess::updateSignal( int sigId, 
			   VHDLProcess *srcId, 
			   VHDLData *data,
			   const ArrayInfo *dInfo, 
			   const ArrayInfo *sInfo ){

  bool eventOccurred = false;	// Set if an event occurs on this signal.
  SignalBase *sig = getVHDLState()->locateSig(sigId); 
  
#ifdef VHDLDBG
  cout << getName() << " updating signal " << sig->name << " driver " 
       << srcId << " slice " << *dInfo << " with value ";
  data->print(cout);
  cout << " slice " << *sInfo << " at " << getTimeNow() << endl;
#endif

  const VHDLData *oldEffVal = sig->readEffVal();
  const VHDLType *newEffVal;

  // Update the driver as specified by srcId, and resolve the signal.
  // The resolve is an overloaded function in ImplicitSignal and Signal.
  // If the Signal has to be resolved, the signal will resolve itself and
  // update the effective value of the signal.
  if(sig->getKind() == ObjectBase::IMPLICIT_SIGNAL){
    newEffVal = (VHDLType *)data;
  }
  else{
    sig->updateDriver( sigId, srcId->getVHDLProcessId(), data, sInfo, dInfo );
    const VHDLData &tempVal = sig->readDriver(srcId)->getObject()->readVal();
    const VHDLData *driverVal = &tempVal;
    sig->updateDrvVal(driverVal, sInfo, dInfo);
    newEffVal = sig->resolve(this);
  }
  // check if there's been an event
  if ( !oldEffVal->operator==(newEffVal->getObject()->readVal()) ){  
    // update sensitivity list
    if (sig->isSensitive() == true) {
      eventSigs.add(sig->getSigId());
    }
    eventOccurred = true;
  }

#ifdef VHDLDBG
  cout << getName() << " updating signal " << sig->name << " with drvVal(";
  sig->readDrvVal()->print(cout);
  cout << ") and effVal(";
  sig->readEffVal()->print(cout);
  cout << ") and oldEffVal(";
  oldEffVal->print(cout);
  cout <<") at " << getTimeNow() << endl;
#endif
  for( int i = 0; i < sig->getNumAttributes(); i++ ){
    switch(sig->getAttributeList()[i]->attrib) {
    case LAST_EVENT: 
      if(eventOccurred == true) {
	updateAttribute( sig->getAttributeList()[i] );
      }
      break;
    case LAST_VALUE: 
      if(eventOccurred == true) {
	updateAttribute( sig->getAttributeList()[i], *oldEffVal);
      }
      break;
    case LAST_ACTIVE:
      updateAttribute( sig->getAttributeList()[i] );
      break;
    case DRIVING :
      cerr << getProcessName()
	   << ": VHDLProcess::updateSignal : Not yet implemented" << endl;
      break;
    case TRANSACTION: 
      updateAttribute( sig->getAttributeList()[i] );
      break;
    case EVENT: 
    case DRIVING_VALUE:
    case INVALID:
    case ACTIVE:
      break;
    case STABLE:
      if( dynamic_cast<SignalBase*>(sig->getAttributeList()[i]->value->getObject())->isSensitive() == true && eventOccurred == true) {
	eventSigs.add(((SignalBase*) sig->getAttributeList()[i]->value->getObject())->getSigId());
      }
      break;
    case QUIET:
      if(((SignalBase*) sig->getAttributeList()[i]->value->getObject())->isSensitive() == true) {
	eventSigs.add(((SignalBase*) sig->getAttributeList()[i]->value->getObject())->getSigId());
      }
      break;
    default:
      cerr << getProcessName()
	   << ": VHDLProcess::updateSignal : (invalid attribute)" << endl;
      break;
    } // switch
  } // for
  if(eventOccurred == true) {
    sig->updateEffVal(newEffVal->getObject()->readVal());
  }
}

void
VHDLProcess::updateGuard(VHDLType *guardSig) {
  SignalBase *signal = (SignalBase*)guardSig->getObject();
  const VHDLData *oldEffVal = signal->readEffVal();
  const VHDLType *newEffVal = NULL;
  bool eventOccurred = false;

  newEffVal = getGuardExpression();

  if ( newEffVal == NULL ){
    return;
  }
  if ( !oldEffVal->operator==(newEffVal->getObject()->readVal()) ){  
    // update sensitivity list
    if (signal->isSensitive() == true) {
      eventSigs.add(signal->getSigId());
    }
    eventOccurred = true;
  }
  
#ifdef VHDLDBG
  cout << getName() << " updating signal " << signal->name
       << " with drvVal(";
  signal->readDrvVal()->print(cout);
  cout << ") and effVal(";
  signal->readEffVal()->print(cout);
  cout << ") and oldEffVal(";
  oldEffVal->print(cout);
  cout <<") at " << getTimeNow() << endl;
#endif
  
  //Update the effective Value of the signal if there is an event
  if(eventOccurred == true) {
    signal->updateEffVal(newEffVal->getObject()->readVal());
  }
}

void
VHDLProcess::executeWait( int wId, const PhysicalType &timeout ){
  VHDLVTime delay = VHDLVTime::getVHDLVTimePositiveInfinity();
  
  // record which wait we're waiting on
  getVHDLState()->waitLabel = wId;
  
  // This is required in update Wait.
  getVHDLState()->setWaitTime( getTimeNow() );

  if ( &timeout != &SAVANT_INFINITY_TIME ){
    delay = dynamic_cast<const PhysicalType &>(timeout).getVTime();
  }

#ifdef VHDLDBG
  cout << getName() << " executing wait "<< wId << " at time " 
       << getTimeNow() << ", will resume in: " << delay << endl;
#endif

  getVHDLState()->waitExpired = false;
  if( delay != VHDLVTime::getVHDLVTimePositiveInfinity() ){
    scheduleWait( wId, getTimeNow() + delay );
  }
}

void
VHDLProcess::updateWait( const VHDLVTime &waitTime ){
  if( getVHDLState()->getWaitTime() == waitTime ){
    getVHDLState()->waitExpired = true;
  }
#ifdef VHDLDBG
  cout << getName() << " updating wait time to " << waitTime
       << ", waiting on "
       << getVHDLState()->waitLabel << " from "
       << getVHDLState()->getWaitTime();
  if(getVHDLState()->waitExpired == true) {
    cout << " wait expired." << endl;
  } else {
    cout << " wait has not expired, waitTime = " << getVHDLState()->getWaitTime();
  }
#endif
}

bool
VHDLProcess::resumeWait( int wId, const EnumerationType &cond ){
  bool resume = false;
  Wait *wait;

  if (wId != getVHDLState()->waitLabel) {
    cerr << getProcessName() << " resumeWait: wId (" << wId 
	 << ") not the one I'm waiting for! (" << getVHDLState()->waitLabel 
	 << ")" << endl;
    abort();
  }

  if( getVHDLState()->waitExpired == true ){
    // timeout has been satisfied
    resume = true;
#ifdef VHDLDBG
    cout << getName() << ": wait timeout has expired-resuming\n";
#endif
  } else {
    if( wId == WAITING_IN_PROC ){
      wait = getVHDLState()->procWait;
    } 
    else {
      wait = (getVHDLState()->wait + wId);
    }
    if( wait->getSensSize() > 0 ){
      // wait has a sensitivity list
      for( int i = 0; i < eventSigs.num(); i++ ){
	if( wait->sensitiveTo(eventSigs.get(i)) ){
	  // this wait is sensitive to a signal that has had an event this
	  // cycle.  Only resume the wait if there was a hit on the
	  // sensList _and_ the condition clause evaluates to true.
	  resume = (cond == SAVANT_BOOLEAN_TRUE);
#ifdef VHDLDBG
	  cout << getName() << " had event on sens. sig " << eventSigs.get(i)
		<< ": condition evals to " << (resume ? "true":"false") << endl;
#endif
	  break;
	} // if
      }	// for
    } // if(wait->getSensSize() > 0)
  } // else 

  if (resume == true) {
    getVHDLState()->waitLabel = NOT_WAITING;
#ifdef VHDLDBG
    cout << getName() << " resuming wait " << wId << " at " << getTimeNow() <<endl;
  } 
  else {
    cout << getName() << " not resuming wait " << wId << " at " 
	 << getTimeNow() << endl;
#endif
  }

  return resume;
}

void
VHDLProcess::updateAttribute( Attribute *attribute, 
			      const VHDLData &value ){
  switch(attribute->attrib) {
  case LAST_EVENT:
  case LAST_ACTIVE:
    dynamic_cast<PhysicalType *>(attribute->value)->getObject()->updateVal(UniversalLongLongInteger(getTimeNow().getMajor()));

//     ((PhysicalType *)attribute->value)->getObject()->updateVal(UniversalLongLongInteger(getSimulationTime().getMajor()));
    break;
  case LAST_VALUE:
    if(attribute->value->is_scalar_type() == true) {
      ((ScalarType *)attribute->value)->getObject()->updateVal(value);
    } else {
      abort();
    }
    break;
  case QUIET:
  case EVENT:
  case ACTIVE:
  case STABLE:
    ((EnumerationType *)attribute->value)->getObject()->updateVal(value);
    break;
  case TRANSACTION:
    attribute->value->updateVal( (dynamic_cast<EnumerationType *>(attribute->value)->vhdlNot()).readVal() );
    break;
  case DRIVING :
  case DRIVING_VALUE:
    cerr << "VHDLProcess::updateAttribute : Not yet implemented" << endl;
    break;
  default:
    cerr << "VHDLProcess::updateAttribute : (unknown attribute)" << endl;
    break;
  } // switch(attribute->attrib);
}
