#include "std_standardPkg.hh"
#include "Signal.hh"
#include "ImplicitSignal.hh"
#include "Variable.hh"
#include "SignalNetinfo.hh"
#include "STDTypes.hh"
#include <sstream>

using std::ostringstream;

const EnumerationType &
EnumerationType::getEnumerationZero(){
  static const EnumerationType ENUMERATION_ZERO(ObjectBase::VARIABLE, 0, SavantbitType_info);
  return ENUMERATION_ZERO;
}

const EnumerationType &
EnumerationType::getEnumerationOne(){
  static const EnumerationType ENUMERATION_ONE(ObjectBase::VARIABLE, 1, SavantbitType_info);
  return ENUMERATION_ONE;
}

const EnumerationType &
EnumerationType::getEnumerationFalse(){
  static const EnumerationType ENUMERATION_FALSE(ObjectBase::VARIABLE, 0, SavantbooleanType_info);
  return ENUMERATION_FALSE;
}

const EnumerationType &
EnumerationType::getEnumerationTrue(){
  static const EnumerationType ENUMERATION_TRUE(ObjectBase::VARIABLE, 1, SavantbooleanType_info);
  return ENUMERATION_TRUE;
}


const EnumerationType &
EnumerationType::toBoolean( bool toConvert ){
  if( toConvert ){
    return getEnumerationTrue();
  }
  else{
    return getEnumerationFalse();
  }
}


EnumerationType::EnumerationType(const EnumerationType& value): ScalarType(value) {
  typeInfo = value.typeInfo;
}


// The new constructors
EnumerationType::EnumerationType(ObjectBase::ObjectType objType,
				 const TypeInfo& tInfo): ScalarType(objType) {
  ASSERT ( tInfo.getKind() == ENUM_INFO );
  const EnumerationTypeInfo &eInfo = (const EnumerationTypeInfo &) tInfo;
  
  UniversalInteger initialValue((int) eInfo.get_left());
  
  switch(objType) {
  case ObjectBase::SIGNAL : 
    setObject( new Signal<UniversalInteger>(initialValue) );
    break;
  case ObjectBase::IMPLICIT_SIGNAL :
    setObject( new ImplicitSignal<UniversalInteger>(initialValue) );
    break;
  case ObjectBase::VARIABLE :
    setObject( new Variable<UniversalInteger>(initialValue) );
    break;
  case ObjectBase::SIGNAL_NETINFO :
    setObject( new SignalNetinfo );
    break;
  default:
    cerr << "ERROR: unknown object type" << endl;
    exit(-1);
  }

  typeInfo = eInfo;
}

EnumerationType::EnumerationType(ObjectBase::ObjectType objType, 
				 const UniversalInteger& val, 
				 const TypeInfo& tInfo): 
  ScalarType(objType) {
  switch(objType) {
  case ObjectBase::SIGNAL: 
    setObject( new Signal<UniversalInteger>(val) );
    break;
  case ObjectBase::IMPLICIT_SIGNAL :
    setObject( new ImplicitSignal<UniversalInteger>(val) );
    break;
  case ObjectBase::VARIABLE:
    setObject( new Variable<UniversalInteger>(val) );
    break;
  case ObjectBase::SIGNAL_NETINFO:
    setObject( new SignalNetinfo );
    break;
  default:
    cerr << "ERROR: unknown object type" << endl;
    exit(-1);
  }

  ASSERT ( tInfo.getKind() == ENUM_INFO );
  typeInfo = (const EnumerationTypeInfo &) tInfo;
}

EnumerationType::EnumerationType(ObjectBase::ObjectType objType,
                                 const IntegerType& value,
				 const TypeInfo& tInfo):
  
  ScalarType(objType) {
  UniversalInteger val = (UniversalInteger&) value.readVal();
  switch(objType) {
  case ObjectBase::SIGNAL :
    setObject( new Signal<UniversalInteger>(val)); 
    break;
  case ObjectBase::IMPLICIT_SIGNAL :
    setObject( new ImplicitSignal<UniversalInteger>(val) );
    break;
  case ObjectBase::VARIABLE :
    setObject( new Variable<UniversalInteger>(val)); 
    break;
  case ObjectBase::SIGNAL_NETINFO :
    setObject( new SignalNetinfo );
    break;
  default:
    cerr << "ERROR: unknown object type" << endl;
    exit(-1);
  }

  ASSERT ( tInfo.getKind() == ENUM_INFO );
  typeInfo = (const EnumerationTypeInfo &) tInfo;
}

EnumerationType::EnumerationType(ObjectBase::ObjectType objType, 
				 const EnumerationType& value,
				 const TypeInfo& tInfo): 
  ScalarType(objType) {
  UniversalInteger val = (UniversalInteger&) value.readVal();
  switch(objType) {
  case ObjectBase::SIGNAL: 
    setObject( new Signal<UniversalInteger>(val) );
    break;
  case ObjectBase::IMPLICIT_SIGNAL:
    setObject( new ImplicitSignal<UniversalInteger>(val) );
    break;
  case ObjectBase::VARIABLE:
    setObject( new Variable<UniversalInteger>(val) );
    break;
  case ObjectBase::SIGNAL_NETINFO:
    setObject( new SignalNetinfo );
    break;
  default:
    cerr << "ERROR: unknown object type" << endl;
    exit(-1);
  }

  ASSERT ( tInfo.getKind() == ENUM_INFO );
  typeInfo = (const EnumerationTypeInfo &) tInfo;
}


EnumerationType::EnumerationType(const EnumerationType& value,
				 const TypeInfo& tInfo): 
  ScalarType(value) {
  ASSERT ( tInfo.getKind() == ENUM_INFO );
  typeInfo = (const EnumerationTypeInfo &) tInfo;
}


EnumerationType::EnumerationType(bool alias, 
				 const EnumerationType& actual,
				 const TypeInfo& tInfo)
  : ScalarType(actual.getObject()->getKind(), alias) {
  setObject( actual.getObject() );
  ASSERT ( tInfo.getKind() == ENUM_INFO );
  typeInfo = (const EnumerationTypeInfo &) tInfo;
}

EnumerationType::~EnumerationType() {
}

EnumerationType&
EnumerationType::operator = (const EnumerationType& value) {
  this->ScalarType::operator=( dynamic_cast<const ScalarType &>(value) );
  typeInfo = dynamic_cast<const EnumerationTypeInfo &>(value.typeInfo);
  return *this;
}

Type 
EnumerationType::get_kind() const {
  return ENUMERATION_TYPE;
}

bool
operator == (const EnumerationType &lhs, const EnumerationType &rhs) {
  return (lhs.getIntValue() == rhs.getIntValue());
}

const EnumerationType &
EnumerationType::vhdlAnd( const EnumerationType &rhs ) const {
  return toBoolean( getIntValue() && rhs.getIntValue());
}

const EnumerationType &
EnumerationType::vhdlOr( const EnumerationType &rhs ) const {
  return toBoolean( getIntValue() || rhs.getIntValue() );
}

const EnumerationType &
EnumerationType::vhdlNot() const {
  return toBoolean( !getIntValue() );
}

const EnumerationType &
EnumerationType::vhdlNand( const EnumerationType &rhs ) const {
  return toBoolean( !(getIntValue() && rhs.getIntValue()) );
}

const EnumerationType &
EnumerationType::vhdlNor(const EnumerationType &rhs ) const {
  return toBoolean( !(getIntValue() || rhs.getIntValue()) );
}

const EnumerationType &
EnumerationType::vhdlXor( const EnumerationType &rhs ) const {
  bool retval = true;
  if( getIntValue() && rhs.getIntValue() ){
    retval = false;
  }
  else if ( (!getIntValue() && !rhs.getIntValue()) ){
    retval = false;
  }
  
  return toBoolean( retval );
}

const EnumerationType &
EnumerationType::vhdlXnor( const EnumerationType &rhs ) const {
  return vhdlXor(rhs).vhdlNot();
}

// The attributes of EnumerationType

EnumerationType 
EnumerationType::LEFT(const TypeInfo& tInfo) {
  ASSERT ( tInfo.getKind() == ENUM_INFO );
  return EnumerationType(ObjectBase::VARIABLE,
			 (int) ((const EnumerationTypeInfo &) tInfo).get_left(), tInfo);
}

EnumerationType
EnumerationType::RIGHT(const TypeInfo& tInfo) {
  ASSERT ( tInfo.getKind() == ENUM_INFO );
  const EnumerationTypeInfo &eInfo = (const EnumerationTypeInfo &) tInfo;
  return EnumerationType(ObjectBase::VARIABLE, (int) eInfo.get_right(), tInfo);
}

EnumerationType
EnumerationType::HIGH(const TypeInfo& tInfo){
  ASSERT ( tInfo.getKind() == ENUM_INFO );
  const EnumerationTypeInfo &rInfo = (const EnumerationTypeInfo &) tInfo;
  
  if (rInfo.get_left() > rInfo.get_right())
    return EnumerationType(ObjectBase::VARIABLE, (int)rInfo.get_left(), rInfo);
  else
    return EnumerationType(ObjectBase::VARIABLE, (int)rInfo.get_right(), rInfo);
}

EnumerationType
EnumerationType::LOW(const TypeInfo& tInfo){
  ASSERT ( tInfo.getKind() == ENUM_INFO );
  const EnumerationTypeInfo &rInfo = (const EnumerationTypeInfo &) tInfo;
  
  if (rInfo.get_left() > rInfo.get_right())
    return EnumerationType(ObjectBase::VARIABLE, (int)rInfo.get_right(),rInfo);
  else
    return EnumerationType(ObjectBase::VARIABLE, (int)rInfo.get_left(), rInfo);
}

EnumerationType
EnumerationType::ASCENDING(const TypeInfo& tInfo){
  ASSERT ( tInfo.getKind() == ENUM_INFO );
  const EnumerationTypeInfo &rInfo = (const EnumerationTypeInfo &) tInfo;
  
  if (rInfo.get_direction() == ArrayInfo::to) {
    return SAVANT_BOOLEAN_TRUE;
  }
  else {
    return SAVANT_BOOLEAN_FALSE;
  }
}

EnumerationType
EnumerationType::SUCC( const EnumerationType &x ){
  int val = x.getIntValue();
  if(val >= x.typeInfo.get_left() && val < x.typeInfo.get_right()){
    val = val + 1;
    return EnumerationType(ObjectBase::VARIABLE, val, x.typeInfo);
  }
  else {
    cerr << "The parameter to `SUCC is.operator== to 'HIGH\n";
    abort();
    return EnumerationType(ObjectBase::VARIABLE, val, x.typeInfo);
  }
}

EnumerationType
EnumerationType::PRED(const EnumerationType & x){
  int val = x.getIntValue();
  if(val > x.typeInfo.get_left() && val <= x.typeInfo.get_right()){
    val = val - 1;
    return EnumerationType(ObjectBase::VARIABLE, val, x.typeInfo);
  }
  else {
    cerr << "The parameter to `PRED is.operator== to 'LOW\n";
    abort();
    return EnumerationType(ObjectBase::VARIABLE, val, x.typeInfo);
  }
}

EnumerationType
EnumerationType::LEFTOF(const EnumerationType & x){
  int val = x.getIntValue();
  if(x.typeInfo.get_direction() == ArrayInfo::to && val > x.typeInfo.get_left() &&
     val <= x.typeInfo.get_right()){
    val = val - 1;
    return EnumerationType(ObjectBase::VARIABLE, val, x.typeInfo);
  }
  else if(x.typeInfo.get_direction() == ArrayInfo::downto && val <= x.typeInfo.get_left() &&
	  val > x.typeInfo.get_right()){
    val = val + 1;
    return EnumerationType(ObjectBase::VARIABLE, val, x.typeInfo);
  }
  else {
    cerr << "The parameter of `LEFTOF attribute not within typeInfo\n";
    abort();
    return EnumerationType(ObjectBase::VARIABLE, val, x.typeInfo);
  }
}

EnumerationType
EnumerationType::RIGHTOF(const EnumerationType & x){
  int val = x.getIntValue();
  if(x.typeInfo.get_direction() == ArrayInfo::to && val >= x.typeInfo.get_left() &&
     val < x.typeInfo.get_right()){
    val = val + 1;
    return EnumerationType(ObjectBase::VARIABLE, val, x.typeInfo);
  }
  else if(x.typeInfo.get_direction() == ArrayInfo::downto && val < x.typeInfo.get_left() &&
	  val >= x.typeInfo.get_right()){
    val = val - 1;
    return EnumerationType(ObjectBase::VARIABLE, val, x.typeInfo);
  }
  else {
    cerr << "The parameter of `RIGHTOF attribute not within typeInfo\n";
    abort();
    return EnumerationType(ObjectBase::VARIABLE, val, x.typeInfo);
  }
}

ArrayType 
EnumerationType::IMAGE(const EnumerationType & x) {
  int value = x.getIntValue();
  if(value < x.typeInfo.get_right() && value >= x.typeInfo.get_left()){
    char *ptr =  x.typeInfo.get_imageMap()[value];
    ArrayType image(ObjectBase::VARIABLE, SavantstringType_info, -1, ptr);
    return image;
  }
  else {
    cerr << "The parameter of `IMAGE attribute not withing typeInfo";
    abort();
  }
  return ArrayType(ObjectBase::VARIABLE, SavantstringType_info, -1, "");
}

EnumerationType
EnumerationType::VALUE(const ArrayType& x, const TypeInfo& tInfo){
  ASSERT ( tInfo.getKind() == ENUM_INFO );
  const EnumerationTypeInfo& eInfo = (const EnumerationTypeInfo &) tInfo;
  
  int length = x.get_number_of_elements();
  ostringstream value;
  for(int i = 0; (i < length); i++) {
    x.get_element(i).print(value);
  }
  UniversalInteger val = getIndex( value.str().c_str(), 
				   eInfo.get_imageMap(),
				   eInfo.get_no_of_elements() );

  if ( val.getIntValue() >= eInfo.get_left() && val.getIntValue() <= eInfo.get_right()){
    return EnumerationType(ObjectBase::VARIABLE, val, eInfo);
  }

  return EnumerationType(ObjectBase::VARIABLE, eInfo);
}

IntegerType
EnumerationType::POS(const EnumerationType & x, const TypeInfo& tInfo) {
  ASSERT ( tInfo.getKind() == ENUM_INFO );
  const EnumerationTypeInfo& eInfo = (const EnumerationTypeInfo &) tInfo;
  
  int val = x.getIntValue();
  val = val -  eInfo.get_left();

  return IntegerType(ObjectBase::VARIABLE, val, tInfo);
}

IntegerType
EnumerationType::POS(const EnumerationType &x ) {
  UniversalInteger val;
  val = x.getObject()->readVal();
  return IntegerType(ObjectBase::VARIABLE, val, x.getTypeInfo());
}


EnumerationType
EnumerationType::VAL(const IntegerType& x, const TypeInfo& tInfo){
  ASSERT ( tInfo.getKind() == ENUM_INFO );
  const EnumerationTypeInfo &eInfo = (const EnumerationTypeInfo &) tInfo;
  
  int value =  x.getIntValue();
  if (value >= eInfo.get_left() &&  value <= eInfo.get_right()){
    value = value + eInfo.get_left();
    return EnumerationType(ObjectBase::VARIABLE, value, tInfo);
  }
  else {
    cerr << "The parameter of `VAL attribute not withing typeInfo";
    abort();
    return EnumerationType(ObjectBase::VARIABLE, value, tInfo);
  }
}

const VHDLData&
EnumerationType::leftValue() {
  static UniversalInteger retValue(0);
  
  retValue = UniversalInteger(typeInfo.get_left());
  
  return retValue;
}

ostream&
operator<<(ostream &os, const EnumerationType &var) {
  if (var.getTypeInfo().get_imageMap() == SavantcharacterType_info.get_imageMap()) {
    // Okay. This is a character type so use the default stuff, please!!
    os << (char)var.getIntValue();
  }
  else {
    os << var.getTypeInfo().get_imageMap()[var.getIntValue()];
  }

  return os;
}

TypeInfo &
EnumerationType::getTypeInfo() const {
  return const_cast<EnumerationTypeInfo &>(typeInfo);
}
