# renice.tcl - a tcl dialog to the renice command
# version 1.0 - Feb 15, 1999
# Author: Samy Zafrany  (samy@netanya.ac.il // samy@math.technion.ac.il)
#         www.netanya.ac.il/~samy
#
namespace eval renice {
    variable renice

    # Define the renice array structure so that all variables are
    # defined for the callbacks in the radiobuttons and checkbuttons.
    array set renice {
       dialog ""
       user_procs_text ""
       all_procs_text ""
       users_text ""
       pgids_text ""
       priority 0
       pids:list {}
       users:list {}
       pgids:list {}
    }
}

# renice::create --
#
#   Method to create the dialog box for the renice command.
#
# Note
#
#   This dialog will not grab focus so the user can keep it open
#   and run other tkWorld dialogs.  Imagine how tedious it would be
#   if you had to close the dialog to run your command, then reopen
#   it to modify it.  By not making this a modal dialog, we do not
#   have to implement any last command saving characteristics since
#   the user can just leave the dialog open.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc renice::create { } {
    global tkWorld
    variable renice

    # Put the focus on the renice dialog if it is already open.
    if [winfo exists $renice(dialog)] {
        wm deiconify $renice(dialog)
	raise $renice(dialog)
	focus $renice(dialog)
	return
    } else {
	set renice(dialog) [dialog::create .renice Renice]
    }

    ############################### FIRST TAB

    # The first tab deals with the standard renice command line options.
    set tab1 [tabnotebook::page [dialog::interior $renice(dialog)] "Nice Value"]

    # Now build the individual radio buttons for the -p (priority) option
    #
    set f1 [frame $tab1.f1 -class TabnotebookFrame]
    set f2 [frame $tab1.f2 -class TabnotebookFrame]

    scale $f1.s -label "Nice value" -from -20 -to +20 -length 5i \
        -orient horizontal \
        -tickinterval 5 \
        -variable renice::renice(priority)
    pack $f1.s -side top

    message $f2.m -text "Use the scale to set a scheduling priority value.\
A negative value means a higher scheduling priority\
(only root can decrease values). Positive values means lower\
scheduling priority." \
        -aspect 400
    pack $f2.m -side top -fill x

    # Build the first tab.
    pack $f1 $f2 \
        -side top \
        -fill x   \
        -padx 5   \
        -pady 5   \
        -ipadx 5  \
        -ipady 5

    ############################### SECOND TAB

    # The second tab has to do with the list of user processes to renice.
    # Only processes that belong to the current user (who runs tkworld)
    # are presented.
    # We use a text widget for presenting this list of processes.
    # The user can select any number of them.
    #
    set tab2 [tabnotebook::page [dialog::interior .renice] "User Processes"]

    set f2 [frame $tab2.f2 -class TabnotebookFrame]
    set t $f2.user_procs_text
    set renice(user_procs_text) $t

    scrollbar $f2.sb -command "$t yview"
    pack $f2.sb -side right -fill y

    text $t -background white \
            -cursor left_ptr  \
            -tabs {2c 4c} \
            -yscrollcommand "$f2.sb set" \
            -height 20 \
            -width 61
    pack $t -side left -expand yes -fill both
    pack $f2 \
       -side  top \
       -fill  x   \
       -padx  5   \
       -pady  5   \
       -ipadx 5   \
       -ipady 5

    ############################### THIRD TAB

    # The third tab has to do with the list of all current processes
    # This is useful only for the root user, since normal users cannot
    # change scheduling priority of processes of other users.
    # We use a text widget for presenting the available list of processes.
    # The (root) user can select any number of them.
    #
    set tab3 [tabnotebook::page [dialog::interior .renice] "All Processes"]

    set f3 [frame $tab3.f3 -class TabnotebookFrame]
    set t $f3.all_procs_text
    set renice(all_procs_text) $t

    scrollbar $f3.sb -command "$t yview"
    pack $f3.sb -side right -fill y

    text $t -background white \
            -cursor left_ptr  \
            -tabs {2c 4c 6.5c} \
            -yscrollcommand "$f3.sb set" \
            -height 20 \
            -width 61
    pack $t -side left -expand yes -fill both
    pack $f3 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    ############################### FOURTH TAB

    # The fourth tab has to do with the list users that are currently
    # logged on (or own at least one process).
    # Only root can assign a new scheduling priority to other users.
    #
    set tab4 [tabnotebook::page [dialog::interior .renice] "Users"]

    set f4 [frame $tab4.f4 -class TabnotebookFrame]
    set t $f4.users_text
    set renice(users_text) $t

    scrollbar $f4.sb -command "$t yview"
    pack $f4.sb -side right -fill y

    text $t -background white \
            -cursor left_ptr  \
            -spacing1 5 \
            -spacing3 5 \
            -tabs {3c 6c 9c 12c} \
            -yscrollcommand "$f4.sb set" \
            -height 12 \
            -width 61
    pack $t -side left -expand yes -fill both
    pack $f4 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    ############################### FIFTH TAB

    # The fifth tab has to do with the list of process group ids.
    # Each pgid consists of a group of processes with the same process
    # group id and owner.
    # Only root can assign a new scheduling priority to other users pgid's.
    #
    set tab5 [tabnotebook::page [dialog::interior .renice] "Process Group Id's"]

    set f5 [frame $tab5.f5 -class TabnotebookFrame]
    set t $f5.pgids_text
    set renice(pgids_text) $t

    scrollbar $f5.sb -command "$t yview"
    pack $f5.sb -side right -fill y

    text $t -background white \
            -cursor left_ptr  \
            -tabs {2c 4.5c 6.5c 8c} \
            -yscrollcommand "$f5.sb set" \
            -height 20 \
            -width 61
    pack $t -side left -expand yes -fill both

    pack $f5 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    ####### Bindings:
    #
    foreach x {
        user_procs_text
        all_procs_text
        users_text
        pgids_text
    } {
       set t $renice($x)
       # First, we disable all default text bindings
       #
       foreach event [bind Text] {
          bind $t $event {break}
       }
       # We keep the following standard bindings:
       bind $t <Enter> {focus %W}
       bind $t <Button-1> {renice::select %W}
       bind $t <Next>  "$t yview scroll  1 pages"
       bind $t <space> "$t yview scroll  1 pages"
       bind $t <Prior> "$t yview scroll -1 pages"
       bind $t <Up>    "$t yview scroll -1 units"
       bind $t <Down>  "$t yview scroll  1 units"
    }

    # We need to keep a translation table between uid to login name
    # This is almost impossible by using ps.
    # I know this is possibly a big waste of memory and time, but I could
    # not find any other way without compromising robustness.
    # We have to open /etc/passwd:
    #
    set file_desc [::open /etc/passwd r]
    while {[gets $file_desc line]>=0} {
      set name [lindex [split $line :] 0]
      set uid [lindex [split $line :] 2]
      set renice(logname,$uid) $name
    }
    ::close $file_desc

    # Finally, we fill those text widgets with their text
    # The info is updated every 30 seconds:
    #
    renice::update
}

############################### PROCEDURES

# Extracting Info from the ps commands is quite hard and probably
# platform dependent (tested on Linux only). The following code may
# look quite messy, quirky, unreliable, and too dependent on the version
# of the ps command.
# But if you got better ideas, I will be happy to hear about them.
# If only the designers of the ps command knew about Tcl, our world would
# look much better .....

proc renice::niceValues {} {
   variable renice

   foreach name [array names renice] {
      if [regexp {^nice,[0-9]+} $name] {
        unset renice($name)
      }
   }

   foreach line [split [exec ps alxw] \n] {
      if [regexp {[ ]*FLAGS[ ]+UID[ ]+PID} $line] {
        set cut [string first RSS $line]
        continue
      }
      if [regexp {ps alxw[ ]*$} $line] {continue}
      set short_line [string range $line 0 $cut]
      set pid [lindex $short_line 2]
      set renice(nice,$pid) [lindex $short_line 5]
   }
}

proc renice::refreshUserProcs {} {
   variable renice

   set t $renice(user_procs_text)
   set yv [lindex [$t yview] 0]
   $t configure -state normal
   $t delete 1.0 end
   $t insert end "PID\tNICE\tCOMMAND\n" title

   foreach line [split [exec ps uxw] \n] {
      if [regexp {[ ]*USER[ ]+PID[ ]+%CPU} $line] {
        set cmd_start [string first COMMAND $line]
        set cut [string first SIZE $line]
        continue
      }
      if [regexp {ps uxw[ ]*$} $line] {continue}
      set short_line [string range $line 0 $cut]
      set pid [lindex $short_line 1]
      set cmd [string range $line $cmd_start [expr $cmd_start+40]]
      if [info exists renice(nice,$pid)] {
          set nice $renice(nice,$pid)
      } else {
          set nice "??"
      }
      $t insert end "$pid\t$nice\t$cmd\n" [list body "pid:$pid"]
   }

   $t tag configure title -font {Helvetica 12 bold}
   $t tag configure title -underline 1

   $t configure -state disabled
   $t yview moveto $yv
}

proc renice::refreshAllProcs {} {
   variable renice

   set t $renice(all_procs_text)
   set yv [lindex [$t yview] 0]
   $t configure -state normal
   $t delete 1.0 end
   $t insert end "PID\tNICE\tOWNER\tCOMMAND\n" title

   set all_pids {}
   foreach line [split [exec ps auxw] \n] {
      if [regexp {[ ]*USER[ ]+PID[ ]+%CPU} $line] {
        set cut [string first SIZE $line]
        set cmd_start [string first COMMAND $line]
        continue
      }
      if [regexp {ps auxw[ ]*$} $line] {continue}
      set short_line [string range $line 0 $cut]
      set owner [lindex $short_line 0]
      set pid [lindex $short_line 1]
      set cmd [string range $line $cmd_start [expr $cmd_start+32]]
      if [info exists renice(nice,$pid)] {
          set nice $renice(nice,$pid)
      } else {
          set nice "??"
      }
      $t insert end "$pid\t$nice\t$owner\t$cmd\n" \
         [list body "pid:$pid"]
      lappend all_pids $pid
   }

   $t tag configure title -font {Helvetica 12 bold}
   $t tag configure title -underline 1

   $t configure -state disabled
   $t yview moveto $yv
   
   # If some of the pid's in renice(pids:list) died, we need to
   # remove them from the list:
   foreach p $renice(pids:list) {
      if {[lsearch $all_pids $p]==-1} {
        set i [lsearch $renice(pids:list) $p]
        set renice(pids:list) [lreplace $renice(pids:list) $i $i]
      }
   }
}

proc renice::refreshUsers {} {
   variable renice

   set t $renice(users_text)
   set yv [lindex [$t yview] 0]
   $t configure -state normal
   $t delete 1.0 end
   $t insert end "\t\tUSERS\n" title

   set users {}
   foreach line [split [exec ps au] \n] {
      if [regexp {[ ]*USER[ ]+PID[ ]+%CPU} $line] {
        set cut [string first SIZE $line]
        continue
      }
      set short_line [string range $line 0 $cut]
      set u [lindex $short_line 0]
      if {[lsearch $users $u]>=0} {continue}
      lappend users $u
   }

   set users [lsort $users]

   set n 0
   foreach u $users {
      if {[expr $n%5]==4} {
           $t insert end "$u\n" [list body "user:$u"]
      } else {
           $t insert end "$u\t" [list body "user:$u"]
      }
      incr n
   }

   $t tag configure title -font {Helvetica 12 bold}
   $t tag configure title -underline 1

   $t configure -state disabled
   $t yview moveto $yv
}

proc renice::refreshPgids {} {
   variable renice

   set t $renice(pgids_text)
   set yv [lindex [$t yview] 0]
   $t configure -state normal
   $t delete 1.0 end
   $t insert end "PGID\tOWNER\tPID\tNICE\tCOMMANDS\n" title

   set all_pgids {}
   catch {unset owner}
   foreach line [split [exec ps jaxw] \n] {
      if [regexp {[ ]*PPID[ ]+PID} $line] {
        set cmd_start [string first COMMAND $line]
        set uid_start [expr [string first STAT $line]+4]
        set uid_end [expr [string first TIME $line]+4]
        set cut [string first TTY $line]
        continue
      }
      if [regexp {ps jaxw[ ]*$} $line] {continue}
      set short_line [string range $line 0 $cut]
      set pid [lindex $short_line 1]
      set pgid [lindex $short_line 2]
      set cmd [string range $line $cmd_start [expr $cmd_start+26]]
      set uid_line [string range $line $uid_start $uid_end]
      set uid [lindex $uid_line 0]
      if [info exists renice(nice,$pid)] {
          set nice $renice(nice,$pid)
      } else {
          set nice "??"
      }
      if {[lsearch $all_pgids $pgid]==-1} {
         lappend all_pgids $pgid
         set owner($pgid) $renice(logname,$uid)
         $t insert end "$pgid\t$owner($pgid)\t$pid\t$nice\t$cmd\n" \
             [list "pgid:$pgid" leader]
         continue
      }
      $t insert end "\t$owner($pgid)\t$pid\t$nice\t$cmd\n" "pgid:$pgid"
   }

   $t tag configure title -font {Helvetica 12 bold}
   $t tag configure title -underline 1
   $t tag configure leader -spacing1 6

   $t configure -state disabled
   $t yview moveto $yv
   
   foreach p $renice(pgids:list) {
      if {[lsearch $all_pgids $p]==-1} {
        set i [lsearch $renice(pgids:list) $p]
        set renice(pgids:list) [lreplace $renice(pids:list) $i $i]
      }
   }
}

proc renice::select {w} {
   variable renice
   set t1 $renice(user_procs_text)
   set t2 $renice(all_procs_text)
   set t3 $renice(users_text)
   set t4 $renice(pgids_text)
   set tags [$w tag names current]
   set i [lsearch -glob $tags *:*]
   if {$i==-1} {return}
   set tg [lindex $tags $i]
   set type [lindex [split $tg :] 0]
   set id [lindex [split $tg :] 1]
   switch $type {
      pid {
        set i [lsearch $renice(pids:list) $id]
        if {$i >= 0} {
          $t1 tag configure $tg -background [$t1 cget -background]
          $t2 tag configure $tg -background [$t2 cget -background]
          set renice(pids:list) [lreplace $renice(pids:list) $i $i]
          return
        }
        $t1 tag configure $tg -background grey75
        $t2 tag configure $tg -background grey75
        lappend renice(pids:list) $id
      }
      user {
        set i [lsearch $renice(users:list) $id]
        if {$i >= 0} {
          $t3 tag configure $tg -background [$t3 cget -background]
          set renice(users:list) [lreplace $renice(users:list) $i $i]
          return
        }
        $t3 tag configure $tg -background grey75
        lappend renice(users:list) $id
      }
      pgid {
        set i [lsearch $renice(pgids:list) $id]
        if {$i >= 0} {
          $t4 tag configure $tg -background [$t4 cget -background]
          set renice(pgids:list) [lreplace $renice(pgids:list) $i $i]
          return
        }
        $t4 tag configure $tg -background grey75
        lappend renice(pgids:list) $id
      }
   }
}

# renice::ok --
#
#   Method to insert the command the user has created into the CC
#   as a Tcl list.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc renice::ok { } {
   global tkWorld
   variable renice

   # Build the full command line
   #
   set cmd "renice $renice(priority)"

   if [llength $renice(pids:list)] {
         set pids [lsort -integer $renice(pids:list)]
         append cmd " "
         eval lappend cmd -p $pids
   }

   if [llength $renice(users:list)] {
         set users [lsort $renice(users:list)]
         append cmd " "
         eval lappend cmd -u $users
   }

   if [llength $renice(pgids:list)] {
         set pgids [lsort -integer $renice(pgids:list)]
         append cmd " "
         eval lappend cmd -g $pgids
   }

   # Insert the Tcl command list in the Command Center with the
   # proper formatting of a space between each argument on the
   # command line.  If there are no options given by the user,
   # then don't display it in the CC.

   $tkWorld(cmd_center) insert insert $cmd

   # Activate the buttons in the toolbar for the command center.
   toolbar::group_state cmd_center active
   toolbar::button_state $toolbar::toolbar(stop) disabled
}

# renice::reset --
#
#   Method to reset the radio and checkbuttons in the dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc renice::reset { } {
   variable renice

   set renice(priority) 0
   renice::clear
}

# renice::clear --
#
#   Method to clear entry items of their text and reset the
#   background and foreground properties.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc renice::clear { } {
   variable renice

   set renice(pids:list) {}
   set renice(users:list) {}
   set renice(pgids:list) {}
   set t1 $renice(user_procs_text)
   set t2 $renice(all_procs_text)
   set t3 $renice(users_text)
   set t4 $renice(pgids_text)
   foreach t [list $t1 $t2 $t3 $t4] {
      foreach tg [$t tag names] {
         $t tag delete $tg
      }
   }
   renice::niceValues
   renice::refreshUserProcs
   renice::refreshAllProcs
   renice::refreshUsers
   renice::refreshPgids
}

proc renice::update {} {
   variable renice

   if ![winfo exists $renice(dialog)] {return}
   renice::niceValues
   renice::refreshUserProcs
   renice::refreshAllProcs
   renice::refreshUsers
   renice::refreshPgids
   after 30000 {renice::update}
}

# renice::help --
#
#   Method to invoke the Renice Command Help.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc renice::help { } {
   global tkWorld

   help::create "help/renice.html" "Renice Command Help"
}

# renice::close --
#
#   Close the dialog up.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc renice::close { } {
   variable renice
   
   balloonhelp::cancel
   destroy $renice(dialog)
}
