# $Id: chats.tcl,v 1.125 2004/07/10 20:05:04 aleksey Exp $

package require textutil

option add *Chat.theyforeground        red3   widgetDefault
option add *Chat.meforeground          blue   widgetDefault
option add *Chat.serverlabelforeground green  widgetDefault
option add *Chat.serverforeground      violet widgetDefault
option add *Chat.infoforeground        blue   widgetDefault
option add *Chat.errforeground         red    widgetDefault
option add *urlforeground              blue   widgetDefault
option add *urlactiveforeground        red    widgetDefault
option add *Chat.inputheight           3      widgetDefault


namespace eval chat {
    array set opened {}
    set urlid 0
    set enrichid 0
    custom::defgroup Chat [::msgcat::mc "Chat options."] -group Tkabber
    custom::defvar options(smart_scroll) 0 \
	    [::msgcat::mc "Enable chat window autoscroll only when last message is shown."] -type boolean -group Chat
    custom::defvar options(stop_scroll) 0 [::msgcat::mc "Stop chat window autoscroll."] -type boolean -group Chat
    custom::defvar options(emphasize) 1 \
	[::msgcat::mc "Enable messages emphasize."] \
	-type boolean -group Chat \
	-command chat::switch_emphasize
    custom::defvar options(display_status_description) 1 \
	[::msgcat::mc "Display description of user status in chat windows."] \
	-type boolean -group Chat
    custom::defvar options(default_message_type) chat \
	[::msgcat::mc "Default message type (if not specified explicitly)."] \
	-type radio -group Chat \
	-values [list normal [::msgcat::mc "Normal"] \
		      chat [string trim [::msgcat::mc "Chat "]]]
    set url_regexp {^(http://|https://|www\.|ftp://|ftp\.)}
}

set chat_width 50
set chat_height 18

plugins::load [file join plugins chat]

proc chat::get_nick {jid type} {
    variable chats

    set nick $jid
    switch -- $type {
	chat {
	    set group [node_and_server_from_jid $jid]
	    foreach connid [jlib::connections] {
		set chatid [chatid $connid $group]
		if {[is_groupchat $chatid]} {
		    return [resource_from_jid $jid]
		} else {
		    set nick [roster::itemconfig $connid \
				  [roster::find_jid $connid $jid] -name]
		    if {$nick != ""} {
			return $nick
		    }
		}
	    }
	    set nick [node_from_jid $jid]
	}
	groupchat {
	    set group [node_and_server_from_jid $jid]
	    set nick [resource_from_jid $jid]
	}
	default {
	}
    }
    return $nick
}

proc chat::winid {chatid} {
    set connid [get_connid $chatid]
    set jid [get_jid $chatid]
    set tag [jid_to_tag $jid]
    return .chat_${connid}_$tag
}

proc chat::chatid {connid jid} {
    return [list $connid $jid]
}

proc chat::get_connid {chatid} {
    return [lindex $chatid 0]
}

proc chat::get_jid {chatid} {
    return [lindex $chatid 1]
}

proc client:message {connid from id type subject body err thread priority x} {
    debugmsg chat "MESSAGE: $connid; $from; $id $type; $subject; $body; $err;\
$thread; $priority; $x"

    if {![cequal [info commands ::ssj::encrypted:input] ""]} {
	set xs {}

	foreach xe $x {
	    jlib::wrapper:splitxml $xe tag vars isempty chdata children

	    if {![cequal [jlib::wrapper:getattr $vars xmlns] \
			 jabber:x:encrypted]} {
		lappend xs $xe
		continue
	    } elseif {[cequal $chdata ""]} {
# in case the sender didn't check the exit code from gpg...
	    } elseif {[catch { ssj::encrypted:input $from $chdata } body]} {
		set body [format [::msgcat::mc ">>> Unable to decipher data: %s <<<"] $body]
	    } else {
		lappend xs $xe
	    }
	}

	set x $xs
    }

    chat::process_message \
	$connid $from $id $type $subject $body $err $thread $priority $x
}

proc chat::process_message {connid from id type subject body err thread priority x} {
    variable options

    if {$type == ""} {
	set type $options(default_message_type)
    }

    set from [tolower_node_and_domain $from]
    hook::run examine_message_hook $from $id $type $subject $body $err \
				   $thread $priority $x

    hook::run process_message_hook \
	$connid $from $id $type $subject $body $err $thread $priority $x
}

proc chat::process_message_fallback {connid from id type subject body err thread priority x} {
    global font
    variable opened
    variable chats

    set chatid [chatid $connid $from]
    hook::run normalize_chat_id_hook chatid $type


    switch -- $type {
	normal {
	    message::show_dialog $connid $from $type $subject $body $thread $priority $x
	    return
	}
	headline {
	    message::show_headline $from $type $subject $body $thread $priority $x
	    return
	}
	chat {
	    if {$thread != ""} {
		set chats(thread,$chatid) $thread
	    }
	    set chats(id,$chatid) $id
	}
	groupchat {
	    set chatid [chatid $connid [node_and_server_from_jid $from]]

	    if {![is_groupchat $chatid]} return

	    if {![cequal $subject ""]} {
		set_subject $chatid $subject
		if {[cequal $body ""]} {
		    set body [format [::msgcat::mc "/me has changed the subject to: %s"] $subject]
		}
	    }
	}
	error {
	    if {![cequal $subject ""]} {
		set body "subject: "
		restore_subject $chatid
	    } else {
		set body ""
	    }
	    append body [error_to_string $err]
	}
	default {
	    debugmsg chat "MESSAGE: UNSUPPORTED message type '$type'"
	}
    }

    #chat::open_window $chatid $type

    chat::add_message $chatid $from $type $body $x
    if {[llength $x] > 0} {
        message::show_dialog $connid $from $type $subject $body $thread $priority $x 0
    }
}

hook::add process_message_hook \
    [namespace current]::chat::process_message_fallback 99


proc chat::window_titles {chatid} {
    variable chats

    set connid [get_connid $chatid]
    set jid [get_jid $chatid]
    set chatname [roster::itemconfig $connid \
		      [roster::find_jid $connid $jid] -name]
    if {$chatname != ""} {
	set tabtitlename $chatname
	set titlename $chatname
    } else {
	set titlename $jid
	if {[is_groupchat $chatid]} {
	    set tabtitlename [node_from_jid $jid]
	} else {
	    set tabtitlename [get_nick $jid chat]
	}
	if {$tabtitlename == ""} {
	    set tabtitlename $titlename
	}
    }
    return [list $tabtitlename [format [::msgcat::mc "Chat with %s"] $titlename]]
}

proc chat::reconnect_groupchats {connid} {
    variable chats
    global grouproster
	
    if {![info exists chats(opened)]} return

    foreach chatid $chats(opened) {
	if {[cequal [get_connid $chatid] $connid] && \
		[is_groupchat $chatid]} {
	    set grouproster(users,$chatid) {}

	    join_group [get_jid $chatid] \
		-nick [get_our_groupchat_nick $chatid] \
		-connection $connid
	}
    }
}

hook::add connected_hook [namespace current]::chat::reconnect_groupchats 99

proc chat::disconnect_groupchats {connid} {
    variable chats

    if {![info exists chats(opened)]} return

    foreach chatid $chats(opened) {
	if {[cequal $connid {}] || [cequal [get_connid $chatid] $connid]} {
	    set chatw [chat_win $chatid]
	    if {[winfo exists $chatw]} {
		add_message $chatid [get_jid $chatid] error \
		    [::msgcat::mc "Disconnected"] {}
	    }
	}
    }
}

hook::add disconnected_hook [namespace current]::chat::disconnect_groupchats 99

proc chat::open_window {chatid type} {
    global font font_bold font_italic font_bold_italic
    global chat_width chat_height
    variable opened
    variable chats
    variable chat_id
    variable options

    set connid [get_connid $chatid]
    set jid [get_jid $chatid]
    set jid [tolower_node_and_domain $jid]
    set chatid [chatid $connid $jid]
    
    set cw [winid $chatid]

    if {[winfo exists $cw]} {
	if {!$::usetabbar && \
		[info exists ::raise_on_activity] && $::raise_on_activity} {
	    if {$type == "chat"} {
		wm deiconify $cw
	    }
	    raise $cw
	}
	return
    }

    hook::run open_chat_pre_hook $chatid $type

    set chats(type,$chatid) $type
    set chats(subject,$chatid) ""
    debugmsg chat "OURJID: [our_jid $chatid]"
    lappend chats(opened) $chatid
    
    set opened($chatid) $cw
    set chat_id($cw) $chatid
    #debugmsg chat $cw
    #debugmsg chat [array names opened_chats]
    #debugmsg chat $opened_chats($chatid)

    lassign [chat::window_titles $chatid] chats(tabtitlename,$chatid) \
	chats(titlename,$chatid)

    add_win $cw -title $chats(titlename,$chatid) \
	-tabtitle $chats(tabtitlename,$chatid) \
	-class Chat -type $type \
	-raisecmd "focus [list $cw.input]
		   tab_set_updated [list $cw]
                   hook::run raise_chat_tab_hook [list $cw] [list $chatid]"

    frame $cw.status
    pack $cw.status -side top -fill x
    if {[cequal $type chat]} {
	set status [get_user_status $connid $jid]
	label $cw.status.icon -image $ifacetk::roster::rostericon(user,$status)
	pack $cw.status.icon -side left
	if {$options(display_status_description)} {
	    Label $cw.status.desc -text "($status)" -font $font \
		-helptext [get_user_status_desc $connid $jid]
	    pack $cw.status.desc -side left
	}
    }
    if {![cequal [info commands ::ssj::encrypt:toggleP] ""]} {
        if {[cequal [set sjid [roster::find_jid $connid $jid]] ""]} {
            set sjid $chatid
        }
        Button $cw.status.encrypted -relief flat \
            -image [message::encrypted:icon $sjid] \
            -helptype balloon \
            -helptext [::msgcat::mc "Toggle encryption"] \
            -command [list ssj::encrypt:toggleP $sjid]
        ssj::encrypted:trace "
            $cw.status.encrypted configure -image \[message::encrypted:icon $sjid\]
        " $sjid
        pack $cw.status.encrypted -side left
    }
    if {[cequal $type chat]} {
	menubutton $cw.status.mb -text $jid -font $font \
	    -menu $cw.status.mb.menu
	create_user_menu $cw.status.mb.menu $chatid
	pack $cw.status.mb -side left
    } else {
        menubutton $cw.status.mb -text [::msgcat::mc "Subject:"] \
	    -direction below \
	    -menu $cw.status.mb.menu -relief flat
	create_conference_menu $cw.status.mb.menu $chatid
	pack $cw.status.mb -side left

	entry $cw.status.subject -font $font
	pack $cw.status.subject -side left -fill x -expand yes

	bind $cw.status.subject <Return> \
	    [list chat::change_subject [double% $chatid]]
    }
    foreach tag [bind Menubutton] {
        if {[string first 1 $tag] >= 0} {
	    regsub -all 1 $tag 3 new
	    bind $cw.status.mb $new [bind Menubutton $tag]
	}
    }

    PanedWin $cw.pw0 -side right -pad 0 -width 8
    pack $cw.pw0 -fill both -expand yes

    set upw [$cw.pw0 add -weight 1 -minsize 0]
    set dow [$cw.pw0 add -weight 0 -minsize 0]

    set isw [ScrolledWindow $cw.isw -scrollbar vertical]
    pack $cw.isw -fill both -expand yes -side bottom -in $dow
    textUndoable $cw.input -width $chat_width \
	-height [option get $cw inputheight Chat] -font $font -wrap word
    $isw setwidget $cw.input
    [winfo parent $dow] configure -height [winfo reqheight $cw.input]
    set chats(inputwin,$chatid) $cw.input

    if {[cequal $type groupchat]} {
	PanedWin $cw.pw -side bottom -pad 2 -width 8
	pack $cw.pw -fill both -expand yes -in $upw

	set cf [$cw.pw add -weight 1 -minsize 0]

	set uw [$cw.pw add -weight 0 -minsize 0]
	set chats(userswin,$chatid) $uw.users

	set rosterwidth [option get . chatRosterWidth [winfo class .]]
	if {$rosterwidth == ""} {
	    set rosterwidth [winfo pixels . 3c]
	}

	ifacetk::roster::create $uw.users -width $rosterwidth \
	    -popup ifacetk::roster::groupchat_popup_menu
	pack $uw.users -fill both -side right -expand yes
	[winfo parent $uw] configure -width $rosterwidth
	
	global grouproster
	
	set grouproster(users,$chatid) {}

	#roster::addline $uw.users group Users users

	client:presence $connid $jid "" "" {}

	set pack_in $cf
    } else {
	set cf $cw
	set pack_in $upw
    }

    set chats(chatwin,$chatid) $cf.chat

    set csw [ScrolledWindow $cf.csw -scrollbar vertical -auto none]
    pack $csw -expand yes -fill both -side left -in $pack_in
    text $cf.chat -width $chat_width -height $chat_height -font $font -wrap word
    $csw setwidget $cf.chat

    $cf.chat tag configure they \
	-foreground [option get $cw theyforeground Chat]
    $cf.chat tag configure me \
	-foreground [option get $cw meforeground Chat]
    $cf.chat tag configure highlight \
	-foreground [option get $cw meforeground Chat]
    $cf.chat tag configure server_lab \
	-foreground [option get $cw serverlabelforeground Chat]
    $cf.chat tag configure server \
	-foreground [option get $cw serverforeground Chat]
    $cf.chat tag configure info \
	-foreground [option get $cw infoforeground Chat]
    $cf.chat tag configure err \
	-foreground [option get $cw errforeground Chat]

    $cf.chat tag configure bold -font $font_bold
    $cf.chat tag configure italic -font $font_italic
    $cf.chat tag configure bold_italic -font $font_bold_italic
    $cf.chat tag configure underlined -underline 1
    $cf.chat tag configure emphasized -elide 0
    $cf.chat tag configure nonemphasized -elide 1


    $cf.chat configure -state disabled
    focus $cw.input

    bind $cw.input <Shift-Key-Return> { }

    bind $cw.input <Key-Return> [double% "
	chat::send_message [list $cw] [list $chatid] [list $type]
	break"]

    bind $cw.input <Control-z> { %W undo }
    bind $cw.input <Control-Z> { %W redo }

    regsub -all %W [bind Text <Prior>] $cf.chat prior_binding
    regsub -all %W [bind Text <Next>] $cf.chat next_binding
    bind $cw.input <Meta-Prior> $prior_binding
    bind $cw.input <Meta-Next> $next_binding
    bind $cw.input <Alt-Prior> $prior_binding
    bind $cw.input <Alt-Next> $next_binding

    bind $cw <Destroy> [list chat::close_window [double% $chatid]]

    hook::run open_chat_post_hook $chatid $type
}


proc chat::close_window {chatid} {
    variable opened
    variable chats
    variable chat_id

    if {![info exists opened($chatid)]} return

    unset opened($chatid)

    set connid [get_connid $chatid]
    set jid [get_jid $chatid]

    set cw [winid $chatid]
    unset chat_id($cw)

    set idx [lsearch -exact $chats(opened) $chatid]
    set chats(opened) [lreplace $chats(opened) $idx $idx]
    debugmsg chat $chats(opened)

    if {[is_groupchat $chatid]} {
	jlib::send_presence \
	    -to $jid/[get_our_groupchat_nick $chatid] \
	    -connection $connid \
	    -type unavailable
	client:presence $connid $jid unavailable "" {}
	foreach jid [get_jids_of_user $connid $jid] {
	    client:presence $connid $jid unavailable "" {}
	}
    }

    destroy $cw
    hook::run close_chat_post_hook $chatid
}

proc chat::opened {} {
    variable chats
    if {[info exists chats(opened)]} {
	return $chats(opened)
    } else {
	return {}
    }
}

proc chat::users_win {chatid} {
    variable chats
    return $chats(userswin,$chatid)
}

proc chat::chat_win {chatid} {
    variable chats
    return $chats(chatwin,$chatid)
}

proc chat::input_win {chatid} {
    variable chats
    return $chats(inputwin,$chatid)
}

proc chat::is_groupchat {chatid} {
    variable chats
    if {[info exists chats(type,$chatid)]} {
	return [cequal $chats(type,$chatid) groupchat]
    } else {
	return 0
    }
}

proc chat::create_user_menu {path chatid} {
    set m [menu $path -tearoff 0]

    set connid [get_connid $chatid]
    set jid [get_jid $chatid]
    $m add command -label [::msgcat::mc "Invite to conference..."] \
	-command [list chat::invite_dialog $jid 0 -connection $connid]
    $m add command -label [::msgcat::mc "Add user..."] \
	-command [list message::send_subscribe_dialog \
		      [node_and_server_from_jid $jid] -connection $connid]
    hook::run chat_create_user_menu_hook $path $chatid
    $m add separator
    $m add command -label [::msgcat::mc "Send users..."] \
	-command [list ifacetk::roster::send_users_dialog $jid -connection $connid]
    ft::create_menu $m $jid -connection $connid
    $m add separator
    $m add command -label [::msgcat::mc "Show info..."] \
	-command [list userinfo::open $jid -connection $connid]
    $m add command -label [::msgcat::mc "Show history..."] \
	-command [list logger::show_log $jid]
    return $path
}

proc chat::create_conference_menu {path chatid} {
    set m [menu $path -tearoff 0]
    set connid [get_connid $chatid]
    set jid [get_jid $chatid]
    # TODO: connid
    $m add command -label [::msgcat::mc "Invite users..."] \
	-command [list chat::invite_dialog2 $jid]
    $m add command -label [::msgcat::mc "Show history..."] \
	-command [list logger::show_log $jid]
    add_conference_custom_presence_menu $m $chatid
    after idle [list hook::run chat_create_conference_menu_hook $path $chatid]
    return $path
}

proc chat::add_conference_custom_presence_menu {m chatid} {
    set mm [menu $m.custom_presence -tearoff 0]

    set jid [get_jid $chatid]
    set connid [get_connid $chatid]

    $mm add command -label [::msgcat::mc "Online"] \
	-command [list send_custom_presence $jid available \
			   -type group -connection $connid]
    $mm add command -label [::msgcat::mc "Free to chat"] \
	-command [list send_custom_presence $jid chat \
			   -type group -connection $connid]
    $mm add command -label [::msgcat::mc "Away"] \
	-command [list send_custom_presence $jid away \
			   -type group -connection $connid]
    $mm add command -label [::msgcat::mc "Extended away"] \
	-command [list send_custom_presence $jid xa \
			   -type group -connection $connid]
    $mm add command -label [::msgcat::mc "Do not disturb"] \
	-command [list send_custom_presence $jid dnd \
			   -type group -connection $connid]
    #$mm add command -label [::msgcat::mc "Offline"] \
    #    -command [list send_custom_presence $jid unavailable \
    #   		   -type group -connection $connid]

    $m add cascad -label [::msgcat::mc "Send custom presence"] -menu $mm
}

proc chat::add_chat_win_popup_menu {m chatwin X Y x y} {
    set tags [$chatwin tag names "@$x,$y"]
    set idx [lsearch -glob $tags url*]
    if {$idx >= 0} {
	$m add command -label [::msgcat::mc "Copy URL to clipboard"] \
	    -command [list [namespace current]::copy_url $chatwin \
			   [lindex $tags $idx]]
    }
}

hook::add chat_win_popup_menu_hook \
    [namespace current]::chat::add_chat_win_popup_menu 10

proc chat::copy_url {chatwin tag} {
    regsub -all %% [$chatwin tag bind $tag <1>] % res
    # assuming "browseurl url"
    clipboard clear -displayof $chatwin
    clipboard append -displayof $chatwin [lrange $res 1 end]
}

proc chat::copy_url2 {chatwin ign url} {
}

proc chat::send_message {cw chatid type} {
    set iw $cw.input

    set connid [get_connid $chatid]

    if {[catch { set user [jlib::connection_user $connid] }]} {
	set user ""
    }

    set body [$iw get 1.0 "end -1 chars"]

    debugmsg chat "SEND_MESSAGE:\
 [list $chatid] [list $user] [list $body] [list $type]"

    set chatw [chat_win $chatid]
    $chatw mark set start_message "end -1 chars"
    $chatw mark gravity start_message left

    hook::run chat_send_message_hook $chatid $user $body $type

    $iw delete 1.0 end
}

proc chat::add_message {chatid from type body x} {
    variable chats
    variable opened
    variable options

    if {[info exists opened($chatid)]} {
	set chatw [chat_win $chatid]

	if {[lindex [$chatw yview] 1] == 1} {
	    set scroll 1
	} else {
	    set scroll 0
	}
    } else {
	set scroll 1
    }

    hook::run draw_message_hook $chatid $from $type $body $x

    if {[info exists opened($chatid)]} {
	set chatw [chat_win $chatid]

	if {![$chatw compare "end -1 chars linestart" == "end -1 chars"]} {
	    $chatw insert end "\n"
	}

	if {$body != "" && !$options(stop_scroll) && \
		(!$options(smart_scroll) || \
		     ($options(smart_scroll) && $scroll) || \
		     [chat::is_our_jid $chatid $from])} {
	    $chatw see end
	}

	$chatw configure -state disabled
    }
}

proc chat::highlighttext {chatw tag color cursor} {
	$chatw configure -cursor $cursor
	$chatw tag configure $tag -foreground $color
}

proc chat::add_emoteiconed_text {chatw body defaulttag} {
    variable url_regexp
    variable options

    set num_of_emoticons 0
    set words [textutil::splitx $body {([\t \r\n]+)}]

    foreach word $words {
	if {$num_of_emoticons < 200} {
	    if {[emoteicons::get $word] != ""} {
		emoteicons::put $chatw $word
		incr num_of_emoticons
		continue
	    }
	}
	if {[regexp -- $url_regexp $word]} {
	    variable urlid
	    set tag url$urlid
	    set urlfg    [option get $chatw urlforeground       Text]
	    set urlactfg [option get $chatw urlactiveforeground Text]
	    $chatw tag configure $tag -foreground $urlfg -underline 1
	    $chatw tag bind $tag <1> [list browseurl [double% $word]]
	    $chatw tag bind $tag <Any-Enter> \
		[list chat::highlighttext $chatw $tag $urlactfg hand2]
	    $chatw tag bind $tag <Any-Leave> \
		[list chat::highlighttext $chatw $tag $urlfg xterm]
	    
	    $chatw insert end $word $tag
	    incr urlid
	} else {
	    if {0 && !$options(emphasize)} {
		$chatw insert end $word $defaulttag
	    } else {
		set enrichedText(bold) 0
		set enrichedText(italic) 0
		set enrichedText(underline) 0

		set tmp $word
		while {[regexp -- "^(_|/|\\*)(.+)(_|/|\\*)$" $tmp wholeMatch e1 subWord e2]} {
		    if { $e1 != $e2 } {
			break
		    }
		    if { "*" == $e1 } {
			if {$enrichedText(bold)} break
			set enrichedText(bold) 1
		    } elseif { "/" == $e1 } {
			if {$enrichedText(italic)} break
			set enrichedText(italic) 1
		    } elseif { "_" == $e1 } {
			if {$enrichedText(underline)} break
			set enrichedText(underline) 1
		    }
		    set tmp $subWord
		}

		set tag $defaulttag
		switch -- $enrichedText(bold),$enrichedText(italic) {
		    1,1 {lappend tag bold_italic}
		    1,0 {lappend tag bold}
		    0,1 {lappend tag italic}
		}
		if {$enrichedText(underline) == 1} {
		    lappend tag underlined
		}
		if {$tmp != $word} {
		    lappend tag emphasized
		    $chatw insert end $tmp $tag
		    $chatw insert end $word [concat nonemphasized $defaulttag]
		} else {
		    $chatw insert end $word $defaulttag
		}
	    }
	}
    }
}

proc chat::switch_emphasize {args} {
    variable options
    variable opened

    foreach chatid [array names opened] {
	set cw [chat_win $chatid]
	if {$options(emphasize)} {
	    $cw tag configure emphasized -elide 0
	    $cw tag configure nonemphasized -elide 1
	} else {
	    $cw tag configure emphasized -elide 1
	    $cw tag configure nonemphasized -elide 0
	}
    }
}

proc chat::open_to_user {connid user args} {
    set msg ""
    foreach {opt val} $args {
	switch -- $opt {
	    -message { set msg $val }
	}
    }

    if {$connid == ""} {
	set connid [jlib::route $user]
    }

    set jid [get_jid_of_user $connid $user]

    if {[cequal $jid ""]} {
	set jid $user
    }

    set chatid [chatid $connid $jid]
    set cw [winid $chatid]
    if {[winfo exists $cw]} {
	if {$::usetabbar} {
	    .nb raise [ifacetk::nbpage $cw]
	}
	focus -force $cw.input
    } else {
	chat::open_window $chatid chat
    }

    if {![cequal $msg ""]} {
	debugmsg chat "SEND_MESSAGE ON OPEN:\
	    [list $chatid] [jlib::connection_user $connid] [list $msg] chat"

	set chatw [chat_win $chatid]
	$chatw mark set start_message "end -1 chars"
	$chatw mark gravity start_message left

	hook::run chat_send_message_hook $chatid [jlib::connection_user $connid] \
	    $msg chat
    }
}

proc chat::change_presence {connid jid status} {
    global grouproster
    variable chats
    variable opened

    set group [node_and_server_from_jid $jid]
    set chatid [chatid $connid $group]
    set nick [get_nick $jid groupchat]

    if {[info exists opened($chatid)]} {
	if {[is_groupchat $chatid]} {
	    debugmsg chat "ST: $connid $jid $status"
	    if {[resource_from_jid $jid] == ""} {
		return
	    }
	    if {[cequal $status unavailable] || [cequal $status error]} {
		if {$status == "error" && [is_our_jid $chatid $jid]} {
		    add_message $chatid $group error \
			[format [::msgcat::mc "Error %s"] \
			     [get_jid_presence_info status $connid $jid]] {}
		}
		if {$status == "unavailable" && [is_our_jid $chatid $jid]} {
		    if {[muc::is_compatible $group] || \
			    ![muc::is_changing_nick $chatid]} {
			add_message $chatid $group error \
			    [::msgcat::mc "Disconnected"] {}
		    }
		}
		debugmsg chat "$jid UNAVAILABLE"
		muc::process_unavailable $chatid $nick
		hook::run chat_user_exit $chatid $nick
		lvarpop grouproster(users,$chatid) \
		    [lsearch -exact $grouproster(users,$chatid) $jid]
		debugmsg chat "GR: $grouproster(users,$chatid)"
		chat::redraw_roster_after_idle $chatid
	    } else {
		set userswin [users_win $chatid]
		if {![lcontain $grouproster(users,$chatid) $jid]} {
		    #roster::addline $userswin jid $nick $jid
		    lappend grouproster(users,$chatid) $jid
		    set grouproster(status,$chatid,$jid) $status

		    muc::process_available $chatid $nick
		    hook::run chat_user_enter $chatid $nick
		}
		set grouproster(status,$chatid,$jid) $status
		ifacetk::roster::changeicon $userswin $jid \
		    $::ifacetk::roster::rostericon(user,$status)
		ifacetk::roster::changeforeground $userswin $jid $status
		chat::redraw_roster_after_idle $chatid
	    }
	}

	#roster:changeicon $w.users [user_from_jid $from] $status
    }

    set cw [winid [chatid $connid $jid]]

    if {[winfo exists $cw.status.icon]} {
	$cw.status.icon configure -image $::ifacetk::roster::rostericon(user,$status)
    }

    if {[winfo exists $cw.status.desc]} {
	$cw.status.desc configure -text "([get_user_status $connid $jid])" \
	    -helptext [get_user_status_desc $connid $jid]
    }

    set user [node_and_server_from_jid $jid]
    set cw [winid [chatid $connid $user]]

    if {[winfo exists $cw.status.icon]} {
	$cw.status.icon configure \
	    -image $::ifacetk::roster::rostericon(user,[get_user_status $connid $user])
    }
}

proc chat::redraw_roster {group} {
    global grouproster

    set connid [get_connid $group]
    set userswin [users_win $group]

    set grouproster(users,$group) [lsort $grouproster(users,$group)]
    ifacetk::roster::clear $userswin 0
    #roster::addline $userswin group Users users

    set levels {}
    foreach jid $grouproster(users,$group) {
	if {[info exists ::muc::users(role,$connid,$jid)]} {
	    if {$::muc::users(role,$connid,$jid) == "" && \
		    $::muc::users(affiliation,$connid,$jid) == ""} {
		set level "aGreen mans"
		lappend levels $level
		lappend levelusers($level) $jid
	    } else {
		if {$::muc::users(role,$connid,$jid) != "" && \
			$::muc::users(role,$connid,$jid) != "none"} {
		    set level $::muc::users(role,$connid,$jid)
		    switch -- $level {
			moderator   {set level 4[::msgcat::mc "Moderators"]}
			participant {set level 5[::msgcat::mc "Participants"]}
			visitor     {set level 6[::msgcat::mc "Visitors"]}
		    }
		    lappend levels $level
		    lappend levelusers($level) $jid
		}

		#if {$::muc::users(affiliation,$jid) != "" && \
		#	$::muc::users(affiliation,$jid) != "none"} {
		#    set level $::muc::users(affiliation,$jid)
		#    switch -- $level {
		#	owner   {set level 0Owners}
		#	admin   {set level 1Admins}
		#	member  {set level 2Members}
		#	outcast {set level 9Outcasts}
		#    }
		#    lappend levels $level
		#    lappend levelusers($level) $jid
		#}
	    }
	} else {
	    set level 2[::msgcat::mc "Users"]
	    lappend levels $level
	    lappend levelusers($level) $jid
	}	    
    }
    set levels [lrmdups $levels]

    foreach level $levels {
	ifacetk::roster::addline $userswin group \
	    "[crange $level 1 end] ([llength $levelusers($level)])" \
	    [crange $level 1 end] [crange $level 1 end] 0
	set jid_nicks {}
	foreach jid $levelusers($level) {
	    lappend jid_nicks [list $jid [get_nick $jid groupchat]]
	}
	set jid_nicks [lsort -index 1 -dictionary $jid_nicks]

	foreach item $jid_nicks {
	    lassign $item jid nick
	    set status $grouproster(status,$group,$jid)

	    ifacetk::roster::addline $userswin jid $nick \
		[list $connid $jid] [crange $level 1 end] 0
	    ifacetk::roster::changeicon $userswin \
		[list $connid $jid] $::ifacetk::roster::rostericon(user,$status)
	    if {$::plugins::nickcolors::options(use_colored_roster_nicks)} {
		ifacetk::roster::changeforeground $userswin \
		    [list $connid $jid] $status \
		    [plugins::nickcolors::get_nick_color $nick]
	    } else {
		ifacetk::roster::changeforeground $userswin \
		    [list $connid $jid] $status
	    }
	}
    }

    ifacetk::roster::update_scrollregion $userswin
}

proc chat::redraw_roster_after_idle {group} {
    variable afterid

    if {[info exists afterid($group)]} \
	return

    set afterid($group) [after idle "
	chat::redraw_roster [list $group]
	unset [list chat::afterid($group)]
    "]
}

proc chat::restore_subject {chatid} {
    variable opened
    variable chats

    set cw [winid $chatid]

    if {[info exists opened($chatid)]} {
	$cw.status.subject delete 0 end
	$cw.status.subject insert 0 $chats(subject,$chatid)
    }
}

proc chat::set_subject {chatid subject} {
    variable opened
    variable chats

    set cw [winid $chatid]

    if {[info exists opened($chatid)]} {
	$cw.status.subject delete 0 end
	$cw.status.subject insert 0 $subject
	set chats(subject,$chatid) $subject
    }
}

proc chat::change_subject {chatid} {
    set cw [winid $chatid]
    set connid [get_connid $chatid]
    set jid [get_jid $chatid]

    set subject [$cw.status.subject get]

    message::send_msg $jid -connection $connid \
	-type groupchat -subject $subject
}

proc chat::is_our_jid {chatid jid} {
    return [cequal [our_jid $chatid] $jid]
}

proc chat::our_jid {chatid} {
    variable chats

    set connid [get_connid $chatid]
    set jid [get_jid $chatid]
    switch -- $chats(type,$chatid) {
	groupchat {
	    return $jid/[get_our_groupchat_nick $chatid]
	}
	chat {
	    set group [node_and_server_from_jid $jid]
	    set groupid [chatid $connid $group]
	    if {[is_groupchat $groupid]} {
		return $group/[get_our_groupchat_nick $groupid]
	    } else {
		return [jlib::connection_jid $connid]
	    }
	}
    }
    return ""
}

proc chat::invite_dialog {user {ignore_muc 0} args} {
    variable chats
    variable opened
    global invite_gc

    foreach {opt val} $args {
	switch -- $opt {
	    -connection { set connid $val }
	}
    }
    if {![info exists connid]} {
	set connid [jlib::route $user]
    }

    set jid [get_jid_of_user $connid $user]

    if {[cequal $jid ""]} {
        set jid $user
    }

    set gw .invite
    catch { destroy $gw }

    if {[catch { set nick [roster::get_label $user] }]} {
	if {[catch {set nick [roster::get_label \
				  [node_and_server_from_jid $user]] }]} {
	    if {[catch { set nick [chat::get_nick $user groupchat] }]} {
		set nick $user
	    }
	}
    }

    set titles {}
    set jids {}
    foreach chatid [lsort [array names opened]] {
        if {[is_groupchat $chatid]} {
	    lappend jids $chatid [get_jid $chatid]
            lappend titles $chatid [node_from_jid [get_jid $chatid]]
	}
    }
    if {[llength $titles] == 0} {
        MessageDlg ${gw}_err -aspect 50000 -icon info \
	    -message [::msgcat::mc "No conferences in progress..."] -type user \
	    -buttons ok -default 0 -cancel 0
        return
    }

    CbDialog $gw [format [::msgcat::mc "Invite %s to conferences"] $nick] \
	[list [::msgcat::mc "Invite"] "chat::invitation [list $jid] 0 $ignore_muc
				       destroy $gw" \
	      [::msgcat::mc "Cancel"] "destroy $gw"] \
	invite_gc $titles $jids
}

proc chat::invite_dialog2 {jid {ignore_muc 0}} {
    variable chats
    variable opened
    global invite_gc

    set gw .invite
    catch { destroy $gw }

    set title [node_from_jid $jid]

    set choices {} 
    set balloons {} 
    foreach c [jlib::connections] { 
	foreach choice [roster::get_jids $c] { 
	    if {![cequal [roster::itemconfig $c $choice -category] conference]} {
		lappend choices [list $c $choice] [roster::get_label $c $choice]
		lappend balloons [list $c $choice] $choice 
	    } 
	} 
    } 
    if {[llength $choices] == 0} {
        MessageDlg ${gw}_err -aspect 50000 -icon info \
	    -message [::msgcat::mc "No users in roster..."] -type user \
	    -buttons ok -default 0 -cancel 0
	return
    }

    CbDialog $gw [format [::msgcat::mc "Invite users to %s"] $title] \
	[list [::msgcat::mc "Invite"] "chat::invitation [list $jid] 1 $ignore_muc
				       destroy $gw" \
	      [::msgcat::mc "Cancel"] "destroy $gw"] \
	invite_gc $choices $balloons
}

proc chat::invitation {jid usersP ignore_muc {reason ""}} {
    global invite_gc

    foreach choice [array names invite_gc] {
        if {$invite_gc($choice)} {
	    lassign $choice con gc
	    if {$usersP} {
		set to $gc
		set chatid $jid
	    } else {            
		set to $jid
		set chatid $gc
	    }
	    if {[cequal $reason ""]} {
		set reas [format [::msgcat::mc "Please join %s"] $chatid]
	    } else {
		set reas $reason
	    }
	    if {!$ignore_muc && [muc::is_compatible [chat::get_jid $chatid]]} {
		muc::invitation $chatid $to $reas
	    } else {
		message::send_msg $to -type normal \
		    -subject "Invitation" \
		    -body $reas \
		    -xlist [list [jlib::wrapper:createtag x \
				-vars [list xmlns jabber:x:conference \
				jid $chatid]]]
	    }
	}
    }
}


