/*   FILE: fmt_ptrn.c --
 * AUTHOR: W. Michael Petullo <new@flyn.org>
 *   DATE: 08 January 2000
 *
 * Copyright (c) 1999 W. Michael Petullo <new@flyn.org>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>
#include <fmt_ptrn.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>
#include <sizes.h>
#include <common.h>

/* ============================ modifier_name_t ============================ */
typedef enum {
    FMT_PTRN_NO_MODIFIER,
    FMT_PTRN_UPPER,
    FMT_PTRN_LOWER,
    FMT_PTRN_BASENAME,
    FMT_PTRN_BEFORE,
    FMT_PTRN_AFTER,
    FMT_PTRN_FN,
    FMT_PTRN_C_DELIM,
    FMT_PTRN_CPP_DELIM,
    FMT_PTRN_SH_DELIM,
    FMT_PTRN_TEX_DELIM,
    FMT_PTRN_TEMPLATE,
    FMT_PTRN_FILE,
    FMT_PTRN_COMMENT,
    FMT_PTRN_REMOVE_UNDERSCORE
} modifier_name_t;

/* ============================ modifier_t ================================= */
typedef struct modifier_t {
    modifier_name_t name;
    char arg[MODIFIER_ARG_LEN + 1];
} modifier_t;

/* ============================ fmt_ptrn_parse_err () ====================== */
int fmt_ptrn_parse_err(const fmt_ptrn_t * x)
{
    return queue_size(&x->parse_errmsg);
}

/* ============================ fmt_ptrn_parse_strerror () ================= */
char *fmt_ptrn_parse_strerror(fmt_ptrn_t * x)
{
    char *errmsg;
    if (queue_dequeue(&x->parse_errmsg, (void **) &errmsg) < 0)
	return strdup("no error");	/* So it can be freed. */
    else
	return errmsg;
}

/* ============================ fmt_ptrn_parse_perror () =================== */
void fmt_ptrn_parse_perror(fmt_ptrn_t * x, const char *msg)
{
    char *errmsg = fmt_ptrn_parse_strerror(x);
    if (msg)
	fprintf(stderr, "%s: %s\n", msg, errmsg);
    else
	fprintf(stderr, "%s\n", errmsg);
    free(errmsg);
}

/* ============================ _enqueue_parse_errmsg () =================== */
static void _enqueue_parse_errmsg(fmt_ptrn_t * x, const char *msg, ...)
{
    char *err = (char *) malloc(sizeof(char) * PARSE_ERR_LEN + 1);
    va_list args;
    va_start(args, msg);
    vsnprintf(err, PARSE_ERR_LEN, msg, args);
    va_end(args);
    queue_enqueue(&x->parse_errmsg, err);
}

/* ============================ stack_t ==================================== */
typedef struct stack_t {
    modifier_t data[STACK_MAX_ITEMS];
    int size;
} stack_t;

/* ============================ _stack_init () ============================= */
static void _stack_init(stack_t * s)
{
    s->size = 0;
}

/* ============================ _stack_size () ============================= */
static int _stack_size(const stack_t s)
{
    return s.size;
}

/* ============================ _stack_push () ============================= */
static int _stack_push(fmt_ptrn_t * x, stack_t * s, const modifier_t data)
{
    if (s->size == STACK_MAX_ITEMS) {
	_enqueue_parse_errmsg(x, "%s: %ld: more than %d modifiers",
			      x->template_path, x->line_num,
			      STACK_MAX_ITEMS);
	return 0;
    }
    s->data[s->size++] = data;
    return 1;
}

/* ============================ _stack_pop () ============================== */
static int _stack_pop(stack_t * s, modifier_t * data)
{
    if (!s->size)
	return 0;
    *data = s->data[--s->size];
    return 1;
}

/* ============================ _stack_contains () ========================= */
static int _stack_contains(const stack_t s, const modifier_name_t n)
{
    int i;
    for (i = 0; i < s.size; i++)
	if (s.data[i].name == n)
	    return 1;
    return 0;
}

/* ============================ _buffer_init () ============================ */
static buffer_t _buffer_init(void)
{
    buffer_t x;
    x.data = (char *) malloc(sizeof(char) * INIT_BUF_SIZE);
    *x.data = 0x00;
    x.size = INIT_BUF_SIZE;
    return x;
}

/* ============================ _buffer_destroy () ========================= */
static void _buffer_destroy(buffer_t b)
{
    free(b.data);
    b.data = NULL;
    b.size = 0;
}

/* ============================ _buffer_eat () ============================= */
static void _buffer_eat(buffer_t buf, size_t n)
/* Eats n characters off the beginning of buffer. */
{
    char *p_1 = buf.data, *p_2;
    if (!n)
	return;
    for (p_2 = p_1 + n; p_2 <= p_1 + strlen(p_1); p_2++)
	*p_1++ = *p_2;
}

/* ============================ ============================================ */
inline size_t _buffer_len(buffer_t buf)
{
    return (buf.size - 1);
}

/* ============================ _pair_compare () =========================== */
static inline int _pair_compare(const pair_t * x, const pair_t * y)
{
    return strcmp(x->key, y->key);
}

/* FIXME: The following two functions should be renamed. */
/* ============================ fmt_ptrn_update_kv_1 () ==================== */
void fmt_ptrn_update_kv_1(fmt_ptrn_t * x, pair_t *p)
{
    bistree_insert(&x->fillers, p);
}

/* ============================ fmt_ptrn_update_kv () ====================== */
void fmt_ptrn_update_kv(fmt_ptrn_t * x, char *key, char *val)
{
    pair_t *p = (pair_t *) malloc(sizeof(pair_t));
    pair_init(p, key, val, free, free);
    fmt_ptrn_update_kv_1 (x, p);
}

/* ============================ _realloc_n_cat () ========================== */
static void _realloc_n_cat(buffer_t * dest, const char *src)
{
    size_t new_len = (dest && dest->data ? strlen(dest->data) : 0) + strlen(src);
    if (!dest->data) {
	dest->size = new_len * 2 + 1;
	dest->data = malloc(sizeof(char) * dest->size);
	*dest->data = 0x00;
    } else if (new_len + 1 > dest->size) {
	dest->size = new_len * 2 + 1;
	dest->data = realloc(dest->data, sizeof(char) * dest->size);
    }
    strcat(dest->data, src);
}

/* ============================ _realloc_n_cpy () ========================== */
static void _realloc_n_cpy(buffer_t * dest, const char *src)
{
    if (dest->data)
	*dest->data = 0x00;
    _realloc_n_cat(dest, src);
}

/* ============================ _realloc_n_ncat () ========================= */
static void _realloc_n_ncat(buffer_t * dest, const char *src,
			    const size_t nc)
{
    size_t src_len = strlen(src);
    size_t new_len =
	(dest && dest->data ? strlen(dest->data) : 0) + (src_len <
						 nc ? src_len : nc);
    if (new_len + 1 > dest->size) {
	dest->size = new_len * 2 + 1;
	dest->data = realloc(dest->data, sizeof(char) * dest->size);
    }
    strncat(dest->data, src, nc);
}

/* ============================ _read_alternate () ========================= */
static void _read_alternate(fmt_ptrn_t * x, char **pattern)
{
    if (!**pattern)		/* Already queued error, hopefully. */
	return;
    if (**pattern == ':') {
	(*pattern)++;
	while (**pattern != ')' && **pattern != 0x00)
	    _realloc_n_ncat(&x->filled_buf, (*pattern)++, 1);
	if (!**pattern)
	    _enqueue_parse_errmsg(x, "%s: %ld: end of input",
				  x->template_path, x->line_num);
    } else
	/* We know there is no value for the format string because
	 * this function was called.  There is also no alternate.
	 * Call this a parse error to be safe.
	 */
	_enqueue_parse_errmsg(x, "%s: %ld: key has no value",
			      x->template_path, x->line_num);
}

/* ============================ _eat_alternate () ========================== */
static void _eat_alternate(fmt_ptrn_t * x, char **pattern)
{
    if (!**pattern || **pattern != ':')
	/* No alternate provided to eat. */
	return;
    while (**pattern != ')' && **pattern != 0x00)
	(*pattern)++;
    if (!**pattern)
	_enqueue_parse_errmsg(x, "%s: %ld: end of input", x->template_path,
			      x->line_num);
}

/* ============================ _read_modifier_arg () ====================== */
static void _read_modifier_arg(fmt_ptrn_t * x, char **pattern,
			       modifier_t * i)
{
    size_t arg_len;
    char *end_quote = strchr(*pattern, '"'), *end_paren =
	strchr(*pattern, ')');
    if (!end_quote || (end_paren && end_quote > end_paren)) {
	_enqueue_parse_errmsg(x, "%s: %ld: no end quote", x->template_path,
			      x->line_num);
	return;
    }
    arg_len = end_quote - *pattern;
    if (arg_len > MODIFIER_ARG_LEN) {
	strncpy(i->arg, *pattern, MODIFIER_ARG_LEN);
	i->arg[MODIFIER_ARG_LEN] = 0x00;
	_enqueue_parse_errmsg(x, "%s: %ld: modifier arg. too long",
			      x->template_path, x->line_num);
    } else {
	strncpy(i->arg, *pattern, arg_len);
	i->arg[arg_len] = 0x00;
    }
    if (*(*pattern + arg_len + 1) != ' ')
	_enqueue_parse_errmsg(x, "%s: %ld: no space after arg",
			      x->template_path, x->line_num);
    *pattern += arg_len + 2;	/* Add 2 for end quote and space. */
}

/* ============================ _read_modifier () ========================== */
static int _read_modifier(fmt_ptrn_t * x, char **ptrn, stack_t * modifier)
{
    modifier_t i;
    if (!strncmp("upper", *ptrn, sizeof("upper") - 1)) {
	*ptrn += sizeof("upper");	/* Skip one space too. */
	i.name = FMT_PTRN_UPPER;
	_stack_push(x, modifier, i);
	return 1;
    } else if (!strncmp("lower", *ptrn, sizeof("lower") - 1)) {
	*ptrn += sizeof("lower");
	i.name = FMT_PTRN_LOWER;
	_stack_push(x, modifier, i);
	return 1;
    } else if (!strncmp("basename", *ptrn, sizeof("basename") - 1)) {
	*ptrn += sizeof("basename");
	i.name = FMT_PTRN_BASENAME;
	_stack_push(x, modifier, i);
	return 1;
    } else if (!strncmp("c_delim", *ptrn, sizeof("c_delim") - 1)) {
	*ptrn += sizeof("c_delim");
	i.name = FMT_PTRN_C_DELIM;
	_stack_push(x, modifier, i);
	return 1;
    } else if (!strncmp("cpp_delim", *ptrn, sizeof("cpp_delim") - 1)) {
	*ptrn += sizeof("cpp_delim");
	i.name = FMT_PTRN_CPP_DELIM;
	_stack_push(x, modifier, i);
	return 1;
    } else if (!strncmp("sh_delim", *ptrn, sizeof("sh_delim") - 1)) {
	*ptrn += sizeof("sh_delim");
	i.name = FMT_PTRN_SH_DELIM;
	_stack_push(x, modifier, i);
	return 1;
    } else if (!strncmp("tex_delim", *ptrn, sizeof("tex_delim") - 1)) {
	*ptrn += sizeof("tex_delim");
	i.name = FMT_PTRN_TEX_DELIM;
	_stack_push(x, modifier, i);
	return 1;
    } else if (!strncmp("fn", *ptrn, sizeof("fn") - 1)) {
	*ptrn += sizeof("fn");
	i.name = FMT_PTRN_FN;
	_stack_push(x, modifier, i);
	return 1;
    } else if (!strncmp("before=\"", *ptrn, sizeof("before=\"") - 1)) {
	*ptrn += sizeof("before=\"") - 1;
	i.name = FMT_PTRN_BEFORE;
	_read_modifier_arg(x, ptrn, &i);
	_stack_push(x, modifier, i);
	return 1;
    } else if (!strncmp("after=\"", *ptrn, sizeof("after=\"") - 1)) {
	*ptrn += sizeof("after=\"") - 1;
	i.name = FMT_PTRN_AFTER;
	_read_modifier_arg(x, ptrn, &i);
	_stack_push(x, modifier, i);
	return 1;
    } else if (!strncmp("template", *ptrn, sizeof("file") - 1)) {
	*ptrn += sizeof("file");
	i.name = FMT_PTRN_TEMPLATE;
	_stack_push(x, modifier, i);
	return 1;
    } else if (!strncmp("file", *ptrn, sizeof("file") - 1)) {
	*ptrn += sizeof("file");
	i.name = FMT_PTRN_FILE;
	_stack_push(x, modifier, i);
	return 1;
    } else if (!strncmp("#", *ptrn, sizeof("#") - 1)) {
	*ptrn += strchr(*ptrn, ')') - *ptrn;
	i.name = FMT_PTRN_COMMENT;
	_stack_push(x, modifier, i);
	return 1;
    } else if (!strncmp("remove_underscore", *ptrn, sizeof("remove_underscore") - 1)) {
	*ptrn += sizeof("remove_underscore");
	i.name = FMT_PTRN_REMOVE_UNDERSCORE;
	_stack_push(x, modifier, i);
	return 1;
    }
    return 0;
}

/* ============================ _read_modifiers () ========================= */
static void _read_modifiers(fmt_ptrn_t * x, char **ptrn,
			    stack_t * modifier)
{
    while (_read_modifier(x, ptrn, modifier)) {
	/* NOOP. */
    }
    return;
}

/* ============================ _read_key () =============================== */
static void _read_key(fmt_ptrn_t * x, char *key, char **p)
{
    int i;
    *key = 0x00;
    for (i = 0; i < KEY_LEN && **p && !strchr(":)", **p); i++)
	strncat(key, (*p)++, 1);
    if (**p && !strchr(":)", **p)) {
	/* Uh oh, key is too many characters, eat the rest. */
	while (**p && **p != ':' && **p != ')')
	    (*p)++;
	_enqueue_parse_errmsg(x, "%s: %ld: key too long", x->template_path,
			      x->line_num);
    }
    if (!**p)
	_enqueue_parse_errmsg(x, "%s: %ld: end of input", x->template_path,
			      x->line_num);
}

/* ============================ _apply_delim () ============================ */
static void _apply_delim(buffer_t * str, const char *start_cmnt,
			 const char *end_cmnt)
{
    /* This one is a bit ugly, but not very interesting. */
    int i;
    char ptr[81];
    size_t start_cmnt_len = strlen(start_cmnt) + 1;	/* + 1 for ' '. */
    size_t end_cmnt_len = end_cmnt ? strlen(end_cmnt) + 1 : 0;
    size_t len;
    if (str->size < 81) {
	str->data = realloc(str->data, sizeof(char) * 81);
	str->size = 81;
    }
    strcpy(ptr, start_cmnt);
    strcat(ptr, " ");
    for (i = 0; i < (31 - start_cmnt_len - 2); i++)	/* - 2 for spaces. */
	strcat(ptr, "=");
    strcat(ptr, " ");
    len = strlen(ptr);
    strncat(ptr, str->data, 80 - len - 2 - end_cmnt_len);	/* - 2 for spaces. */
    strcat(ptr, " ");
    len = strlen(ptr);
    for (i = 0; i < (80 - len - end_cmnt_len); i++)
	strcat(ptr, "=");
    strcat(ptr, end_cmnt ? " " : "");
    strcat(ptr, end_cmnt ? end_cmnt : "");
    strcpy(str->data, ptr);
}

/* ============================ _apply_before () =========================== */
static void _apply_before(buffer_t * dest, const char *src)
{
    /* Done with no malloced tmp. */
    size_t i, j, old_len = strlen(dest->data), src_len =
	strlen(src), new_len = old_len + src_len;
    if (new_len + 1 > dest->size) {
	dest->size = new_len + 1;
	dest->data = realloc(dest->data, sizeof(char) * dest->size);
    }
    /* Shift dest over to make room for src. */
    for (i = old_len - 1, j = new_len - 1; i > 0; i--, j--)
	dest->data[j] = dest->data[i];
    dest->data[src_len] = dest->data[0];
    dest->data[new_len] = 0x00;
    /* Copy src into dest. */
    for (i = 0; i < src_len; i++)
	dest->data[i] = src[i];
}

/* ============================ _remove_underscore () ====================== */
static void _remove_underscore(buffer_t * dest)
{
    int i;
    for (i = 0; i < dest->size; i++)
        if (dest->data[i] == '_')
	    dest->data[i] = '-';
}

/* ============================ _apply_after () ============================ */
static void _apply_after(buffer_t * dest, const char *src)
{
    /* Too easy. */
    _realloc_n_cat(dest, src);
}

/* ============================ _apply_modifiers () ======================== */
static void _apply_modifiers(fmt_ptrn_t * x, buffer_t * str,
			     stack_t * modifier)
{
    modifier_t i;
    while (_stack_pop(modifier, &i)) {
	switch (i.name) {
	    char *ptr;
	case FMT_PTRN_NO_MODIFIER:
	    break;
	case FMT_PTRN_UPPER:
	    ptr = str->data;
	    while (*ptr)
		*ptr++ = toupper(*ptr);
	    break;
	case FMT_PTRN_LOWER:
	    ptr = str->data;
	    while (*ptr)
		*ptr++ = tolower(*ptr);
	    break;
	case FMT_PTRN_BASENAME:
	    ptr = strchr(str->data, '.');
	    if (ptr)
		*ptr = 0x00;
	    break;
	case FMT_PTRN_C_DELIM:
	    _apply_delim(str, "/*", "*/");
	    break;
	case FMT_PTRN_CPP_DELIM:
	    _apply_delim(str, "//", NULL);
	    break;
	case FMT_PTRN_SH_DELIM:
	    _apply_delim(str, "#", NULL);
	    break;
	case FMT_PTRN_TEX_DELIM:
	    _apply_delim(str, "%", NULL);
	    break;
	case FMT_PTRN_FN:
	    _apply_after(str, " ()");
	    break;
	case FMT_PTRN_BEFORE:
	    _apply_before(str, i.arg);
	    break;
	case FMT_PTRN_AFTER:
	    _apply_after(str, i.arg);
	    break;
	case FMT_PTRN_REMOVE_UNDERSCORE:
	    _remove_underscore(str);
	    break;
	default:
	    _enqueue_parse_errmsg(x, "%s: %ld: invalid modifier",
				  x->template_path, x->line_num);
	    break;
	}
    }
}

/* ============================ _lookup () ================================= */
int _lookup(const fmt_ptrn_t * x, char *key, buffer_t * value)
{
    pair_t *tmp;
    pair_init(x->kv, key, NULL, NULL, free);
    tmp = bistree_lookup(&x->fillers, (void *) x->kv);
    if (tmp && tmp->val) {
	_realloc_n_cpy(value, tmp->val);
	return 1;
    } else
	return 0;
}

/* ============================ _add_file () =============================== */
static int _add_file(fmt_ptrn_t * x, const char *path)
/* This function handles the case where the FMT_PTRN_FILE modifier is 
 * used. 
 */
{
    char b[BUFSIZ];
#ifdef HAVE_LIBZ
    gzFile f;
    if (!(f = gzopen(path, "rb"))) {
#else
    FILE *f;
    if (!(f = fopen (path, "rb"))) {
#endif
	_enqueue_parse_errmsg(x, "%s: %ld: could not open %s",
			      x->template_path, x->line_num, path);
	return 0;
    }
#ifdef HAVE_LIBZ
    while (gzgets(f, b, BUFSIZ) != Z_NULL)
#else
    while (fgets(b, BUFSIZ, f) != NULL)
#endif
    	_realloc_n_cat(&x->filled_buf, b);	
#ifdef HAVE_LIBZ
    gzclose (f);
#else
    fclose (f);
#endif
    return 1;
}

/* ============================ _fill_file () ============================== */
static int _fill_file(fmt_ptrn_t * x, const char *path)
/* This function handles the case where the FMT_PTRN_TEMPLATE modifier is 
 * used. 
 */
{
    fmt_ptrn_t f;
    char b[BUFSIZ];
    if (!fmt_ptrn_open(path, &f)) {
	_enqueue_parse_errmsg(x, "%s: %ld: could not open %s",
			      x->template_path, x->line_num, path);
	return 0;
    }
    initialize_fillers(&f);
    while (fmt_ptrn_gets(b, BUFSIZ, &f))
	_realloc_n_cat(&x->filled_buf, b);
    while (fmt_ptrn_parse_err(&f))
	/* Copy parse error messages into the main fmt_ptrn_t data structure. */
	_enqueue_parse_errmsg(x, fmt_ptrn_parse_strerror(&f));
    fmt_ptrn_close(&f);
    return 1;
}

/* ============================ _modifier_is_lb_include () ================= */
/* FIXME: see FIXME in _handle_fmt_str. */
static modifier_name_t _modifier_is_lb_include(fmt_ptrn_t *x, char **p, stack_t mod)
{
    if (_stack_contains(mod, FMT_PTRN_TEMPLATE)) {
	if (_stack_size(mod) > 1)
	    _enqueue_parse_errmsg(x, "%s: %ld: template not only modifier", x->template_path, x->line_num);
	return FMT_PTRN_TEMPLATE;
    } else if (_stack_contains(mod, FMT_PTRN_FILE)) {
	if (_stack_size(mod) > 1)
	    _enqueue_parse_errmsg(x, "%s: %ld: file not only modifier", x->template_path, x->line_num);
	return FMT_PTRN_FILE;
    }
    return FMT_PTRN_NO_MODIFIER;
}


/* ============================ _apply_lb_include () ======================= */ 
/* FIXME: see FIXME in _handle_fmt_str. */
static void _apply_lb_include(fmt_ptrn_t *x, modifier_name_t n, char **p)
{
    char key[KEY_LEN + 1];
    if (n == FMT_PTRN_TEMPLATE) {
	_read_key(x, key, p);
	_fill_file(x, key);
    } else if (n == FMT_PTRN_FILE) {
	_read_key(x, key, p);
	_add_file(x, key);
    } else
	_enqueue_parse_errmsg(x, "%s: %ld: non-#include style modifier in _apply_lb_include ()", x->template_path, x->line_num);
}

/* ============================ _handle_fmt_str () ========================= */
static void _handle_fmt_str(fmt_ptrn_t * x, char **p)
{
    /* format string -> %(<modifier_0> ... <modifier_n> <key>:<alt>) */
    stack_t modifier;
    modifier_name_t mod_name;
    char key[KEY_LEN + 1];
    _stack_init(&modifier);
    *p += 2;			/* Skip "%(". */
    _read_modifiers(x, p, &modifier);
    /* FIXME: #include style modifiers are handled seperately as they
     * should not appear with any other modifiers.  This may not be a
     *good idea.  Is there an instance one would want to modify a file or
     * template modifier?
     */
    if ((mod_name = _modifier_is_lb_include(x, p, modifier)))
        _apply_lb_include(x, mod_name, p);
    else if (_stack_contains(modifier, FMT_PTRN_COMMENT)) {
	/* NOOP. */
    } else {
	_read_key(x, key, p);
	if (_lookup(x, key, &x->lookup_buf)) {
	    _eat_alternate(x, p);
	    _apply_modifiers(x, &x->lookup_buf, &modifier);
	    _realloc_n_cat(&x->filled_buf, x->lookup_buf.data);
	} else
	    _read_alternate(x, p);
    }
    if (**p)
	(*p)++;			/* Skip ')'. */
}

/* ============================ _fill_it () ================================ */
static int _fill_it(fmt_ptrn_t * x, const char *p)
{
    char *pattern, *orig_ptr;
    pattern = orig_ptr = strdup((char *) p);
    while (*pattern != 0x00) {
	if (*pattern == '%' && *(pattern + 1) == '%') {
	    /* Handle %%(...), which should be filled as %(...). */
	    _realloc_n_ncat(&x->filled_buf, pattern, 1);
            pattern += 2;
	} else if (*pattern == '%' && *(pattern + 1) == '(')
	    _handle_fmt_str(x, &pattern);
	else {
	    if (*pattern == '\n')
		x->line_num++;
	    _realloc_n_ncat(&x->filled_buf, pattern++, 1);
	}
    }
    free(orig_ptr);
    return 1;
}

/* ============================ fmt_ptrn_filled () ========================= */
char *fmt_ptrn_filled(fmt_ptrn_t * x, const char *p)
{
    *x->filled_buf.data = 0x00;
    if (!_fill_it(x, p))
	return NULL;
    return strdup (x->filled_buf.data);
}

/* ============================ fmt_ptrn_init () =========================== */
int fmt_ptrn_init(fmt_ptrn_t * x)
/* Alternative to open; does everything but open the file.  This 
 * should be used when filling strings instead of files.
 */
{
    strcpy(x->errmsg, "no error");
    queue_init(&x->parse_errmsg, NULL);
    bistree_init(&x->fillers, (void *) _pair_compare, (void *) pair_destroy);
    x->template_fp = NULL;
    x->line_num = 1;
    x->raw_buf = _buffer_init();
    x->filled_buf = _buffer_init();
    x->lookup_buf = _buffer_init();
    x->kv = (pair_t *) malloc(sizeof(pair_t));	/* Here so it is persistent. */
    strcpy(x->template_path, "string");
    return 1;
}

/* ============================ fmt_ptrn_open () =========================== */
int fmt_ptrn_open(const char *path, fmt_ptrn_t * x)
{
#ifdef HAVE_LIBZ
    gzFile in_file;
    if (!(in_file = gzopen(path, "rb"))) {
#else
    FILE *in_file;
    if (!(in_file = fopen(path, "rb"))) {
#endif
	return 0;
    }
    if (!fmt_ptrn_init(x))
	return 0;
    x->template_fp = in_file;	/* init sets this to NULL. */
    strcpy(x->template_path, path);	/* init sets this to "string". */
    return 1;
}

/* ============================ fmt_ptrn_gets () =========================== */
char *fmt_ptrn_gets(char *buf, size_t size, fmt_ptrn_t * x)
{
    if (!strlen(x->filled_buf.data)) {
	/* FIXME: potentially, a buffer could be filled with only 
	 * half of a format string. 
	 */
	/* Here buf is used as a temp. buffer. */
#ifdef HAVE_LIBZ
	if (gzgets(x->template_fp, buf, size) == Z_NULL)
#else
	if (fgets(buf, size, x->template_fp) == NULL)
#endif
	    return NULL;
	_fill_it(x, buf);
    }
    strncpy(buf, x->filled_buf.data, size - 1);
    buf[size - 1] = 0x00;
    _buffer_eat(x->filled_buf, strlen(buf));
    return buf;
}

/* ============================ fmt_ptrn_close () ========================== */
int fmt_ptrn_close(fmt_ptrn_t * x)
{
    queue_destroy(&x->parse_errmsg);
    bistree_destroy(&x->fillers);
    _buffer_destroy(x->raw_buf);
    _buffer_destroy(x->filled_buf);
    _buffer_destroy(x->lookup_buf);
    free(x->kv);
    /* x->template_fp == NULL if fmt_ptrn_init was used instead of 
     * fmt_ptrn_open.
     */
#ifdef HAVE_LIBZ
    return x && x->template_fp ? gzclose(x->template_fp) : 1;
#else
    return x && x->template_fp ? fclose(x->template_fp) : 1;
#endif
}

/* ============================ fmt_ptrn_perror () ========================= */
void fmt_ptrn_perror(const fmt_ptrn_t * x, const char *msg)
{
    fprintf(stderr, "%s: %s\n", msg, x->errmsg);
}

/* ============================ fmt_ptrn_strerror () ======================= */
const char *fmt_ptrn_strerror(const fmt_ptrn_t * x)
{
    return x->errmsg;
}
