/*
 * getaddrinfo.c - replacement functions for getaddrinfo (& co.)
 * This is not thread-safe!!!
 * $Id: getaddrinfo.c,v 1.3 2003/01/26 09:09:08 rdenisc Exp $
 */

/***********************************************************************
 *  Copyright (C) 2002-2003 Rmi Denis-Courmont.                       *
 *  This program is free software; you can redistribute and/or modify  *
 *  it under the terms of the GNU General Public License as published  *
 *  by the Free Software Foundation; version 2 of the license.         *
 *                                                                     *
 *  This program is distributed in the hope that it will be useful,    *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of     *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.               *
 *  See the GNU General Public License for more details.               *
 *                                                                     *
 *  You should have received a copy of the GNU General Pulic License   *
 *  along with this program; if not, you can get it from:              *
 *  http://www.gnu.org/copyleft/gpl.html                               *
 ***********************************************************************/
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "getaddrinfo.h"

#include <stddef.h> /* size_t */
#include <string.h> /* strncpy(), strlen(), memcpy(), memset(), strchr() */
#include <stdlib.h> /* malloc(), free(), strtoul() */
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_ARPA_INET_H
# include <arpa/inet.h>
#endif
#ifdef HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif
#ifdef HAVE_ERRNO_H
# include <errno.h>
#endif

#if !HAVE_GAI_STRERROR
static const char *
__gai_errlist [] = {
	"Error 0",
	"Invalid flag used",
	"Host or service not found",
	"Temporary name service failure",
	"Non-recoverable name service failure",
	"No data for host name",
	"Unsupported address family",
	"Unsupported socket type",
	"Incompatible service for socket type",
	"Unavailable address family for host name",
	"Memory allocation failure",
	"System error"
};

static const char *
__gai_unknownerr =
	"Unrecognized error number";
# define _EAI_POSITIVE_MAX 11

/*
 * Converts an EAI_* error code into human readable english text.
 */
const char *
gai_strerror (int errnum)
{
	unsigned int errc;

	errc = (unsigned int)(-errnum);

	if (errc > _EAI_POSITIVE_MAX)
		return __gai_unknownerr;

	return __gai_errlist[errc];
}
# undef _EAI_POSITIVE_MAX
#endif /* if !HAVE_GAI_STRERROR */

#if !(HAVE_GETNAMEINFO && HAVE_GETADDRINFO)
/*
 * Converts the current herrno error value into an EAI_* error code.
 * That error code is normally returned by getnameinfo() or getaddrinfo().
 */
static int
gai_error_from_herrno (void)
{
	switch(h_errno)
	{
		case HOST_NOT_FOUND:
			return EAI_NONAME;

		case NO_ADDRESS:
# if (NO_ADDRESS != NO_DATA)
		case NO_DATA:
# endif
			return EAI_NODATA;

		case NO_RECOVERY:
			return EAI_FAIL;

		case TRY_AGAIN:
			return EAI_AGAIN;
	}
	return EAI_SYSTEM;
}
#endif /* if !(HAVE_GETNAMEINFO && HAVE_GETADDRINFO) */

#if !HAVE_GETNAMEINFO
/*
 * getnameinfo() non-thread-safe IPv4-only implementation,
 * Address-family-independant address to hostname translation
 * (reverse DNS lookup in case of IPv4).
 *
 * This is meant for use on old IP-enabled systems that are not IPv6-aware,
 * and probably do not have getnameinfo(), but have the old gethostbyaddr()
 * function.
 *
 * GNU C library 2.0.x is known to lack this function, even though it defines
 * getaddrinfo().
 */
int
getnameinfo (const struct sockaddr *sa, int salen,
	     char *host, int hostlen, char *serv, int servlen, int flags)
{
	if ((salen < sizeof (struct sockaddr_in))
	 || (sa->sa_family != AF_INET))
		return EAI_FAMILY;
	else if (flags & ~_NI_MASK)
		return EAI_BADFLAGS;
	else
	{
		const struct sockaddr_in *addr;
		
		addr = (const struct sockaddr_in *)sa;

		if (host != NULL)
		{
			const char *hostname;
			int solved = 0;
			
			/* host name resolution */
			if (!(flags & NI_NUMERICHOST))
			{
				struct hostent *hent;

				if ((hent = gethostbyaddr (
						(const void*)&addr->sin_addr,
						4, AF_INET)) != NULL)
				{
					hostname = hent->h_name;
					solved = 1;
				}
				else if (flags & NI_NAMEREQD)
					return gai_error_from_herrno ();
			}
			if (!solved)
				hostname = inet_ntoa (addr->sin_addr);

			/* host name copy */
			strncpy (host, hostname, hostlen);
			host[hostlen - 1] = 0;
			
			if ((flags & NI_NOFQDN) && solved)
			{
				char *ptr;

				ptr = strchr (host, '.');
				if (ptr != NULL)
					*ptr = 0;
			}
		}

		if (serv != NULL)
		{
			const char *servname = NULL;
	
			/* service name resolution */
			if (!(flags & NI_NUMERICSERV))
			{
				struct servent *sent;

				sent = getservbyport(addr->sin_port,
						     (flags & NI_DGRAM)
						      ? "udp" : "tcp");
				if (sent != NULL)
					servname = sent->s_name;
			}
			/* service name copy */
			if (servname == NULL)
				snprintf (serv, servlen, "%u",
					  (unsigned int)ntohs (addr->sin_port));
			else
				strncpy (serv, servname, servlen);
			serv[servlen - 1] = 0;
		}
	}
	return 0;
}
#endif /* if !HAVE_GETNAMEINFO */


#if !HAVE_GETADDRINFO
/*
 * This functions must be used to free the memory allocated by getaddrinfo().
 */
void
freeaddrinfo (struct addrinfo *res)
{
	if (res != NULL)
	{
		if (res->ai_canonname != NULL)
			free (res->ai_canonname);
		if (res->ai_addr != NULL)
			free (res->ai_addr);
		free (res);
	}
}

/*
 * Internal function that builds an addrinfo struct.
 */
static struct addrinfo *
makeaddrinfo (int af, int type, int proto,
		const struct sockaddr *addr, size_t addrlen,
		const char *canonname)
{
	struct addrinfo *res;

	res = (struct addrinfo *)malloc (sizeof (struct addrinfo));
	if (res != NULL)
	{
		memset (res, 0, sizeof (struct addrinfo));
		res->ai_family = af;
		res->ai_socktype = type;
		res->ai_protocol = proto;
		res->ai_addrlen = addrlen;
		res->ai_addr = malloc(addrlen);
		res->ai_next = NULL; /* not needed - for clarity */
			
		if (res->ai_addr != NULL)
		{
			memcpy (res->ai_addr, addr, addrlen);
			
			if (canonname != NULL)
			{
				size_t namelen;

				namelen = strlen (canonname);
				res->ai_canonname = (char *)malloc (++namelen);
				if (res->ai_canonname != NULL)
				{
					memcpy (res->ai_canonname, canonname,
						namelen);
					return res;
				}
			}
			else
				return res;
			
			/* memory error */
			free (res->ai_addr);
		}
		free (res);
	}
	freeaddrinfo (res);
	return NULL;
}

static struct addrinfo *
makeipv4info (int type, int proto, u_long ip, u_short port, const char *name)
{
	struct sockaddr_in addr;

	memset (&addr, 0, sizeof (addr));
	addr.sin_family = AF_INET;
	addr.sin_port = port;
	addr.sin_addr.s_addr = ip;
	
	return makeaddrinfo (PF_INET, type, proto,
				(struct sockaddr*)&addr, sizeof (addr), name);
}

/*
 * getaddrinfo() non-thread-safe IPv4-only implementation
 * Address-family-independant hostname to address resolution.
 *
 * This is meant for IPv6-unaware systems that do probably not provide
 * getaddrinfo(), but still have old function gethostbyname().
 *
 * Only UDP and TCP over IPv4 are supported here.
 */		
int getaddrinfo (const char *node, const char *service,
		const struct addrinfo *hints, struct addrinfo **res)
{
	struct addrinfo *info;
	u_long ip;
	u_short port;
	int protocol = 0, flags = 0;
	const char *name = NULL;
	
	if (hints != NULL)
	{
		flags = hints->ai_flags;
		
		if (flags & ~_AI_MASK)
			return EAI_BADFLAGS;
		/* only accept AF_INET and AF_UNSPEC */
		if (hints->ai_family && (hints->ai_family != AF_INET))
			return EAI_FAMILY;

		/* protocol sanity check */
		switch (hints->ai_socktype)
		{
			case SOCK_STREAM:
				protocol = IPPROTO_TCP;
				break;

			case SOCK_DGRAM:
				protocol = IPPROTO_UDP;
				break;

			case SOCK_RAW:
			case 0:
				break;

			default:
				return EAI_SOCKTYPE;
		}
		if (hints->ai_protocol && protocol
		 && (protocol != hints->ai_protocol))
			return EAI_SERVICE;
	}

	*res = NULL;

	/* default values */
	if (node == NULL)
	{
		if (flags & AI_PASSIVE)
			ip = htonl (INADDR_ANY);
		else
			ip = htonl (INADDR_LOOPBACK);
	}
	else
	if ((ip = inet_addr (node)) == INADDR_NONE)
	{
		struct hostent *entry = NULL;
		
		/* hostname resolution */
		if (!(flags & AI_NUMERICHOST))
			entry = gethostbyname (node);
		
		if (entry == NULL)
			return EAI_NONAME;

		if ((entry->h_length != 4) || (entry->h_addrtype != AF_INET))
			return EAI_FAMILY;

		ip = *((u_long *) entry->h_addr);
		if (flags & AI_CANONNAME)
			name = entry->h_name;
	}

	/* service resolution */
	if (service == NULL)
		port = 0;
	else
	{
		long d;
		char *end;
		
		d = strtoul (service, &end, 0);
		if (end[0] /* service is not a number */
		 || (d > 65535))
		{
			struct servent *entry;
			const char *protoname;

			switch (protocol)
			{
				case IPPROTO_TCP:
					protoname = "tcp";
					break;

				case IPPROTO_UDP:
					protoname = "udp";
					break;

				default:
					protoname = NULL;
			}

			entry = getservbyname (service, protoname);
			if (entry == NULL)
				return EAI_SERVICE;

			port = entry->s_port;
		}
		else
			port = htons ((u_short)d);
	}

	/* building results... */
	if ((!protocol) || (protocol == IPPROTO_UDP))
	{
		info = makeipv4info (SOCK_DGRAM, IPPROTO_UDP, ip, port, name);
		if (info == NULL)
		{
			errno = ENOMEM;
			return EAI_SYSTEM;
		}
		if (flags & AI_PASSIVE)
			info->ai_flags |= AI_PASSIVE;
		*res = info;
	}
	if ((!protocol) || (protocol == IPPROTO_TCP))
	{
		info = makeipv4info (SOCK_STREAM, IPPROTO_TCP, ip, port, name);
		if (info == NULL)
		{
			errno = ENOMEM;
			return EAI_SYSTEM;
		}
		info->ai_next = *res;
		if (flags & AI_PASSIVE)
			info->ai_flags |= AI_PASSIVE;
		*res = info;
	}

	return 0;
}
#endif /* if !HAVE_GETADDRINFO */

