/*
   SwingWT
   Copyright(c)2003-2004, R. Rawson-Tetley

   For more information on distributing and using this program, please
   see the accompanying "COPYING" file.

   Contact me by electronic mail: bobintetley@users.sourceforge.net

   $Log: JDesktopPane.java,v $
   Revision 1.28  2004/10/30 20:11:57  bobintetley
   Code cleanup

   Revision 1.27  2004/05/11 16:36:30  bobintetley
   Reverted to tabbed emulation for now

   Revision 1.26  2004/05/11 09:41:41  bobintetley
   Fixes to JInternalFrame ordering

   Revision 1.25  2004/05/10 20:51:20  bobintetley
   Event chain for making internal frames active in their JDesktopPane when a child is clicked/receives the focus

   Revision 1.24  2004/05/10 14:33:17  bobintetley
   MDI support and JPanel border fixes

   Revision 1.23  2004/05/06 12:35:22  bobintetley
   Parity with Swing constants for Binary Compatibility + fixes to JDesktopPane

   Revision 1.22  2004/05/05 13:24:32  bobintetley
   Bugfixes and Laurent's patch for binary compatibility on Container.add()

   Revision 1.21  2004/04/28 08:38:11  bobintetley
   Hierarchy fixes, code cleanup for base classes, additional javadocs and use of flag to identify JComponent descendants with peers

   Revision 1.20  2004/03/04 15:32:28  bobintetley
   JInternalFrame methods now modify their peers after creation

   Revision 1.19  2004/01/30 10:49:49  bobintetley
   JInternalFrame/JDesktopPane fixes

   Revision 1.18  2004/01/30 10:38:10  bobintetley
   Fixes to JDesktopPane that caused frames to be added multiple times

   Revision 1.17  2004/01/26 08:11:00  bobintetley
   Many bugfixes and addition of SwingSet

   Revision 1.16  2004/01/20 07:38:05  bobintetley
   Bug fixes and compatibility methods

   Revision 1.15  2004/01/15 15:20:30  bobintetley
   Java2D work

   Revision 1.14  2003/12/16 18:42:38  bobintetley
   More thread safety

   Revision 1.13  2003/12/16 12:36:08  bobintetley
   Fix to closing of internal frames

   Revision 1.12  2003/12/16 09:19:02  bobintetley
   Various small fixes to match Swing more closely

   Revision 1.11  2003/12/15 18:29:57  bobintetley
   Changed setParent() method to setSwingWTParent() to avoid conflicts with applications

   Revision 1.10  2003/12/14 09:13:38  bobintetley
   Added CVS log to source headers

*/


package swingwtx.swing;

import swingwt.awt.*;
import swingwtx.custom.*;
import swingwtx.custom.event.*;

import java.util.*;

public class JDesktopPane extends JLayeredPane {
    
    /**
     * Cache of JInternalFrames in the JDesktopPane
     */
    protected Vector frames = new Vector();
    
    /** This value determines how the JDesktopPane is displayed. If this is
     *  set to true, a series of closable tabs are used to represent the
     *  JInternalFrames.
     *  Otherwise, "real" MDI emulation is used.
     */
    protected boolean useTabbedEmulation = true;
    /** If we are using tabbed emulation, a reference to the closable
     *  tabbed pane we are using to contain JInternalFrames
     */
    protected JClosableTabbedPane jtp = null;
    /** Cache of the currently selected frame */
    protected JInternalFrame pSelectedFrame = null;

    public JDesktopPane() { super(); }
    /** SwingWT specific - allows you to choose whether to use "real" MDI or tabbed emulation of MDI */
    public JDesktopPane(boolean useTabbedEmulation) { super(); this.useTabbedEmulation = useTabbedEmulation; }
    
    /** Overridden add method to deal with JInternalFrames */
    public Component add(final Component c) {
        
        /** If it's not a JInternalFrame, or we aren't using tabbed emulation, just
          * add the component to us as normal. */
        if (!(c instanceof JInternalFrame)) {
            super.add(c);
            return c;
        }   
        else
        {
            // Add the frame to the cache of internal frames we
            // are holding and set a reference to us on the frame itself.
            frames.add((JInternalFrame) c);
            ((JInternalFrame) c).setParentPane(this);
            
            if (useTabbedEmulation) {
                return (Component) addInternalFrameToTabbedPane((JInternalFrame) c);
            }
            else {
                // Define initial size/location
                // Size should be overriden by the subclass anyway
                Point loc = getNextFrameLocation();
                c.setBounds(loc.x, loc.y, 320, 200);
                super.add(c);
                return c;
            }   
        }
    }
    
    /** Overridden add method to deal with JInternalFrames */
    public void add(Component c, Object layoutModifier) {
        add(c);
    }
    
    /** Overridden add method to deal with JInternalFrames */
    public Component add(String name, Component c) {
        c.setName(name);
        return add(c);
    }
    
    /**
      * TABBED EMULATION ONLY: 
      * Adds a JInternalFrame to the tabbed pane */
    protected JInternalFrame addInternalFrameToTabbedPane(final JInternalFrame frame) {
        
        frame.setDrawDecoration(false); // Tabbed pane emulation doesn't want the frame to draw
                                        // window decorations.
        if (jtp == null) return frame;
        
        final JDesktopPane me = this;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (jtp != null) {
                    jtp.addTab(frame.getTitle(), frame.getFrameIcon(), frame, frame.getTitle());
                    // Set focus to the added frame
                    jtp.setSelectedIndex(jtp.getTabCount() - 1);
                }
            }
        });
        return frame;
    }
    
    /**
     * Changes the JDesktopPane to use tabbed emulation if set
     * to true. Tabbed emulation emulates an MDI interface by
     * using closable tabs to represent the JInternalFrame
     * objects inside.
     */
    public void setUseTabbedEmulation(boolean b) {
        useTabbedEmulation = b;
    }
    
    protected Point nextLocation = null;
    
    /** Returns the next location a frame should appear at. This
     *  is just a quick hack and it steps down 24 pixels in a cascade
     *  for 5 windows, before going back to 0, 0
     */
    public Point getNextFrameLocation() {
        if (nextLocation == null)
            nextLocation = new Point(0, 0);
        else if ((nextLocation.x) == 0)
            nextLocation.setLocation(24, 24);
        else if ((nextLocation.x) == 24)
            nextLocation.setLocation(48, 48);
        else if ((nextLocation.x) == 48)
            nextLocation.setLocation(72, 72);
        else if ((nextLocation.x) == 72)
            nextLocation.setLocation(0, 0);
        return nextLocation;
    }
    
    public void setSwingWTParent(swingwt.awt.Container parent) throws Exception { 
        
        descendantHasPeer = true;
        
        // Create peer
        super.setSwingWTParent(parent);
        
        // If we are using the tabbed stuff, we need to create
        // a closable tabbed pane and add it as a single child
        // of this component, then fill it with the frames.
        if (useTabbedEmulation) {

            // Add items onto a tabbed pane, filling this
            // container.
            setLayout(new BorderLayout());
            jtp = new JClosableTabbedPane();
            jtp.setTabPlacement(SwingConstants.TOP);
            add(jtp, BorderLayout.CENTER);

            Iterator i = frames.iterator();
            while (i.hasNext()) {
                JInternalFrame frame = (JInternalFrame) i.next();
                jtp.addTab(frame.getTitle(), frame.getFrameIcon(), frame, frame.getTitle());
            }

            // Listen for close events
            jtp.addTabCloseListener(new TabCloseListener() {
                public boolean tabClosed(int index) {
                    return ((JInternalFrame) frames.get(index)).processFrameClosing();
                }
            });
            
        }
        else {
            
            // If we aren't using tabbed emulation, then
            // the JInternalFrames can draw themselves on the
            // JLayeredPane - this is already taken care of in
            // the call to super.setSwingWTParent() as they are
            // regular JComponents
            
        }
    }
    
    /**
     * TABBED EMULATION ONLY:
     * Updates a frame's components from it's cached properties.
     */
    protected void refreshFrame(final JInternalFrame frame) {
        
        // If we aren't using tabbed emulation, stop now
        if (!useTabbedEmulation) return;
        
        // If no frame specified, drop out 
        if (frame == null) return;
        
        // Go onto event dispatch thread to prevent frames being
        // closed whilst we process
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                
                // Get all frames to find this one (drop out if none exist)
                final JInternalFrame[] frames = getAllFrames();
                if (frames == null) return;
                
                for (int i = 0; i < frames.length; i++)
                    if (frames[i] == frame)
                        updateFrameAt(i, frame);
            }
        });
    }
    
    /**
     * TABBED EMULATION ONLY:
     * Copies an internal frame's properties onto it's component
     */
    protected void updateFrameAt(int index, JInternalFrame frame) {
        jtp.setTitleAt(index, frame.getTitle());
        jtp.setIconAt(index, frame.getFrameIcon());
    }
    
    /** Removes an internal frame */
    protected void removeFrame(JInternalFrame frame) {
        if (useTabbedEmulation) {
            int i = frames.indexOf(frame);
            if (i != -1) {
                jtp.removeTabAt(i);
                frames.remove(frame);
            }
        }
        else {
            frames.remove(frame);
            remove(frame);
        }
    }
    
    public JInternalFrame[] getAllFrames() {
        JInternalFrame[] ret = new JInternalFrame[frames.size()];
        Object[] jf = frames.toArray();
        for (int i = 0; i < jf.length; i ++) {
            ret[i] = (JInternalFrame) jf[i];    
        }
        jf = null;
        return ret;
    }
    
    public JInternalFrame getSelectedFrame() {
        if (useTabbedEmulation) {
            if (jtp == null) return null;
            return (JInternalFrame) frames.get(jtp.getSelectedIndex());
        }
        else
        {
            // We cache the currently selected frame
            // in setSelectedFrame. If none is cached, we
            // return the first one if there is one, 
            // otherwise return null.
            if (pSelectedFrame != null)
                return pSelectedFrame;
            else if (frames.size() > 0)
                return (JInternalFrame) frames.get(0);
            else
                return null;
        }
    }
    
    public void setSelectedFrame(JInternalFrame frame) {
        if (useTabbedEmulation) {
            if (jtp == null) return;
            int i = frames.indexOf(frame);
            if (i != -1)
                jtp.setSelectedIndex(i);
        }
        else {
            // Mark the JInternalFrame as selected and
            // bring it to the front of the layered pane.
	    // 
	    // This is called back from events in JComponent
	    // when they detect they are added to a JInternalFrame
	    // that isn't using tabbed emulation
            pSelectedFrame = frame;
            moveToFront(frame);
        }
    }
    
}
