/*==================================================================
 * SwamiUIMultiList.c - User interface multi IPItem action object
 *
 * A widget to create user interface lists that deal with actions on
 * a batch of multiple sound font items.
 *
 * Swami
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Swami homepage: http://swami.sourceforge.net
 *==================================================================*/
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <gtk/gtk.h>
#include <instpatch.h>

#include "SwamiUIMultiList.h"

#include "i18n.h"

static void swamiui_multilist_class_init (SwamiUIMultiListClass *klass);
static void swamiui_multilist_init (SwamiUIMultiList *multilist);
static void destroynotify_unref_items (gpointer data);
static void cb_listbtn_clicked (GtkWidget *btn, gpointer data);

guint
swamiui_multilist_get_type (void)
{
  static guint obj_type = 0;

  if (!obj_type)
    {
      GtkTypeInfo obj_info = {
	"SwamiUIMultiList",
	sizeof (SwamiUIMultiList),
	sizeof (SwamiUIMultiListClass),
	(GtkClassInitFunc) swamiui_multilist_class_init,
	(GtkObjectInitFunc) swamiui_multilist_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };
      obj_type = gtk_type_unique (gtk_window_get_type (), &obj_info);
    }

  return obj_type;
}

static void
swamiui_multilist_class_init (SwamiUIMultiListClass *klass)
{
}

static void
swamiui_multilist_init (SwamiUIMultiList *multi)
{
  GtkWidget *box;
  GtkWidget *lbl;
  GtkWidget *frame;
  GtkWidget *scroll_win;

  gtk_object_set (GTK_OBJECT (multi), "type", GTK_WINDOW_DIALOG, NULL);
  gtk_widget_set_usize (GTK_WIDGET (multi), 360, 200);

  /* main dialog vbox */
  multi->vbox = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (multi->vbox);
  gtk_container_add (GTK_CONTAINER (multi), multi->vbox);

  /* information label */
  lbl = multi->label = gtk_label_new (NULL);
  gtk_widget_show (lbl);
  gtk_box_pack_start (GTK_BOX (multi->vbox), lbl, FALSE, FALSE, 0);

  /* frame for list and list buttons */
  frame = gtk_frame_new (NULL);
  gtk_widget_show (frame);
  gtk_box_pack_start (GTK_BOX (multi->vbox), frame, TRUE, TRUE, 0);

  /* vbox in frame */
  box = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (box);
  gtk_container_add (GTK_CONTAINER (frame), box);

  /* scroll window for list */
  scroll_win = multi->scroll_win = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll_win),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_widget_show (scroll_win);
  gtk_box_pack_start (GTK_BOX (box), scroll_win, TRUE, TRUE, 0);

  /* list button box */
  multi->list_btnbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (multi->list_btnbox);
  gtk_box_pack_start (GTK_BOX (box), multi->list_btnbox, FALSE, FALSE, 0);

  /* action button box */
  multi->action_btnbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (multi->action_btnbox);
  gtk_box_pack_start (GTK_BOX (multi->vbox), multi->action_btnbox,
		      FALSE, FALSE, 0);

  multi->ok_button = gtk_button_new_with_label (_("OK"));
  gtk_widget_show (multi->ok_button);
  gtk_box_pack_start (GTK_BOX (multi->action_btnbox), multi->ok_button,
		      TRUE, FALSE, 0);

  multi->cancel_button = gtk_button_new_with_label (_("Cancel"));
  gtk_signal_connect_object (GTK_OBJECT (multi->cancel_button), "clicked",
			     gtk_object_destroy, GTK_OBJECT (multi));
  gtk_widget_show (multi->cancel_button);
  gtk_box_pack_start (GTK_BOX (multi->action_btnbox), multi->cancel_button,
		      TRUE, FALSE, 0);
}

/**
 * Create a new multi list object
 * @title Title of dialog
 * @columns Number of columns in list
 * @first_label First column label, followed by remaining labels ...
 * Returns: new widget of type SwamiUIMultiList
 */
GtkWidget *
swamiui_multilist_new (char *title, char *ques, int columns,
		       char *first_label, ...)
{
  SwamiUIMultiList *multi;
  va_list ap;
  int i;

  g_return_val_if_fail (columns > 0, NULL);

  multi = gtk_type_new (swamiui_multilist_get_type ());
  if (title) gtk_window_set_title (GTK_WINDOW (multi), title);

  gtk_label_set_text (GTK_LABEL (multi->label), ques);

  /* create GtkCList here (rather than init), can't change column count */
  multi->clist = gtk_clist_new (columns);
  gtk_widget_show (multi->clist);
  gtk_container_add (GTK_CONTAINER (multi->scroll_win), multi->clist);

  gtk_clist_column_titles_show (GTK_CLIST (multi->clist));

  gtk_clist_set_column_auto_resize (GTK_CLIST (multi->clist), 0, TRUE);
  gtk_clist_set_column_title (GTK_CLIST (multi->clist), 0, first_label);

  va_start (ap, first_label);

  i = 0;
  while (++i < columns)
    {
      gtk_clist_set_column_auto_resize (GTK_CLIST (multi->clist), i, TRUE);
      gtk_clist_set_column_title (GTK_CLIST (multi->clist), i,
				  va_arg (ap, char *));
    }

  va_end (ap);

  return (GTK_WIDGET (multi));
}

/**
 * Set a list of patch items to use for the duration of a Multi list
 *   object
 * @multi Multi list object
 * @items List of items to reference for the duration of the multi list
 *   object. This list will be freed when multi list object is destroyed,
 *   make sure this list isn't used afterwards.
 *
 * The ref_count of each item in the list is incremented. The `items'
 * pointer in the multi list object is set to the item list.  When the
 * multi list dialog is destroyed each ref_count is removed and the
 * item list is destroyed.
 */
void
swamiui_multilist_set_items (SwamiUIMultiList *multi, GList *items)
{
  GList *p;

  g_return_if_fail (multi != NULL);
  g_return_if_fail (SWAMIUI_MULTILIST (multi));

  p = items;
  while (p)
    {
      instp_item_ref (INSTP_ITEM (p->data));
      p = g_list_next (p);
    }

  multi->items = items;

  gtk_object_set_data_full (GTK_OBJECT (multi), "_items", items,
			    destroynotify_unref_items);
}

/* callback to unref the list of items and free the list apon multi list
   object destroy */
static void
destroynotify_unref_items (gpointer data)
{
  GList *items = (GList *)data;
  GList *p;

  p = items;
  while (p)
    {
      instp_item_unref (INSTP_ITEM (p->data));
      p = g_list_next (p);
    }

  g_list_free (items);
}

/**
 * Add a list button to a multi list object
 * @multi Multi list object
 * @label Label of new button
 * @func Callback function for when button is clicked
 * @func_data Callback data
 */
void
swamiui_multilist_new_listbtn (SwamiUIMultiList *multi, char *label,
			       SwamiUIMultiListCallback func,
			       gpointer func_data)
{
  GtkWidget *btn;

  g_return_if_fail (multi != NULL);
  g_return_if_fail (SWAMIUI_IS_MULTILIST (multi));
  g_return_if_fail (label != NULL);
  g_return_if_fail (func != NULL);

  btn = gtk_button_new_with_label (label);
  gtk_widget_show (btn);
  gtk_box_pack_start (GTK_BOX (multi->list_btnbox), btn, TRUE, TRUE, 0);

  gtk_object_set_data (GTK_OBJECT (btn), "func", func);
  gtk_object_set_data (GTK_OBJECT (btn), "func_data", func_data);

  gtk_signal_connect (GTK_OBJECT (btn), "clicked",
		      GTK_SIGNAL_FUNC (cb_listbtn_clicked), multi);
}

static void
cb_listbtn_clicked (GtkWidget *btn, gpointer data)
{
  SwamiUIMultiList *multi = SWAMIUI_MULTILIST (data);
  GtkSignalFunc func;
  gpointer func_data;

  func = (GtkSignalFunc)gtk_object_get_data (GTK_OBJECT (btn), "func");
  func_data = gtk_object_get_data (GTK_OBJECT (btn), "func_data");

  (*func)(multi, func_data);
}
