inherit "/kernel/secure_mapping.pike";

#include <macros.h>
#include <database.h>
#include <attributes.h>
#include <classes.h>

string html_show_annotations(array(object) annotations)
{
    array(object) next_annotations;
    string html = "<UL>\n";
    
    foreach ( annotations, object annotation ) {
	if ( !objectp(annotation) ) continue;
	    
	next_annotations = annotation->get_annotations();
	html += "<LI><strong>"+annotation->query_attribute(OBJ_NAME)+"</strong> ( id: " + annotation->get_object_id()+") by "+
	    "<i><a href=\"JavaScript:open_userdetails('" + 
	    annotation->get_creator()->get_object_id() +"');\">"+
	    annotation->get_creator()->get_identifier() + "</a>"+
	    "</i> on "+
	    ctime(annotation->query_attribute(OBJ_CREATION_TIME))+
	    "<BR/>\n";
	switch ( annotation->query_attribute(DOC_MIME_TYPE) ) {
	case "text/plain":
	case "text/html":
	    html += annotation->get_content();
	    break;
	case "image/jpeg":
	case "image/gif":
	    html += "<image src=\"/scripts/get.pike?object="+annotation->get_object_id()+"\"/>\n";
	    break;
	default:
	    html += "<a href=\"/scripts/get.pike?object="+annotation->get_object_id()+"\">"+annotation->get_identifier()+"</a>\n";
	    break;
	}
	
	html += "</LI>\n";
	    annotation->get_content()+"</LI>\n";
	html += html_show_annotations(next_annotations);
    }
    html += "</UL>\n";
    return html;
}

string pikeWiki(object obj, object fp, string pcode)
{
  string result;

  sscanf(pcode, "@@PIKE%s@@", pcode);
  string code = "#include <macros.h>\n#include <attributes.h>\n#include <classes.h>\n#include <database.h>\nmixed exec(object env, object fp) {\n" + pcode + "\n}";
  
  object e = master()->ErrorContainer();
  master()->set_inhibit_compile_errors(e);
  mixed err = catch { 
    program prg = compile_string(code); 
    object o = new(prg);
    result = o->exec(obj, fp);
  };
  if ( err != 0 ) {
    return "<!-- error calling function:"+sprintf("%O\n%O\n",err, e->get())+"-->\n";
  }
  return result;
}

string embedWiki(object obj, object fp, string embed)
{
  string link, prefix, args;
  mapping      vars = ([ ]);
  
  if ( sscanf(embed, "{%s}", link) == 0 )
    return "<!-- unable to embed " + embed + " -->\n";

  if ( sscanf(link, "%s:%s", prefix, link) ) {
    object namespace = get_wiki_room(prefix);
    if ( objectp(namespace) )
      obj = namespace;
  }
  if ( sscanf(link, "%s$%s", link, args) == 2 ) {
    array params = args / ",";
    foreach(params, string param) {
      string key, val;
      if ( sscanf(param, "%s=%s", key, val) == 2 )
	vars[key] = val;
    }
  }

  object o = fp->resolve_path(obj, link);
  if ( !objectp(o) )
    return "<!-- object " + embed + " not found !-->\n";
  
  if ( o->get_object_class() & CLASS_SCRIPT )
    return o->execute( vars );


  string mtype = o->query_attribute(DOC_MIME_TYPE);

  if ( search(mtype, "image") >= 0 )
    return "<img src='"+fp->object_to_filename(o)+"'/>";
  
  if ( search(mtype, "text") == -1 && search(mtype, "source") == -1 )
    return "<a href='"+fp->object_to_filename(o)+"'>"+
      o->get_identifier()+"</a>\n";

  string embed_class = replace(mtype, "/", "_");

  string maint, subt;
  sscanf(mtype, "%s/%s", maint, subt);

  return "<div class='embedded'><div class='"+maint+"'><div class='"+subt+"'>"+
    o->get_content()+"</div></div></div>\n";
}

string annotationWiki(object obj, object fp, string ann) 
{
  string     text;
  object        o;
  int          id;
  
  if ( sscanf(ann, "[%s[#%d]]", text, id) == 2 ) {
    o   = find_object(id);
    if ( objectp(o) ) {
      ann = o->get_content();
      replace(ann, "\"", "'");
    }
  }
  else {
    id = time();
    sscanf(ann, "[%s[%s]]", text, ann);
  }
  
  return sprintf("<div class='annotate'><div class='annotated'>%s</div><div class='annotation'>%s</div></div>", text, ann);
}

string linkInternalWiki(object obj, object fp, string link) 
{
  string desc, prefix, space, image;
  space = "";

  if ( strlen(link) > 1 && link[0] == ' ' ) {
    link = link[1..];
    space = " ";
  }
  
  if ( sscanf(link, "[[%s|%s]]", link, desc) == 0 ) {
      sscanf(link, "[[%s]]", link);
      desc = link;
  }
  else {
  }
  if ( sscanf(link, "%s:%s", prefix, link) ) {
    object namespace = get_wiki_room(prefix);
    if ( objectp(namespace) )
      obj = namespace;
  }
  
  if ( sscanf(desc, "{%s}", image) == 1 ) 
    desc = embedWiki(obj, fp, desc);
  
  object lnk = fp->resolve_path(obj, link);
  if ( !objectp(lnk) ) {
    link += ".wiki";  
    lnk = fp->resolve_path(obj, link);
  }

  if ( !objectp(lnk) ) {
    link = fp->object_to_filename(obj) + "/"+link;
    return space+"<a class=\"notexistant\" href=\"/scripts/wikiedit.pike?path="+
      link + "\">"+desc+"</a>";
  }
  
  return space+"<a class=\"internal\" href=\""+link+"\">"+desc+"</a>" + (!objectp(lnk)?"?":"");
}

string hyperlinkWiki(object obj, object fp, string link) 
{
  string dest, desc;
  if ( sscanf(link, "[%s|%s]", dest, desc) != 2 && 
       sscanf(link, "[%s %s]", dest, desc) != 2 ) 
  {
    sscanf(link, "[%s]", link);
    dest = link;
    desc = link;
  }
  if ( sscanf(desc, "{%*s}") )
    desc = embedWiki(obj, fp, desc);
  return "<a class=\"external\" href=\""+dest+"\">"+desc+"</a>";
}

string barelinkWiki(object obj, object fp, string link) 
{
  return "<a class=\"external\" href=\""+link+"\">"+link+"</a>";
}

string imageWiki(object obj, object fp, string link) 
{
  string img, alt;
  sscanf(link, "[[image:%s|%s", img, alt);
  return "<img alt=\""+alt+"\" src=\""+img+"\">";
}

string wiki_to_html_plain(object doc, void|object fp, void|mapping vars)
{
  if ( doc->query_attribute(DOC_MIME_TYPE) != "text/wiki" )
    return "<!-- Source Document is not a wiki file !-->\n";

  if ( !objectp(fp) )
    fp = _FILEPATH;

  object WParser = wiki.Parser(this_object());
  string res = WParser->wiki_to_html(doc->get_environment(), fp, doc->get_content());
  res = "<div class='html'>"+res+"</div>";
  return res;
}

string wiki_to_html(object doc, void|object fp, void|mapping vars)
{
  string html = "<html><head>"+
    "<meta content=\"text/html; charset=utf-8\" http-equiv=\"Content-Type\">"+
    "<LINK REL=\"stylesheet\" HREF=\"/gui_css/wiki.css\"/></head><body>";
  
#if constant(wiki.Parser)

  if ( !objectp(fp) )
    fp = _FILEPATH;

  object WParser = wiki.Parser(this_object());
  string path = fp->object_to_path(doc);
  object cuser = doc->query_attribute(DOC_USER_MODIFIED);
  string content = doc->get_content();
  string result = "";
  if ( stringp(content) ) 
    result = WParser->wiki_to_html(doc->get_environment(), fp, content);
  
  html += "<div class='wiki_content'>\n"+result+"</div>\n";
  
  object wikiDoc = find_object("/documents/wiki.wiki");
  if ( objectp(wikiDoc) ) {
    html += WParser->wiki_to_html(doc->get_environment(),fp,wikiDoc->get_content());
  }
  else {
    html += "<div class='engine'><img src='/images/RoomWiki.gif' border='0'/><em>sTeam WIKI engine</em> &nbsp; "+
      "<a href='/scripts/wikiedit.pike?path="+fp->object_to_filename(doc)+"'>EditText</a>"+
      " (last changed " + ctime(doc->query_attribute(DOC_LAST_MODIFIED)) +
      (objectp(cuser) ? " by " + cuser->get_identifier() : "") + "),"+
      "&nbsp; <a href='"+path+"'>View</a> the Room or <a href='/scripts/browser.pike?_action=search'>Search</a> for Documents.";
    html += "</div>";
  }
  html += "<div class='annotations'>";
  html += html_show_annotations(doc->get_annotations());
  html += "</div></body></html>";
  return html;

#else

  return doc->get_content();

#endif
}

void register_wiki(string wiki, object room)
{
  object reg = get_value(wiki);
  if ( objectp(reg) ) 
    steam_error("Wiki namespace is already registered for " +
		reg->get_identifier());
  set_value(wiki, room);
  set_value(room->get_object_id(), wiki);
}

object get_wiki_room(string wiki)
{
  return get_value(wiki);
}

string get_room_wiki(object room)
{
  string wiki = get_value(room->get_object_id());
  return wiki;
}

string get_identifier() { return "wiki"; }







