//  libsigcperl -- a helper library for writing XSUB wrappers of libsigc++
//  Copyright (C) 2002 Ron Steinke
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Library General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Library General Public License for more details.
//
//  You should have received a copy of the GNU Library General Public
//  License along with this library; if not, write to the 
//  Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
//  Boston, MA  02111-1307  USA.

#include "slot.h"

#include <sigc++/object_slot.h>
#include <sigc++/slot.h>
#include <sigc++/bind.h>

bool SigCPerl::VoidContext(I32 flags)
{
  const I32 context_mask = G_VOID | G_SCALAR | G_ARRAY;
  return (flags & context_mask) == G_VOID;
}

SigCPerl::Slot& SigCPerl::Slot::bind(const Data &data) throw()
{
  m_data = Data(data, m_data);
  return *this;
}

SigCPerl::Data SigCPerl::Slot::call(const Data &data, I32 flags) throw()
{
  return m_slot(data, flags, m_data);
}

SigCPerl::Slot::SlotType SigCPerl::Slot::slot() const throw()
{
  // Need to copy m_slot because bind invalidates the copy it uses
  return SigC::bind(InternalType(m_slot), m_data);
}

SigCPerl::Slot SigCPerl::ParseToSlot(const Data &data) throw(Slot::BadParams)
{
  assert(data.size() >= 1);

  SV *first_arg = data[0];

  if(!sv_isobject(first_arg) || !sv_isa(first_arg, "SigC::Slot"))
    return Slot(data);

  Slot *slot = (Slot*)SvIV((SV*)SvRV(first_arg));

  if(data.size() == 1) // Only have the slot
    return *slot;

  // Bind the remaining parameters into the slot
  return Slot(*slot).bind(Data(data.begin() + 1, data.end()));
}

// stuff used in Slot constructor

namespace SigCPerl {

class FuncBox {
 public:
  FuncBox(SV *func) : m_func(newSVsv(func)) {}
  FuncBox(const FuncBox &f) : m_func(f.m_func) {SvREFCNT_inc(m_func);}
  ~FuncBox() {SvREFCNT_dec(m_func);}

  int call(SV* instance, I32 flags) const {
    if(instance) {
      //std::cerr << "Calling method on instance\n";
      return call_method(SvPV_nolen(m_func), flags);
    }
    else {
      //std::cerr << "Calling standard function\n";
      return call_sv(m_func, flags);
    }
  }

 private:
  SV* m_func;
};

static Data
callback(const Data &data, I32 flags, Data call_data,
	 FuncBox func, SV *instance) throw()
{
  //std::cerr << "Starting callback\n";

  dSP;
  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  if(instance)
    XPUSHs(sv_mortalcopy(instance));
  PUTBACK;

  //std::cerr << "Pushing " << data.size() + call_data.size() << " args\n";
  data.push_stack();
  call_data.push_stack();

  //std::cerr << "Calling perl function\n";

  int count = func.call(instance, flags);

  //std::cerr << "Called perl function\n";

  Data data_out;

  if(VoidContext(flags)) {
//    if(count != 0)
//      std::cerr << "Got a count of " << count << " from a void context call\n";
  }
  else
    data_out = Data::pop_stack(count);

  SPAGAIN;
  FREETMPS;
  LEAVE;

  //std::cerr << "Finished callback\n";

  return data_out;
}

static Slot::InternalType GetFuncSlot(const FuncBox &fbox) throw()
{
  return SigC::bind(SigC::slot(&callback), fbox, (SV*) 0);
}

class ObjectCallback : virtual public SigC::Object
{
  // This doesn't hold a reference to "instance", as it will
  // be destroyed when instance is
  ObjectCallback(SV *instance) throw() : m_instance(newRV_noinc(SvRV(instance))) {}

  Data call(const Data &data, I32 flags, Data call_data, FuncBox fbox) throw()
    {return callback(data, flags, call_data, fbox, m_instance);}

 public:
  static Slot::InternalType getSlot(SV *instance, const FuncBox &fbox)
	throw(Slot::BadParams);

 private:
  SV *m_instance;
};

} // namespace SigCPerl

SigCPerl::Slot::InternalType SigCPerl::ObjectCallback::getSlot(SV *instance,
	const FuncBox &fbox) throw(Slot::BadParams)
{
  // Make this string long and annoying. Nobody else should ever
  // touch this hash element, and we only use it in this function.
  const char key[] = "_SigC_Perl_Object_Destroy_Notify_Key_";
  const U32 key_len = sizeof(key) - 1;
  const char class_name[] = "SigC::_ObjectCallback";

  HV *hash = (HV*) SvRV(instance);
  assert(SvTYPE((SV*) hash) == SVt_PVHV);

  SV** obj_sv_ptr = hv_fetch(hash, key, key_len, 0);

  ObjectCallback *obj;

  // Insert an ObjectCallback into the hash, if there isn't one already.
  // We insert it as a SigC::Object, so that we can tell perl to call
  // its destructor in the XSUB code without sticking the ObjectCallback
  // definition in the headers.

  if(obj_sv_ptr) {
    SV* obj_sv = *obj_sv_ptr;
    if(!sv_isobject(obj_sv) || !sv_isa(obj_sv, class_name))
      throw Slot::BadParams(); // someone messed with the namespace
    obj = dynamic_cast<ObjectCallback*>((SigC::Object*) SvIV((SV*)SvRV(obj_sv)));
    if(!obj) // someone messed with the namespace
      throw Slot::BadParams();
  }
  else {
    obj = new ObjectCallback(instance);
    SV* obj_sv = newSViv(0);
    sv_setref_pv(obj_sv, class_name, (void*) static_cast<SigC::Object*>(obj));
    hv_store(hash, key, key_len, obj_sv, 0);
  }

  return SigC::bind(SigC::slot(*obj, &ObjectCallback::call), fbox);
}

SigCPerl::Slot::Slot(const Data &data) throw(BadParams)
{
  //std::cerr << "Called Slot constructor\n";

  if(data.size() < 1)
    throw BadParams();
  //std::cerr << "Got at least one argument\n";

  SV *first_arg = data[0];
  svtype type = (svtype) SvTYPE(first_arg);
  if(type == SVt_RV)
    type = (svtype) SvTYPE((SV*) SvRV(first_arg));

  switch(type) {
    case SVt_PV: // string for function name
      //std::cerr << "Got a function called " << SvPV_nolen(first_arg) << std::endl;
      if(!gv_fetchmethod(gv_stashpv("main", 0), SvPV_nolen(first_arg)))
        throw BadParams();
      // fall through
    case SVt_PVCV: // function pointer
      {
        //std::cerr << "Got a function pointer\n";
        m_slot = GetFuncSlot(first_arg);
        m_data = Data(data.begin() + 1, data.end());
        return;
      }
    case SVt_PVHV: // class instance
      {
        //std::cerr << "Got an instance\n";
        if(!sv_isobject(first_arg)) 
          throw BadParams();

        if(data.size() < 2)
          throw BadParams();
        SV *second_arg = data[1];
        if(SvTYPE(second_arg) != SVt_PV)
          throw BadParams();

        //std::cerr << "Got a method\n";
        if(!gv_fetchmethod(SvSTASH((SV*) SvRV(first_arg)), SvPV_nolen(second_arg)))
          throw BadParams();

        m_slot = ObjectCallback::getSlot(first_arg, second_arg);
        m_data = Data(data.begin() + 2, data.end());
        return;
      }
    case SVt_PVMG: // perhaps a wrapped SigC::Object?
      {
        // void *ptr = (void*)SvIV(first_arg);
        // SigC::Object *obj = dynamic_cast<SigC::Object*>(ptr);
        // FIXME do something useful for obj != 0
        throw BadParams();
      }
    default:
      throw BadParams();
  }
}
