/* sort.c.  The dialog box can only be open for one file at a time,
  but in the middle of editing one file another file could do a 
  column sort */

#include "includes.h"
#include "globals.h"

#define FROM_LIST_WIDTH   180
#define FROM_LIST_HEIGHT  180

typedef struct {
  union { 
    double number;
    char* string;
  } level [MAX_SORT_NESTING];
  int old_row;
} sort_row; 

sort_row* big_array;

sort_info curr_sort;
sort_info edit_sort;
char select_mode;
gint levels; /* how many sort levels there are */
gint level_type [MAX_SORT_NESTING + 1];
gint level_ascending [MAX_SORT_NESTING + 1]; /* +1 or -1 */
gint level_formatting [MAX_SORT_NESTING +1];

/* prototypes */
void ascending_clicked (GtkObject *object, gpointer *entry);
void build_report_box ();
void cancel_clicked (GtkObject *object, gpointer *entry);
void descending_clicked (GtkObject *object, gpointer *entry);
void display_edit_window ();
void do_the_nasty_sort ();
void field_change (GtkObject *object, gpointer *entry);

GtkWidget* field_menu (gint level);
void ok_clicked (GtkObject *object, gpointer *entry);
static  void select_ok (GtkWidget *widget, gint row, 
		     gint col, GdkEventButton *event, gpointer data);
gint sort_compare (const void* fir, const void* sec);


/* _________________________________
   |                                |
   |      ascending_clicked         |
   |________________________________|
   User clicked ascending at some level */
void ascending_clicked (GtkObject *object,  gpointer *entry){
  gint level = (gint) entry; 
  edit_sort.line [level].ascending = TRUE;
}


/* _________________________________
   |                                |
   |    build_select_box            |
   |________________________________| */
void build_select_box () {
  gint sortx;
  char* newrow [2];
   GtkWidget *top_vbox;
  
  GtkWidget *hbox2; /* clists and header */
  GtkWidget *scrwin_from;
  
  GtkWidget *hbox6; /* same as action area */
  GtkWidget *cancel_button;
  
  gchar* from_titles [1] = {"Double-click to select a sort"};
  make_basic_dialog1 ();

  gtk_window_set_title (GTK_WINDOW(dialog1_win), 
			_("Select a sort"));
  top_vbox = GTK_DIALOG(dialog1_win) -> vbox;
  gtk_box_set_spacing (GTK_BOX (top_vbox), 10);
  hbox6 = GTK_DIALOG (dialog1_win) -> action_area;

  /* now the area containing clist of fields that can be added */
  /*  The horizontal box containing the from and to listboxes  */
  hbox2 = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX (top_vbox), hbox2, TRUE, TRUE, 0);

  /* The sort names clist */
  front->clist_from = gtk_clist_new_with_titles (1, from_titles);     

  gtk_clist_set_column_width (GTK_CLIST (front->clist_from), 0, 
			       FROM_LIST_WIDTH - 30);  
  gtk_widget_set_usize (front->clist_from, FROM_LIST_WIDTH, 
			FROM_LIST_HEIGHT);
   gtk_signal_connect (GTK_OBJECT (front->clist_from), "select_row",
		      (GtkSignalFunc) select_ok, NULL);
  gtk_clist_column_titles_passive (GTK_CLIST (front->clist_from));

  scrwin_from = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER (scrwin_from), front->clist_from);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrwin_from),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_container_set_border_width (GTK_CONTAINER (scrwin_from), 5);
  gtk_box_pack_start (GTK_BOX (hbox2), scrwin_from, TRUE, TRUE, 0);

  cancel_button = gtk_button_new_with_label (_("Cancel"));
  gtk_box_pack_start (GTK_BOX (hbox6),
		      cancel_button, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT(cancel_button),"clicked", 
		      GTK_SIGNAL_FUNC (cancel_level1), NULL);
  gtk_signal_connect (GTK_OBJECT  (dialog1_win),
		      "delete_event",
		      (GtkSignalFunc) cancel_level1, NULL);

  /* now populate the clist of sort names */
  newrow [1] = NULL;
  for (sortx = 0; sortx < front->sort_ct; sortx++) {
    newrow [0] = front->sorts [sortx].name;
    gtk_clist_append (GTK_CLIST (front->clist_from), newrow);
  }
  gtk_widget_show_all (dialog1_win);  
} /* End of build_select_box */ 

/* _________________________________
   |                                |
   |        cancel_clicked          |
   |________________________________|
   User clicked cancel button in editing dialog box */
void cancel_clicked (GtkObject *object, gpointer *entry) {;}


/* _________________________________
   |                                |
   |     descending_clicked         |
   |________________________________|
   User clicked ascending at some level */
void descending_clicked (GtkObject *object, gpointer *entry){
  gint level = (gint) entry; 
  edit_sort.line [level].ascending = FALSE;
}

/* ________________________________
  |                                |
  |      display_edit_window       |
  |________________________________| */
void display_edit_window () {
  GtkWidget *ok_button,
    *cancel_button,
    *top_box,
    *button_box,
    *temp_box,
    *hbox1,
    *name_label;
  gint sortx;

  /* let's clean up the sort rules for display purposes.  Even there may be 
     only two lines, all six display.  For that reason, add 1 to all entries
     so that 0 displays as None and 1 displays field 0 */
  for (sortx = 0; sortx < MAX_SORT_NESTING; sortx++) {
    if (sortx < edit_sort.line_ct)
      edit_sort.line [sortx].field++;
    else {
      edit_sort.line [sortx].field = 0;
      edit_sort.line [sortx].ascending = TRUE;}
  }

  make_basic_dialog1 ();  
  gtk_window_set_title (GTK_WINDOW (dialog1_win), "Add or edit sort rules");
  top_box = GTK_DIALOG(dialog1_win) -> vbox;
  button_box = GTK_DIALOG (dialog1_win) -> action_area;
  gtk_box_set_spacing (GTK_BOX (top_box), 5);
  gtk_signal_connect (GTK_OBJECT  (dialog1_win),
		      "delete_event",
		      (GtkSignalFunc) cancel_level1, NULL);
  /* Show and allow changes to sort name */
  hbox1 = gtk_hbox_new (FALSE, 5);
  name_label = gtk_label_new (_("Sort Name"));
  gtk_box_pack_start (GTK_BOX (hbox1), name_label, FALSE, FALSE, 0);
  
  front->entry1 = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (hbox1), front->entry1, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (top_box), hbox1, FALSE, FALSE, 0); 
  gtk_entry_set_text (GTK_ENTRY (front->entry1), edit_sort.name);

  /* make six entries for levels of sorting */
  for (sortx = 0; sortx < MAX_SORT_NESTING; sortx++) {
    temp_box = field_menu (sortx);
    gtk_container_add (GTK_CONTAINER (top_box), temp_box);
  }

  ok_button = gtk_button_new_with_label (_("OK"));
  gtk_signal_connect (GTK_OBJECT (ok_button), 
                      "clicked", 
                      GTK_SIGNAL_FUNC (ok_clicked), NULL);

  cancel_button = gtk_button_new_with_label (_("Cancel"));
  gtk_signal_connect (GTK_OBJECT (cancel_button), 
                      "clicked",
                      GTK_SIGNAL_FUNC (cancel_level1), NULL);

  gtk_container_add (GTK_CONTAINER (button_box), ok_button);
  gtk_container_add (GTK_CONTAINER (button_box), cancel_button);
  gtk_widget_show_all (dialog1_win);
} /* end of display_edit_window */

/* ________________________________
  |                                |
  |      do_the_nasty_sort         |
  |________________________________|
  This function does this:
  1.  Build an array of elements to sort
  2.  Sort them
  3.  Copy the sorted rows to the end of the sheet
  4.  Delete the original sorted rows  */
void do_the_nasty_sort () {

  gint fieldx;
  gint rows, rowx, fromrow, torow;
  gboolean visible, hold_visible;
  gint  colx;
  gint levelx;
  gint local_last_field;
  char* text;
  char* hold1row [MAX_FIELDS + 2];

  for (fieldx = 0; fieldx < MAX_FIELDS; fieldx++)
    hold1row [fieldx] = NULL;
  local_last_field = front->last_field;
  rows = front->last_row; /* omits last row, which is blank */
  levels = curr_sort.line_ct;
  if (rows < 2)
    return;
  for (levelx = 0; levelx < levels; levelx++) {
    fieldx = curr_sort.line [levelx].field;
    curr_sort.line [levelx].column = front->fields [fieldx].sheet_column;
    level_type [levelx] = front->fields [fieldx].type;
    level_formatting [levelx] = front->fields [fieldx].formatting;
    if (curr_sort.line [levelx].ascending)
      level_ascending [levelx] = 1;
    else
      level_ascending [levelx] = -1;
  }

  /* go get enough storage to sort this mess */
  big_array = (sort_row*) g_malloc (rows * sizeof (sort_row));

  /* Start the big loop to build array that qsort will sort */
  for (rowx = 0; rowx < rows; rowx++) {
    big_array [rowx].old_row = rowx;
    for (levelx = 0; levelx < levels; levelx++) {
       text = gtk_sheet_cell_get_text (GTK_SHEET (front->sheet), rowx,
				    curr_sort.line [levelx].column);
       if (text) 
	 switch (level_type [levelx]) {
	 case FIELD_TYPE_TEXT: 
	   big_array [rowx].level [levelx].string = text;
	   break;
	 default:
	   big_array [rowx].level [levelx].number = 
	     qls2d (text, level_type [levelx], 
		    level_formatting [levelx]);
	   break;
        }
       else {
	 if (level_type [levelx] == FIELD_TYPE_TEXT)
	   big_array [rowx].level [levelx].string = NULL;
	 else
	   big_array [rowx].level [levelx].number = HUGE_VAL;
       }
    }
  } /* end of building array of pointers */

  /* this is sure easy.  sort them */
  qsort (big_array, (size_t) rows, sizeof (sort_row), sort_compare);

  /* rows are rearranged.  move them in sheet */ 
  big_draw_start ();

  /* the plan is to move aside one row, then move in the row that goes 
     there, then move in the next row into the vacated position, and so on */
  for (rowx = 0; rowx < rows; rowx++) {
    if (big_array [rowx].old_row < 0) /* already moved */
      continue;
    if (rowx ==  big_array [rowx].old_row)  /* this row doesn't move */
      continue;

    /* save this first row */
    hold_visible = row_is_visible (rowx);
   for (colx = 0; colx <= local_last_field; colx++) {
      text = gtk_sheet_cell_get_text (GTK_SHEET (front->sheet), 
				      rowx, colx);
      if (text) {
	 hold1row [colx] = g_strdup (text);
	 gtk_sheet_cell_clear (GTK_SHEET (front->sheet), rowx, colx);
      }
    }
    torow= rowx;
    fromrow = big_array [rowx].old_row;
    big_array [rowx].old_row = -1; 

    /* copy row that goes here from where it is now */
    do {
       visible = row_is_visible (fromrow);
       gtk_sheet_row_set_visibility (GTK_SHEET (front->sheet), 
				     torow, visible);
       for (colx = 0; colx <= local_last_field; colx++) {
	 text = gtk_sheet_cell_get_text (GTK_SHEET (front->sheet), 
					 fromrow, colx);
	 if (text) { 
	   gtk_sheet_set_cell_text (GTK_SHEET (front->sheet), torow, colx,
				    text);
	   gtk_sheet_cell_clear (GTK_SHEET (front->sheet), fromrow, colx);
	 }
      }
      torow = fromrow;
      fromrow = big_array [fromrow].old_row;
      big_array [torow].old_row = -1;
    }
    while (big_array [fromrow].old_row >= 0);
    
    /* got to the end of a chain, put back the stored row */
     gtk_sheet_row_set_visibility (GTK_SHEET (front->sheet), 
				     torow, hold_visible);
     for (colx = 0; colx <= local_last_field; colx++) {
       if (hold1row [colx]) {
	 gtk_sheet_set_cell_text (GTK_SHEET (front->sheet), torow, colx,
				  hold1row [colx]);
	 g_free (hold1row [colx]);
	 hold1row [colx] = NULL;
      }
    }
  } /* end of big looop to move all rows */
  g_free (big_array);  
  big_draw_end ();
  front_is_changed ();
} /* end of do_the_nasty_sort () */


/* _________________________________
   |                                |
   |      field_change               |
   |________________________________|
   User clicked to change a field at some level */
void field_change (GtkObject *object, gpointer *entry){
  gint level = (gint) entry / 100; 
  gint fieldx = (gint) entry % 100;
  edit_sort.line [level].field = fieldx;
}

/* ________________________________
  |                                |
  |          field_menu            |
  |________________________________|
  Build the field button for the sort dialog box  */
GtkWidget* field_menu (gint level) {
  GtkWidget *temp_box,
    *ascend_button,
    *descend_button,
    *menu,
    *menu_item,
    *name_label;
  gint ret_value;
  gint fieldx;
  GtkWidget *field_button;

  menu = gtk_menu_new ();
  menu_item = gtk_menu_item_new_with_label ("-Unused-");
  gtk_widget_show (menu_item);
  ret_value = level * 100;
  gtk_signal_connect (GTK_OBJECT (menu_item),
			"select",
			GTK_SIGNAL_FUNC (field_change), 
			(gpointer) ret_value);
    gtk_menu_append (GTK_MENU(menu), menu_item);

    /* add the individual fields to the menu */
    for (fieldx = 0; fieldx <= front->last_field; fieldx++) {
      menu_item = gtk_menu_item_new_with_label (front->fields [fieldx].name);
      gtk_widget_show (menu_item);
      ret_value = level*100 + fieldx + 1;
	gtk_signal_connect (GTK_OBJECT (menu_item),
			    "select",
			    GTK_SIGNAL_FUNC (field_change), 
			    (gpointer) ret_value);
      gtk_menu_append (GTK_MENU(menu), menu_item);  
    }
    
    field_button = gtk_option_menu_new ();
    gtk_option_menu_set_menu (GTK_OPTION_MENU (field_button), menu);
    if (edit_sort.line [level].field >= 0)
      gtk_option_menu_set_history (GTK_OPTION_MENU (field_button), 
				   edit_sort.line [level].field);

   /* Pack Widgets into boxes */
   temp_box = gtk_hbox_new (FALSE, 5);
   
   /* now ascending and descending buttons */
   /* ad_vbox = gtk_vbox_new (FALSE, 5); */
   ascend_button = gtk_radio_button_new_with_label (NULL, _("Ascending"));
   descend_button = 
     gtk_radio_button_new_with_label (gtk_radio_button_group 
				      (GTK_RADIO_BUTTON (ascend_button)),
				      _("Descending"));
   if (edit_sort.line [level].ascending)
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ascend_button), 1);
   else
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (descend_button), 1);
   
   /* Callbacks for radio buttons */
   gtk_signal_connect (GTK_OBJECT (ascend_button), "clicked", 
		       GTK_SIGNAL_FUNC (ascending_clicked), (gpointer) level);
   gtk_signal_connect (GTK_OBJECT (descend_button), "clicked",
		       GTK_SIGNAL_FUNC (descending_clicked),
		       (gpointer) level);
   
   gtk_box_pack_end (GTK_BOX (temp_box), descend_button, FALSE,
		      FALSE, 0);
   gtk_box_pack_end (GTK_BOX (temp_box), ascend_button, FALSE,
		       FALSE, 0);
   gtk_box_pack_end (GTK_BOX (temp_box), field_button, FALSE, FALSE, 0);
   
   /* now right justify the name label */
   if (level)
     name_label = gtk_label_new (_("then on: "));
   else
     name_label = gtk_label_new (_("Primary sort on: "));
   gtk_box_pack_end (GTK_BOX (temp_box), name_label, FALSE, FALSE, 0);

   return (temp_box);
} /* end of field_menu () */


/* _________________________________
   |                                |
   |        ok_clicked              |
   |________________________________| 
  User clicked OK to finish editing a sort layout.  What we do now
  depends on how we got here.*/
void ok_clicked (GtkObject *object, gpointer *entry) {
  char* text;
  gint sortx;
  gboolean gotone;
  
  /* save the changed name */
  text = gtk_entry_get_text (GTK_ENTRY (front->entry1));
  if (check_entry (text))
    return;
  front_is_changed ();
  strcpy (edit_sort.name, text);

  /* check to see if even one field.  Don't start moving until we
     get past this point */
  gotone = FALSE;
  for (sortx = 0; sortx < MAX_SORT_NESTING; sortx++)
    if (edit_sort.line [sortx].field > 0)
      gotone = TRUE;

  if (!gotone) {
    level2_error ("You must specify at least one column to sort on.",
		  "Go back");
    return;
  }

  /* there is at least one field.  Now clean up and convert all back to valid
     values. */
  edit_sort.line_ct = 0;
  for (sortx = 0; sortx < MAX_SORT_NESTING; sortx++) 
    if (edit_sort.line [sortx].field-- > 0) 
      edit_sort.line [edit_sort.line_ct++] = edit_sort.line [sortx];
 
  curr_sort = edit_sort;
  destroy_dialog ();
  do_the_nasty_sort ();
 
 if (select_mode == 'A')  /* add */
   front->sort_ct++;
 *front->sort_ptr = curr_sort;
 dim_list_sort_menu ();
 return;
}


/* _________________________________
   |                                |
   |      select_ok                 |
   |________________________________| 
  User clicked OK to select a specific sort layout.  What we do now
  depends on how we got here.*/
static  void select_ok (GtkWidget *widget, gint row, 
		     gint col, GdkEventButton *event, gpointer data) {
  gint sortx;
  gint reportx;

  if (event->type != GDK_2BUTTON_PRESS)
    return;
      
  front->sort_ptr = &front->sorts [row];
  destroy_dialog ();
  if (select_mode == 'E') {
    edit_sort = front->sorts [row];
    display_edit_window ();
    return;
  }
  if (select_mode == 'S') {
    curr_sort = front->sorts [row];
    do_the_nasty_sort ();
    return;
  }
  /* have to do delete here */ 
  for (sortx = row; sortx < front->sort_ct - 1; sortx++)
     front->sorts [sortx] = front->sorts [sortx + 1];
  front->sort_ct--;

    /* now go through reports */
  for (reportx = 0; reportx < front->report_ct; reportx++) {
    if (front->reports [reportx].sort == row)
      front->reports [reportx].sort = -1;
    else if (front->reports [reportx].sort > row)
      front->reports [reportx].sort--;
  }
  dim_list_sort_menu ();
}


/* ________________________________
  |                                |
  |        sort_add                |
  |________________________________|
  Add a new sort to the file */
void sort_add (GtkWidget* object, gpointer *data) {
  if (check_if_changed ())
      return;
  memset (&edit_sort, '\0', sizeof (sort_info));
  strcpy (edit_sort.name, "Untitled sort");
  select_mode = 'A';
  front->sort_ptr = &front->sorts [front->sort_ct]; /* add at end */
  display_edit_window ();
}
 
/* ________________________________
  |                                |
  |        sort_do_it              |
  |________________________________|
  Called by report.  Just sort the file */
void sort_do_it (gint sortno) {
  if (check_if_changed ())
    return;
  curr_sort = front->sorts [sortno];
  do_the_nasty_sort (); 
}

/* ________________________________
  |                                |
  |        sort_compare            |
  |________________________________|
  This routine compares two big_array elements to return hi, low, equal  */
gint sort_compare (const void* fir, const void* sec) {
  char* local_string1;
  char* local_string2;
  double local_double1;
  double local_double2;
  gint levelx;
  gint temp_return;
  double temp_double;
  sort_row *first = (sort_row *)(fir), *second = (sort_row *)(sec);
  for (levelx = 0; levelx < levels; levelx++) {
    if (level_type [levelx] == FIELD_TYPE_TEXT) {
      local_string1 = first->level [levelx].string;
      local_string2 = second->level [levelx].string;
      if (!local_string1) {
	if (!local_string2)
	  continue; /* they are equal */
	return (level_ascending [levelx]); /* left is less than right */
      }
      /* first is not zero, maybe second is */
      if (!local_string2)
	return (-level_ascending [levelx]);
      
      temp_return = strcmp (local_string1, local_string2);
      if (temp_return)
	return (level_ascending [levelx] * temp_return);
    }
    
    else { /* all numeric types */
      local_double1 = first->level [levelx].number;
      local_double2 = second->level [levelx].number;   
       if (local_double1 == HUGE_VAL) {
	if (local_double2 == HUGE_VAL)
	  continue; /* they are equal */
	return (level_ascending [levelx]); /* left is less than right */
      }
      /* first is not zero, maybe second is */
      if (local_double2 == HUGE_VAL)
	return (-level_ascending [levelx]);   
      temp_double =  first->level [levelx].number - 
	second->level [levelx].number;
      if (temp_double > 0)
	return (level_ascending [levelx]);
      if (temp_double < 0)
	return (-level_ascending [levelx]);
    }  
  }
    
  /* got through everything.  They must be equal */
  return (0);
}

 
/* ________________________________
  |                                |
  |        sort_by_column          |
  |________________________________| */
void sort_by_column (GtkWidget *w, gpointer *data) {
  GtkSheetRange range;
  if (check_if_changed ())
      return;
  memset (&curr_sort, '\0', sizeof (sort_info));
  curr_sort.line_ct = 1;
  curr_sort.line [0].field = front->col_to_field [front->sel_range.col0];
  curr_sort.line [0].ascending = TRUE;
  do_the_nasty_sort (); 
  range.col0 = range.coli = front->sel_range.col0;
  range.row0 = range.rowi = 0;
  gtk_sheet_select_range (GTK_SHEET (front->sheet), &range);
}

/* _________________________________
   |                                |
   |         sort_delete            |
   |________________________________| */
void sort_delete (GtkWidget *w, gpointer *data){
  if (check_if_changed ())
    return;
  select_mode = 'D';
  build_select_box ();
}


/* _________________________________
   |                                |
   |         sort_edit              |
   |________________________________| */
void sort_edit (GtkWidget *w, gpointer *data){
  if (check_if_changed ())
    return;  
  select_mode = 'E';
  build_select_box ();
}

/* _________________________________
   |                                |
   |         sort_apply             |
   |________________________________| */
void sort_apply (GtkWidget *w, gpointer *data){
  if (check_if_changed ())
    return;  
  select_mode = 'S';
  build_select_box ();
}








