/* *************************************************************************
  Module:        pcommands.c
  Author:        Matt Simpson
                 Arlington, TX
                 matthewsimpson@home.com
  Date:          August, 2000
  Description:
                 Printer commands. 
  Changes:

****************************************************************************
                 COPYRIGHT (C) 1999, 2000 Matt Simpson
                 GNU General Public License
                 See lexgui.c for full notice.
****************************************************************************
                 Portions of this file include printer calls
                 from the 'Lexmark Printer Technical Reference'
                 manual, Version 1.1, February 1999. With permission.

                 Statement from Lexmark:
                 All printer command strings in this application have been
                 released for public distribution.

                 The commands I am talking about are all PJL commands
                 as well as the string in the uel() function. 
**************************************************************************** */

#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include "lexgui.h"

/* -------------------------------------------------------------------------
        uel() Issues universal exit command.
   ------------------------------------------------------------------------- */
void uel(FILE *fp)
{
  fprintf(fp, "%c%%-12345X", 27);
}
/* -------------------------------------------------------------------------
        cr() Prints a carriage return if output is a file
   ------------------------------------------------------------------------- */
void cr(FILE *fp)
{
  extern int devtype;
  if(devtype == 0)
     fprintf(fp, "\n");
} 
/* -------------------------------------------------------------------------
        print_PS_fonts()
   ------------------------------------------------------------------------- */
void print_PS_fonts(topwin_struct *top, msgbox_struct *msgbox)
{
  extern FILE *fp;
  gchar *command_str;

  command_str = get_family_command(1, top->prefs.family, msgbox);

  if(strcmp(command_str, "\0"))
  {
    uel(fp);
    fprintf(fp, "%s\n", command_str);
    uel(fp);
    cr(fp);
    if(!flush_dev(msgbox))
      put_msg(msgbox, "Sent print PS fonts command.", GREEN, 0);
  }
}
/* -------------------------------------------------------------------------
        print_PCL_fonts()
   ------------------------------------------------------------------------- */
void print_PCL_fonts(topwin_struct *top, msgbox_struct *msgbox)
{
  extern FILE *fp;
  gchar *command_str;

  command_str = get_family_command(2, top->prefs.family, msgbox);

  if(strcmp(command_str, "\0"))
  {
    uel(fp);
    fprintf(fp, "%s\n", command_str);
    uel(fp);
    cr(fp);
    if(!flush_dev(msgbox))
      put_msg(msgbox, "Sent print PCL fonts command.", GREEN, 0);
  }
}
/* -------------------------------------------------------------------------
        print_menu1()
   ------------------------------------------------------------------------- */
void print_menu1(topwin_struct *top, msgbox_struct *msgbox)
{
  extern FILE *fp;
  gchar *command_str;

  command_str = get_family_command(0, top->prefs.family, msgbox);

  if(strcmp(command_str, "\0"))
  {
    uel(fp);
    fprintf(fp, "%s\n", command_str);
    uel(fp);
    cr(fp);
    if(!flush_dev(msgbox))
      put_msg(msgbox, "Sent the print settings command.", GREEN, 0);
  }
}
/* -------------------------------------------------------------------------
        print_demo()
   ------------------------------------------------------------------------- */
void print_demo(topwin_struct *top, msgbox_struct *msgbox)
{
  extern FILE *fp;
  gchar *command_str;

  command_str = get_family_command(3, top->prefs.family, msgbox);

  if(strcmp(command_str, "\0"))
  {
    uel(fp);
    fprintf(fp, "%s\n", command_str);
    uel(fp);
    cr(fp);
    if(!flush_dev(msgbox))
      put_msg(msgbox, "Sent the demo command.", GREEN, 0);
  }
}
/* -------------------------------------------------------------------------
        send_commands() Issues query commands to printer.
                        The output device was already determined
                        to be a device and not a file before this
                        function is called.
   ------------------------------------------------------------------------- */
int send_commands(io_struct *io)
{
  extern FILE *fp;

  clear_junk();

  if(opendev(&(io->msgbox), 2))
    return(1);

  /* Note all PJL command sets must be preceeded and followed by a UEL */
  uel(fp);
  fprintf(fp, "@PJL ECHO PUP_START\n"); /* Test start string */

  switch (io->command)
  {
    case 0:
      /* Note-- If these are changed, also change the *info[] string
         in put_text() */
      fprintf(fp, "@PJL INFO ID\n");
      fprintf(fp, "@PJL INFO STATUS\n");
      fprintf(fp, "@PJL INFO CONFIG\n");
      fprintf(fp, "@PJL INFO PAGECOUNT\n");
      fprintf(fp, "@PJL INFO VARIABLES\n");
      break;
    case 1:
      uel(fp); /* termination for ECHO PUP_START above */
      fprintf(fp, "@PJL ENTER LANGUAGE = POSTSCRIPT\n");
      fprintf(fp, "%c", 0x14); /* ctrl-T */
      fprintf(fp, "%c", 0x4);  /* ctrl-D */
      uel(fp); /* preceeding for ECHO PUP_END below. */
      break;
    case 2:
      uel(fp); /* termination for ECHO PUP_START above */
      fprintf(fp, "@PJL ENTER LANGUAGE = POSTSCRIPT\n");
      fprintf(fp, "%%!PS-Adobe-3.0\n");
      fprintf(fp, "serverdict begin 0 exitserver systemdict /quit get exec");
      /*fprintf(fp, "%c", 0x3);*/ /* ctrl-C */
      fprintf(fp, "%c", 0x4); /* ctrl-D */
      uel(fp); /* preceeding for ECHO PUP_END below. */
      break;
    case 3: /* Query for dynamic setting of defaults */
      fprintf(fp, "@PJL INFO VARIABLES\n");
      break;
  }

  fprintf(fp, "@PJL ECHO PUP_END\n"); /* Test end string */
  uel(fp);

  if(flush_dev(&(io->msgbox)))
  {
    closedev();
    return(1);
  }
  closedev();
  return(0);
} 
/* -------------------------------------------------------------------------
        reset_factory() Sends command to reset printer to factory defaults.
   ------------------------------------------------------------------------- */
void reset_factory(io_struct *io)
{
  extern FILE *fp;

  if(opendev(&(io->msgbox), 2))
    return;
  uel(fp);
  fprintf(fp, "@PJL INITIALIZE\n");
  uel(fp);
  cr(fp);
  if(!flush_dev(&(io->msgbox)))
    put_msg(&(io->msgbox), "Factory Defaults Applied !", GREEN, 0);
  closedev();
  return;
}
/* -------------------------------------------------------------------------
        adtimer() Timer for applying user selected changes. Looks for a change
                  to make, then exits to come back on the next time interval
                  for the next change.
   ------------------------------------------------------------------------- */
gint adtimer(io_struct *io)
{
  gint i, lookfor;
  dc_struct *dc;
  static gchar COMMAND[8];

  dc = io->dc_ptr;

  /* The part after the || is not really necessary but is included for safety */
  if(dc->numchanged >= dc->numtochange ||
     dc->ad_count >= dc->gcount)
  {
    uel(dc->fp); /* Ending UEL */
    cr(dc->fp);
    flush_dev(dc->msgbox_ptr);
    closedev();
    io->busy = 0;
    io->command = 3;
    if(!dc->fixq) /* if dynamic */
      send_get_build(io);
    else
      build_choices(io);
    return(FALSE); /* returning FALSE will automatically remove the timer */
  }

  put_query_message(dc->msgbox_ptr, "  Sending Data  ");
  lookfor = 1;
  while(lookfor)
  {
    i = dc->ad_count;
    if(dc->gr[i].changed) /* we are only interested in the changed ones */
    {

#ifdef NOTUSED
      if(dc->gr[i].sonly == 1)
        strcpy(COMMAND, "SET\0");
      else
#endif
        strcpy(COMMAND, "DEFAULT\0");

      if(dc->gr[i].choicetype == 0) /* enumerated choice */
      {
        fprintf(dc->fp, "@PJL %s %s=%s\n", 
                COMMAND, dc->gr[i].name, dc->gr[i].new);
      }
      else /* range choice */
      {
        if(dc->gr[i].rtype == 0)
        {
          fprintf(dc->fp, "@PJL %s %s=%d\n", COMMAND, dc->gr[i].name,
                          (int)(GTK_ADJUSTMENT(dc->gr[i].adjustment))->value);
        }
        else
        {
          fprintf(dc->fp, "@PJL %s %s=%.2f\n", COMMAND, dc->gr[i].name,
                          (GTK_ADJUSTMENT(dc->gr[i].adjustment))->value);
        }
      }
      dc->numchanged++;
      lookfor = 0; /* since change made, break out to use the timer delay */
    }
    else
      dc->ad_count++; /* increment ad_count and continue search */
    if(dc->ad_count >= dc->gcount) /* for safety */
      lookfor = 0; 
  }
  /* Made a change so increment ad_count for next and exit timer with a TRUE
     so it will come back again after waiting until the next time interval. */ 
  dc->ad_count++;
  return(TRUE);
}
/* -------------------------------------------------------------------------
        adtimer_call() Calls adtimer() with timeout. This effectively
                       slows down the process to allow the printer
                       to digest the commands. 
   ------------------------------------------------------------------------- */
void adtimer_call(io_struct *io)
{
  if(io->clock[3])
  {
    gtk_timeout_remove(io->clock[3]);
    io->clock[3] = 0;
  }
  io->busy = 1;
  /* call adtimer every 100 ms */
  io->clock[3] = gtk_timeout_add(100, (GtkFunction)adtimer, io);
}
/* -------------------------------------------------------------------------
        apply_defaults() Sends commands to set new defaults based on
                         choices picked. 
   ------------------------------------------------------------------------- */
void apply_defaults(io_struct *io)
{
  gint i;
  extern FILE *fp;
  dc_struct *dc;

  dc = io->dc_ptr;
  dc->numtochange = 0;
  dc->numchanged = 0;
  dc->ad_count = 0;

  /* Determine the number of changes */
  for(i = 0; i < dc->gcount; i++)
  {
    if(dc->gr[i].changed)
      dc->numtochange++;
  }
  /* Make the changes */
  if(dc->numtochange)
  {
    if(opendev(dc->msgbox_ptr, 2))
      return;
    dc->fp = fp;
    uel(dc->fp); /* Beginning UEL */
    adtimer_call(io);
  }
  else
  {
    put_msg(dc->msgbox_ptr, "Nothing to Change!", RED, 0);
    gtk_widget_set_sensitive(dc->ac_button, FALSE);
    gtk_widget_set_sensitive(dc->fc_button, FALSE);
  }
}
