#!/bin/sh
# the next line restarts using wish \
exec wish "$0" "$@"

# PgAccess versioning
set PgAcVar(MAIN_VERSION) "0.98.8"
set PgAcVar(LAST_BUILD) "20030520"

proc get_version {} {
	# on weekly/beta release
	return "$::PgAcVar(MAIN_VERSION).$::PgAcVar(LAST_BUILD)"
	
	# in devel cvs
	# return "$::PgAcVar(MAIN_VERSION)+.dev(cvs)"
}

image create bitmap dnarw -data  {
#define down_arrow_width 15
#define down_arrow_height 15
static char down_arrow_bits[] = {
	0x00,0x80,0x00,0x80,0x00,0x80,0x00,0x80,
	0x00,0x80,0xf8,0x8f,0xf0,0x87,0xe0,0x83,
	0xc0,0x81,0x80,0x80,0x00,0x80,0x00,0x80,
	0x00,0x80,0x00,0x80,0x00,0x80
	}
}

#----------------------------------------------------------
# registerPlugin --
#
#    Takes care of registering the plugin
#
# Arguments:
#    name_    the name to appear in the plugins menu
#    cmd_     the command to invoke the plugin
#
# Results:
#    none returned.
#----------------------------------------------------------
#
proc registerPlugin {name_ cmd_} {

    if {![info exists ::PgAcVar(plugin,list)]} {
        set ::PgAcVar(plugin,list) [list]
    }
   
    lappend ::PgAcVar(plugin,list) $name_
    set ::PgAcVar(plugin,$name_) $cmd_

    return
    
}; # end proc registerPlugin

proc say {msg} {
	tk_messageBox -message $msg
}

proc intlmsg {msg} {
global PgAcVar Messages
	if {$PgAcVar(pref,language)=="english"} { return $msg }
	if { ! [array exists Messages] } { return $msg }
	if { ! [info exists Messages($msg)] } {
		puts "Please translate: $msg"
		return $msg
	}
	return $Messages($msg)
}

proc PgAcVar:clean {prefix} {
global PgAcVar
	foreach key [array names PgAcVar $prefix] {
		set PgAcVar($key) {}
		unset PgAcVar($key)
	}
}

proc set_defaults {} {
global PgAcVar CurrentDB
	puts "\nSetting the default values:"

	array set def_vars {
		PgAcVar(currentdb,host) ""
		PgAcVar(currentdb,pgport) 5432
		CurrentDB ""
		PgAcVar(tablist) {Tables Queries Views Sequences Functions Reports Graphs Forms Scripts Images Usergroups Diagrams}
		PgAcVar(activetab) {}
		PgAcVar(query,tables) {}
		PgAcVar(query,links) {}
		PgAcVar(query,results) {}
		PgAcVar(mwcount) 0
		PgAcVar(PGACCESS_HIDDEN) 0
		PgAcVar(PGACCESS_LOGIN) 0
		PgAcVar(PGACCESS_SMLOGIN) 0
		PgAcVar(PGACCESS_NOAUTO) 0
		PgAcVar(PGACCESS_NOSCRIPT) 0
		PgAcVar(PGACCESS_CONNLOAD) 0
		PgAcVar(opendb,host) "localhost"
		PgAcVar(opendb,pgport) "5432"
		PgAcVar(opendb,dbname) ""
		PgAcVar(opendb,username) ""
		PgAcVar(opendb,password) ""
	}

	foreach i [array names def_vars] {
		if {![info exists $i]} {
			set $i $def_vars($i)
			puts "\t$i: $def_vars($i)"
		}
	}

	# the script's home dir
	if {![info exists PgAcVar(PGACCESS_HOME)]} {
		set home [file dirname [info script]]
		switch [file pathtype $home] {
			absolute {set PgAcVar(PGACCESS_HOME) $home}
			relative {set PgAcVar(PGACCESS_HOME) [file join [pwd] $home]}
			volumerelative {
				set curdir [pwd]
				cd $home
				set PgAcVar(PGACCESS_HOME) [file join [pwd] [file dirname [file join [lrange [file split $home] 1 end]]]]
				cd $curdir
			}
		}
		puts "\tPGACCESS_HOME: $PgAcVar(PGACCESS_HOME)"
	}
	
	# The path to the libpgtcl shared object lib.
    # Windows XP is seen as NT, but the Windows directory is called 'windows' and not 'winnt' -
    # change the first switch according to the Windows version you run. (Iavor)
    # also make sure that we arent using PGINTCL

	if {![info exists PgAcVar(PGLIB)] && ![info exists PgAcVar(PGINTCL)]} {
		switch $::tcl_platform(platform) {
            windows {
                switch $::tcl_platform(os) {                    
                    "Windows NT" {
                        set PgAcVar(PGLIB) "c:/windows/system32/" 
                    }
                    "Windows 95" {
                        set PgAcVar(PGLIB) "c:/windows/system/"
                    }
                    default {
                        set PgAcVar(PGLIB) ""
                    }
                }
            }
			unix {
				set PgAcVar(PGLIB) "/usr/lib"
			}
			MacOS {
				# How is this on Macintosh platform?
				set PgAcVar(PGLIB) ""
			}
			default {
				# for all other cases
				set PgAcVar(PGLIB) ""
			}
		}
		puts "\tPGLIB: $PgAcVar(PGLIB)"
	}
}

proc load_conf_file {filename} {
global PgAcVar
	puts "\nLoading the $filename config file:"
	if {![file exists $filename]} {
		return 0
	}
	set fid [open $filename r]
	if {$fid == ""} {
		return 0
	}
	set line_no 0
	while {![eof $fid]} {
		incr line_no
		catch {
			set line [string trim [gets $fid]]
			if {![string equal -length 1 $line "#"] && ![string equal $line ""]} {
				if {[llength [split $line "="]] == 2} {
					set var [string trim [lindex [split $line "="] 0]]
					set value [string trim [lindex [split $line "="] 1]]
					if {$var != ""} {
						set PgAcVar($var) $value
						puts "\t$var: $value"
					} else {
						puts "\tSyntax error ($filename,$line_no): null variable name"
					}
				} else {
					puts "\tSyntax error ($filename,$line_no): too many tokens"
				}
			}
		}
	}
	close $fid
	return 1
}

proc load_env_vars {} {
global PgAcVar env
	puts "\nLoading the enviroment variables:"
	set var_list {PGACCESS_HOME PGLIB PGPORT}

	foreach i $var_list {
		if {![info exists PgAcVar($i)]} {
			if {[info exists env($i)]} {
				set PgAcVar($i) $env($i)
				puts "\t$i: $PgAcVar($i)"
			}
		}
	}
}

proc parse_cmd_line {} {
global PgAcVar argv
    puts "Parsing the command line parameters:"
    if {[catch {package require cmdline} msg]} {
        puts "\tCouldn't find cmdline package elsewhere on your machine."
        puts "\tNow trying to load the cmdline package from PgAccess."
        puts "\tBut first I need to load some default values to find it..."
        flush stdout
        set_defaults
        if {[info exists PgAcVar(PGACCESS_HOME)]} {
            lappend ::auto_path [file join $PgAcVar(PGACCESS_HOME) lib] [file join $PgAcVar(PGACCESS_HOME) lib widgets]
            if {[catch [
                puts ""
                puts "\tcmdline: [package require cmdline]"
                puts "\tGot cmdline package supplied by PgAccess !"
                puts ""
                flush stdout
            ] msg]} {
                puts "\tFAILED"
                puts "Error: $msg"
                puts "Please install the tcllib package (http:://www.sourceforge.net/projects/tcllib)"
                flush stdout
                exit
            }
        }
	}

	array set cmdline_args {
		pglib {.arg "Path to libpgtcl shared object (dll/so) file" {
					set PgAcVar(PGLIB) $val
					puts "\tPGLIB: $PgAcVar(PGLIB)"
				}
			}
		home {.arg "PGACCESS_HOME" {
					set PgAcVar(PGACCESS_HOME) $val
					puts "\tPGACCESS_HOME: $PgAcVar(PGACCESS_HOME)"
				}
			}
		version {"" "Show version information" {
					puts "\tPGACCESS_VERSION: $PgAcVar(VERSION)"
					exit 0
				}
			}
		hide {"" "Hide main window" {
					set PgAcVar(PGACCESS_HIDDEN) $val
					puts "\tPGACCESS_HIDDEN: $PgAcVar(PGACCESS_HIDDEN)"
				}
			}
        noauto {"" "Disables automatic load of connections at startup" {
                    set PgAcVar(PGACCESS_NOAUTO) $val
                    puts "\tPGACCESS_NOAUTO: $PgAcVar(PGACCESS_NOAUTO)"
                }
            }
        noscript {"" "Disables execution of the 'autoexec' script" {
                    set PgAcVar(PGACCESS_NOSCRIPT) $val
                    puts "\tPGACCESS_NOSCRIPT: $PgAcVar(PGACCESS_NOSCRIPT)"
                }
            }
        conn {"" "Automatically loads the connection specified on the command line" {
                    set PgAcVar(PGACCESS_CONNLOAD) $val
                    puts "\tPGACCESS_CONNLOAD: $PgAcVar(PGACCESS_CONNLOAD)"
                }
            }
        login {"" "Displays login dialog at startup" {
                    set PgAcVar(PGACCESS_LOGIN) $val
                    puts "\tPGACCESS_LOGIN: $PgAcVar(PGACCESS_LOGIN)"
                }
            }
        host {.arg "Database host" {
                    set PgAcVar(opendb,host) $val
                    puts "\topendb,host: $PgAcVar(opendb,host)"
                }
            }
        pgport {.arg "Database port" {
                    set PgAcVar(opendb,pgport) $val
                    puts "\topendb,pgport: $PgAcVar(opendb,pgport)"
                }
            }
        dbname {.arg "Database name" {
                    set PgAcVar(opendb,dbname) $val
                    puts "\topendb,dbname: $PgAcVar(opendb,dbname)"
                }
            }
        username {.arg "Database user" {
                    set PgAcVar(opendb,username) $val
                    puts "\topendb,username: $PgAcVar(opendb,username)"
                }
            }
        password {.arg "Database password" {
                    set PgAcVar(opendb,password) $val
                    puts "\topendb,password: $PgAcVar(opendb,password)"
                }
            }
        pgintcl {"" "Uses pgin.tcl, a pure Tcl-PG interface, instead of a dll/so (overrides PGLIB)" {
                    set PgAcVar(PGINTCL) $val
                    puts "\tPGINTCL: $PgAcVar(PGINTCL)"
                }
            }
        temp {"" "Creates only TEMPorary PgAccess tables so your database doesn't get cluttered by them." {
                    set PgAcVar(PGACCESS_TEMP) $val
                    puts "\tPGACCESS_TEMP: $PgAcVar(PGACCESS_TEMP)"
                }
            }
        smlogin {"" "Shrinks the login window to ask for only items not specified on the command line" {
                    set PgAcVar(PGACCESS_SMLOGIN) $val
                    puts "\tPGACCESS_SMLOGIN: $PgAcVar(PGACCESS_SMLOGIN)"
                }
            }

	}

	foreach i [array names cmdline_args] {
		lappend getopt_list "$i[lindex $cmdline_args($i) 0]"
	}

	while {[set ok [cmdline::getopt argv $getopt_list opt val]] > 0} {
		if {[catch {eval [lindex $cmdline_args($opt) 2]} msg]} {
			puts "Error: $msg ..."
		}
	}

	if {$ok < 0} {
		puts "Error: wrong option"
		foreach i [array names cmdline_args] {
			lappend usage_list "$i \{[lindex $cmdline_args($i) 1]\}"
		}
		puts [cmdline::usage $usage_list]
		exit 1
	}
	return 1
}

proc init {argc argv} {
global PgAcVar CurrentDB env
	set PgAcVar(VERSION) [get_version]
	if {[info exists env(HOME)]} {
		set PgAcVar(HOME) $env(HOME)
	} else {
		set PgAcVar(HOME) ""
	}

	parse_cmd_line
	load_env_vars

	if {![info exists PgAcVar(PGACCESS_DIR)]} {
		# PGACCESS_DIR - the directory for the conf files
		set PgAcVar(PGACCESS_DIR) [file join $PgAcVar(HOME) .pgaccess]
		if {![file exists $PgAcVar(PGACCESS_DIR)]} {
			puts -nonewline "Creating $PgAcVar(PGACCESS_DIR) ..."
			if {[catch [file mkdir $PgAcVar(PGACCESS_DIR)] msg]} {
				puts ""
				puts "Warning: the $PgAcVar(PGACCESS_DIR) directory for storing the config stuff cannot be created ($msg)."
			}
			puts " Done."
		}
	}

	if {![info exists PgAcVar(PGACCESS_CFG)]} {
		# PGACCESS_CFG - the main config file
		set PgAcVar(PGACCESS_CFG) [file join $PgAcVar(PGACCESS_DIR) pgaccess.cfg]
		if {![file exists $PgAcVar(PGACCESS_CFG)]} {
			puts -nonewline "Creating $PgAcVar(PGACCESS_CFG) ..."
			if {[catch {
				set fid [open $PgAcVar(PGACCESS_CFG) w]
				puts $fid "# PgAccess main config file"
				puts $fid ""
				puts $fid "# path to libpgtcl"
				if {[info exists PgAcVar(PGLIB)]} {
					puts $fid "# PGLIB = $PgAcVar(PGLIB)"
				} else {
					puts $fid "# PGLIB = /usr/lib"
				}
				puts $fid ""
				puts $fid "# script path"
				if {[info exists PgAcVar(PGACCESS_HOME)]} {
					puts $fid "# PGACCESS_HOME = $PgAcVar(PGACCESS_HOME)"
				} else {
					puts $fid "# PGACCESS_HOME = /usr/lib/pgaccess"
				}
				puts $fid ""
				close $fid
			} msg]} {
				puts ""
				puts "Warning: the $PgAcVar(PGACCESS_CFG) config file cannot be created ($msg)."
			}
			puts " Done."
		}
	}

	if {![info exists PgAcVar(PGACCESS_RC)]} {
		# PGACCESS_RC - the rc file
		set PgAcVar(PGACCESS_RC) [file join $PgAcVar(PGACCESS_DIR) pgaccessrc]
		if {![file exists $PgAcVar(PGACCESS_RC)]} {
			puts -nonewline "Creating $PgAcVar(PGACCESS_RC) ..."
			if {[catch {
				set fid [open $PgAcVar(PGACCESS_RC) w]
				puts $fid ""
				close $fid
			} msg]} {
				puts ""
				puts "Warning: the $PgAcVar(PGACCESS_RC) file cannot be created ($msg)."
			}
			puts " Done."
		}
	}

	if {![info exists PgAcVar(PGACCESS_CONN)]} {
		# PGACCESS_CONN - th connections file
		set PgAcVar(PGACCESS_CONN) [file join $PgAcVar(PGACCESS_DIR) connections]
		if {![file exists $PgAcVar(PGACCESS_CONN)]} {
			puts -nonewline "Creating $PgAcVar(PGACCESS_CONN) ..."
			if {[catch {
				set fid [open $PgAcVar(PGACCESS_CONN) w]
				puts $fid ""
				close $fid
			} msg]} {
				puts ""
				puts "Warning: the $PgAcVar(PGACCESS_CONN) file cannot be created ($msg)."
			}
			puts " Done."
		}
	}

	if {![load_conf_file $PgAcVar(PGACCESS_CFG)]} {
		puts "Main config file [file join $PgAcVar(PGACCESS_DIR) $PgAcVar(PGACCESS_CFG)] not found falling back to default settings."
	}

	set_defaults

    # only load PGLIB if PGINTCL not specified on command line
    if {![info exists PgAcVar(PGINTCL)]} {
		set shlib [file join $PgAcVar(PGLIB) libpgtcl][info sharedlibextension]
		if {![file exists $shlib]} {
			puts "\nError: Shared library file: '$shlib' does not exist. \n\
				Check this file, or check PGLIB variable (in pgaccess.cfg)\n"
			#exit
            puts "====> Using pgin.tcl instead\n"
            set PgAcVar(PGINTCL) 1
		}

		if {[catch {load $shlib} err]} {
			puts "Error: can not load $shlib shared library."
			puts "You need to make sure that the library exists and"
			puts "the environment variable PGLIB points to the directory"
			puts "where it is located.\n"
			puts "If you use Windows, be sure that the needed libpgtcl.dll"
			puts "and libpq.dll files are copied in the Windows/System"
			puts "directory"
			puts "\nERROR MESSAGE: $err\n"
			#exit
            puts "====> Using pgin.tcl instead\n"
            set PgAcVar(PGINTCL) 1
		}
    }

	# Loading all defined namespaces
	puts -nonewline "\nLoading namespaces:"
	foreach module {mainlib stack syntax database debug tables queries visualqb forms views functions reports scripts usergroups sequences diagrams help preferences printer importexport connections graphs pgackages images} {
		puts -nonewline " $module"
		flush stdout
		if {[catch {source [file join $PgAcVar(PGACCESS_HOME) lib $module.tcl]} msg]} {
			puts "\nERROR: $msg"
			puts "Please check your installation or set the PGACCESS_HOME ($PgAcVar(PGACCESS_HOME)) variable properly!" 
			exit -1
		}
	}
	puts ""

	##  Loadin plugins
	puts -nonewline "\nLoading plugins:"
	foreach plugin [glob -nocomplain [file join $PgAcVar(PGACCESS_HOME) lib plugins *.tcl]] {
        # special check for pgin.tcl
        if {[string match "pgin" [file rootname [file tail $plugin]]] && ![info exists PgAcVar(PGINTCL)]} {
            puts -nonewline " [file rootname [file tail $plugin]]-(NOT LOADED)"
            flush stdout
        } else {
            puts -nonewline " [file rootname [file tail $plugin]]"
            flush stdout
            source $plugin
        }
	}
	puts ""

	Preferences::load

	Connections::load

	#  Loading the required packages
	puts "\nLoading the required packages:"
	lappend ::auto_path [file join $PgAcVar(PGACCESS_HOME) lib] [file join $PgAcVar(PGACCESS_HOME) lib widgets]
	if {[catch [
		puts "\ttablelist: [package require tablelist 2.7]"
		puts "\tBWidget: [package require BWidget]"
		puts "\ticons: [package require icons]"
		puts "\tbase64: [package require base64]"
		puts "\tcsv: [package require csv]"
		puts "\tfileutil: [package require fileutil]"
		puts "\ttkwizard: [package require tkwizard]"
	] msg]} {
		puts $msg
		puts "\n[intlmsg {Please install the required packages:}]\n"
		puts "\ttablelist: http:://www.nemethi.de"
		puts "\ttcllib: http:://www.sourceforge.net/projects/tcllib"
		puts "\tBWidget: http:://www.sourceforge.net/projects/tcllib"
        puts "\ttkwizard: http://www2.clearlight.com/~oakley/tcl/tkwizard/"
	}

    puts ""

	# Creating icons
	::icons::icons create \
		-file [file join $PgAcVar(PGACCESS_HOME) lib widgets icons1.0 tkIcons-sample.slick] \
		{filenew-22 fileopen-22 edit-22 back-22 forward-22 reload-22 fileclose-22 editcopy-22 edittrash-22 move-22 connect_creating-22 filter1-22 cancel-22 down-22 up-22 configure-22 decrypted-22 encrypted-22 connect_no-22 exit-22 people-16 system-16 network_local-16 misc-16 thumbnail-16 txt-16 desktop_flat-16 widget_doc-16 shellscript-16 queue-16 completion-16 edit-16 1rightarrow-22 1leftarrow-22 fileprint-22 wizard-22 run-22 rever-22 filesave-22 colorize-16 view_tree-16 font_truetype-16 view_icon-16 view_text-16 xapp-16 news-16 go-16 spellcheck-16 stop-16 2rightarrow-22 2leftarrow-22 editcut-16 hotlistadd-16 hotlistdel-16 filesaveas-22 imagegallery-22 editdelete-22 start-22 finish-22 player_stop-22 player_eject-22 kpresenter-16 package_toys-16 user-16 krayon-16 contents2-16 help-22 key_bindings-16}

}; # end proc init

proc wpg_exec {db cmd} {
global PgAcVar
	set PgAcVar(pgsql,cmd) "never executed"
	set PgAcVar(pgsql,status) "no status yet"
	set PgAcVar(pgsql,errmsg) "no error message yet"
	if {[catch {
		Mainlib::sqlw_display "-----BEGIN----------------------------------------"
		Mainlib::sqlw_display $cmd
		set PgAcVar(pgsql,cmd) $cmd
		set PgAcVar(pgsql,res) [pg_exec $db $cmd]
		set PgAcVar(pgsql,status) [pg_result $PgAcVar(pgsql,res) -status]
		set PgAcVar(pgsql,errmsg) [pg_result $PgAcVar(pgsql,res) -error]
	} tclerrmsg]} {
		showError [format [intlmsg "Tcl error executing pg_exec %s\n\n%s"] $cmd $tclerrmsg]
		return 0
	}
	Mainlib::sqlw_display $PgAcVar(pgsql,res)
	Mainlib::sqlw_display "-----END-----------------------------------------"
	return $PgAcVar(pgsql,res)
}


proc wpg_select {args} {
	Mainlib::sqlw_display "-----BEGIN----------------------------------------"
	Mainlib::sqlw_display "[lindex $args 1]"
	Mainlib::sqlw_display "-----END-----------------------------------------"
	uplevel pg_select $args
}


proc create_drop_down {base x y w} {
global PgAcVar
	if {[winfo exists $base.ddf]} return;
	frame $base.ddf -borderwidth 1 -height 75 -relief raised -width 55
	listbox $base.ddf.lb -background #fefefe -foreground #000000 -selectbackground #c3c3c3 -borderwidth 1  -font $PgAcVar(pref,font_normal)  -highlightthickness 0 -selectborderwidth 0 -yscrollcommand [subst {$base.ddf.sb set}]
	scrollbar $base.ddf.sb -borderwidth 1 -command [subst {$base.ddf.lb yview}] -highlightthickness 0 -orient vert
	place $base.ddf -x $x -y $y -width $w -height 185 -anchor nw -bordermode ignore
	place $base.ddf.lb -x 1 -y 1 -width [expr $w-18] -height 182 -anchor nw -bordermode ignore
	place $base.ddf.sb -x [expr $w-15] -y 1 -width 14 -height 183 -anchor nw -bordermode ignore
}


proc setCursor {{type NORMAL}} {
	if {[lsearch -exact "CLOCK WAIT WATCH" [string toupper $type]] != -1} {
		set type watch
	} else {
		set type left_ptr
	}
	foreach wn [winfo children .] {
		catch {$wn configure -cursor $type}
	}
	update ; update idletasks 
}


proc parameter {msg} {
global PgAcVar
	Window show .pgaw:GetParameter
	focus .pgaw:GetParameter.e1
	set PgAcVar(getqueryparam,var) ""
	set PgAcVar(getqueryparam,flag) 0
	set PgAcVar(getqueryparam,msg) $msg
	bind .pgaw:GetParameter <Destroy> "set PgAcVar(getqueryparam,flag) 1"
	grab .pgaw:GetParameter
	tkwait variable PgAcVar(getqueryparam,flag)
	if {$PgAcVar(getqueryparam,result)} {
		return $PgAcVar(getqueryparam,var)
	} else {
		return ""
	}
}


# show the error
# provide a link to help if one was specified
proc showError {emsg {helptopic {}}} {
global PgAcVar
    set blist [list "OK" "Help"]
    bell ;
    puts $emsg
    if {$PgAcVar(pref,erroremailuse)} {
        lappend blist "Report"
    }
    set res [MessageDlg .pgacErrorDlg \
        -title [intlmsg "Error"] \
        -icon error \
        -message $emsg \
        -type user \
        -buttons $blist]
    switch $res {
        0 {
            # they hit ok
        }
        1 {
            # they need help
            if {[string length $helptopic]>0} {
                Help::load $helptopic
            } else {
                Help::load index
            }
        }
        2 {
            # they want to report the error
            tk_messageBox -message "Soon you might be able to email bug reports.\nBut not yet."
        }
        default {
            # what did they hit
            puts "You should never see this."
        }
    }; # end switch
}


#----------------------------------------------------------
# executes a line of sql
#----------------------------------------------------------
#
proc sql_exec {how cmd {dbh_ ""}} {

    global PgAcVar CurrentDB

    if {[string match "" $dbh_]} {
        set dbh_ $CurrentDB
    }

    if {[set pgr [wpg_exec $dbh_ $cmd]]==0} {
        return 0
    }

    if {($PgAcVar(pgsql,status)=="PGRES_COMMAND_OK") || ($PgAcVar(pgsql,status)=="PGRES_TUPLES_OK")} {
        pg_result $pgr -clear
        return 1
    }

    if {$how != "quiet"} {
        showError [format [intlmsg "Error executing query\n\n%s\n\nPostgreSQL error message:\n%s\nPostgreSQL status:%s"] $cmd $PgAcVar(pgsql,errmsg) $PgAcVar(pgsql,status)]
    }

    pg_result $pgr -clear

    return 0

}



proc main {argc argv} {

    global PgAcVar CurrentDB tcl_platform env

    Window show .pgaw:Main

    set tdb ""
    set thost ""

    if {[info exists PgAcVar(pref,lastdb)] \
        && [info exists PgAcVar(pref,lasthost)]} {
        set tdb $PgAcVar(pref,lastdb)
        set thost $PgAcVar(pref,lasthost)
    }

    # dont squash the commandline args if they were provided
    if {![info exists PgAcVar(opendb,host)]} {
        set PgAcVar(opendb,host) {}
    }
    if {![info exists PgAcVar(opendb,dbname)]} {
        set PgAcVar(opendb,dbname) {}
    }
    if {![info exists PgAcVar(opendb,pgport)]} {
        set PgAcVar(opendb,pgport) 5432
    }
    if {![info exists PgAcVar(opendb,username)]} {
        set PgAcVar(opendb,username) {}
    }
    set PgAcVar(opendb,password) {}

    ##
    ## Check to see if connections should be opened automagically
    ##
    if {$PgAcVar(pref,autoload) && !$PgAcVar(PGACCESS_NOAUTO)} {
        ##
        ##  Load all dbs that are in the connections file
        ##
        foreach H [::Connections::getHosts] {

            foreach d [::Connections::getDbs $H] {

                set i [::Connections::getIds $H $d]

                if {[llength $i] != 1} {
                    puts stderr "\nERROR: There seems to be a problem with your connections file."
                    puts stderr "A host/db combination should be unique and the db should not be empty string"
                    puts stderr "Check host/db:  $H/$d with ids: $i"
                    puts stderr "Try removing the ~/.pgaccess/connections file"
                    puts stderr "Skipping this host/db combination\n"
                    continue
                }

                if {![info exists ::Connections::Conn(autoload,$i)]} {
                    set ::Connections::Conn(autoload,$i) 1
                }

                if {$::Connections::Conn(autoload,$i) == 0} {
                     continue
                }

                if {![::Connections::openConn $i]} {
                    puts "$::Connections::Msg($i)"
                    # display the open conn window
                    if {![::Connections::openConn $i 1]} {
                        continue
                    }
                }

                ::Mainlib::addDbNode $H $d

            }; # end foreach DB

        }; # end foreach host

        #Connections::check

       ##
       ##  Open up the tree node for the last db
       ##
       if {[string match "" $thost]} {set thost sockets}

       if {(![string match "" $tdb]) && ([$::Mainlib::Win(tree) exists __db__-${thost}-${tdb}])} {

               $::Mainlib::Win(tree) opentree __host__-$thost 0
               $::Mainlib::Win(tree) opentree __db__-${thost}-$tdb
               ::Mainlib::select 1 __db__-${thost}-$tdb
       }

    }; # end if autoload (or if NO_AUTO doesnt exist)

    wm protocol . WM_DELETE_WINDOW {
        ::Mainlib::Exit
    }

    return
}


proc Window {args} {
global vTcl
	set cmd [lindex $args 0]
	set name [lindex $args 1]
	set newname [lindex $args 2]
	set rest [lrange $args 3 end]
	if {$name == "" || $cmd == ""} {return}
	if {$newname == ""} {
		set newname $name
	}
	set exists [winfo exists $newname]
	switch $cmd {
		show {
			if {$exists == "1" && $name != "."} {wm deiconify $name; return}
			if {[info procs vTclWindow(pre)$name] != ""} {
				eval "vTclWindow(pre)$name $newname $rest"
			}
			if {[info procs vTclWindow$name] != ""} {
				eval "vTclWindow$name $newname $rest"
			}
			if {[info procs vTclWindow(post)$name] != ""} {
				eval "vTclWindow(post)$name $newname $rest"
			}
		}
		hide    { if $exists {wm withdraw $newname; return} }
		iconify { if $exists {wm iconify $newname; return} }
		destroy { if $exists {destroy $newname; return} }
	}
}

proc vTclWindow. {base} {
	if {$base == ""} {
		set base .
	}
	wm focusmodel $base passive
	wm geometry $base 1x1+0+0
	wm maxsize $base 1009 738
	wm minsize $base 1 1
	wm overrideredirect $base 0
	wm resizable $base 1 1
	wm withdraw $base
	wm title $base "vt.tcl"
}

init $argc $argv
main $argc $argv

if {$PgAcVar(PGACCESS_HIDDEN)} {
	Window hide .
	# wm iconify .
}

# checking if we should connect based on the given command line parameters
# this looks a little gross, but some modules rely on the GUI a bit too much
if {$PgAcVar(PGACCESS_CONNLOAD) || $PgAcVar(PGACCESS_LOGIN)} {
    if {[::Connections::openConn 0 $PgAcVar(PGACCESS_LOGIN) $PgAcVar(PGACCESS_SMLOGIN)]} {
        set H $PgAcVar(opendb,host)
        set d $PgAcVar(opendb,dbname)
        if {![$::Mainlib::Win(tree) exists __db__-${H}-${d}]} {
            ::Mainlib::addDbNode $H $d
        }
        $::Mainlib::Win(tree) opentree __host__-$H 0
        $::Mainlib::Win(tree) opentree __db__-${H}-$d
        ::Mainlib::select 1 __host__-$H
        ::Mainlib::select 1 __db__-${H}-$d
        # ::Mainlib::cmd_Tables
    } else {
        showError [intlmsg "Cannot login, possible bad connection parameters"]
    }
}





