/* ----------------------------------------------- */
/* company : pentamedia
 * author  : elcomski 
 * file    : monitor.c 
 * date    : 2001-10-08
 * modify  : 2001-12-13 by elcomski
 * 
 */   
/* ----------------------------------------------- */
#include <stdio.h>
#include <stdlib.h>
#include <termios.h>
#include <term.h>
#include <curses.h>
#include <unistd.h>
#include <string.h>

#include <signal.h>
#include <sys/time.h>

#include "../../share/xptype.h"
#include "../../share/ifdef.h"
#include "../libif/pentapi.h"


/*----------------------------------------- */
/* Constant definitions */
/*----------------------------------------- */
#define PENTAVALM_VER "1.0.1"
#define Y_POS 5
#define X_POS 0


/*----------------------------------------- */
/* Variable definitions */
/*----------------------------------------- */
static struct termios initial_settings, new_settings;
static int peek_char = -1;
static int ferr = 0;
char g_dev_name[10] = "pentanet0";


/*----------------------------------------- */
/* Function definitions */
/*----------------------------------------- */

/*
 * Func : init_keyb
 * Context :
 */
void init_keyb( void )
{
	tcgetattr( 0, &initial_settings );
	new_settings = initial_settings;
	new_settings.c_lflag &=  ~ICANON;
	new_settings.c_lflag &=  ~ECHO;
	new_settings.c_lflag &=  ~ISIG;
	new_settings.c_cc[VMIN] = 1;
	new_settings.c_cc[VTIME] = 0;
	tcsetattr( 0, TCSANOW, &new_settings );
}

/*
 * Func : close_keyb
 * Context :
 */
void close_keyb( void )
{
	tcsetattr( 0, TCSANOW, &initial_settings );
}


/*
 * Func : hit_keyb
 * Context :
 */
int hit_keyb( void )
{
	char ch;
	int nread;

	if( peek_char != -1 ){
		return -1;
	}

	new_settings.c_cc[VMIN] = 0;
	tcsetattr( 0, TCSANOW, &new_settings );
	nread = read( 0, &ch, 1);
	new_settings.c_cc[VMIN] = 1;
	tcsetattr( 0, TCSANOW, &new_settings );

	if( nread == 1 ){
		peek_char = ch;
		return 1;
	}
	return 0;
}


/*
 * Func : readch
 * Context :
 */
int readch( void )
{
	char ch;

	if( peek_char != -1 ){
		ch = peek_char;
		peek_char = -1;
		return ch;
	}
	read( 0, &ch, 1);
	return ch;
}


/*
 * Func : display_info
 * Context :
 */
void display_info( void )
{
	move( 0, 0 );
	attron(A_BOLD);
	attrset( COLOR_PAIR(2));
	printw("\n ----------------------------------\n" );
	printw(" Monitor for Pent@NET (Ver %s)\n", PENTAVALM_VER );
	printw(" ----------------------------------\n\n" );
	attroff(A_BOLD);
	attrset( COLOR_PAIR(1));
	refresh( );
}


/*
 * Func : display_status 
 * Context :
 */
int display_status( void )
{
	unsigned char bMacAddr[6];
	int iretDrv,iretLock,iretRxPackets,iretBer,iretUnRSerr,iretRSerr;
	unsigned long lretVer,lretBPS;
	double doSigQual;
	float fBPS;

	/*------------------------------*/
	/* Driver/Device Info */
	/*------------------------------*/
	move( Y_POS, X_POS );
	attrset( COLOR_PAIR(4));
	printw("[Driver/Device info]\n");
	attrset( COLOR_PAIR(1));
	refresh( );

	/*------------------------------*/
	/* Device name */
	/*------------------------------*/
	move( Y_POS+1, X_POS );
	attrset( COLOR_PAIR(3));
   	printw("# Device name : %s\n", g_dev_name );
	attrset( COLOR_PAIR(1));
	refresh( );
	
	/*------------------------------*/
	/* Driver status */
	/*------------------------------*/
	move( Y_POS+2, X_POS );
	attrset( COLOR_PAIR(3));
	iretDrv = sm_get_device_status(g_dev_name);
   	if(iretDrv > 0){
	       	printw("# Driver status : Enable\n");
		attrset( COLOR_PAIR(1));
		refresh( );
	}else{
	       	printw("# Driver status : ");
		if( iretDrv == ERR_OPEN_SOCKET){
			printw("**ERROR** network interface open error\n");
		}else if(iretDrv == ERR_NOT_DEVICE){
			printw("**ERROR** device not found\n");
		}else if(iretDrv == ERR_NOT_DEVICE_OPEN){
			printw("**ERROR** device not open\n");
		}else{
			printw("**ERROR** unknown error code[%d]\n",iretDrv);
		}
		refresh( );
		attrset( COLOR_PAIR(1));
		sleep(1);
		return 1;
	}

	/*------------------------------*/
	/* Driver Version */
	/*------------------------------*/
	move( Y_POS+3, X_POS );
	attrset( COLOR_PAIR(3));
    	lretVer = sm_get_device_version( g_dev_name );
   	printw("# Driver version : %x.%x.%x\n", ((lretVer>>16)&0xff),((lretVer>>8)&0xff),(lretVer&0xff));
	attrset( COLOR_PAIR(1));
	refresh( );
	
	/*------------------------------*/
	/* MAC address */
	/*------------------------------*/
	move( Y_POS+4, X_POS );
	attrset( COLOR_PAIR(3));
	sm_get_mac_address( g_dev_name, bMacAddr );
	printw("# MAC address : %02x:%02x:%02x:%02x:%02x:%02x\n", bMacAddr[0],bMacAddr[1],bMacAddr[2], bMacAddr[3],bMacAddr[4],bMacAddr[5]);
	attrset( COLOR_PAIR(1));
	refresh( );
   
	/*------------------------------*/
	/* Rx data Info */
	/*------------------------------*/
	move( Y_POS+6, X_POS );
	attrset( COLOR_PAIR(4));
	printw("[Rx data info]\n");
	attrset( COLOR_PAIR(1));
	refresh( );

	/*------------------------------*/
	/* Total Rx packets */
	/*------------------------------*/
	move( Y_POS+7, X_POS );
	attrset( COLOR_PAIR(3));
 	iretRxPackets = sm_get_received_packets( g_dev_name );
	if( iretRxPackets >= 0 ){
	       	printw("# Total RX packets : %d \n", iretRxPackets );	
	}
	attrset( COLOR_PAIR(1));
	refresh( );

	/*------------------------------*/
	/* Rx Data Rate */
	/*------------------------------*/
	move( Y_POS+8, X_POS );
	attrset( COLOR_PAIR(3));
	lretBPS = sm_get_down_speed( g_dev_name );
	fBPS = (float)(lretBPS * 8);
	fBPS = fBPS / (1024 * 1024);
	printw("# RX data rate : %.3f[Mbps]\n", fBPS );	
	attrset( COLOR_PAIR(1));
	refresh( );


	/*------------------------------*/
	/* Etc Info */
	/*------------------------------*/
	move( Y_POS+10, X_POS );
	attrset( COLOR_PAIR(4));
	printw("[Tuner info]\n");
	attrset( COLOR_PAIR(1));
	refresh( );

	/*------------------------------*/
	/* Lock status */
	/*------------------------------*/
	move( Y_POS+11, X_POS );
	attrset( COLOR_PAIR(3));
	iretLock = sm_tuner_get_status( g_dev_name);
	if( iretLock ){
		printw("# LNB Lock status: lock\n" );
	}else{
		printw("# LNB Lock status: unlock\n" );
	}
	attrset( COLOR_PAIR(1));
	refresh( );
	
	/*------------------------------*/
	/* Bit Error Rate */
	/*------------------------------*/
	move( Y_POS+12, X_POS );
	attrset( COLOR_PAIR(3));
	iretBer = sm_get_bit_error_rate ( g_dev_name );
	if( iretBer > 0){
	       	printw("# Viterbi decoded BER : %e\n", (float)iretBer/1048576);
	}else if( iretBer == 0 ){
	       	printw("# Viterbi decoded BER : %d\n", iretBer);
	}else{
	       	printw("# Viterbi decoded BER : Not supported\n");
	}
	attrset( COLOR_PAIR(1));
	refresh( );

	/*------------------------------*/
	/* Signal-to-Noise Ratio */
	/*------------------------------*/
	move( Y_POS+13,X_POS );
	attrset( COLOR_PAIR(3));
	doSigQual = sm_get_signal_quality( g_dev_name );
	printw("# Signal to Noise Ratio: %f [dB]\n", doSigQual);	
	attrset( COLOR_PAIR(1));
	refresh( );
    
	/*---------------------------------*/
	/* Reed Solomon Corrected errors */
	/*---------------------------------*/
	move( Y_POS+14, X_POS );
	attrset( COLOR_PAIR(3));
	iretRSerr = sm_get_corrected_rs_error( g_dev_name);
	if( iretRSerr >= 0 ){
		printw("# Reed Solomon Corrected errors: %d\n", iretRSerr);	
	}else{
		printw("# Reed Solomon Corrected errors: Not supported\n");	
	}
	attrset( COLOR_PAIR(1));
	refresh( );

	/*---------------------------------*/
	/* Reed Solomon Uncorrected errors */
	/*---------------------------------*/
	move( Y_POS+15, X_POS );
	attrset( COLOR_PAIR(3));
	iretUnRSerr = sm_get_uncorrected_rs_error( g_dev_name);
	if( iretUnRSerr >= 0 ){
		printw("# Reed Solomon Uncorrected errors: %d\n", iretUnRSerr);	
	}else{
		printw("# Reed Solomon Uncorrected errors: Not supported\n");	
	}
	attrset( COLOR_PAIR(1));
	refresh( );


	
	move( Y_POS+20, X_POS );
	attrset( COLOR_PAIR(5));
	printw("Press the 'q' key to finish it !\n" );
	attrset( COLOR_PAIR(1));
	refresh( );

	return 0;
}


/*
 * Func : timer_interrupt
 * Context :
 */
void timer_interrupt( int signum )
{
	display_info( );
	ferr = display_status( );
}


/*
 * Func : main
 * Context :
 */
int main( int argc, char *argv[] )
{
	int ch;
	struct itimerval itv;
	int opt;
	
	initscr( );

	if( !has_colors( ) ){
		endwin( );
		exit(1);
	}

	if( start_color( ) != OK ){
		endwin( );
		exit(1);
	}

	init_pair( 1, COLOR_WHITE, COLOR_BLACK );
	init_pair( 2, COLOR_WHITE, COLOR_BLUE );
	init_pair( 3, COLOR_YELLOW, COLOR_BLACK );
	init_pair( 4, COLOR_BLUE, COLOR_GREEN );
	init_pair( 5, COLOR_RED, COLOR_BLACK );
	
	clear( );

	while((opt = getopt(argc, argv, "d:")) != -1){
		switch( opt ){
			case 'd':
			{
				strcpy( g_dev_name, optarg );
			}
			break;
		}
	}
	
	itv.it_value.tv_sec = 1;
	itv.it_value.tv_usec = 0;
	itv.it_interval.tv_sec = 1;
	itv.it_interval.tv_usec = 0;
	signal( SIGPROF, timer_interrupt );
	setitimer( ITIMER_PROF, &itv, NULL );
	

	/* Check keyboard routine */
	ferr = 0;
	ch = 0;
	init_keyb( );
	while( ch != 'q' ){
		if( hit_keyb( ) ){
			ch = readch( );
		}

		if( ferr ) break;
	};
	close_keyb( );

	endwin( );

	exit( EXIT_SUCCESS );
}
