(*---------------------------------------------------------------------------*
  IMPLEMENTATION  cf_message.ml

  Copyright (c) 2002-2004, James H. Woodyatt
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

    Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
  FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
  COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
  INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
  OF THE POSSIBILITY OF SUCH DAMAGE. 
 *---------------------------------------------------------------------------*)

type t = (string * int * int) list

let sub_invariant_ (s, pos, len as sub) =
    let _ = s.[pos] in
    let _ = s.[pos + len - 1] in
    len > 0

let normalize = List.filter sub_invariant_

let rec sub_blit_loop_ ?(i = 0) dest =
    function
    | [] ->
        ()
    | (s, pos, len as sub) :: tl ->
        assert begin
            try sub_invariant_ (dest, i, len)
            with Invalid_argument _ -> false
        end;
        String.blit s pos dest i len;
        let i = i + len in
        sub_blit_loop_ ~i dest tl

let create s =
    let n = String.length s in
    if n > 0 then [ s, 0, n ] else []

let length =
    let f acc (_, _, len as sub) =
        assert (sub_invariant_ sub);
        acc + len
    in
    List.fold_left f 0

let contents =
    function
    | [] ->
        ""
    | tl ->
        let n = length tl in
        let s = String.create n in
        sub_blit_loop_ s tl;
        s

let copy m = create (contents m)

let flatten = function
    | _ :: _ :: _ as m -> copy m
    | m -> m

let x_pos_negative_ = "pos < 0"
let x_pos_range_ = "pos > length"

let rec split_aux_ (i, l1, l2) =
    function
    | [] ->
        if i > 0 then invalid_arg x_pos_range_;
        List.rev l1, List.rev l2
    | hd :: tl ->
        assert (sub_invariant_ hd);
        let acc =
            let s, pos, len = hd in
            match i with
            | 0 ->
                0, l1, hd :: l2
            | _ when i < len ->
                0, (s, pos, i) :: l1, (s, pos + i, len - i) :: l2
            | _ ->
                i - len, hd :: l1, []
        in
        split_aux_ acc tl

let split ~pos =
    match pos with
    | pos when pos < 0 -> invalid_arg x_pos_negative_
    | pos when pos > 0 -> split_aux_ (pos, [], [])
    | _ -> (fun m -> [], m)

let rec truncate_aux_ l0 i =
    function
    | [] ->
        if i > 0 then invalid_arg x_pos_range_;
        List.rev l0
    | hd :: tl ->
        assert (sub_invariant_ hd);
        let s, pos, len = hd in
        match i with
        | 0 -> List.rev l0
        | _ when i < len -> List.rev ((s, pos, i) :: l0)
        | _ -> truncate_aux_ (hd :: l0) (i - len) tl

let truncate ~pos =
    match pos with
    | pos when pos < 0 -> invalid_arg x_pos_negative_
    | pos when pos > 0 -> truncate_aux_ [] pos
    | _ -> (fun _ -> [])

let rec shift_aux_ i =
    function
    | [] ->
        if i > 0 then invalid_arg x_pos_range_;
        []
    | hd :: tl ->
        assert (sub_invariant_ hd);
        let s, pos, len = hd in
        if i < len then
            (s, pos + i, len - i) :: tl
        else
            shift_aux_ (i - len) tl

let shift ~pos =
    match pos with
    | 0 -> (fun x -> x)
    | pos when pos < 0 -> invalid_arg x_pos_negative_
    | pos -> shift_aux_ pos

let rec insert_aux_ l0 l1 i =
    function
    | [] ->
        if i > 0 then invalid_arg x_pos_range_;
        List.rev_append l0 l1
    | hd :: tl ->
        assert (sub_invariant_ hd);
        let s, pos, len = hd in
        if i = 0 then begin
            assert (not true); []
        end
        else if i < len then begin
            let sub1 = s, pos, i and sub2 = s, pos + i, len - i in
            let l0 = sub1 :: l0 and tl = sub2 :: tl in
            insert_aux_ l0 (l1 @ tl) 0 []
        end
        else
            insert_aux_ (hd :: l0) l1 (i - len) tl

let insert ~pos =
    match pos with
    | 0 -> (fun ~m m0 -> m @ m0)
    | pos when pos < 0 -> invalid_arg x_pos_negative_
    | pos -> (fun ~m -> insert_aux_ [] m pos)

let unsafe_shift1 = function
    | (buf, pos, len) :: tl when len > 1 -> (buf, pos + 1, len - 1) :: tl
    | (buf, pos, len) :: tl -> tl
    | [] -> []

let rec unsafe_to_seq m =
    lazy begin
        match m with
        | [] ->
            Cf_seq.Z
        | (buf, pos, _) :: _ as m ->
            let hd = String.unsafe_get buf pos in
            let tl = unsafe_to_seq (unsafe_shift1 m) in
            Cf_seq.P (hd, tl)
    end

let rec unsafe_to_seq2 m =
    lazy begin
        match m with
        | [] ->
            Cf_seq.Z
        | (buf, pos, _) :: _ as m ->
            let tail = unsafe_shift1 m in
            let hd = String.unsafe_get buf pos, tail in
            let tl = unsafe_to_seq2 tail in
            Cf_seq.P (hd, tl)
    end

let to_seq m = unsafe_to_seq (normalize m)
let to_seq2 m = unsafe_to_seq2 (normalize m)

(*--- End of File [ cf_message.ml ] ---*)
