//
// C++ Interface: packagesearchimpl
//
// Description: 
//
//
// Author: Benjamin Mesing <bensmail@gmx.net>, (C) 2004
//
// Copyright: See COPYING file that comes with this distribution
//
//

#ifndef __PACKAGESEARCHIMPL_H_2004_03_21
#define __PACKAGESEARCHIMPL_H_2004_03_21

#include <utility>
#include <vector>

#include <OpSet.h>

#include "packagesearch.h"
#include "iprovider.h"
#include "ipluginuser.h"

class QTabDialog;
class QCloseEvent;

class TagSelection;

namespace NPlugin
{
	class Plugin;
	class SearchPlugin;
	class InformationPlugin;
	class ShortInformationPlugin;
	class PluginManager;
}

using namespace std;

/** I had to leave the ui.h approach as the TagcollBuilder wants to have a handle maker on construction and
  * as I did not like to mess around with pointers I was forced to switch to the subclassing approach.
  */
class PackageSearchImpl : public PackageSearch, public NPlugin::IProvider, public NPlugin::IPluginUser
{
    Q_OBJECT
public:
	PackageSearchImpl( QWidget* parent = 0, const char* name = 0, WFlags fl = WType_TopLevel );
	~PackageSearchImpl();
	struct Exception
	{
		Exception(const string& errorMsg) : _errorMsg(errorMsg) {}
		Exception(const char* errorMsg) : _errorMsg(errorMsg) {}
		string _errorMsg;
	};
	/** Shows the short information for the given packages. 
	  *
	  * @param packages the packages to be shown in the short information listing. */
	void updateShortInformation(Tagcoll::OpSet<int>& packages);
	/** Actions to perform on close. */
	virtual void closeEvent(QCloseEvent* pE);
	/** @name IProvider interface
	  *
	  * These functions implement the IProvider interface.
	  */
	//@{	
	// documented in base class
	virtual void setEnabled(bool enabled);
	// documented in base class
	virtual QPopupMenu* systemMenu() const;
	// documented in base class
	virtual void reportError(const QString& title, const QString& message);
	// documented in base class
	virtual void reportWarning(const QString& title, const QString& message);
	// documented in base class
	virtual Tagcoll::HandleMaker<string>& handleMaker() const;
	// documented in base class
	virtual void reportBusy(NPlugin::Plugin* pPlugin, const QString& message);
	// documented in base class
	virtual void reportReady(NPlugin::Plugin* pPlugin);
	// documented in base class
	virtual QMainWindow* mainWindow() { return this; }
	// documented in base class
	virtual QStatusBar* statusBar() { return QMainWindow::statusBar(); }
	// documented in base class
	virtual IPluginUser* pluginUser() { return this; }
	// documented in base class
	virtual const set<string>& packages();
	// documented in base class
	virtual NUtil::IProgressObserver* progressObserver();
	//@}
	/** @name IPluginUser interface
	  *
	  * These functions implement the IPluginUser interface.
	  */
	//@{	
	// documented in base class
	virtual void addPlugin(NPlugin::Plugin* pPlugin);
	/** @brief Adds the given plugin to the SearchWindow.
	  *
	  * @param pPlugin the plugin to be added
	  */
	virtual void addPlugin(NPlugin::PluginContainer* pPlugin);
	/** @brief Finds every reference to plugin and remove it so that pPlugin can be safely deleted.
	  *
	  * @param pPlugin the plugin to be removed 
	  */
	void removePlugin(NPlugin::Plugin* pPlugin);

	//@}
public slots:
	/** Call this function after the dialog is shown to initialize the whole system. */
	void initialize();
protected slots:
	/** @brief This let the views show all information available about the package.
	  *
	  * This request all plugins to update their information widgets and gets their
	  * information text.\n
	  * If package is empty, the views will be cleared. 
	  * @note Do not call this function directly, but call setSelectedPackage() instead.
	  */
	virtual void showPackageInformation(const QString & package);
	/** @brief Evaluates the plugin search results and displays them in the GUI.
	  *
	  * This is called whenever the search of a SearchPlugin changes.
	  * @param pPlugin the plugin where the search changed or 0 if no originator is known
	  */
	virtual void onSearchChanged(NPlugin::SearchPlugin* pPlugin);
	virtual void onClearSearch();
	/** This is called whenever the user selects another information page. */
	virtual void onInformationPageChanged( QWidget * pPage );
	/** This shows the control dialog for the plugins. */
	virtual void onControlPlugins();
	/** This shows a settings dialog for the plugins. */
	virtual void onPluginSettings();
	virtual void saveSettings();
	virtual void loadSettings();
	/** @brief This refreshes #_shownShortInformation and  #_hiddenShortInformation according to 
	  * the state of this class. 
	  *
	  * #_shownShortInformation will be updated according to the listed columns in #_pPackageListView
	  * also reflecting its columns order, and from #_hiddenShortInformation all entries will be
	  * removed where no plugin is available wich have this text as caption.
	  */
	void refreshShownAndHidden();

protected:
	/** @brief Funtion object to filter search results with plugins which uses the filter 
	  * technique.  */
	class FilterPackages
	{
		NPlugin::SearchPlugin* _pPlugin;
	public:
		/** @param pPlugin the plugin to be used for filtering. 
		  * @pre pPlugin.isInactive() == false ^ pPlugin.usesFilterTechnique() == true  */
		FilterPackages(NPlugin::SearchPlugin* pPlugin) : _pPlugin(pPlugin)	{};
		bool operator()(int packageID);
		/** Call this after you've finished all filtering - the results will be written to the 
		  * OpSet handed in the constructor. */
// 		void finishFiltering();
	};
	
	/** @brief Updates search interface.
	  *
	  * Shows the search input for all available search plugins in the correct order,
	  * removing outdated inputs. */
	void updateSearchPluginGui();
	/** @brief Adds the widgets of the search plugin to the GUI. 
	  * 
	  * @pre pPlugin != 0  */
	void addSearchPluginToGui(NPlugin::SearchPlugin* pPlugin);
	/** @brief Updates information interface.
	  *
	  * Shows the information widget for all available information plugins in the correct 
	  * order, removing outdated widgets. */
	void updateInformationPluginGui();
	/** @brief Adds the elements of the information plugin to the GUI. 
	  *
	  * @pre pPlugin != 0  */
	void addInformationPluginToGui(NPlugin::InformationPlugin* pPlugin);
	/** @brief Updates short information interface.
	  *
	  * Shows the short information for all available short information plugins in the correct 
	  * order, removing outdated ones. */
	void updateShortInformationPluginGui();
	/** @brief Adds the given action (menuEntry.second) to the menu with the given name (menuEntry.first).
	  *
	  * If the menu is not yet existent it will be created.
	  * @param menuEntry \li menuEntry.first name of the menu the action will be added to
	  * 	\li menuEntry.second action to be added
	  *
	  */
	void addMenuEntry( pair<QString, QAction*> menuEntry );
protected:
	void onPackageViewContextMenu(QListViewItem* pItem, const QPoint& pos);
	
private:
	/** @brief This is the class that takes care of the plugins loading and unloading. 
	  *
	  * <tt>this</tt> will be added to it as a IPluginUser do it will be informed about
	  * changes in the plugin. */
	NPlugin::PluginManager* _pPluginManager;
	
	typedef vector<NPlugin::Plugin*> PluginContainer;
	typedef vector<NPlugin::SearchPlugin*> SearchPluginContainer;
	typedef vector<NPlugin::InformationPlugin*> InformationPluginContainer;
	typedef vector<NPlugin::ShortInformationPlugin*> ShortInformationPluginContainer;

	/** All plugins currently installed. 
	  * @invariant \f$plugins = searchPlugins \cup informationPlugins \cup shortInformationPlugins\f$<br>
	  * \f$searchPlugins \subseteq plugins\f$ <br>
	  * \f$informationPlugins \subseteq plugins\f$<br>
	  * \f$shortInformationPlugins \subseteq plugins\f$
	  */
	PluginContainer _plugins;
	/** All search plugins currently installed.
	  *
	  * They are kept sorted by there priority in ascending order.
	  * @see _plugins for invariants. */
	SearchPluginContainer _searchPlugins;
	/** @brief All information plugins currently installed.
	  *
	  * They are kept sorted by there priority in ascending order.
	  * @see _plugins for invariants. */
	InformationPluginContainer _informationPlugins;
	/** @brief All short information plugins currently installed.
	  *
	  * They are kept sorted by there priority in decending order.
	  * @see _plugins for invariants. */
	ShortInformationPluginContainer _shortInformationPlugins;
	
	
	/** @brief This maps the plugin to the short information column it fills. 
	  *
	  * Additionally the plugins contained in the map are the ones which are shown.
	  */
	map<NPlugin::ShortInformationPlugin*, int> _shortInformationColumn;
	/** @brief Lists the name of the caption for the short information plugins.
	  * 
	  * This does not always reflect the order of the corresponding columns.
	  */
	QStringList _shownShortInformation;
	/** @brief Lists the name of the caption for the short information plugins which
	  * are hidden.  */
	QStringList _hiddenShortInformation;
	/** @brief This maps the integers to the corresponding package names. */
	mutable Tagcoll::HandleMaker<string>* _pHandleMaker;
	/** @brief This holds a pointer to the page for the details section. */
	QWidget* _pInformationDetailsPage;
	/** Holds the priority for the details page. */
	const uint _DETAILS_INFORMATION_PRIORITY;
	/** Holds the name of the package currently shown in the information container. */
//	QString _currentPackage;
	/** Holds the name of the settings file to be used ($HOME/.packagesearch)*/
	QString _settingsFilename;
	/** @brief This holds all packages available on the system.
	  *
	  * It bases on the apt-cache so it assumes those packages as all. It is
	  * lazy initialized.
	  */
	set<string> _packages;
	/** @brief This plugin is used to display the package names in the short description. */
	NPlugin::Plugin* _pPackageNamePlugin;
};

#endif // __PACKAGESEARCHIMPL_H_2004_03_21
