/*
 * rtp-sourcegen.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1999-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "timer.h"
#include "media-timer.h"
#include "source.h"
#include "pktbuf.h"

class RTPSourceGenerator : public PacketHandler, public MediaTimer, public Timer {
public:
  RTPSourceGenerator();
  pktbuf *buf_head_;
  pktbuf *buf_tail_;
  pktbuf *buf_cur_;
  double frame_rate_;
  double bit_rate_;
  u_int32_t ssrc_;
  int state_;
  u_int16_t next_seqno_;
  int marker_seen_;
  int first_time_;
  u_int32_t last_out_ts_;
  u_int32_t last_rec_ts_;

  void stop();
  void record();
  void play();

  virtual int command(int argc, const char*const* argv);
  virtual void timeout();
  virtual void recv(pktbuf *pb);
};

static int IDLE = 0;
static int RECORDING = 1;
static int PLAYING = 2;

//static class RTPSourceGeneratorClass : public TclClass {
//public:
//  RTPSourceGeneratorClass() : TclClass("RTPSourceGenerator") {}
//  TclObject * create(int argc, const char*const* argv) {
//    return (new RTPSourceGenerator());
//  }
//} rtp_source_generator_class_;

DEFINE_OTCL_CLASS(RTPSourceGenerator, "RTPSourceGenerator") {
}

RTPSourceGenerator::RTPSourceGenerator() :
  PacketHandler(0)
{
  buf_head_ = 0;
  buf_tail_ = 0;
  buf_cur_ = 0;
  frame_rate_ = 30.0;
  bit_rate_ = 1000000.0;
  ssrc_ = 1;
  state_ = IDLE;
  next_seqno_ = -1;
};

void
RTPSourceGenerator::stop() {
  cancel();
  state_ = IDLE;
};

void
RTPSourceGenerator::record() {
  if (state_ == RECORDING)
    return;
  stop();
  if (buf_head_ != 0) {
    buf_cur_ = buf_head_;
    while (buf_cur_ != buf_tail_) {
      buf_head_ = buf_cur_->next;
      buf_cur_->release();
      buf_cur_ = buf_head_;
    }
    buf_cur_->release();
    buf_head_ = buf_tail_ = buf_cur_ = 0;
  }
  marker_seen_ = 0;
  state_ = RECORDING;
}

void
RTPSourceGenerator::play() {
  if (state_ == PLAYING)
    return;
  stop();
  if ((buf_head_ == 0) || (target_ == 0))
    return;
  state_ = PLAYING;
  first_time_ = 1;
  msched(1);
}


int
RTPSourceGenerator::command(int argc, const char*const* argv)
{
  if (argc == 2) {
    if (strcmp(argv[1], "play") == 0) {
      play();
      return TCL_OK;
    }
    if (strcmp(argv[1], "stop") == 0) {
      stop();
      return TCL_OK;
    }
    if (strcmp(argv[1], "record") == 0) {
      record();
      return TCL_OK;
    }
  }
  if (argc == 3) {
    if (strcmp(argv[1], "ssrc") == 0) {
      ssrc_ = atoi(argv[2]);
      return TCL_OK;
    }
    if (strcmp(argv[1], "framerate") == 0) {
      double new_framerate = atof(argv[2]);
      if (new_framerate > 0) {
	frame_rate_ = new_framerate;
      }
      return TCL_OK;
    }
    if (strcmp(argv[1], "bitrate") == 0) {
      double new_bitrate = atof(argv[2]);
      if (new_bitrate > 0) {
	bit_rate_ = new_bitrate;
      }
      return TCL_OK;
    }
  }
  return (PacketHandler::command(argc,argv));
}

void
RTPSourceGenerator::timeout() {
  if ((state_ == IDLE) || (buf_cur_ == 0) || (target_ == 0))
    return;

  rtphdr *rh = (rtphdr *) buf_cur_->dp;

  if ((ntohl(rh->rh_ts) != last_rec_ts_) || first_time_) {
    u_int32_t now = media_ts();
    if (!first_time_) {
      /* Make sure under frame rate restriction, otherwise schedule. */
      u_int32_t diff = now - last_out_ts_;
      int wait = ((int) (90000.0/frame_rate_)) - ((int) diff);

      if (wait >= 90) {
	msched(wait/90);
	return;
      }
      last_out_ts_ += ((int) (90000.0/frame_rate_));

    } else {
      last_out_ts_ = now;
      first_time_ = 0;
    }
    last_rec_ts_ = ntohl(rh->rh_ts);
    /* last_out_ts_ = now; */
  }

  while ((ntohl(rh->rh_ts)) == last_rec_ts_) {
    pktbuf *pb = (pktbuf *) buf_cur_->copy();
    buf_cur_ = buf_cur_->next;
    rh = (rtphdr *) pb->dp;

    rh->rh_ts = htonl(last_out_ts_);
    rh->rh_ssrc = htonl(ssrc_);
    rh->rh_seqno = htons(next_seqno_);
    next_seqno_++;

    target_->recv(pb);
    rh = (rtphdr *) buf_cur_->dp;
  }
  msched(1);
  return;

#ifdef 0

  int len = pb->len;

  /* Calculate next transmit time. */

  int wait = (int) (((len * 8) * 1000) / bit_rate_);
  msched(wait);

  target_->recv(pb);
  return;
#endif
}

void
RTPSourceGenerator::recv(pktbuf* pb) {
  if (state_ != RECORDING) {
    pb->release();
  } else {
    if (!marker_seen_) {
      rtphdr *rh = (rtphdr *) pb->dp;
      if (rh->rh_flags && htons(RTP_M)) {
	marker_seen_ = 1;
      }
      pb->release();
      return;
    }

    if (buf_head_ == 0) {
      buf_head_ = buf_tail_ = buf_cur_ = pb;
      pb->next = buf_head_;
    } else {
      buf_tail_->next = pb;
      pb->next = buf_head_;
      buf_tail_ = pb;
    }
  }
}

