/*
 * rep-encoder-semicompressed2sc.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <errno.h>
#ifdef WIN32
#else
#include <sys/time.h>
#endif
#include "inet.h"
#include "net.h"
#include "rtp.h"
#include "codec/dct.h"
#include "bsd-endian.h"
#include "Tcl.h"
#include "pktbuf-rtp.h"
#include "codec/module.h"
#include "vidreps.h"

#define HDRSIZE (sizeof(rtphdr) + 18)

class SemicompressedToSCEncoder : public EncoderModule {
public:
  SemicompressedToSCEncoder() : EncoderModule(), block_width_(0),
    block_height_(0) {
      setq(30);
      last_size_ = 0;
      bind("last_size_", (int *)&last_size_);
  };
  void size(int w, int h);
  void recv(Buffer*);
  void recv(Semicompressed* semi_input);

protected:
  int command(int argc, const char*const* argv);
  int encode(const Semicompressed* vf);
  int block_width_;
  int block_height_;

  u_char quant_;
  int lqt_[64];
  int cqt_[64];
  int last_size_;

  u_int32_t translate(u_int32_t block_addr, const Semicompressed *vf);
  void setq(int q);
};

static class SemicompressedToSCEncoderClass : public TclClass {
public:
  SemicompressedToSCEncoderClass() : TclClass("Module/VideoEncoder/SemicompressedToSC") {}
  TclObject* create(int, const char*const*) {
    return (new SemicompressedToSCEncoder);
  }
} semiccompressed_to_sc_encoder_class_;

static const int stdlqt[] = {
  16, 11, 10, 16, 24, 40, 51, 61,
  12, 12, 14, 19, 26, 58, 60, 55,
  14, 13, 16, 24, 40, 57, 69, 56,
  14, 17, 22, 29, 51, 87, 80, 62,
  18, 22, 37, 56, 68, 109, 103, 77,
  24, 35, 55, 64, 81, 104, 113, 92,
  49, 64, 78, 87, 103, 121, 120, 101,
  72, 92, 95, 98, 112, 100, 103, 99 };

static const int stdcqt[] = {
  17, 18, 24, 47, 99, 99, 99, 99,
  18, 21, 26, 66, 99, 99, 99, 99,
  24, 26, 56, 99, 99, 99, 99, 99,
  47, 66, 99, 99, 99, 99, 99, 99,
  99, 99, 99, 99, 99, 99, 99, 99,
  99, 99, 99, 99, 99, 99, 99, 99,
  99, 99, 99, 99, 99, 99, 99, 99,
  99, 99, 99, 99, 99, 99, 99, 99 };


void
SemicompressedToSCEncoder::setq(int q)
{
  quant_ = q;

  int s = (q>50) ? 200-q*2 : 5000/q;
  for (int i=0; i<64; i++) {
    int v = (stdlqt[i] * s + 50)/ 100;
    if (v<1)
      lqt_[i] = 1;
	else if (v>255)
	  lqt_[i] = 255;
    else
      lqt_[i] = v;

    v = (stdcqt[i] * s + 50)/ 100;
    if (v<1)
      cqt_[i] = 1;
    else if (v>255)
      cqt_[i] = 255;
    else
      cqt_[i] = v;
  }
}

void
SemicompressedToSCEncoder::size(int w, int h)
{
  FrameModule::size(w, h);
  if (w&0x7 || h&0x7) {
    fprintf(stderr, "SemicompressedToSCEncoder: bad geometry: %dx%d\n", w, h);
    exit(1);
  }
  block_width_ = w >> 3;
  block_height_ = h >> 3;
}

int
SemicompressedToSCEncoder::command(int argc, const char*const* argv)
{
  if (argc == 3 && strcmp(argv[1], "q") == 0) {
    setq(atoi(argv[2]));
    return (TCL_OK);
  }
  if (argc == 3 && strcmp(argv[1], "recv") == 0) {
    Semicompressed *input = (Semicompressed *) TclObject::lookup(argv[2]);
    if (input != 0) {
      recv(input);
    }
    return (TCL_OK);
  }
  return (EncoderModule::command(argc, argv));
}

void
SemicompressedToSCEncoder::recv(Semicompressed* semi_input)
{
  if (!(semi_input->w_ == width_ &&
	semi_input->h_ == height_)) {
    size(semi_input->w_, semi_input->h_);
  }
  encode(semi_input);
}

void
SemicompressedToSCEncoder::recv(Buffer* bp)
{
  recv((Semicompressed *) bp);
  return;
}

#ifdef  DC_SIZE
#undef  DC_SIZE
#endif
#define DC_SIZE 2
#define MAX_PER_AC_USE 4
#define BLOCK_END_SIZE 4

int
SemicompressedToSCEncoder::encode(const Semicompressed* vf)
{
  int max_space_needed, space_used;
  int16_t *dp;
  u_int16_t block_addr;
  int num_lum_blocks, num_chr_blocks, num_blocks;
  ScBlock *scb;
  u_int16_t run, code;
  int last_pos;

  last_size_ = 0;

  pktbuf* pb = pool_->alloc(htonl(vf->ts_), RTP_PT_SC);
  int fragsize = mtu_ - HDRSIZE;

  /* Timing stuff
  struct timeval tv1, tv2;
  ::gettimeofday(&tv1, 0);
  */

  /* RTP/SC header */
  rtphdr* rh = (rtphdr*)pb->data;
  u_int16_t* h = (u_int16_t*)(rh+1);
  h[0] = htons(vf->w_);
  h[1] = htons(vf->h_);
  h[2] = htons(vf->true_w_);
  h[3] = htons(vf->true_h_);
  h[4] = htons(vf->xoff_);
  h[5] = htons(vf->yoff_);
  h[6] = htons((vf->h_subsample_ << 8) | (vf->v_subsample_ & 0xff));

  u_int32_t translated_block_addr = translate(0, vf);

  h[7] = htons((u_int16_t) (translated_block_addr >> 16));
  h[8] = htons((u_int16_t) (translated_block_addr & 0x0000ffff));

  dp = (int16_t *)&(h[9]);

  space_used = 0;

  block_addr = 0;
  num_lum_blocks = (vf->w_/8)*(vf->h_/8);
  num_chr_blocks = ((vf->w_/(8*vf->h_subsample_)) *
		    (vf->h_/(8*vf->v_subsample_)));
  num_blocks = num_lum_blocks + 2*num_chr_blocks;

  while (block_addr < num_blocks) {
    int *qt;

    if (block_addr < num_lum_blocks) {
      scb = &(vf->lum_->firstBlock[block_addr]);
      qt = lqt_;
    } else if (block_addr < num_lum_blocks + num_chr_blocks) {
      scb = &(vf->cr_->firstBlock[block_addr-num_lum_blocks]);
      qt = cqt_;
    } else {
      scb = &(vf->cb_->firstBlock[block_addr-num_lum_blocks-num_chr_blocks]);
      qt = cqt_;
    }

    /*
     * Calculate if I'll fit in remaining packet space. If not,
     * send packet and get a new one.
     */

    max_space_needed = DC_SIZE + MAX_PER_AC_USE * scb->numOfAC + BLOCK_END_SIZE;
    if (max_space_needed > fragsize - space_used) {
      pb->len = space_used + HDRSIZE;

      /* Timing stuff
      ::gettimeofday(&tv2, 0);
      fprintf(stderr, "%u sec + %u usec to %u sec + %u usec\n",
	      tv1.tv_sec, tv1.tv_usec, tv2.tv_sec, tv2.tv_usec);
	      */

      last_size_ += pb->len;
      target_->recv(pb);


      pb = pool_->alloc(htonl(vf->ts_), RTP_PT_SC);

      /* Timing stuff
      ::gettimeofday(&tv1, 0);
      */

      rh = (rtphdr*)pb->data;
      h = (u_int16_t*)(rh+1);
      h[0] = htons(vf->w_);
      h[1] = htons(vf->h_);
      h[2] = htons(vf->true_w_);
      h[3] = htons(vf->true_h_);
      h[4] = htons(vf->xoff_);
      h[5] = htons(vf->yoff_);
      h[6] = htons((vf->h_subsample_ << 8) | (vf->v_subsample_ & 0xff));

      translated_block_addr = translate(block_addr, vf);
      h[7] = htons((u_int16_t) (translated_block_addr >> 16));
      h[8] = htons((u_int16_t) (translated_block_addr & 0x0000ffff));

      dp = (int16_t *)&(h[9]);
      space_used = 0;
    }

    *dp++ = htons(scb->dc);
    space_used += 2;

    last_pos = 0;

    for(int i=0; i<scb->numOfAC; i++) {
      if (!((scb->value[i] >= -1 * qt[i]) && (scb->value[i] <= qt[i]))) {
	run = scb->index[i] - last_pos;
	if ((run < 1) || (run > 63)) {
	  fprintf(stderr, "Serious problem with RLE calc\n");
	  exit(1);
	}
	if (run > 15) {
	  /* Escape for large run values */
	  code = 0xf000 | ((run-1) & 0x00ff);
	  *dp++ = htons(code);
	  *dp++ = htons(scb->value[i]);
	  space_used += 4;
	} else {
	  code = ((run-1) << 12) | (scb->value[i] & (0x0fff));
	  *dp++ = htons(code);
	  space_used += 2;
	}
	last_pos = scb->index[i];
      }
    }

    block_addr++;
    u_int32_t addr_diff = (translate(block_addr, vf) -
			   translated_block_addr - 1);
    translated_block_addr += addr_diff + 1;

    if (addr_diff < 0x400) {
      /* Address difference will fit in available 10 bits. */
      *dp++ = htons(0xf800 | ((u_int16_t) addr_diff));
      space_used += 2;
    } else {
      *dp++ = htons(0xfc00 | ((u_int16_t) ((addr_diff >> 16) & 0x00ff)));
      *dp++ = htons((u_int16_t) (addr_diff & 0xffff));
      space_used += 4;
    }
  }

  rh->rh_flags |= htons(RTP_M);
  pb->len = space_used + HDRSIZE;

  /* Timing stuff
  ::gettimeofday(&tv2, 0);
  fprintf(stderr, "%u sec + %u usec to %u sec + %u usec\n",
	  tv1.tv_sec, tv1.tv_usec, tv2.tv_sec, tv2.tv_usec);
	  */

  last_size_ += pb->len;
  target_->recv(pb);

  return 0;
}

u_int32_t
SemicompressedToSCEncoder::translate(u_int32_t block_addr,
				     const Semicompressed *vf)
{
  int brow;
  int bcol;
  int bwidth;

  int lum_block_range = (vf->w_ / 8) * (vf->h_ / 8);
  int cr_block_range = lum_block_range +
    (lum_block_range / (vf->h_subsample_ * vf->v_subsample_));
  int cb_block_range = cr_block_range + (cr_block_range - lum_block_range);

  if (block_addr < lum_block_range) {
    bwidth = (vf->w_ / 8);

    brow = block_addr / bwidth;
    bcol = block_addr % bwidth;

    brow += (vf->yoff_ / 8);
    bcol += (vf->xoff_ / 8);

    return (brow * (vf->true_w_ / 8) + bcol);

  } else if (block_addr < cr_block_range) {
    block_addr -= lum_block_range;

    bwidth = (vf->w_ / (8 * vf->h_subsample_));

    brow = block_addr / bwidth;
    bcol = block_addr % bwidth;

    brow += (vf->yoff_ / (8 * vf->v_subsample_));
    bcol += (vf->xoff_ / (8 * vf->h_subsample_));

    return ((brow * (vf->true_w_ / (8 * vf->h_subsample_)) + bcol) +
	    ((vf->true_w_ / 8) * (vf->true_h_ / 8)));

  } else if (block_addr < cb_block_range) {
    block_addr -= cr_block_range;

    bwidth = (vf->w_ / (8 * vf->h_subsample_));

    brow = block_addr / bwidth;
    bcol = block_addr % bwidth;

    brow += (vf->yoff_ / (8 * vf->v_subsample_));
    bcol += (vf->xoff_ / (8 * vf->h_subsample_));

    return ((brow * (vf->true_w_ / (8 * vf->h_subsample_)) + bcol) +
	    ((vf->true_w_ / 8) * (vf->true_h_ / 8)) +
	    ((vf->true_w_ / (8 * vf->h_subsample_)) *
	     (vf->true_h_ / (8 * vf->v_subsample_))));
  }

  return 0;
}

