/*
 * ztrace.cc --
 *
 *      Buffer output until a SIGINT is caught; when caught, flush the output
 *      and exit. The implementation that allows the ZTrace object to be
 *      "compiled away" is from Yatin's MTrace impl.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef ZTRACE

#include "ztrace.h"
#include "signal.h"

#ifndef WIN32
#include <sys/types.h>
#include <sys/param.h>
#include <unistd.h>
#endif

ZTraceObject *ztrace=NULL;

void
bye(int)
{
	/* We wouldn't be here if there weren't a global object initialized */
	ztrace->flush();
	exit(0);
}

static class ZTraceObjectClass : public TclClass {
public:
	ZTraceObjectClass() : TclClass("ZTrace") { }
	TclObject *create(int /*argc*/, const char*const* /*argv*/) {
		if (ztrace!=NULL) return ztrace;
		else {
			ztrace = new ZTraceObject;
			return ztrace;
		}
	}
} ztraceObjectClass;

#define BUFFER_LEN 10000

ZTraceObject::ZTraceObject() : buflen_(0), TclObject()
{
	if ((buffer_ = new char[BUFFER_LEN]) == NULL) {
		fprintf(stderr, "ZTrace buffer couldn't be allocated\n");
		exit(1);
	}

	signal(SIGINT, bye);
}

void
ZTraceObject::flush(void)
{
	buffer_[buflen_] = '\0';
	fprintf(stderr, "%s", buffer_);
	buflen_ = 0;
}

int
ZTraceObject::command(int argc, const char *const* argv)
{
	Tcl &tcl = Tcl::instance();

        if (!strcmp(argv[1], "trace")) {
		unsigned slen = strlen(argv[2]);

		if (slen  > BUFFER_LEN) {
			tcl.resultf("String too long to ztrace");
			return TCL_ERROR;
		 }

		if (buflen_ + slen > BUFFER_LEN) {
			flush();
		}

		/*
		 * Copy all but the braces around the string. If
		 * we ever use this from c files, change line to:
		 * strncpy(&buffer_[buflen_], argv[2], slen - 1);
		 * OR test for the presence of braces.
		 */
		strncpy(&buffer_[buflen_], argv[2]+1, slen - 2);
		buflen_ += slen - 2;
		buffer_[buflen_++] = '\n';
                return TCL_OK;
        }

	return TclObject::command(argc, argv);
}

#endif // ZTRACE

