/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <signal.h>

#include <netdude/nd.h>
#include <netdude/nd_dialog.h>
#include <netdude/nd_tcpdump.h>
#include <netdude/nd_prefs.h>
#include <netdude/nd_misc.h>
#include <netdude/nd_protocol.h>
#include <netdude/nd_protocol_inst.h>
#include <support.h>
#include <interface.h>

#ifdef HOST_BIGENDIAN
static int host_bigendian = 1;
#else
static int host_bigendian = 0;
#endif

#define	SWAP_LONG(y) \
        ((((y)&0xff)<<24) | (((y)&0xff00)<<8) | (((y)&0xff0000)>>8) | (((y)>>24)&0xff))
#define	SWAP_SHORT(y) \
	((((y)&0xff)<<8) | ((u_short)((y)&0xff00)>>8))

/* Tcpdump communication handling */

#define TCPDUMP_OPS_HARDCODED     "-l -r"

static char tcpdump_dyn_opt[MAXPATHLEN];

/**
 * nd_tcpdump_fill_in_options - creates a vector of options to pass to tcpdump.
 * @opt: string to paste parameters into
 * @opt_vec: array of char*s to receive options
 * @opt_size: length of @opt_vec
 *
 * This function fills in the given array of strings
 * by analyzing the command line options and setting
 * each string in the array to one option.
 *
 * Returns: number of options used.
 */
static int
tcpdump_fill_in_options(char *opt, char **opt_vec, int opt_size)
{
  int   flag;
  int   index = 0;
  char *p;

  sprintf(opt, "tcpdump ");
  p = opt + strlen(opt);

  if (nd_prefs_get_int_item(ND_DOM_NETDUDE, "tcpdump_resolve", &flag))
    {
      if (!flag)
	{
	  sprintf(p, "-n ");
	  p = p + strlen(p);
	}
    }

  if (nd_prefs_get_int_item(ND_DOM_NETDUDE, "tcpdump_domains", &flag))
    {
      if (!flag)
	{
	  sprintf(p, "-N ");
	  p = p + strlen(p);
	}
    }

  if (nd_prefs_get_int_item(ND_DOM_NETDUDE, "tcpdump_quick", &flag))
    {
      if (flag)
	{
	  sprintf(p, "-q ");
	  p = p + strlen(p);
	}
    }

  if (nd_prefs_get_int_item(ND_DOM_NETDUDE, "tcpdump_print_link", &flag))
    {
      if (flag)
	{
	  sprintf(p, "-e ");
	  p = p + strlen(p);
	}
    }

  if (nd_prefs_get_int_item(ND_DOM_NETDUDE, "tcpdump_print_timestamp", &flag))
    {
      if (!flag)
	{
	  sprintf(p, "-t ");
	  p = p + strlen(p);
	}
    }

  sprintf(p, tcpdump_dyn_opt);
  p = p + strlen(p);

  sprintf(p, " " TCPDUMP_OPS_HARDCODED " -");
  p = opt;

  while (TRUE)
    {
      if (index == opt_size - 2)
	break;

      if (*p == 0)
	break;
      
      if (*p == ' ')
	{
	  *p = 0;
	  p++;
	}
      else
	{
	  opt_vec[index++] = p;
	  
	  while (*p != ' ' && *p != 0)
	    p++;
	}
    }

  /* Last string must be NULL for execv */
  opt_vec[index++] = NULL;

  return index;
}

int
nd_tcpdump_init(void)
{
  char *tcpdump_path;

  if (nd_prefs_get_str_item(ND_DOM_NETDUDE, "tcpdump_path", &tcpdump_path))
    {
      if (nd_misc_can_exec(tcpdump_path))
	return (TRUE);
    }
  
  tcpdump_dyn_opt[0] = '\0';

  return (FALSE);
}


static gboolean
tcpdump_init_tracefile(ND_Trace *trace)
{
  FILE *f;
  struct pcap_file_header *pfh;

  if (!trace || !trace->filename)
    return FALSE;
  
  pfh = &(trace->tcpdump.pfh);

  /* Check if we can read the tracefile */
  if ( (f = fopen(trace->filename, "r")) == NULL)
    {
      fprintf(stderr, "tcpdump init error\n");
      return FALSE;
    }

  /* Read trace file header */
  if (fread(pfh, sizeof(struct pcap_file_header), 1, f) != 1)
    fprintf(stderr, "tcpdump init error\n");
  
  /* Swap endianness if necessary */
  if (((pfh->magic == 0xd4c3b2a1) && !host_bigendian) ||
      ((pfh->magic == 0xa1b2c3d4) && host_bigendian)  ||
      ((pfh->magic == 0x34cdb2a1) && !host_bigendian) ||
      ((pfh->magic == 0xa1b2cd34) && host_bigendian))
    {
      /* We need to swap the header: */
      pfh->magic = SWAP_LONG(pfh->magic);
      pfh->version_major = SWAP_SHORT(pfh->version_major);
      pfh->version_minor = SWAP_SHORT(pfh->version_minor);
      pfh->thiszone = SWAP_LONG(pfh->thiszone);
      pfh->sigfigs  = SWAP_LONG(pfh->sigfigs);
      pfh->snaplen  = SWAP_LONG(pfh->snaplen);
      pfh->linktype = SWAP_LONG(pfh->linktype);
    }

  pfh->magic = 0xa1b2c3d4;

  fclose(f);

  return TRUE;
}


int
nd_tcpdump_open(ND_Trace *trace)
{
  int     fd[2];
  char   *tcpdump_path;
  char    message[MAXPATHLEN];

  if (!trace)
    return FALSE;

  if (!nd_prefs_get_str_item(ND_DOM_NETDUDE, "tcpdump_path", &tcpdump_path))
    return FALSE;

  if (!nd_misc_can_exec(tcpdump_path))
    {
      g_snprintf(message, MAXPATHLEN,
	       _("An error occurred when trying to run\n"
		 "%s\n"
		 "Make sure the file exists and\n"
		 "you have correct permissions."),
		 tcpdump_path);
      
      nd_dialog_message(_("Tcpdump problem."), message, TRUE);
      return FALSE;
    }

  if (!nd_trace_initialized(trace))
    {
      if (trace->filename && !tcpdump_init_tracefile(trace))
	{
	  nd_dialog_message(_("Tcpdump problem."),
			    _("An error occurred when trying to read\n"
			      "the beginning of the trace file.\n"
			      "Make sure the file exists and\n"
			      "you have correct permissions."), TRUE);
	  return FALSE;
	}
    }

  if (socketpair(AF_UNIX, SOCK_STREAM, 0, fd) < 0)
    {
      fprintf(stderr, "Could not create socket pair -- exiting.\n");
      exit(1);
    }
  
  if ( (trace->tcpdump.pid = fork()) < 0)
    {
      fprintf(stderr, "Fork error -- exiting.\n");
      exit(1);
    }
  else if (trace->tcpdump.pid > 0)
    {
      close(fd[1]);
      trace->tcpdump.fd = fd[0];

      if (fcntl(trace->tcpdump.fd, F_SETFL, O_NONBLOCK) < 0)
	{
	  fprintf(stderr, "Can not fcntl socket -- exiting.\n");
	  exit(-1);
	}

      if (nd_trace_initialized(trace))
	nd_tcpdump_send_header(trace);      
    }
  else
    {
      /* The child replaces the process image with a new tcpdump */

      char  options[MAXPATHLEN + 50];
      char *options_vec[20];
      int   options_index;

      options_index = tcpdump_fill_in_options(options, options_vec, 20);

      close(fd[0]);
      if (fd[1] != STDIN_FILENO)
	{
	  if (dup2(fd[1], STDIN_FILENO) != STDIN_FILENO)
	    fprintf(stderr, "stdin pipe error\n");
	}
      if (fd[1] != STDOUT_FILENO)
	{
      	  if (dup2(fd[1], STDOUT_FILENO) != STDOUT_FILENO)
	    fprintf(stderr, "stdout pipe error\n");
	}
      
      if (execv(tcpdump_path, options_vec) < 0)
	{
	  perror("Ooops");
	  fprintf(stderr, "tcpdump error\n");	
	}
    }

  return TRUE;
}


/**
 * nd_tcpdump_send_header - writes a pcap file header to a tcpdump process
 * @trace: the trace whose tcpdump and save file header should be used
 */
void 
nd_tcpdump_send_header(ND_Trace *trace)
{
  /* The parent process writes the pcap trace file header
     to the tcpdump child, as the beginning of the file:
  */
  
  if (nd_misc_writen(trace->tcpdump.fd, (guchar *) &(trace->tcpdump.pfh),
		     sizeof(struct pcap_file_header))
      != sizeof(struct pcap_file_header))
    fprintf(stderr, "Write error in pipe when sending header\n");
}


void
nd_tcpdump_close(ND_Trace *trace)
{
  if (!trace)
    return;

  if (trace->tcpdump.pid <= 0)
    return;

  kill(trace->tcpdump.pid, SIGKILL);
  close(trace->tcpdump.fd);

  if (waitpid(trace->tcpdump.pid, NULL, 0) != trace->tcpdump.pid)
    {
      nd_dialog_message(_("Tcpdump problem."),
			_("An error occurred while the connection\n"
			  "to tcpdump was closed."), TRUE);
    }

  trace->tcpdump.pid = 0;
  trace->tcpdump.fd  = 0;
}


static void
tcpdump_update_line_cb(ND_Packet *packet,
		       ND_ProtoData *pd,
		       void *user_data)
{
  char *line = (char *) user_data;

  if (!packet || !pd)
    return;

  if (pd->inst.proto->is_stateful)
    pd->inst.proto->update_tcpdump_line(packet, line);
}


static gboolean
tcpdump_send_packet(int td_fd,
		    const struct pcap_pkthdr *hdr,
		    const guchar *data)
{
  fd_set       fdset;

  FD_ZERO(&fdset);
  FD_SET(td_fd, &fdset);

  if (select(td_fd + 1, NULL, &fdset, NULL, NULL) <= 0)
    goto error_exit;

  if (write(td_fd, (guchar *) hdr, sizeof(struct pcap_pkthdr)) != sizeof(struct pcap_pkthdr))
    goto error_exit;
  
  if (write(td_fd, data, hdr->caplen) != (int) hdr->caplen)
    goto error_exit;

  fsync(td_fd);


  return TRUE;

 error_exit:
  fprintf(stderr, "Write error in pipe\n");
  return FALSE;
}


/**
 * tcpdump_drain_pipe - causes remaining data in the tcpdump buffer to be sent
 * @td_fd: tcpdump file descriptor
 *
 * This function is basically a gross hack to avoid problems with tcpdump
 * output for a single packet that spans multiple lines. We cannot rely on
 * receiving all the output in one burst, and there's currently no way to
 * force an fflush() on the tcpdump side from the outside (might be worth
 * adding a signal handler for that, or add an option that causes a flush
 * after each packet -- should only be a couple lines). What this function
 * does is sending a tiny dummy packet to tcpdump that'll cause the remaining
 * data to arrive. This is far from perfect to say the least, but it's
 * better than doing nothing ...
 */
static void
tcpdump_drain_pipe(int td_fd)
{
  struct pcap_pkthdr pkthdr;
  guchar             data;
  fd_set             fdset;
  char               s[MAXPATHLEN];
  int                n;

  /* Initialize our pseudo-packet data: */
  memset(&pkthdr, 0, sizeof(struct pcap_pkthdr));
  pkthdr.len = 100;
  pkthdr.caplen = 1;
  data = 0;
  
  tcpdump_send_packet(td_fd, &pkthdr, &data);
      
  FD_ZERO(&fdset);
  FD_SET(td_fd, &fdset);
  
  if (select(td_fd + 1, &fdset, NULL, NULL, NULL) > 0)
    {
      for ( ; ; )
	{
	  if ((n = read(td_fd, s, MAXPATHLEN-1)) < 0)
	    {
	      if (errno == EINTR)
		continue;
	    }

	  break;
	}
      
      /*
	s[n] = '\0';
	D(("Drained: '%s'\n", s));
      */
    }
} 


void
nd_tcpdump_get_packet_line(const ND_Packet *packet,
			   char *dest, gboolean find_context)
{
  fd_set       fdset;
  static char  s[MAXPATHLEN];
  char        *sp, *eol = NULL, *result = NULL;
  int          td_fd, n = 0, remaining;
  gboolean     success;

  if (!packet)
    return;
  
  td_fd = packet->trace->tcpdump.fd;

  tcpdump_drain_pipe(td_fd);
  tcpdump_send_packet(td_fd, &(packet->ph), packet->data);

  sp = s;
  remaining = MAXPATHLEN-1;

  FD_ZERO(&fdset);
  FD_SET(td_fd, &fdset);
  select(td_fd + 1, &fdset, NULL, NULL, NULL);

  success = FALSE;
  result = s;

  while (remaining > 0)
    {
      if ((n = read(td_fd, sp, MAXPATHLEN-1)) > 0)
	{
	  eol = strchr(s, '\n');
	  if (eol)
	    {
	      *eol = '\0';
	      success = TRUE;
	      break;
	    }
	  
	  sp += n;
	  remaining -= n;
	}
    }

  if (success && find_context)
    {
      nd_packet_foreach_proto((ND_Packet *) packet,
			      tcpdump_update_line_cb,
			      result);
    }

  strncpy(dest, result, MAXPATHLEN);
}


void    
nd_tcpdump_options_reset(void)
{
  tcpdump_dyn_opt[0] = '\0';
}


void    
nd_tcpdump_options_add(const char *option)
{
  if (!option || !*option)
    return;

  strncat(tcpdump_dyn_opt, " ", MAXPATHLEN - strlen(tcpdump_dyn_opt));
  strncat(tcpdump_dyn_opt, option, MAXPATHLEN - strlen(tcpdump_dyn_opt));

  D(("Dynamic tcpdump options now '%s'\n", tcpdump_dyn_opt));
}
