/*
 *   Copyright (C) 1991-2000 by Jonathan Naylor HB9DRD/G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <string.h>

#include <gtk/gtk.h>

#include "global.h"

static struct {
	char   *Name;
	char   *File_Name;
	double Min_Longitude;
	double Min_Latitude;
	double Max_Longitude;
	double Max_Latitude;
} MapInfo[] = {
	{"Africa",      "africa.pbm",      -29.4440, -54.7660,  74.0970,  41.5830},
	{"Asia",        "asia.pbm",         32.6750, -11.1440, 180.3679,  79.6949},
	{"Australasia", "australasia.pbm", 112.9570, -50.9990, 184.5289, -10.7920},
	{"Europe",      "europe.pbm",      -25.0520,  31.0960, 105.0430,  81.4910},
	{"N.America",   "namerica.pbm",   -178.0709,  25.2050, -52.8140,  83.1390},
	{"Pacific",     "pacific.pbm",    -177.7920, -27.5060, -91.1169,  28.3850},
	{"S.America",   "samerica.pbm",   -177.7920, -55.7040, -34.7290,  32.4820},
	{"USA",         "usa.pbm",        -124.6833,  25.1326, -67.0096,  49.3810},
	{"World",       "world.pbm",      -179.1379, -85.5060, 180.3679,  83.7540}
};

static void map_select_cb(GtkWidget *, gpointer);
static void map_undisp_cb(GtkWidget *, gpointer);
static void map_modif_cb(GtkWidget *, gpointer);
static void map_ttable_cb(GtkWidget *, gpointer);
static void map_visib_cb(GtkWidget *, gpointer);
static void map_elems_cb(GtkWidget *, gpointer);
static void map_beep_cb(GtkWidget *, gpointer);

static GtkItemFactoryEntry map_menu_items[] = {
	{"/_Select",             NULL, map_select_cb,      0, NULL},
	{"/_Remove",             NULL, map_undisp_cb,      0, NULL},
	{"/_Modify...",          NULL, map_modif_cb,       0, NULL},
	{"/sep",                 NULL, NULL,               0, "<Separator>"},
	{"/_Tracking Table...",  NULL, map_ttable_cb,      0, NULL},
	{"/_Visibility",         NULL, map_visib_cb,       0, NULL},
	{"/_Updated Elements",   NULL, map_elems_cb,       0, NULL},
	{"/sep",                 NULL, NULL,               0, "<Separator>"},
	{"/Toggle AOS _Beep",    NULL, map_beep_cb,        0, NULL}
};

GtkItemFactory *map_itemfactory;

GdkGC *BackgroundGC;
GdkGC *CitiesGC;
GdkGC *LandGC;
GdkGC *FootprintGC;
GdkGC *SatelliteGC;
GdkGC *UserGC;

static GdkGC *TextFgGC;
static GdkGC *TextBgGC;

static GdkFont *MapFont;
static GdkFont *TextFont;

static GdkPixmap *Map;
static GdkPixmap *World;
static GdkPixmap *Text;

static void     Init_Map_Data(void);
static void     Init_Text_Data(void);
static void     Load_Map(void);
static void     Copy_Map(void);
static void     Plot_User(void);
static void     Plot_Cities(void);
static void     Redraw_Text(int, int, int, int);
static void     Redraw_Map(int, int, int, int);
static void     Redraw_Satellite(struct Sat_Struct *);
static void     Redraw_Footprint(struct Sat_Struct *);
static double   Calc_X_Scale(void);
static double   Calc_Y_Scale(void);
static int      Long_To_X(double, double);
static int      Lat_To_Y(double, double);
static void     Draw_Line(int, int, int, int);

static struct Sat_Struct *Click_Sat = NULL;

static void map_select_cb(GtkWidget *w, gpointer data)
{
	struct Sat_Struct *Sat;

	if (Click_Sat == NULL)
		return;
	
	if ((Sat = Find_Current()) != NULL) {
		if (Sat == Click_Sat)
			return;
	
		Sat->Current = FALSE;
	}

	Click_Sat->Current = TRUE;

	Track_Satellite(time(NULL));  /* fixme: this is missing "+ Time_Offset" compared to version in mtrack.c */
}

static void map_undisp_cb(GtkWidget *w, gpointer data)
{
	struct Sat_Struct *Sat;
	GSList *Sat_Iterator;

	Num_Chosen--;

	if (Click_Sat == NULL)
		return;

	Click_Sat->Display = FALSE;

	if (Click_Sat->Current) {
		Click_Sat->Current = FALSE;

		/* Now give the Current flag to another selected satellite */
		Sat_Iterator = Sat_List;

		while (Sat_Iterator != NULL) {
			Sat = (struct Sat_Struct *)Sat_Iterator->data;

			if (Sat->Display) {
				Sat->Current = TRUE;
				break;
			}

			Sat_Iterator = g_slist_next(Sat_Iterator);
		}
	}

	Track_Satellite(time(NULL));  /* fixme: this is missing "+ Time_Offset" compared to version in mtrack.c */
}

static void map_modif_cb(GtkWidget *w, gpointer data)
{
	if (Click_Sat == NULL)
		return;

	Update_Sat(Click_Sat);
}

static void map_ttable_cb(GtkWidget *w, gpointer data)
{
	if (Click_Sat == NULL)
		return;

	List_Sat(Click_Sat);
}

static void map_visib_cb(GtkWidget *w, gpointer data)
{
	if (Click_Sat == NULL)
		return;

	Show_Locations(Click_Sat);
}

static void map_elems_cb(GtkWidget *w, gpointer data)
{
	if (Click_Sat == NULL)
		return;

	Show_Elements(Click_Sat);
}

void map_beep_cb(GtkWidget *w, gpointer data)
{
	if (Click_Sat == NULL)
		return;

	Click_Sat->Beep = !Click_Sat->Beep;
}

void MapCb(GtkWidget *w, gpointer data)
{
	int n = (int)data;
	
	if (User_Data.Map_No != n) {
		User_Data.Map_No = n;
		Load_Map();
		Redraw_Satellites();
	}
}

gint maparea_buttonpress_cb(GtkWidget *w, GdkEventButton *ev, gpointer dummy)
{
	double x_scale = Calc_X_Scale();
	double y_scale = Calc_Y_Scale();
	int plot_x, plot_y;
	long Dist, Best_Dist = 1000000000L;
	GSList *Sat_Iterator;
	struct Sat_Struct *Sat;

	/* search satellite closest to cursor */
	Click_Sat    = NULL;
	Sat_Iterator = Sat_List;
	
	while (Sat_Iterator != NULL) {
		Sat = (struct Sat_Struct *)Sat_Iterator->data;
	
		if (Sat->Display) {
			plot_x = Long_To_X(Sat->SSP_Longitude, x_scale);
			plot_y = Lat_To_Y(Sat->SSP_Latitude, y_scale);

			if (plot_x >= 0 && plot_x < MAP_WIDTH && plot_y >= 0 && plot_y < MAP_HEIGHT) {
				Dist = (ev->x - plot_x) * (ev->x - plot_x) + (ev->y - plot_y) * (ev->y - plot_y);

				if (Dist < Best_Dist) {
					Best_Dist = Dist;
					Click_Sat = Sat;
				}
			}
		}

		Sat_Iterator = g_slist_next(Sat_Iterator);
	}

	if (Click_Sat == NULL)
		return TRUE;

	if (ev->button == 1) {   /* left mouse button pressed */
		if ((Sat = Find_Current()) != NULL) {
			if (Sat == Click_Sat)
				return TRUE;
			Sat->Current = FALSE;
		}

		Click_Sat->Current = TRUE;
		Track_Satellite(time(NULL));	/* fixme: this is missing "+ Time_Offset" compared to version in mtrack.c */
	} else if (ev->button == 3) {   /* right mouse button pressed */
		gint x, y;
		gdk_window_get_origin(w->window, &x, &y);
		gtk_item_factory_popup(map_itemfactory, x + ev->x, y + ev->y, ev->button, ev->time);
	}

	return TRUE;
}

gint MapAreaCb(GtkWidget *w, GdkEventExpose *event)
{
	static int First_Time = TRUE;

	if (First_Time) {
		Init_Map_Data();
		First_Time = FALSE;

		Map   = gdk_pixmap_new(MapArea->window, MAP_WIDTH, MAP_HEIGHT, -1);
		World = gdk_pixmap_new(MapArea->window, MAP_WIDTH, MAP_HEIGHT, -1);

		Load_Map();

		map_itemfactory = gtk_item_factory_new(GTK_TYPE_MENU, "<popup>", NULL);
		gtk_item_factory_create_items(map_itemfactory, sizeof(map_menu_items) / sizeof(GtkItemFactoryEntry), map_menu_items, NULL);

		/*
		 * Draw everything except the satellites, this will keep the
		 * screen busy until the timeout occurs and everything else
		 * is drawn.
		 */
		Copy_Map();
		Plot_Cities();
		Plot_User();
		Redraw_Map(0, 0, MAP_WIDTH, MAP_HEIGHT);
	}

	Redraw_Map(event->area.x, event->area.y, event->area.width, event->area.height);

	return FALSE;
}

void Clear_Text(void)
{
	gdk_draw_rectangle(Text, TextBgGC, TRUE, 0, 0, MAP_WIDTH, MAP_TEXT_HEIGHT);
}

void Refresh_Text(void)
{
	Redraw_Text(0, 0, MAP_WIDTH, MAP_TEXT_HEIGHT);
}

void Write_Text(int pos, char *text)
{
	int x, y;

	switch (pos) {
		case TEXT_SAT:
			x = (MAP_WIDTH / 12) * 0 + 5;
			y = (MAP_TEXT_HEIGHT / 4) * 1 - 5;
			break;
		case TEXT_DATETIME:
			x = (MAP_WIDTH / 12) * 3;
			y = (MAP_TEXT_HEIGHT / 4) * 1 - 5;
			break;
		case TEXT_AZIMUTH:
			x = (MAP_WIDTH / 12) * 8;
			y = (MAP_TEXT_HEIGHT / 4) * 1 - 5;
			break;
		case TEXT_ELEVATION:
			x = (MAP_WIDTH / 12) * 10;
			y = (MAP_TEXT_HEIGHT / 4) * 1 - 5;
			break;
		case TEXT_MA:
			x = (MAP_WIDTH / 12) * 10;
			y = (MAP_TEXT_HEIGHT / 4) * 2 - 5;
			break;
		case TEXT_MODE:
			x = (MAP_WIDTH / 12) * 0 + 5;
			y = (MAP_TEXT_HEIGHT / 4) * 2 - 5;
			break;
		case TEXT_RANGE:
			x = (MAP_WIDTH / 12) * 3;
			y = (MAP_TEXT_HEIGHT / 4) * 2 - 5;
			break;
		case TEXT_ORBIT:
			x = (MAP_WIDTH / 12) * 6;
			y = (MAP_TEXT_HEIGHT / 4) * 2 - 5;
			break;
		case TEXT_SQUINT:
			x = (MAP_WIDTH / 12) * 8;
			y = (MAP_TEXT_HEIGHT / 4) * 2 - 5;
			break;
		case TEXT_ALOS1:
			x = (MAP_WIDTH / 12) * 0 + 5;
			y = (MAP_TEXT_HEIGHT / 4) * 3 - 5;
			break;
		case TEXT_ALOS2:
			x = (MAP_WIDTH / 12) * 6;
			y = (MAP_TEXT_HEIGHT / 4) * 3 - 5;
			break;
		case TEXT_FREQ1:
			x = (MAP_WIDTH / 12) * 0 + 5;
			y = (MAP_TEXT_HEIGHT / 4) * 4 - 5;
			break;
		case TEXT_FREQ2:
			x = (MAP_WIDTH / 12) * 6;
			y = (MAP_TEXT_HEIGHT / 4) * 4 - 5;
			break;
		default:
			g_error("mtrack: unknown text position %d\n", pos);
			return;
	}

	gdk_draw_string(Text, TextFont, TextFgGC, x, y, text);
}

gint MapTextCb(GtkWidget *w, GdkEventExpose *event)
{
	static int First_Time = TRUE;

	if (First_Time) {
		Init_Text_Data();
		First_Time = FALSE;

		Text = gdk_pixmap_new(MapText->window, MAP_WIDTH, MAP_TEXT_HEIGHT, -1);
		gdk_draw_rectangle(Text, TextBgGC, TRUE, 0, 0, MAP_WIDTH, MAP_TEXT_HEIGHT);
	}

	Redraw_Text(event->area.x, event->area.y, event->area.width, event->area.height);

	return FALSE;
}

static void Init_Map_Data(void)
{
	Allocate_Colours();

	MapFont = gdk_font_load("-adobe-helvetica-medium-r-normal-*-*-80-*-*-p-*-iso8859-1");

	BackgroundGC = gdk_gc_new(MapArea->window);
	gdk_gc_set_foreground(BackgroundGC, &User_Data.Colours[COLOUR_BACKGROUND]);

	CitiesGC = gdk_gc_new(MapArea->window);
	gdk_gc_set_foreground(CitiesGC, &User_Data.Colours[COLOUR_CITIES]);

	FootprintGC  = gdk_gc_new(MapArea->window);
	gdk_gc_set_foreground(FootprintGC, &User_Data.Colours[COLOUR_FOOTPRINT]);

	LandGC       = gdk_gc_new(MapArea->window);
	gdk_gc_set_foreground(LandGC, &User_Data.Colours[COLOUR_LAND]);

	SatelliteGC  = gdk_gc_new(MapArea->window);
	gdk_gc_set_foreground(SatelliteGC, &User_Data.Colours[COLOUR_SATELLITE]);
	gdk_gc_set_font(SatelliteGC, MapFont);

	UserGC       = gdk_gc_new(MapArea->window);
	gdk_gc_set_foreground(UserGC, &User_Data.Colours[COLOUR_USER]);
	gdk_gc_set_font(UserGC, MapFont);
}

static void Init_Text_Data(void)
{
	Allocate_Colours();

	TextFont = gdk_font_load("-adobe-helvetica-medium-r-normal-*-*-120-*-*-p-*-iso8859-1");

	TextBgGC = gdk_gc_new(MapText->window);
	gdk_gc_set_foreground(TextBgGC, &Text_Bg_Colour);

	TextFgGC = gdk_gc_new(MapText->window);
	gdk_gc_set_foreground(TextFgGC, &Text_Fg_Colour);
	gdk_gc_set_font(TextFgGC, TextFont);
}

static void Copy_Map(void)
{
	gdk_draw_pixmap(Map, BackgroundGC, World, 0, 0, 0, 0, MAP_WIDTH, MAP_HEIGHT);
}

static void Redraw_Map(int x, int y, int Width, int Height)
{
	gdk_draw_pixmap(MapArea->window, BackgroundGC, Map, x, y, x, y, Width, Height);
}

static void Redraw_Text(int x, int y, int Width, int Height)
{
	gdk_draw_pixmap(MapText->window, TextBgGC, Text, x, y, x, y, Width, Height);
}

static void Load_Map(void)
{
	unsigned char Buffer[98];
	FILE *fp;
	int  i;
	int  x;
	int  y = 0;

	sprintf(Buffer, "%s/%s", MAPDIR, MapInfo[User_Data.Map_No].File_Name);

	gdk_draw_rectangle(World, BackgroundGC, TRUE, 0, 0, MAP_WIDTH, MAP_HEIGHT);

	if ((fp = fopen(Buffer, "r")) == NULL) {
		sprintf(Buffer, "Cannot open map file %s", MapInfo[User_Data.Map_No].File_Name);
		Error_Box(Buffer);
		return;
	}

	fgets(Buffer, 40, fp);
	
	if (strcmp(Buffer, "P4\n") != 0) {
		sprintf(Buffer, "Map file %s is corrupt", MapInfo[User_Data.Map_No].File_Name);
		Error_Box(Buffer);
		return;
	}

	fgets(Buffer, 40, fp);
	
	if (strcmp(Buffer, "780 400\n") != 0) {
		sprintf(Buffer, "Map file %s is corrupt", MapInfo[User_Data.Map_No].File_Name);
		Error_Box(Buffer);
		return;
	}

	while (fread(Buffer, 98, 1, fp) == 1) {
		for (x = 0; x < 98; x++)
			for (i = 0; i < 8; i++)
				if ((Buffer[x] & (0x80 >> i)) != 0x00)
					gdk_draw_point(World, LandGC, x * 8 + i, y);

		y++;
	}

	fclose(fp);
}

static void Plot_User(void)
{
	int x, y;

	x = Long_To_X(User_Data.Longitude, Calc_X_Scale());
	y = Lat_To_Y(User_Data.Latitude, Calc_Y_Scale());

	if (x > 0 && y > 0 && x < MAP_WIDTH && y < MAP_HEIGHT)
		gdk_draw_rectangle(Map, UserGC, TRUE, x - 2, y - 2, 4, 4);

	x -= gdk_string_width(MapFont, User_Data.Callsign) / 2;
	y += gdk_string_height(MapFont, User_Data.Callsign) + 3;

	gdk_draw_string(Map, MapFont, UserGC, x, y, User_Data.Callsign);
}

static void Plot_Cities(void)
{
	GSList *Loc;
	struct Loc_Struct *Data;
	int x, y;

	Loc = Loc_List;

	while (Loc != NULL) {
		Data = (struct Loc_Struct *)Loc->data;

		x = Long_To_X(Data->Longitude, Calc_X_Scale());
		y = Lat_To_Y(Data->Latitude, Calc_Y_Scale());

		if (x > 0 && y > 0 && x < MAP_WIDTH && y < MAP_HEIGHT)
			gdk_draw_point(Map, CitiesGC, x, y);

		Loc = g_slist_next(Loc);
	}
}

static void Redraw_Satellite(struct Sat_Struct *Sat)
{
	double x_scale = Calc_X_Scale();
	double y_scale = Calc_Y_Scale();
	int plot_x, plot_y;
	char Buffer[24];

	plot_x = Long_To_X(Sat->SSP_Longitude, x_scale);
	plot_y = Lat_To_Y(Sat->SSP_Latitude, y_scale);

	if (plot_x > 0 && plot_y > 0 && plot_x < MAP_WIDTH && plot_y < MAP_HEIGHT) {
		gdk_draw_rectangle(Map, SatelliteGC, TRUE, plot_x - 3, plot_y - 3, 6, 6);

		strcpy(Buffer, Sat->Name);
		
		if (Sat->Beep) {
			strcat(Buffer, " (");
			if (Sat->Beep) strcat(Buffer, "B");
			strcat(Buffer, ")");
		}

		plot_x -= gdk_string_width(MapFont, Buffer) / 2;
		plot_y += gdk_string_height(MapFont, Buffer) + 5;

		gdk_draw_string(Map, MapFont, SatelliteGC, plot_x, plot_y, Buffer);
	}
}

static void Redraw_Footprint(struct Sat_Struct *Sat)
{
	int No_Points;
	double Theta;
	double Increment;
	double Lat;
	double Long;
	double cla, sla;
	double clo, slo;
	double sra, cra;
	double X, Y, Z;
	double x, y, z;
	int plot_x, plot_y;
	int last_x1, last_x2, last_y1, last_y2;
	double x_scale = Calc_X_Scale();
	double y_scale = Calc_Y_Scale();

	No_Points = (int)(Sat->Half_Angle + 0.5) + 3;
	Increment = PI / (double)(No_Points - 1);

	cla = cos(RAD(Sat->SSP_Latitude));
	sla = sin(RAD(Sat->SSP_Latitude));
	clo = cos(RAD(Sat->SSP_Longitude));
	slo = sin(RAD(Sat->SSP_Longitude));
	cra = cos(RAD(Sat->Half_Angle));
	sra = sin(RAD(Sat->Half_Angle));

	Long = Sat->SSP_Longitude;
	Lat  = Sat->SSP_Latitude + Sat->Half_Angle;

	if (Lat > 90.0) {
		Long += 180.0;
		Lat  =  180.0 - Lat;

		if (Long >= 360.0) Long -= 360.0;
	}

	last_x1 = last_x2 = Long_To_X(Long, x_scale);
	last_y1 = last_y2 = Lat_To_Y(Lat, y_scale);

	for (Theta = Increment; Theta < PI; Theta += Increment) {
		X = cra;
		Y = sra * sin(Theta);
		Z = sra * cos(Theta);

		x = X * cla - Z * sla;
		y = Y;
		z = X * sla + Z * cla;

		X = x * clo - y * slo;
		Y = x * slo + y * clo;

		Lat  = DEG(asin(z));

		Long = DEG(atan2(Y, X));

		plot_x = Long_To_X(Long, x_scale);
		plot_y = Lat_To_Y(Lat, y_scale);

		Draw_Line(last_x1, last_y1, plot_x, plot_y);

		last_x1 = plot_x;
		last_y1 = plot_y;

		X = x * clo + y * slo;
		Y = x * slo - y * clo;

		Long = DEG(atan2(Y, X));

		plot_x = Long_To_X(Long, x_scale);
		plot_y = Lat_To_Y(Lat, y_scale);

		Draw_Line(last_x2, last_y2, plot_x, plot_y);

		last_x2 = plot_x;
		last_y2 = plot_y;
	}

	Long = Sat->SSP_Longitude;
	Lat  = Sat->SSP_Latitude - Sat->Half_Angle;

	if (Lat < -90.0) {
		Long += 180.0;
		Lat  = -180.0 - Lat;

		if (Long >= 360.0) Long -= 360.0;
	}

	plot_x = Long_To_X(Long, x_scale);
	plot_y = Lat_To_Y(Lat, y_scale);

	Draw_Line(last_x1, last_y1, plot_x, plot_y);
	Draw_Line(last_x2, last_y2, plot_x, plot_y);
}

void Redraw_Satellites(void)
{
	GSList *Sat_Iterator;
	struct Sat_Struct *Sat;

	Copy_Map();
		
	Plot_Cities();

	Sat_Iterator = Sat_List;
	
	while (Sat_Iterator != NULL) {
		Sat = (struct Sat_Struct *)Sat_Iterator->data;

		if (Sat->Display)
			Redraw_Satellite(Sat);

		if (Sat->Current)
			Redraw_Footprint(Sat);

		Sat_Iterator = g_slist_next(Sat_Iterator);
	}

	Plot_User();

	Redraw_Map(0, 0, MAP_WIDTH, MAP_HEIGHT);
}

static void Draw_Line(int x1, int y1, int x2, int y2)
{
	/* Too long */
	if (abs(x1 - x2) > 100)
		return;

	/* Both ends are outside the map */
	if ((x1 < 0 || x1 >= MAP_WIDTH || y1 < 0 || y1 >= MAP_HEIGHT) &&
	    (x2 < 0 || x2 >= MAP_WIDTH || y2 < 0 || y2 >= MAP_HEIGHT))
		return;

	gdk_draw_line(Map, FootprintGC, x1, y1, x2, y2);
}

static double Calc_X_Scale(void)
{
	double x_scale;

	if (MapInfo[User_Data.Map_No].Max_Longitude > 0.0)
		x_scale = (double)MAP_WIDTH / (MapInfo[User_Data.Map_No].Max_Longitude - MapInfo[User_Data.Map_No].Min_Longitude);
	else
		x_scale = (double)MAP_WIDTH / fabs(MapInfo[User_Data.Map_No].Min_Longitude - MapInfo[User_Data.Map_No].Max_Longitude);

	return x_scale;
}

static double Calc_Y_Scale(void)
{
	double y_scale;

	if (MapInfo[User_Data.Map_No].Max_Latitude > 0.0)
		y_scale = (double)MAP_HEIGHT / (MapInfo[User_Data.Map_No].Max_Latitude - MapInfo[User_Data.Map_No].Min_Latitude);
	else
		y_scale = (double)MAP_HEIGHT / fabs(MapInfo[User_Data.Map_No].Min_Latitude - MapInfo[User_Data.Map_No].Max_Latitude);

	return y_scale;
}

static int Long_To_X(double Longitude, double X_Scale)
{
	int x;

	if (Longitude > 180.0)  Longitude -= 360.0;
	if (Longitude < -180.0) Longitude += 360.0;

	x = (int)(X_Scale * (-Longitude - MapInfo[User_Data.Map_No].Min_Longitude) + 0.5);

	return x;
}

static int Lat_To_Y(double Latitude, double Y_Scale)
{
	int y;

	y = MAP_HEIGHT - (int)(Y_Scale * (Latitude - MapInfo[User_Data.Map_No].Min_Latitude) + 0.5);

	return y;
}
