/* outboxjobkeys.cpp

    copyright   : (C) 2001 by Martin Preuss
    email       : openhbci@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/



#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "assert.h"
#include "outboxjobkeys.h"
#include "mediumrdhbase.h"
#include <accountjobs.h>
namespace HBCI {

// This const_cast is used frequently in this file since the
// Pointer to the customer won't be changed (thus doesn't violate
// his const declaration) but it is impossible to express this with
// the Pointer, unfortunately.
static Pointer<Customer>
custPointer_const_cast(const Customer *c)
{
    Pointer<Customer> cp = const_cast<Customer*>(c);
    cp.setAutoDelete(false);
    return cp;
}


OutboxJobKeys::OutboxJobKeys(const API *api, 
			     Pointer<Customer> c,
			     bool sendkeys)
:OutboxJob(c)
,_sendkeys(sendkeys)
,_hbciapi(api)
{
}


OutboxJobKeys::~OutboxJobKeys(){
}


bool OutboxJobKeys::createHBCIJobs(Pointer<MessageQueue> mbox, int n){
  if (n==0) {
    // create appropriate init job
    if (_sendkeys)
      _initjob=new JOBFirstInit(_cust);
    else
      _initjob=new JOBDialogInit(_cust,
				 true,       // anon
				 false,      // crypt
				 false,      // sign
				 !_sendkeys, // getkeys
				 false);     // sync

    mbox.ref().addJob(_initjob);
    addSignersToQueue(mbox);
    return true;
  }
  else if (n==1) {
    // create exit job
    _exitjob=new JOBDialogEnd(_cust,mbox.ref().dialogId(),_sendkeys,false);
    mbox.ref().addJob(_exitjob);
    addSignersToQueue(mbox);
    return true;
  }
  else
    return false;
}


bool OutboxJobKeys::evaluate(){
  if (!_initjob.isValid() || !_exitjob.isValid())
    return false;
  if (_initjob.ref().hasErrors() || _exitjob.ref().hasErrors())
    _result=HBCI_JOB_RESULT_FAILED;
  else
    _result=HBCI_JOB_RESULT_SUCCESS;

  return _result==HBCI_JOB_RESULT_SUCCESS;
}


bool OutboxJobKeys::commit(int msgNumber){
  if (!_initjob.isValid())
    return false;

  if (_sendkeys)
    // nothing to commit if this is an object of class
    // OutboxJobSendKeys. FIXME: This would be better solved by
    // defining a commit() method in each derived class.
    return true;

  // Cheat around the const-correctness. FIXME: There must be a
  // better way around that -- probably we shouldn't save a pointer
  // to const API but rather one to a non-const API.
  API * a = const_cast<API*>(_hbciapi);
  // This is a workaround for as long as we have one method that
  // takes care of each derived class. We checked above (_sendkeys)
  // that this is never a pointer to a JOBFirstInit.
  JOBDialogInit &j = dynamic_cast<JOBDialogInit&>(_initjob.ref());
  return a->postProcessInitJob(j).isOk();
}

list<int> OutboxJobKeys::resultCodes() const{
  list<int> res, exitres;
  if (_initjob.isValid())
    res = resultCodesFromJob(_initjob.ref());
  if (_exitjob.isValid())
    exitres = resultCodesFromJob(_exitjob.ref());
  // Note: this is not the most efficient implementation. However,
  // since this is only called when something went wrong, it isn't
  // used often.
  res.insert(res.end(), exitres.begin(), exitres.end());
  return res;
}


/*************************************************************************
 *** Get Keys ***********************************************************
 *************************************************************************/
OutboxJobGetKeys::OutboxJobGetKeys(const API *api, 
								   Pointer<Customer> c)
:OutboxJobKeys(api,c,false)
{
}


OutboxJobGetKeys::~OutboxJobGetKeys(){
}

} // namespace HBCI
HBCI_OutboxJobGetKeys *
HBCI_OutboxJobGetKeys_new(const HBCI_API *api, const HBCI_Customer *c)
{
  assert(api);
  assert(c);
  return new HBCI_OutboxJobGetKeys(api, custPointer_const_cast(c));
}
HBCI_OutboxJob *
HBCI_OutboxJobGetKeys_OutboxJob(HBCI_OutboxJobGetKeys *j)
{
  return j;
}

/*************************************************************************
 *** Send Keys ***********************************************************
 *************************************************************************/
namespace HBCI {

OutboxJobSendKeys::OutboxJobSendKeys(const API *api, 
				     Pointer<Customer> c)
:OutboxJobKeys(api,c,true)
{
}


OutboxJobSendKeys::~OutboxJobSendKeys(){
}

} // namespace HBCI
HBCI_OutboxJobSendKeys *
HBCI_OutboxJobSendKeys_new(const HBCI_API *api, const HBCI_Customer *c)
{
    assert(api);
    assert(c);
    return new HBCI_OutboxJobSendKeys(api, custPointer_const_cast(c));
}
HBCI_OutboxJob *
HBCI_OutboxJobSendKeys_OutboxJob(HBCI_OutboxJobSendKeys *j)
{
    return j;
}

/*************************************************************************
 *** Disable Keys ***********************************************************
 *************************************************************************/
namespace HBCI {
OutboxJobDisableKeys::OutboxJobDisableKeys(const API *api, 
				   Pointer<Customer> c)
:OutboxJobKeys(api,c,false)
{
}


OutboxJobDisableKeys::~OutboxJobDisableKeys(){
}

bool OutboxJobDisableKeys::commit(int msgNumber){
  if (!_job.isValid())
    return false;

  if (0 == msgNumber) {
    // Cheat around the const-correctness. FIXME: There must be a
    // better way around that -- probably we shouldn't save a pointer
    // to const API but rather one to a non-const API.
    API * a = const_cast<API*>(_hbciapi);
    // This is a workaround for as long as we have one method that
    // takes care of each derived class. We checked above (_sendkeys)
    // that this is never a pointer to a JOBFirstInit.
    JOBDialogInit &j = dynamic_cast<JOBDialogInit&>(_initjob.ref());
    return a->postProcessInitJob(j).isOk();
  } else
    return true;
}

bool OutboxJobDisableKeys::createHBCIJobs(Pointer<MessageQueue> mbox, int n){
  fprintf(stderr, "OutboxJobDisableKeys: n is %d\n", n);
  if (n==0) {
    // create appropriate init job
    fprintf(stderr, "Creating init job.\n");
    _initjob=new JOBDialogInit(_cust,
			       false, // anon
			       true,  // crypt
			       true,  // sign
			       false, // getkeys
			       false); // sync
    mbox.ref().addJob(_initjob);
    addSignersToQueue(mbox);
    return true;
  }
  if (n==1) {
    // create lock-job
    _job = new JOBPublicKeysDisable(_cust);
    addSignersToQueue(mbox);
    mbox.ref().addJob(_job);
    return true;
  }
  else if (n==2) {
    // create exit job (encrypted but not signed!)
    _exitjob=new JOBDialogEnd(_cust,mbox.ref().dialogId(), true, false);
    mbox.ref().addJob(_exitjob);
    addSignersToQueue(mbox);
    return true;
  }
  else
    return false;
}

} // namespace HBCI
HBCI_OutboxJobDisableKeys *
HBCI_OutboxJobDisableKeys_new(const HBCI_API *api, const HBCI_Customer *c)
{
    assert(api);
    assert(c);
    return new HBCI_OutboxJobDisableKeys(api, custPointer_const_cast(c));
}
HBCI_OutboxJob *
HBCI_OutboxJobDisableKeys_OutboxJob(HBCI_OutboxJobDisableKeys *j)
{
    return j;
}


/*************************************************************************
 *** Disable Lost Keys ***********************************************************
 *************************************************************************/
namespace HBCI {
OutboxJobDisableLostKeys::OutboxJobDisableLostKeys(const API *api, 
				   Pointer<Customer> c, int keyNumber, int keyVersion)
:OutboxJobDisableKeys(api,c)
{
  _keyNumber = keyNumber;
  _keyVersion = keyVersion;
}


OutboxJobDisableLostKeys::~OutboxJobDisableLostKeys(){
}

bool OutboxJobDisableLostKeys::createHBCIJobs(Pointer<MessageQueue> mbox, int n){
  if (n==0) {
    // create appropriate init job
    _initjob=new JOBDialogInit(_cust,
			       false,  // anon
			       true,   // crypt
			       false,  // sign
			       false,  // getkeys
			       false); // sync
    mbox.ref().addJob(_initjob);

    // create lock-job
    _job = new JOBPublicKeysDisable(_cust, _keyVersion, _keyNumber);
    mbox.ref().addJob(_job);
    return true;

  }
  else if (n==1) {
    // create exit job
    _exitjob=new JOBDialogEnd(_cust,mbox.ref().dialogId(), true, false);
    mbox.ref().addJob(_exitjob);
    return true;
  }
  else
    return false;
}


} // namespace HBCI
HBCI_OutboxJobDisableLostKeys *
HBCI_OutboxJobDisableLostKeys_new(const HBCI_API *api, const HBCI_Customer *c,
								  int keyNumber, int keyVersion)
{
    assert(api);
    assert(c);
    return new HBCI_OutboxJobDisableLostKeys(api, custPointer_const_cast(c),
											 keyNumber, keyVersion);
}
HBCI_OutboxJob *
HBCI_OutboxJobDisableLostKeys_OutboxJob(HBCI_OutboxJobDisableLostKeys *j)
{
    return j;
}

/*************************************************************************
 *** Change Keys ***********************************************************
 *************************************************************************/
namespace HBCI {
OutboxJobChangeKeys::OutboxJobChangeKeys(const API *api, 
										 Pointer<Customer> c)
:OutboxJobKeys(api, c, false)
{}


OutboxJobChangeKeys::~OutboxJobChangeKeys(){
}

bool OutboxJobChangeKeys::createHBCIJobs(Pointer<MessageQueue> mbox, int n){
  if (n==0) {
    // create appropriate init job
    _initjob=new JOBDialogInit(_cust,
			       false, // anon
			       true,  // crypt
			       true,  // sign
			       false, // getkeys
			       false); // sync
    mbox.ref().addJob(_initjob);
    return true;
  }
  else if (n==1) {
    // create lock-job
    _job = new JOBPublicKeysChange(_cust);
    mbox.ref().addJob(_job);
    return true;
  }
  else if (n==2) {
    // create exit job
    _exitjob=new JOBDialogEnd(_cust,mbox.ref().dialogId(), true, true);
    mbox.ref().addJob(_exitjob);
    return true;
  }
  else
    return false;
}

bool OutboxJobChangeKeys::commit(int msgNumber){
    if (!_job.isValid())
        return false;

	if (0 == msgNumber) {
	  // Cheat around the const-correctness. FIXME: There must be a
	  // better way around that -- probably we shouldn't save a pointer
	  // to const API but rather one to a non-const API.
	  API * a = const_cast<API*>(_hbciapi);
	  // This is a workaround for as long as we have one method that
	  // takes care of each derived class. We checked above (_sendkeys)
	  // that this is never a pointer to a JOBFirstInit.
	  JOBDialogInit &j = dynamic_cast<JOBDialogInit&>(_initjob.ref());
	  return a->postProcessInitJob(j).isOk();
	} else
	  return true;
}

} // namespace HBCI
HBCI_OutboxJobChangeKeys *
HBCI_OutboxJobChangeKeys_new(const HBCI_API *api, const HBCI_Customer *c)
{
    assert(api);
    assert(c);
    return new HBCI_OutboxJobChangeKeys(api, custPointer_const_cast(c));
}
HBCI_OutboxJob *
HBCI_OutboxJobChangeKeys_OutboxJob(HBCI_OutboxJobChangeKeys *j)
{
    return j;
}



