/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "debug.h"
#include "error.h"
#include "util.h"

Sigfunc *
Signal(int signo, Sigfunc *func)
{
	struct sigaction	act, oact;

	act.sa_handler = func;
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	if (signo == SIGALRM)
	{
		#ifdef SA_INTERRUPT
			act.sa_flags |= SA_INTERRUPT;
		#endif
	}
	else
	{
		#ifdef SA_RESTART
			act.sa_flags |= SA_RESTART;
		#endif
	}

	if (sigaction(signo, &act, &oact) < 0)
		return (SIG_ERR);

	return (oact.sa_handler);
}

static void
alarmed(int signo)
{
	return;
}

int
connect_timeout(int sockfd, const SA *saptr, socklen_t salen, int secs)
{
	Sigfunc	*ofunc;
	int	rc;

	ofunc = Signal(SIGALRM, alarmed);

	if (alarm(secs) != 0)
		error(0, "Alarm was already set!");

	if ( (rc = connect(sockfd, saptr, salen)) < 0)
	{
		close(sockfd);
		if (errno == EINTR)
			errno = ETIMEDOUT;
	}

	alarm(0);
	Signal(SIGALRM, ofunc);

	return (rc);
}

int
read_timeout(int fd, char *buf, int len, int secs)
{
	int		rc;
	fd_set		rfds;
	struct timeval	tv;

	if (fd < 0)
		return -1;
	FD_ZERO(&rfds);
	FD_SET(fd, &rfds);
	tv.tv_sec = secs;
	tv.tv_usec = 0;
	rc = select(fd+1, &rfds, NULL, NULL, &tv);
	if (rc == 0)
		return -1;
	if (rc < 0)
		return -1;
	if (!FD_ISSET(fd, &rfds))
		return -1;

	return read(fd, buf, len);
}

/*
 * This is a hack for Darwin OS X, whose gethostbyname()
 * function sadly doesn't accept dotted addresses, and for
 * recent versions of glibc (2.3?) which causes dynamic linking
 * problems with libnss.
 */
struct hostent *
mygethostbyname(const char *name)
{
	static long	ipaddr;
	char		buf[512];
	FILE		*fp;
	#ifndef INADDR_NONE
	    #define INADDR_NONE -1
	#endif

	/*
	 * See if its a dotted decimal...
	 */
	ipaddr = inet_addr(name);
	if (ipaddr != INADDR_NONE)
	{
		static struct hostent	he;
		static char	*addrs[2];
	    dotted:
		he.h_name = NULL;
		he.h_aliases = NULL;
		he.h_addrtype = AF_INET;
		he.h_length = 4;
		he.h_addr_list = addrs;
		he.h_addr_list[0] = (char *) &ipaddr;
		he.h_addr_list[1] = NULL;
		return (&he);
	}

	/*
	 * Try to use the 'host' command.  This avoids dynamic linking
	 * problems introduced in recent versions (2.3?) of glibc.
	 */
	sprintf(buf, "host -W20 '%s' 2>/dev/null", name);
	fp = popen(buf, "r");
	if (fp)
	{
	    while (fgets(buf, sizeof(buf), fp) != NULL)
	    {
		char	*p;
		p = strrchr(buf, '\n'); if (p) *p = 0;
		p = strrchr(buf, ' ');
		if (!p) continue;
		ipaddr = inet_addr(p+1);
		if (ipaddr != INADDR_NONE)
		{
		    pclose(fp);
		    debug(5, "Resolved '%s' to '%s'\n", name, p+1);
		    goto dotted;
		}
	    }
	    pclose(fp);
	}

	/*
	 * Last resort, use the system gethostbyname()
	 */
	return gethostbyname(name);
}

/*
 * Write a timestamp to a file
 */
void
timestamp(FILE *fp)
{
	struct	timeval		tv;
	static struct timeval	start;

	gettimeofday(&tv, NULL);
	if (start.tv_sec == 0)
		start = tv;
	fprintf(fp, "%6ld.%03ld ",
		(long) (tv.tv_sec - start.tv_sec),
		(long) (tv.tv_usec / 1000));
}

/*
 * Hexdump stream data
 */
void
hexdump(FILE *fp, char *lbl1, char *lbln, const void *vdata, int length)
{
	int			s;
	int			i;
	int			n;
	unsigned char		c;
	unsigned char		buf[16];
	const unsigned char	*data = vdata;

	if (length == 0)
	{
		fprintf(fp, "%s [length 0]\n", lbl1);
		return;
	}
	for (s = 0; s < length; s += 16)
	{
		fprintf(fp, "%s", s ? lbln : lbl1);
		n = length - s; if (n > 16) n = 16;
		for (i = 0; i < 16; ++i)
		{
			if (i == 8)
				fprintf(fp, " ");
			if (i < n)
				fprintf(fp, " %02x", buf[i] = data[s+i]);
			else
				fprintf(fp, "   ");
		}
		fprintf(fp, "  ");
		for (i = 0; i < n; ++i)
		{
			if (i == 8)
				fprintf(fp, " ");
			c = buf[i];
			if (c >= ' ' && c < 0x7f)
				fprintf(fp, "%c", c);
			else
				fprintf(fp, ".");
		}
		fprintf(fp, "\n");
	}
}

/*
 * Get N byte value in big endian format
 */
int
getBE(FILE *fp, void *vp, int n)
{
	unsigned char   *p = (unsigned char *) vp;
	int		c;

	while (n--)
	{
		c = fgetc(fp);
		if (c == EOF)
			return 1;
		p[n] = c;
	}
	return 0;
}

void
add_to_PATH(char *dir)
{
	char	*newpath;
	char	*oldpath;

	oldpath = getenv("PATH");
	if (!oldpath)
		oldpath = "/bin:/usr/bin";
	newpath = malloc(5 + strlen(oldpath) + 1 + strlen(dir) + 1);
	if (!newpath)
		error(1, "Cannot get memory to set PATH");
	sprintf(newpath, "PATH=%s:%s", oldpath, dir);
	putenv(newpath);
}
		
/*
 * Substitute a string
 */
char *
strsub(char *d, int dlen, char *s, char *search, char *replace)
{
	char	*p;
	int	len = strlen(s);
	int	slen = strlen(search);
	int	rlen = strlen(replace);

	// Insure we have room
	if ( (len+rlen-slen) >= (dlen-1))
	{
		strncpy(d, s, dlen-1);
		d[dlen-1] = 0;
		return NULL;
	}

	// Find search string
	p = strstr(s, search);
	if (!slen || !p)
	{
		strncpy(d, s, dlen-1);
		d[dlen-1] = 0;
		return NULL;
	}

	// copy first part
	len = p - s;
	memcpy(d, s, len); d[len] = 0;

	// copy replacement
	strcat(d, replace);

	// copy last part
	strcat(d, p+slen);
	return (d+len);
}

/*
 * Return last character in a string
 */
char
strlast(char *s)
{
	char	*p;
	p = strchr(s, 0);
	if (p == s)
		return 0;
	else
		return p[-1];
}

/*
 * Split a string
 */
int
strsplit(char **strary, int nary, char *str, char sep)
{
	int	i;
	char	*p;

	for (i = 0; i < nary; ++i)
	{
		strary[i] = str;
		p = strchr(str, sep);
		if (!p)
			return i+1;
		*p = 0;
		str = p+1;
	}
	return i;
}

/*
 * Convert ascii date/time strings to unix time
 *
 * Acceptable formats for date:
 * 	NULL	(use todays date)
 * 	mm/dd
 * 	mm/dd/yy
 * 	mm/dd/yyyy
 * 	yymmdd
 * Acceptable formats for time:
 * 	NULL	(use current time)
 * 	hh:mm
 * 	hh:mm:ss
 * 	hhmm
 * 	hhmmss
 */
time_t
datetime2unix(char *datestr, char *timestr)
{
	struct tm	tm;
	struct tm	*tmp;
	char		*p;
	int		val;
	time_t		now;

	memset(&tm, 0, sizeof(tm));

	if (datestr)
	{
		p = strchr(datestr, '/');
		if (p)
		{
			// mm/dd/yy format
			tm.tm_mon = atoi(datestr) - 1;
			tm.tm_mday = atoi(p+1);
			p = strchr(p+1, '/');
			if (p)
			{
				val = atoi(p+1);
				if (val < 100)
					tm.tm_year = val + 2000 - 1900;
				else
					tm.tm_year = val - 1900;
			}
			else
			{
				time(&now);
				tmp = localtime(&now);
				tm.tm_year = tmp->tm_year;
			}
		}
		else
		{
			// yymmdd format
			val = atoi(datestr);
			tm.tm_year = val/10000 + 2000 - 1900;
			tm.tm_mon = (val/100)%100 - 1;
			tm.tm_mday = val%100;
		}
	}
	else
	{
		time(&now);
		tmp = localtime(&now);
		tm.tm_year = tmp->tm_year;
		tm.tm_mon = tmp->tm_mon;
		tm.tm_mday = tmp->tm_mday;
	}

	if (timestr)
	{
		p = strchr(timestr, ':');
		if (p)
		{
			tm.tm_hour = atoi(timestr);
			tm.tm_min = atoi(p+1);
			p = strchr(p+1, ':');
			if (p)
			{
				// hh:mm:ss format
				tm.tm_sec = atoi(p+1);
			}
			else
			{
				// hh:mm format
				tm.tm_sec = 0;
			}
		}
		else
		{
			val = atoi(timestr);
			if (strlen(timestr) > 4)
			{
				// hhmmss format
				tm.tm_hour = val / 10000;
				tm.tm_min = (val / 100) % 100;
				tm.tm_sec = val % 100;
			}
			else
			{
				// hhmm format
				tm.tm_hour = val / 100;
				tm.tm_min = val % 100;
				tm.tm_sec = 0;
			}
		}
	}
	else
	{
		time(&now);
		tmp = localtime(&now);
		tm.tm_hour = tmp->tm_hour;
		tm.tm_min = tmp->tm_min;
		tm.tm_sec = tmp->tm_sec;
	}

	tm.tm_isdst = -1;

	return mktime(&tm);
}

/*
 * Given a struct tm, figure out the current Monday
 */
struct tm *
tm2monday(struct tm *today)
{
    time_t	timetoday;
    struct tm	*tmp;

    if (today->tm_hour < 0 || today->tm_hour >= 24)
	today->tm_hour = 0;
    if (today->tm_min < 0 || today->tm_min >= 60)
	today->tm_min = 0;
    if (today->tm_sec < 0 || today->tm_sec >= 60)
	today->tm_sec = 0;

    timetoday = mktime(today);
    tmp = localtime(&timetoday);
    if (tmp->tm_wday == 0)
	tmp->tm_wday = 7;
    timetoday -=  (tmp->tm_wday-1) * 24*60*60;
    return localtime(&timetoday);
}
