#!/bin/bash

#
#	Display charts from web
#
#

PROGNAME="$0"

usage() {
	cat <<EOF
Usage:
	`basename $PROGNAME` [options] symbols ...

	Display charts from web

Options:
	-b	Intraday *and* daily charts
	-i	Intraday charts (else daily charts)
	-w	Weekly charts (else daily charts)
	-T	Taller charts
	-W	Wider charts
	-s src	Chart source: quote.com, yahoo.com, stockcharts.com,
		prophetfinance.com, pcquote.com
	-D lvl	Debug level
EOF

	exit 1
}

#
#       Report an error and exit
#
error() {
	echo "`basename $PROGNAME`: $1" >&2
	exit 1
}

#
#	Display one chart
#
do_one() {
	SYM="$1"
	FREQ="$2"

	ALERTS=
	GT=
	LT=
	while read symbol en var op val act
	do
		if [ "$symbol" != "$SYM" ]; then continue; fi
		if [ "$en" = 0 ]; then continue; fi
		if [ "$var" != l ]; then continue; fi
		ALERTS="$ALERTS $op$val"
		if [ "$op" = ">" ]; then GT=$val; fi
		if [ "$op" = "F" ]; then GT=$val; fi
		if [ "$op" = "<" ]; then LT=$val; fi
		if [ "$op" = "f" ]; then LT=$val; fi
	done < alerts

	case "$SRC" in
	stockcharts.com)
		case "$SYM" in
		'$DJI')		SYM='$INDU';;
		'$COMP')	SYM='$COMPQ';;
		esac

		if [ $WIDE = 1 ]; then
			WID=800
			CODE=dacayacd
			DUR=c
			#a=2m,b=3m,c=6m,d=12m,e=24m,f=36m,g=2m,h=48m,i=5y
		else
			WID=345
			CODE=daolyacd
			DUR=g
		fi
		SC_URL="http://stockcharts.com/def/servlet"
		SC_URL="$SC_URL/SharpChartv05.ServletDriver"
		SC_URL="$SC_URL?chart=$SYM"
		case "$FREQ" in
		weekly)
			SC_URL="$SC_URL,W"
			;;
		*)
			SC_URL="$SC_URL,uu[$WID,A]$CODE[pb20!b50][d$DUR]"
			;;
		esac

		cat <<-EOF
		<A HREF="http://stockcharts.com/webcgi/gall.exe?gall.web+$SYM">
			<IMG SRC="$SC_URL"
			BORDER=0
			ALT="$ALERTS">
		</A>
		EOF
		;;

	yahoo.com)
		case "$SYM" in
		'$COMP')	SYM='^IXIC';;
		'$SPX')		SYM='^GSPC';;
		'$SOX')		SYM='^SOXX';;
		'$TICK')	SYM='^TIC.N';;
		'$TRIN')	SYM='^STI.N';;
		'$TICKQ')	SYM='^TIC.O';;
		'$TRINQ')	SYM='^STI.O';;
		'$TICKA')	SYM='^TIC.AQ';;
		*)		SYM=`echo "$SYM" | tr '$' '^' `;;
		esac

		if [ $WIDE = 1 ]; then
			case "$FREQ" in
			weekly)	DUR=2y;;
			*)	DUR=6m;;
			esac
			SIZE=l
		else
			case "$FREQ" in
			weekly)	DUR=1y;;
			*)	DUR=3m;;
			esac
			SIZE=m
		fi
		YAHOO_C="http://cchart.yimg.com/z?"
		YAHOO_F="http://finance.yahoo.com/q?d=c&k=c3&"
		YAHOO_P="a=vm&p=m10,e20,m50,m200&t=$DUR&l=off&z=$SIZE&q=c"
		if [ "$FREQ" = intra ]; then
			# Hmm, check this for action during trading hours
			#YAHOO_C="http://ichart.yahoo.com/b?"
			:
		fi

		cat <<-EOF
			<A HREF="$YAHOO_F$YAHOO_P&s=$SYM">
			<IMG SRC="$YAHOO_C$YAHOO_P&s=$SYM"
			BORDER=1
			 ALT="$ALERTS">
			</A>
		EOF
		;;

	quote.com)
		case "$SYM" in
		'$DJI')		SYM='$INDU';;
		'$DJT')		SYM='$TRAN';;
		'$DJU')		SYM="$SYM";;
		'$COMP')	SYM='$COMPX';;
		'$TRIN')	SYM='$TRIN';;
		'$TICK')	SYM='$TICK';;
		'$TRINQ')	SYM='$TRINQ';;
		'$'*)		SYM="$SYM.x";;
		esac

		DATE=`date "+%y%m%d%H%M%S0"`
		QC_URL="http://charts-d.quote.com:443"
		QC_URL="$QC_URL/$DATE?User=demo&Pswd=demo"
		case "$FREQ" in
		intra)	QC_URL="$QC_URL&DataType=GIF&Interval=5";;
		weekly)	QC_URL="$QC_URL&DataType=GIF&Interval=7200";;
		*)	QC_URL="$QC_URL&DataType=GIF&Interval=1440";;
		esac
		QC_URL="$QC_URL&Study=Volume&Display=3"
		if [ $TALL = 1 ]; then
			HT=450
		else
			HT=220
		fi
		if [ $WIDE = 1 ]; then
			QC_URL="$QC_URL&Wd=800&Ht=$HT"
		else
			QC_URL="$QC_URL&Wd=345&Ht=$HT"
		fi

		FC_URL="http://finance.lycos.com/home/stocks/charts.asp"
		FC_URL="$FC_URL?symbols=$SYM&days=1&display=2&study=vol"

		cat <<-EOF
			<A HREF="$FC_URL">
				<IMG SRC="$QC_URL&Symbol=$SYM"
				BORDER=0
				 ALT="$ALERTS">
			</A>
		EOF
		;;

	prophetfinance.com)
		case "$SYM" in
		'$COMP')	SYM='$COMPX';;
		esac

		URL=
		URL="$URL?symbol=$SYM"
		if [ $TALL = 1 ]; then
			HT=450
		else
			HT=300
		fi
		if [ $WIDE = 1 ]; then
			URL="$URL&width=800&height=$HT"
			IDUR=2d
			DUR=6m
			WDUR=2y
		else
			URL="$URL&width=345&height=$HT"
			IDUR=1d
			DUR=2m
			WDUR=1y
		fi
		URL="$URL&price.display=3"
		URL="$URL&redbars=1&showspreads=1"
		URL="$URL&VOL=&snapquote=1&service=ProphetSnapCharts"
		case "$FREQ" in
		intra)
			URL="$URL&duration=$IDUR&interval=5"
			;;
		weekly)
			URL="$URL&EVNT=&MA=200,,&EMA=20,,&,,&=,,&=,,"
			URL="$URL&duration=$WDUR&frequency=1"
			;;
		*)
			URL="$URL&EVNT=&MA=200,,&EMA=20,,&,,&=,,&=,,"
			URL="$URL&duration=$DUR&frequency=0"
			;;
		esac

		LURL="http://www.prophetfinance.com/charts/sc.asp$URL"
		URL="http://www.prophetfinance.com/charts/scauth.asp$URL"

		cat <<-EOF
			<A HREF="$LURL">
				<IMG SRC="$URL" BORDER=0 ALT="$ALERTS">
			</A>
		EOF
		;;

	pcquote.com)
		case "$SYM" in
		'$DJI')		;;
		'$DJT')		;;
		'$DJU')		;;
		'$TICK')	;;
		'$TRIN')	;;
		*)		SYM=`echo "$SYM" | tr -d '$' `;;
		esac

		PCQ_URL="http://mscharts.pcquote.com/servlet/ChartEngine"
		PCQ_URL="$PCQ_URL?symbol=$SYM&ticker=$SYM"
		PCQ_URL="$PCQ_URL&pri="
		PCQ_URL="$PCQ_URL&type=4"
		PCQ_URL="$PCQ_URL&CD=&TC="
		if [ $TALL = 1 ]; then
			HT=450
		else
			HT=300
		fi
		if [ $WIDE = 1 ]; then
			PCQ_URL="$PCQ_URL&width=800&height=$HT"
			DUR=6m
			IDUR=2
			WDUR=2y
		else
			PCQ_URL="$PCQ_URL&width=345&height=$HT"
			DUR=2m
			IDUR=1
			WDUR=1y
		fi

		case "$FREQ" in
		intra)
			PCQ_URL="$PCQ_URL&dur=$DUR"
			PCQ_URL="$PCQ_URL&freq=6"
			PCQ_URL="$PCQ_URL&intr=$IDUR,5"
			;;
		weekly)
			PCQ_URL="$PCQ_URL&dur=$WDUR"
			PCQ_URL="$PCQ_URL&freq=3"
			PCQ_URL="$PCQ_URL&intr="
			;;
		*)
			PCQ_URL="$PCQ_URL&dur=$DUR"
			PCQ_URL="$PCQ_URL&freq=6"
			PCQ_URL="$PCQ_URL&intr="
			;;
		esac
		PCQ_URL="$PCQ_URL&indicator="
			PCQ_URL="${PCQ_URL}1,10,0,0;"
			PCQ_URL="${PCQ_URL}2,20,0,0;"
			PCQ_URL="${PCQ_URL}1,50,0,0;"
			PCQ_URL="${PCQ_URL}1,200,0,0;"
		PCQ_URL="$PCQ_URL&vol="
		PCQ_URL="$PCQ_URL&nodata="
		#PCQ_URL="$PCQ_URL&skin=pcquote"
		PCQ_URL="$PCQ_URL&skin=marketscreen"
		PCQ_L_URL="http://www.pcquote.com/stocks/chart.php?ticker=$SYM"

		# Mark triggers
		DATE="03_28_2002"
		DATE=`date -d "last monday" +'%m_%d_%Y'`
		DATE=`date -d "3 days ago" +'%m_%d_%Y'`
		if [ "$LT" != "" ]; then
			PCQ_URL="$PCQ_URL&curve=${DATE};l,-5,$LT,0,$LT;"
		fi
		if [ "$GT" != "" ]; then
			PCQ_URL="$PCQ_URL&curve=${DATE};l,-5,$GT,0,$GT;"
		fi
		# Other examples...
		# PCQ_URL="$PCQ_URL&curve=03_28_2002;l,5,10.00,0,10.00;"
		# PCQ_URL="$PCQ_URL&curve=l,5,14.00,0,14.00;"

		#<option value='1,7,,'>Simple Moving Average
		#<option value='2,7,,'>Expotential Moving Average
		#<option value='3,7,,'>Weighted Moving Average
		#<option value='4,,,'>Accumulation Distribution
		#<option value='5,18,,'>Commodity Channel Index
		#<option value='8,12,26,9'>MACD
		#<option value='9,10,,'>Williams' %R
		#<option value='11,,,'>On Balance Volume
		#<option value='12,7,,'>Linear Regression Channel
		#<option value='13,12,,'>Momentum
		#<option value='14,,,'>MoneyFlow
		#<option value='15,20,2,'>Bollinger Bands
		#<option value='16,10,,'>Price Rate-Of-Change
		#<option value='17,12,26,9'>MACD Histogram
		#<option value='18,14,,'>Relative Strength Index
		#<option value='19,10,,'>Parabolic Stop and Reverse
		#<option value='20,7,13,'>Directional Movement Index
		#<option value='21,10000,5,'>Ease Of Movement
		#<option value='22,,,'>Chaikin Oscillator
		#<option value='23,10,,'>Detrended Price Oscillator
		#<option value='24,25,,'>Mass Index
		#<option value='25,,,'>Median Price
		#<option value='26,,,'>Negative Volume Index
		#<option value='27,,,'>Performance
		#<option value='28,7,50,'>Price Oscillator
		#<option value='29,,,'>Positive Volume Index
		#<option value='30,,,'>Price and Volume Trend
		#<option value='31,5,1,3'>Fast Stochastic Oscillator
		#<option value='32,5,3,3'>Slow Stochastic Oscillator
		#<option value='33,,,'>Fibonacci cycle
		#<option value='34,,,'>Fibonacci Retracement
		#<option value='35,,,'>Gann Line
		#<option value='36,7,,'>ZigZag
		#<option value='39,7,13,'>Volume Oscillator
		#<option value='40,28,,'>Vertical Horizontal Filter
		#<option value='41,,,'>Tirone Levels
		#<option value='42,5,,'>Quadrant Lines
		#<option value='43,,,'>Swing Index
		#<option value='44,,,'>Accumulation Swing Index
		#<option value='45,7,,'>Standart Deviation
		#<option value='46,7,,'>TRIX
		#<option value='47,10,15,'>Volatility, Chaikin's
		#<option value='48,,,'>Speed Resistance Lines
		#<option value='49,7,,'>Average True Range
		#<option value='50,,,'>Typical Price
		#<option value='51,10,,'>Volume Rate-of-change
		#<option value='52,,,'>Weighted Close
		#<option value='53,,,'>Williams Accumulation/Distribution
		#<option value='54,7,,'>Percent of Resistance
		#<option value='55,10,15,'>MA Envelope
		#<option value='56,,,'>Fibonacci Fans
		#<option value='57,,,'>Fibonacci Time Zones
		#<option value='58,,,'>Four Percent Model
		#<option value='59,38,50,62'>Percent Retracement
		#<option value='60,20,,'>Time Series Forecast

		cat <<-EOF
			<A HREF="$PCQ_L_URL">
				<IMG SRC="$PCQ_URL" BORDER=1 ALT="$ALERTS">
			</A>
		EOF

		;;
	esac
}

#
#       Process the options
#
DEBUG=0
FREQUENCY=daily
BOTH=0
FILE=daily.html
WIDE=0
TALL=0
SRC=
unset OPTIND
while getopts "biwTWs:D:h?" opt
do
	case $opt in
	b)	BOTH=1; FILE=both.html;;
	i)	FREQUENCY=intra; FILE=intra.html;;
	w)	FREQUENCY=weekly; FILE=weekly.html;;
	T)	TALL=1;;
	W)	WIDE=1;;
	s)	SRC="$OPTARG";;
	D)	DEBUG="$OPTARG";;
	h|\?)	usage;;
	esac
done
shift `expr $OPTIND - 1`

#
#	Main Program
#

cd $HOME/.linuxtrade || exit 1

if [ "$SRC" = "" ]; then
	SRC=stockcharts.com
	while read tag eq val
	do
		if [ "$tag" = "web_charts" ]; then
			case "$val" in
			q*)		SRC=quote.com;;
			y*)		SRC=yahoo.com;;
			s*)		SRC=stockcharts.com;;
			pr*)		SRC=prophetfinance.com;;
			p*)		SRC=pcquote.com;;
			default)	SRC=stockcharts.com;;
			esac
		fi
	done < rc
else
	case "$SRC" in
	q*)		SRC=quote.com;;
	y*)		SRC=yahoo.com;;
	s*)		SRC=stockcharts.com;;
	pr*)		SRC=prophetfinance.com;;
	p*)		SRC=pcquote.com;;
	default)	SRC=stockcharts.com;;
	esac
fi

cat <<-EOF >$FILE
	<html>
	<body>
EOF

if [ ! -f alerts ]; then
	touch alerts
fi

for SYMBOL in $*
do
	if [ $BOTH = 1 ]; then
		if [ $FREQUENCY = weekly ]; then
			do_one $SYMBOL weekly
			do_one $SYMBOL daily
		else
			do_one $SYMBOL daily
			do_one $SYMBOL intra
		fi
	else
		do_one $SYMBOL $FREQUENCY
	fi
done >>$FILE

cat <<-EOF >>$FILE
	</body>
	</html>
EOF

URL="file:`pwd`/$FILE"

browser="gnome-moz-remote --newwin %s"
while read tag eq val
do
	if [ "$tag" = URL_launch ]; then
		browser="$val"
	fi
done < rc

cmd=`echo "$browser" | sed "s#%s#$URL#"`

exec $cmd
