/* -*- c++ -*-
 *
 * mldonkeyappletgui.cpp
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@kmldonkey.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */


#include <klocale.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <kpushbutton.h>
#include <kiconloader.h>
#include <kglobalsettings.h>

#include "mldonkeyappletgui.h"
#include "mldonkeyapplet.h"

// MLDonkeyAppletGUILabel

MLDonkeyAppletGUILabel::MLDonkeyAppletGUILabel(MLDonkeyAppletGUI* parent, const char* name, bool islabel)
    : QObject(parent, name)
{
    m_label = NULL;
    this->islabel = islabel;
}

MLDonkeyAppletGUILabel::~MLDonkeyAppletGUILabel()
{
}

QLabel* MLDonkeyAppletGUILabel::label()
{
    return m_label;
}

void MLDonkeyAppletGUILabel::setEnabled()
{
    if(m_label) setDisabled();

    MLDonkeyAppletGUI* gui = (MLDonkeyAppletGUI*)parent();

    m_label = new QLabel(gui);
    m_label->setSizePolicy( QSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed, 0, 0, m_label->sizePolicy().hasHeightForWidth() ) );
    m_label->setMaximumSize( QSize( 32767, 20 ) );
    m_label->setMargin( 2 );
    m_label->setFont( ((MLDonkeyApplet*)gui->parent())->appletfont );

    if(islabel) {
        m_label->setAlignment( int( QLabel::AlignVCenter | QLabel::AlignRight ) );
    }
    else {
        m_label->setAlignment( int( QLabel::AlignVCenter | QLabel::AlignLeft ) );
        m_label->setFrameShape( QLabel::Panel );
        m_label->setFrameShadow( QLabel::Sunken );
        m_label->setPaletteBackgroundColor( KGlobalSettings::baseColor() );
    }

    setText(text);
    setTooltip(tooltip);
    m_label->show();
}

void MLDonkeyAppletGUILabel::setDisabled()
{
    delete m_label;
    m_label = NULL;
}

void MLDonkeyAppletGUILabel::setText(const QString& t)
{
    text = t;
    if(m_label) m_label->setText( (t.isEmpty() ? QString("...") : t) );
}

void MLDonkeyAppletGUILabel::setTooltip(const QString& t)
{
    tooltip = t;
    if(m_label) {
        QToolTip::remove(m_label);
        if(! t.isEmpty()) QToolTip::add(m_label, t);
    }
}

// MLDonkeyAppletGUI

MLDonkeyAppletGUI::MLDonkeyAppletGUI(QWidget* parent, const char*, WFlags)
    : QWidget(parent, "MLDonkeyAppletGUI")
{
    icons.addAppDir("mldonkeyapplet");

    mysize = -1;
    horiz = NULL;
    grid = NULL;
    LaunchButton = NULL;
    MuteButton = NULL;

    FirstLabel = new MLDonkeyAppletGUILabel(this, "FirstLabel", true);
    FirstStatus = new MLDonkeyAppletGUILabel(this, "FirstStatus", false);
    SecondLabel = new MLDonkeyAppletGUILabel(this, "SecondLabel", true);
    SecondStatus = new MLDonkeyAppletGUILabel(this, "SecondStatus", false);

    setSizePolicy( QSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed, 0, 0, sizePolicy().hasHeightForWidth() ) );
    //relayoutWidgets(0);
}

MLDonkeyAppletGUI::~MLDonkeyAppletGUI()
{
}

void MLDonkeyAppletGUI::updateLayout()
{
    MLDonkeyApplet* app = (MLDonkeyApplet*)parent();
    int count = app->activeDisplays.count();

    setUpdatesEnabled(false);

    FirstLabel->setDisabled();
    FirstStatus->setDisabled();
    SecondLabel->setDisabled();
    SecondStatus->setDisabled();

    delete LaunchButton; LaunchButton = NULL;
    delete MuteButton; MuteButton = NULL;

    delete horiz; horiz = NULL;
    delete grid; grid = NULL;

    if (!mysize) {
        int pos = 0;
	grid = new QGridLayout(this, 2, 1, 0, 0);

	if (count > 0) {
	    if (app->showLabels) {
                FirstLabel->setEnabled();
		if (count > 1) {
                    SecondLabel->setEnabled();
		    grid->addWidget(FirstLabel->label(), 0, pos, Qt::AlignAuto);
		    grid->addWidget(SecondLabel->label(), 1, pos, Qt::AlignAuto);
		} else {
		    grid->addMultiCellWidget(FirstLabel->label(), 0, 1, pos, pos, Qt::AlignAuto);
		}
		pos++;
	    }

	    FirstStatus->setEnabled();
	    if (count > 1) {
		SecondStatus->setEnabled();
		grid->addWidget(FirstStatus->label(), 0, pos, Qt::AlignAuto);
		grid->addWidget(SecondStatus->label(), 1, pos, Qt::AlignAuto);
	    } else {
		grid->addMultiCellWidget(FirstStatus->label(), 0, 1, pos, pos, Qt::AlignAuto);
	    }
	    pos++;
	}

        createLaunchButton();
        grid->addWidget( LaunchButton, 0, pos, Qt::AlignAuto );

        createMuteButton();
        grid->addWidget( MuteButton, 1, pos, Qt::AlignAuto );
    }
    else {
	horiz = new QHBoxLayout(this);

	if (count) {
	    if (app->showLabels && !app->showDouble) {
		FirstLabel->setEnabled();
		horiz->addWidget(FirstLabel->label());
	    }
	    FirstStatus->setEnabled();
	    horiz->addWidget(FirstStatus->label());

	    if (count > 1) {
		if (app->showLabels && !app->showDouble) {
		    SecondLabel->setEnabled();
		    horiz->addWidget(SecondLabel->label());
		}
		SecondStatus->setEnabled();
		horiz->addWidget(SecondStatus->label());
	    }
	}

        createLaunchButton();
	horiz->addWidget(LaunchButton);

	if (app->showMute) {
            createMuteButton();
	    horiz->addWidget(MuteButton);
	}
    }

    adjustSize();
    setUpdatesEnabled(true);
    //update();
}

void MLDonkeyAppletGUI::relayoutWidgets(int small)
{
    if (mysize == small) return;
    mysize = small;
    updateLayout();
}

bool MLDonkeyAppletGUI::isMute()
{
    return MuteButton && MuteButton->isOn();
}

void MLDonkeyAppletGUI::mute(bool m)
{
    if (MuteButton && MuteButton->isOn() != m)
        MuteButton->toggle();
}

void MLDonkeyAppletGUI::updateLabels(const QString& first, const QString& second)
{
    FirstLabel->setText(first);
    SecondLabel->setText(second);
}

void MLDonkeyAppletGUI::updateStatus(const QString& first, const QString& second)
{
    FirstStatus->setText(first);
    SecondStatus->setText(second);
}

void MLDonkeyAppletGUI::updateTooltips(const QString& first, const QString& second)
{
    FirstStatus->setTooltip(first);
    SecondStatus->setTooltip(second);
}

void MLDonkeyAppletGUI::donkeyDisconnected()
{
    SecondStatus->setText();
    FirstStatus->setText();
}

void MLDonkeyAppletGUI::setLaunchState(bool state)
{
    if(LaunchButton) {
        blockSignals(true);
        LaunchButton->setOn(state);
        blockSignals(false);
    }
}

void MLDonkeyAppletGUI::toggleLaunch(bool t)
{
    emit toggledLaunch(t);
}

void MLDonkeyAppletGUI::toggleMute(bool t)
{
    emit toggledMute(t);
}

void MLDonkeyAppletGUI::createLaunchButton()
{
    LaunchButton = new KPushButton( this, "LaunchButton" );
    LaunchButton->setSizePolicy( QSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed, 0, 0, LaunchButton->sizePolicy().hasHeightForWidth() ) );
    LaunchButton->setMaximumSize( QSize( 20, 20 ) );
    LaunchButton->setFocusPolicy( QPushButton::NoFocus );
    LaunchButton->setIconSet( icons.loadIconSet("mld-launchgui", KIcon::User) );
    LaunchButton->setToggleButton(true);
    LaunchButton->setFlat(true);
    QToolTip::add(LaunchButton, i18n("Show/hide the KMLDonkey interface"));
    LaunchButton->show();
    connect(LaunchButton, SIGNAL(toggled(bool)), this, SLOT(toggleLaunch(bool)));
}

void MLDonkeyAppletGUI::createMuteButton()
{
    QIconSet muteIcon(icons.loadIcon("mld-mutedonkey", KIcon::User));
    muteIcon.setPixmap(icons.loadIcon("mld-unmutedonkey", KIcon::User), QIconSet::Small, QIconSet::Normal, QIconSet::On);

    MuteButton = new KPushButton( this, "MuteButton" );
    MuteButton->setSizePolicy( QSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed, 0, 0, MuteButton->sizePolicy().hasHeightForWidth() ) );
    MuteButton->setMaximumSize( QSize( 20, 20 ) );
    MuteButton->setFocusPolicy( QPushButton::NoFocus );
    MuteButton->setIconSet(muteIcon);
    MuteButton->setToggleButton(true);
    MuteButton->setFlat(true);
    QToolTip::add(MuteButton, i18n("Toggle mute mode"));
    MuteButton->show();
    connect(MuteButton, SIGNAL(toggled(bool)), this, SLOT(toggleMute(bool)));
}

#include "mldonkeyappletgui.moc"
