"""Kid - Pythonic XML Template Language

Kid is a simple Python based template language for generating and
transforming XML vocabularies. Kid was spawned as a result of a kinky love
triangle between XSLT, TAL, and PHP. We believe many of the best features
of these languages live on in Kid with much of the limitations and
complexity stamped out (well, eventually :).

Templates are compiled to native Python byte-code and may be imported and
used like normal Python modules.

"""

__version__ = "0.6.3"
__revision__ = "$Rev: 140 $"
__date__ = "$Date: 2005-03-14 19:32:26 -0500 (Mon, 14 Mar 2005) $"
__author__ = "Ryan Tomayko (rtomayko@gmail.com)"
__copyright__ = "Copyright 2004-2005, Ryan Tomayko"
__license__ = "MIT <http://www.opensource.org/licenses/mit-license.php>"

import sys
import os

from kid.util import xml_sniff, QuickTextReader
from kid.namespace import Namespace
from kid.pull import ElementStream, Element, SubElement, Fragment, \
                     XML, document, _coalesce
from kid.et import ElementTree, Comment, ProcessingInstruction
from kid.parser import KID_XMLNS
from kid.serialization import Serializer, XMLSerializer, HTMLSerializer

def enable_import():
    """Enable the kid module loader and import hooks.
    
    This function must be called before importing kid templates if templates
    are not pre-compiled.
    
    Note that if your application uses ZODB, you will need to import ZODB
    before calling this function as ZODB's import hooks have some issues if
    installed after the kid import hooks.
    
    """
    import kid.importer
    kid.importer.install()

#
# Turn on import hook if KID_IMPORT is set
#
if os.environ.get('KID_IMPORT', None) is not None:
    enable_import()


def import_template(name):
    """Import template by name.

    This is identical to calling enable_import() followed by an import
    statement. For example, importing a template named foo using the normal
    import mechanism looks like this:

        import kid
        kid.enable_import()
        import foo

    This function can be used to achieve the same result as follows:

        import kid
        foo = kid.import_template('foo')

    This is sometimes useful when the name of the template is available only
    as a string.
    """
    enable_import()
    mod = __import__(name)
    components = name.split('.')
    for comp in components[1:]:
        mod = getattr(mod, comp)
    return mod

def load_template(file, name='', cache=1, encoding=None):
    """Bypass import machinery and load a template module directly.

    This can be used as an alternative to accessing templates using the native
    python import mechanisms.

    Arguments:
    file  -- Can be a filename, a kid template string, or an open file object.
    name  -- Optionally specifies the module name to use for this template. This
             is a hack to enable relative imports in templates.
    cache -- Whether to look for a byte-compiled version of the template. If
             no byte-compiled version is found, an attempt is made to dump a
             byte-compiled version after compiling. This argument is ignored if
             file is not a filename.
    """

    if name and cache and sys.modules.has_key(name):
        return sys.modules.get(name)

    if isinstance(file, (str, unicode)):
        if xml_sniff(file):
            fo = QuickTextReader(file)
            filename = '<string>'
        else:
            fo = None
            filename = file
    else:
        fo = file
        filename = '<string>'
    import kid.compiler as compiler
    if filename == '<string>':
        code = compiler.compile(fo, filename, encoding)
    else:
        template = compiler.KidFile(filename, 0, encoding)
        code = template.compile(dump_code=cache)

    import kid.importer as importer
    mod = importer._create_module(code, name, filename, store=cache)
    return mod

# create some default serializers..
output_methods = {
    'xml'          : XMLSerializer(decl=1),
    'xhtml'        : XMLSerializer(decl=0, doctype='xhtml'),
    'xhtml-strict' : XMLSerializer(decl=0, doctype='xhtml-strict'),
    'html'         : HTMLSerializer(doctype='html'),
    'html-strict'  : HTMLSerializer(doctype='html-strict') }

def Template(file=None, source=None, name=None, **kw):
    """Get a Template class quickly given a module name, file, or string.

    This is a convenience function for getting a template in a variety of
    ways. One and only one of the arguments name or file must be specified.

    file -- The template module is loaded by calling:
    load_template(file, name='', cache=1).

    name -- The kid import hook is enabled and the template module is located
    using the normal Python import mechanisms.

    source -- String containing the templates source.

    Once the template module is obtained, a new instance of the module's
    Template class is created with the keyword arguments passed to this
    function.
    """
    if name:
        mod = import_template(name)
    elif file is not None:
        mod = load_template(file)
    elif source is not None:
        mod = load_template(QuickTextReader(source))
    else:
        raise Exception("Must specify one of name, file, or source.")
    mod.Template.module = mod
    return mod.Template(**kw)

from kid.filter import transform_filter

class BaseTemplate(object):

    """Base class for compiled Templates.

    All kid template modules expose a class named `Template` that extends this
    class, making methods defined here available on all Template subclasses.

    This class should not be instantiated directly.
    """

    # the serializer to use when writing output
    serializer = output_methods['xml']
    _filters = [transform_filter]
    
    def __init__(self, **kw):
        """Initialize a template with instance attributes specified by keyword
           arguments.
           
        Keyword arguments are available to the template using self.var notation.
        """
        self.__dict__.update(kw)
        
    def write(self, file, encoding=None, fragment=0, output=None):
        """Execute template and write output to file.
        
        Arguments:
        file     -- A filename or a file like object (must support write()).
        encoding -- The output encoding. Default: utf-8.
        fragment -- Controls whether prologue information (such as <?xml?>
                    declaration and DOCTYPE should be written). Set to 1
                    when generating fragments meant to be inserted into
                    existing XML documents.
        output   -- A string specifying an output method ('xml', 'html',
                    'xhtml') or a Serializer object.
        """
        serializer = self._get_serializer(output)
        return serializer.write(self, file, encoding, fragment)
    
    def serialize(self, encoding=None, fragment=0, output=None):
        """Execute a template and return a single string.
        
        Arguments:
        encoding -- The output encoding. Default: utf-8.
        fragment -- Controls whether prologue information (such as <?xml?>
                    declaration and DOCTYPE should be written). Set to 1
                    when generating fragments meant to be inserted into
                    existing XML documents.
        output   -- A string specifying an output method ('xml', 'html',
                    'xhtml') or a Serializer object.
        
        This is a convienence method, roughly equivalent to:
        ''.join([x for x in obj.generate(encoding, fragment, output)]
        
        """
        serializer = self._get_serializer(output)
        return serializer.serialize(self, encoding, fragment)

    def generate(self, encoding=None, fragment=0, output=None):
        """Execute template and generate serialized output incrementally.
        
        This method returns an iterator that yields an encoded string for each
        iteration. The iteration ends when the template is done executing.
        
        Arguments:
        encoding -- The output encoding. Default: utf-8.
        fragment -- Controls whether prologue information (such as <?xml?>
                    declaration and DOCTYPE should be written). Set to 1
                    when generating fragments meant to be inserted into
                    existing XML documents.
        output   -- A string specifying an output method ('xml', 'html',
                    'xhtml') or a Serializer object.
        """
        serializer = self._get_serializer(output)
        return serializer.generate(self, encoding, fragment)
    
    def __iter__(self):
        return iter(self.transform())
    
    def __str__(self):
        return self.serialize()
    
    def __unicode__(self):
        return unicode(self.serialize(encoding='utf-16'), 'utf-16')
    
    def pull(self):
        """Returns an iterator over the items in this template."""
        # create stream and apply filters
        stream = ElementStream(_coalesce(self._pull()))
        return stream
    
    def _pull(self):
        """Generate events for this template.
        
        Compiled templates implement this method.
        """
        return []
    
    def transform(self, stream=None, filters=[]):
        """Execute the template and apply any match transformations.
        
        If stream is specified, it must be one of the following:
        
           Element       - An ElementTree Element.
           ElementStream - An ElementStream or other iterator that yields
                           stream events.
           string        - A file or URL unless the string starts with
                           '<' in which case it is considered an XML document
                           and processed as if it had been an Element.
        
        By default, the pull() method is called to obtain the stream.
        """
        if stream is None:
            stream = self.pull()
        elif isinstance(stream, (str, unicode)):
            if xml_sniff(stream):
                stream = XML(stream, fragment=0)
            else:
                stream = document(stream)
        elif hasattr(stream, 'tag'):
            stream = ElementStream(stream)
        else:
            stream = ElementStream.ensure(stream)
        for f in filters + self._filters:
            stream = f(stream, self)
        return stream
            
    def _get_match_templates(self):
        # XXX: use inspect instead of accessing __mro__ directly
        try:
            rslt = self._match_templates_cached
        except AttributeError:
            rslt = []
            mro = self.__class__.__mro__
            for C in mro:
                try:
                    templates = C._match_templates
                except AttributeError:
                    continue
                rslt += templates
            self._match_templates_cached = rslt
        return rslt

    def _get_serializer(self, serializer):
        if serializer is None:
            return self.serializer
        elif isinstance(serializer, (str, unicode)):
            return output_methods[serializer]
        else:
            return serializer
        

    
__all__ = ['KID_XMLNS', 'BaseTemplate', 'Template',
           'enable_import', 'import_template', 'load_template',
           'Element', 'SubElement', 'XML', 'document', 'Namespace',
           'Serializer', 'XMLSerializer', 'HTMLSerializer', 'output_methods']
