/***************************************************************************
                          dvbconfig.cpp  -  description
                             -------------------
    begin                : Wed Apr 21 2004
    copyright            : (C) 2004-2005 by Christophe Thommeret
    email                : hftom@free.fr
    last modified        : $Date: 2005/02/15 18:27:00 $ by $Author: hftom $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <fcntl.h>
#include <sys/ioctl.h>
#include <unistd.h>

#include <linux/dvb/frontend.h>

#include <qdir.h>
#include <qlayout.h>
#include <qgroupbox.h>

#include <klocale.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <ktar.h>
#include <kstandarddirs.h>

#include "dvbconfig.h"
#include "gdvb.h"



MSpinBox::MSpinBox( QWidget *parent, int devNum ) : QSpinBox( 1, 4, 1, parent )
{
	deviceNumber = devNum;
	connect( this, SIGNAL(valueChanged(int)), this, SLOT(slotValueChanged(int)) );
}



void MSpinBox::slotValueChanged( int value )
{
	emit signalValueChanged( value, deviceNumber );
}



Device::Device( int num, fe_type_t t, QString n )
{
	adapter = num;
	type = t;
	name = n;
	source = "";
}



DVBconfig::DVBconfig( KConfig *conf, QString dvbConf )
{
	downProgress = 0;
	dvbConfigDir = dvbConf;
	config = conf;
	startup();
	readConfig();
}



bool DVBconfig::haveDvbDevice()
{
	int i, res, fdFrontend=0;
	struct dvb_frontend_info info;
	bool ret = false;

	for ( i=0; i<4; i++ ) {
		fdFrontend = open( QString("/dev/dvb/adapter%1/frontend0").arg( i ), O_RDWR);
		if ( !(fdFrontend<0) ) {
			if ( !(res = ioctl( fdFrontend, FE_GET_INFO, &info ) < 0) ) {
				ret = true;
				close( fdFrontend );
			}
		}
	}
	return ret;
}



void DVBconfig::startup()
{
	int i, res, fdFrontend=0;
	struct dvb_frontend_info info;

	for ( i=0; i<4; i++ ) {
		fdFrontend = open( QString("/dev/dvb/adapter%1/frontend0").arg( i ), O_RDWR);
		if ( fdFrontend<0 ) {
			fprintf(stderr,"Card %d :", i);
			perror("openFe :");
		}
		else {
			if ( (res = ioctl( fdFrontend, FE_GET_INFO, &info ) < 0) ) {
				fprintf(stderr,"Card %d :", i);
				perror("FE_GET_INFO: ");
			}
			else {
				fprintf(stderr,"Card %d : opened ( %s )\n", i, info.name );
				close( fdFrontend );
				devList.append( new Device( i, info.type, info.name ) );
			}
		}
	}
	//devList.append( new Device( 1, FE_OFDM, "Dummy T" ) );
	//devList.append( new Device( 2, FE_QAM, "Dummy C" ) );
}



void DVBconfig::setDownloadResult( Job *job )
{
	if ( downProgress && job ) {
		delete downProgress;
		downProgress = 0;
	}
}



void DVBconfig::setDownloadPercent( Job *job, unsigned long percent )
{
	if ( downProgress && job ) downProgress->progressBar()->setProgress( percent );
}



bool DVBconfig::loadDvbData( QWidget *parent )
{
	QString s="";
	FileCopyJob *job;
	QFile f( dvbConfigDir+"dvbdata.tar.gz" );

	//if ( f.exists() ) f.remove();
	downProgress = new KProgressDialog( parent, "progress", i18n("Downloading ... "), i18n("Copying data files ..."), true );
	downProgress->progressBar()->setTotalSteps( 100 );
	//job = file_copy( KURL( "http://hftom.free.fr/kaxtv/dvbdata.tar.gz" ), KURL( dvbConfigDir+"dvbdata.tar.gz" ), -1, true, false, false );
	job = file_copy( KURL( "http://hftom.free.fr/kaxtv/dvbdata.tar.gz" ), KURL( dvbConfigDir+"dvbdata.tar.gz" ), -1, true, false, false );
	connect( job, SIGNAL(result(KIO::Job*)), this, SLOT(setDownloadResult(KIO::Job*)) );
	connect( job, SIGNAL(percent(KIO::Job*,unsigned long)), this, SLOT(setDownloadPercent(KIO::Job*,unsigned long)) );
	downProgress->exec();
	disconnect( job, SIGNAL(result(KIO::Job*)), this, SLOT(setDownloadResult(KIO::Job*)) );
	disconnect( job, SIGNAL(percent(KIO::Job*,unsigned long)), this, SLOT(setDownloadPercent(KIO::Job*,unsigned long)) );
	if ( downProgress ) {
		delete downProgress;
		downProgress = 0;
	}
	KTar tar( dvbConfigDir+"dvbdata.tar.gz");
	if ( tar.open( IO_ReadOnly ) ) {
		tar.directory()->copyTo( dvbConfigDir );
		return true;
	}
	else return false;
}



bool DVBconfig::localData()
{
	QString s = locate("data","kaffeine/dvbdata.tar.gz");
	KTar tar( s );
	if ( tar.open( IO_ReadOnly ) ) {
		tar.directory()->copyTo( dvbConfigDir );
		return true;
	}
	else return false;
}



bool DVBconfig::haveData()
{
	if ( !QDir( dvbConfigDir+"dvb-s" ).exists() || !QDir( dvbConfigDir+"dvb-c" ).exists() || !QDir( dvbConfigDir+"dvb-t" ).exists() ) {
		loadDvbData(0);
		if ( !QDir( dvbConfigDir+"dvb-s" ).exists() || !QDir( dvbConfigDir+"dvb-c" ).exists() || !QDir( dvbConfigDir+"dvb-t" ).exists() ) {
			if ( !localData() ) return false;
		}
	}
	return true;
}



QStringList DVBconfig::getSourcesList( int devNum )
{
	QString s;
	QStringList list;

	switch ( devList.at(devNum)->type ) {
		case FE_QPSK : s = "dvb-s"; break;
		case FE_QAM : s = "dvb-c"; break;
		case FE_OFDM : s = "dvb-t";
	}
	list = QDir( dvbConfigDir+s ).entryList( QDir::Files, QDir::Name );
	return list;
}



void DVBconfig::readConfig()
{
	QSize size;
	QString s;
	int i;

	config->setGroup( "DVB Options" );
	size = QSize(600, 350);
	epgSize = config->readSizeEntry( "EPG Geometry", &size );
	size = QSize(600, 300);
	timerSize = config->readSizeEntry( "Timers Geometry", &size );
	size = QSize(300, 300);
	scanSize = config->readSizeEntry( "Scan Geometry", &size );
	beginMargin = config->readNumEntry( "BeginMargin", 5 );
	endMargin = config->readNumEntry( "EndMargin", 10 );
	instantDuration = config->readNumEntry( "InstantDuration", 120 );
	format = config->readNumEntry( "PrefFormat", OutTS );
	recordDir = config->readEntry( "RecordDir", QDir::homeDirPath() );
	if ( !recordDir.endsWith("/") ) recordDir+= "/";
	shiftDir = config->readEntry( "ShiftDir", QDir::homeDirPath() );
	if ( !shiftDir.endsWith("/") ) shiftDir+= "/";
	for ( i=0; i<(int)devList.count(); i++ ) devList.at(i)->source = config->readEntry( QString("DVB%1").arg(i), "" );
}



void DVBconfig::saveConfig()
{
	int i;

	config->setGroup( "DVB Options" );
	config->writeEntry( "EPG Geometry", epgSize );
	config->writeEntry( "Timers Geometry", timerSize );
	config->writeEntry( "Scan Geometry", scanSize );
	config->writeEntry( "BeginMargin", beginMargin );
	config->writeEntry( "EndMargin", endMargin );
	config->writeEntry( "InstantDuration", instantDuration );
	config->writeEntry( "PrefFormat", format );
	config->writeEntry( "RecordDir", recordDir );
	config->writeEntry( "ShiftDir", shiftDir );
	for ( i=0; i<(int)devList.count(); i++ ) config->writeEntry( QString("DVB%1").arg(i), devList.at(i)->source );
	config->sync();
}



bool DVBconfig::firstRun()
{
	config->setGroup( "DVB Options" );
	if ( config->readNumEntry( "FirstRun", 0 )<2 ) {
		config->writeEntry( "FirstRun", 2 );
		return true;
	}
	return false;
}



DvbConfigDialog::DvbConfigDialog( DVBconfig *dc, QWidget *parent ) :
	KDialogBase ( IconList, i18n("DVB settings"), Ok|Cancel, Ok, parent, "dvbConfigDialog", true, true )
{
	QLabel *lab;
	KIconLoader *icon = new KIconLoader();
	QHBoxLayout *h1;
	QString s;
	int i;
	QVBoxLayout *vb;
	QGroupBox *gb;
	QGridLayout *grid;
	QLabel *ident;
	QLabel *dvbType;

	dvbConfig = dc;

	dvbPage[0]=dvbPage[1]=dvbPage[2]=dvbPage[3]=0;

	for ( i=0; i<(int)dvbConfig->devList.count(); i++ ) {
		dvbPage[i] = addPage( i18n("DVB device")+" "+s.setNum( dvbConfig->devList.at(i)->adapter ), i18n("Device settings"),
			KGlobal::instance()->iconLoader()->loadIcon( "hwinfo", KIcon::NoGroup, KIcon::SizeMedium ) );
		vb = new QVBoxLayout( dvbPage[i], 6, 6 );
		gb = new QGroupBox( "", dvbPage[i] );
		grid = new QGridLayout( gb, 1, 1, 20, 6 );

		lab = new QLabel( i18n("<qt><b>Name :</b></qt>"), gb );
		grid->addWidget( lab, 0, 0 );
		ident = new QLabel( dvbConfig->devList.at(i)->name, gb );
		grid->addMultiCellWidget( ident, 0, 0, 1, 3 );

		lab = new QLabel( i18n("<qt><b>Type :</b></qt>"), gb );
		grid->addWidget( lab, 1, 0 );
		dvbType = new QLabel( gb );
		switch ( dvbConfig->devList.at(i)->type ) {
			case FE_QAM : dvbType->setText( i18n("DVB-C") ); break;
			case FE_OFDM : dvbType->setText( i18n("DVB-T") ); break;
			case FE_QPSK : dvbType->setText( i18n("DVB-S") ); break;
			default : dvbType->setText( i18n("Unknown") );
		}
		grid->addMultiCellWidget( dvbType, 1, 1, 1, 3 );

		if ( dvbConfig->devList.at(i)->type==FE_QPSK ) {
			lab = new QLabel( i18n("Number of LNBs :"), gb );
			grid->addWidget( lab, 2, 0 );
			satNumber[i] = new MSpinBox( gb, i );
			connect( satNumber[i], SIGNAL(signalValueChanged(int,int)), this, SLOT(satNumberChanged(int,int)));
			grid->addWidget( satNumber[i], 2, 1 );

			lab = new QLabel( i18n("1st sat :"), gb );
			grid->addWidget( lab, 3, 0 );
			sat0[i] = new QComboBox( gb );
			sat0[i]->setEnabled(true);
			sat0[i]->insertStringList( dvbConfig->getSourcesList(i) );
			setSource( sat0[i], dvbConfig->devList.at(i)->source, 0 );
			grid->addWidget( sat0[i], 3, 1 );

			lab = new QLabel( i18n("2nd sat :"), gb );
			grid->addWidget( lab, 3, 2 );
			sat1[i] = new QComboBox( gb );
			sat1[i]->setEnabled(false);
			sat1[i]->insertStringList( dvbConfig->getSourcesList(i) );
			setSource( sat1[i], dvbConfig->devList.at(i)->source, 1 );
			grid->addWidget( sat1[i], 3, 3 );

			lab = new QLabel( i18n("3rd sat :"), gb );
			grid->addWidget( lab, 4, 0 );
			sat2[i] = new QComboBox( gb );
			sat2[i]->setEnabled(false);
			sat2[i]->insertStringList( dvbConfig->getSourcesList(i) );
			setSource( sat2[i], dvbConfig->devList.at(i)->source, 2 );
			grid->addWidget( sat2[i], 4, 1 );

			lab = new QLabel( i18n("4th sat :"), gb );
			grid->addWidget( lab, 4, 2 );
			sat3[i] = new QComboBox( gb );
			sat3[i]->setEnabled(false);
			sat3[i]->insertStringList( dvbConfig->getSourcesList(i) );
			setSource( sat3[i], dvbConfig->devList.at(i)->source, 3 );
			grid->addWidget( sat3[i], 4, 3 );

			satNumber[i]->setValue( dvbConfig->devList.at(i)->source.contains("|") );
		}
		else {
			lab = new QLabel( i18n("Source :"), gb );
			grid->addWidget( lab, 2, 0 );
			sat0[i] = new QComboBox( gb );
			sat0[i]->insertStringList( dvbConfig->getSourcesList(i) );
			setSource( sat0[i], dvbConfig->devList.at(i)->source, 0 );
			grid->addWidget( sat0[i], 2, 1 );
		}

		vb->addWidget( gb );
		vb->addItem( new QSpacerItem( 20, 20, QSizePolicy::Ignored, QSizePolicy::Ignored ) );
	}

	recordPage = addPage(i18n("Recording"),i18n("DVB Recording options"),
		KGlobal::instance()->iconLoader()->loadIcon( "hdd_unmount", KIcon::NoGroup, KIcon::SizeMedium ) );
	vb = new QVBoxLayout( recordPage, 6, 6 );
	gb = new QGroupBox( "", recordPage );
	grid = new QGridLayout( gb, 1, 1, 20, 6 );

	lab = new QLabel( i18n("Records directory :"), gb );
	grid->addWidget( lab, 0, 0 );
	recordDirLe = new QLineEdit( gb );
	recordDirLe->setReadOnly( true );
	grid->addWidget( recordDirLe, 0, 1 );
	recordDirBtn = new QToolButton( gb );
	recordDirBtn->setIconSet( icon->loadIconSet("fileopen", KIcon::Small) );
	grid->addWidget( recordDirBtn, 0, 2 );

	lab = new QLabel( i18n("Time shifting directory :"), gb );
	grid->addWidget( lab, 1, 0 );
	shiftDirLe = new QLineEdit( gb );
	shiftDirLe->setReadOnly( true );
	grid->addWidget( shiftDirLe, 1, 1 );
	shiftDirBtn = new QToolButton( gb );
	shiftDirBtn->setIconSet( icon->loadIconSet("fileopen", KIcon::Small) );
	grid->addWidget( shiftDirBtn, 1, 2 );

	lab = new QLabel( i18n("Begin margin :"), gb );
	grid->addWidget( lab, 2, 0 );
	beginSpin = new QSpinBox( gb );
	h1 = new QHBoxLayout();
	h1->addWidget( beginSpin );
	lab = new QLabel( i18n("(minutes)"), gb );
	h1->addWidget( lab );
	grid->addLayout( h1, 2, 1 );

	lab = new QLabel( i18n("End margin :"), gb );
	grid->addWidget( lab, 3, 0 );
	endSpin = new QSpinBox( gb );
	h1 = new QHBoxLayout();
	h1->addWidget( endSpin );
	lab = new QLabel( i18n("(minutes)"), gb );
	h1->addWidget( lab );
	grid->addLayout( h1, 3, 1 );

	lab = new QLabel( i18n("Instant record duration :"), gb );
	grid->addWidget( lab, 4, 0 );
	instantDurationSpin = new QSpinBox( 1, 1440, 1, gb );
	h1 = new QHBoxLayout();
	h1->addWidget( instantDurationSpin );
	lab = new QLabel( i18n("(minutes)"), gb );
	h1->addWidget( lab );
	grid->addLayout( h1, 4, 1 );

	lab = new QLabel( i18n("Preferred format :"), gb );
	grid->addWidget( lab, 5, 0 );
	formatComb = new QComboBox( gb );
	grid->addWidget( formatComb, 5, 1 );

	vb->addWidget( gb );
	vb->addItem( new QSpacerItem( 20, 20, QSizePolicy::Ignored, QSizePolicy::Ignored ) );

	recordDirLe->setText( dvbConfig->recordDir );
	shiftDirLe->setText( dvbConfig->shiftDir );
	beginSpin->setValue( dvbConfig->beginMargin );
	endSpin->setValue( dvbConfig->endMargin );
	instantDurationSpin->setValue( dvbConfig->instantDuration );
	formatComb->insertItem( "TS" );
	formatComb->insertItem( "MPEG_PES" );
	formatComb->insertItem( "MPEG_PS" );
	switch ( dvbConfig->format ) {
		case OutTS : formatComb->setCurrentItem( 0 ); break;
		case OutPES : formatComb->setCurrentItem( 1 ); break;
		case OutPS : formatComb->setCurrentItem( 2 ); break;
	}

	miscPage = addPage(i18n("Misc"),i18n("Misc"),
		KGlobal::instance()->iconLoader()->loadIcon( "misc", KIcon::NoGroup, KIcon::SizeMedium ) );
	vb = new QVBoxLayout( miscPage, 6, 6 );
	gb = new QGroupBox( "", miscPage );
	grid = new QGridLayout( gb, 1, 1, 20, 6 );

	lab = new QLabel( i18n("Update scan data :"), gb );
	grid->addWidget( lab, 0, 0 );
	updateBtn = new KPushButton( gb );
	updateBtn->setGuiItem( KGuiItem(i18n("Download"), icon->loadIconSet("khtml_kget", KIcon::Small) ) );
	grid->addWidget( updateBtn, 0, 1 );

	vb->addWidget( gb );
	vb->addItem( new QSpacerItem( 20, 20, QSizePolicy::Ignored, QSizePolicy::Ignored ) );

	connect( recordDirBtn, SIGNAL(clicked()), this, SLOT(setRecordDir()) );
	connect( shiftDirBtn, SIGNAL(clicked()), this, SLOT(setShiftDir()) );
	connect( updateBtn, SIGNAL(clicked()), this, SLOT(downloadData()) );
}



void DvbConfigDialog::downloadData()
{
	int ret;

loop:
	if ( !dvbConfig->loadDvbData(0) ) {
		ret = KMessageBox::questionYesNo( this, i18n("<qt>Can't get DVB data from http://hftom.free.fr/kaxtv/dvbdata.tar.gz !<br>\
			Check your internet connection, and say Yes to try again.<br>\
			Or say No to cancel.<br> Should I try again ?</qt>") );
		if ( ret==KMessageBox::Yes ) goto loop;
		return;
	}
}



void DvbConfigDialog::setSource( QComboBox *box, QString s, int sat )
{
	QString c;
	int pos, i;

	for ( i=0; i<sat+1; i++ ) {
		pos = s.find("|");
		s = s.right( s.length()-pos-1 );
	}
	pos = s.find("|");
	c = s.left(pos);
	for ( i=0; i<(int)box->count(); i++ ) {
		if ( box->text(i)==c ) {
			box->setCurrentItem(i);
			break;
		}
	}
}



void DvbConfigDialog::satNumberChanged( int value, int devNum )
{
	sat0[devNum]->setEnabled( value > 0 );
	sat1[devNum]->setEnabled( value > 1 );
	sat2[devNum]->setEnabled( value > 2 );
	sat3[devNum]->setEnabled( value > 3 );
}



void DvbConfigDialog::setRecordDir()
{
	QString s = KFileDialog::getExistingDirectory( recordDirLe->text().stripWhiteSpace() );
	if ( s!="" ) recordDirLe->setText( s );
}



void DvbConfigDialog::setShiftDir()
{
	QString s = KFileDialog::getExistingDirectory( shiftDirLe->text().stripWhiteSpace() );
	if ( s!="" ) shiftDirLe->setText( s );
}



void DvbConfigDialog::accept()
{
	QString s;
	int i;

	if ( recordDirLe->text().stripWhiteSpace()=="" ) {
		KMessageBox::sorry( this, i18n("Invalid records directory.") );
		recordDirLe->setFocus();
		return;
	}
	if ( shiftDirLe->text().stripWhiteSpace()=="" ) {
		KMessageBox::sorry( this, i18n("Invalid time shifting directory.") );
		shiftDirLe->setFocus();
		return;
	}

	for ( i=0; i<(int)dvbConfig->devList.count(); i++ ) {
		if ( dvbConfig->devList.at(i)->type==FE_QPSK ) {
			s = "S";
			if ( sat0[i]->isEnabled() ) s+="|"+sat0[i]->currentText();
			if ( sat1[i]->isEnabled() ) s+="|"+sat1[i]->currentText();
			if ( sat2[i]->isEnabled() ) s+="|"+sat2[i]->currentText();
			if ( sat3[i]->isEnabled() ) s+="|"+sat3[i]->currentText();
		}
		else {
			if ( dvbConfig->devList.at(i)->type==FE_QAM ) s = "C";
			else s = "T";
			s+="|"+sat0[i]->currentText();
		}
		dvbConfig->devList.at(i)->source = s;
	}

	dvbConfig->recordDir = recordDirLe->text();
	if ( !dvbConfig->recordDir.endsWith("/") ) dvbConfig->recordDir+= "/";
	dvbConfig->shiftDir = shiftDirLe->text();
	if ( !dvbConfig->shiftDir.endsWith("/") ) dvbConfig->shiftDir+= "/";
	dvbConfig->beginMargin = beginSpin->value();
	dvbConfig->endMargin = endSpin->value();
	dvbConfig->instantDuration = instantDurationSpin->value();
	switch ( formatComb->currentItem() ) {
		case 0 : dvbConfig->format = OutTS; break;
		case 1 : dvbConfig->format = OutPES; break;
		case 2 : dvbConfig->format = OutPS; break;
	}
	dvbConfig->saveConfig();
	done( Accepted );
}



DvbConfigDialog::~DvbConfigDialog()
{
}
