<?php
/**
 * Maintenance module that renames the sent-mail folder.
 *
 * $Horde: imp/lib/Maintenance/Task/rename_sentmail_monthly.php,v 1.28.10.1 2005/01/03 12:25:36 jan Exp $
 *
 * Copyright 2001-2005 Michael Slusarz <slusarz@bigworm.colorado.edu>
 *
 * See the enclosed file COPYING for license information (GPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/gpl.html.
 *
 * @author  Michael Slusarz <slusarz@bigworm.colorado.edu>
 * @version $Revision: 1.28.10.1 $
 * @since   IMP 2.3.7
 * @package Horde_Maintenance
 */
class Maintenance_Task_rename_sentmail_monthly extends Maintenance_Task {

    /**
     * Renames the old sent-mail folders.
     *
     * @access public
     *
     * @return boolean  Whether all sent-mail folders were renamed.
     */
    function doMaintenance()
    {
        global $imp, $notification, $prefs;

        $success = true;

        include_once 'Horde/Identity.php';
        include_once IMP_BASE . '/lib/Folder.php';

        $identity = &Identity::singleton(array('imp', 'imp'));
        $imp_folder = &IMP_Folder::singleton();
        $folder_list = $identity->getAllSentmailfolders();

        /* Get rid of duplicates. */
        $folder_list = array_unique($folder_list);

        foreach ($folder_list as $sent_folder) {
            /* Display a message to the user and rename the folder.
               Only do this if sent-mail folder currently exists. */
            if ($imp_folder->exists($imp['stream'], $sent_folder)) {
                $old_folder = Maintenance_Task_rename_sentmail_monthly::_renameSentmailMonthlyName($sent_folder);
                $notification->push(sprintf(_("%s folder being renamed at the start of the month."), IMP::displayFolder($sent_folder)), 'horde.message');
                if ($imp_folder->exists($imp['stream'], $old_folder)) {
                    $notification->push(sprintf(_("%s already exists. Your %s folder was not renamed."), IMP::displayFolder($old_folder), IMP::displayFolder($sent_folder)), 'horde.warning');
                    $success = false;
                } else {
                    if ($imp_folder->rename($imp['stream'], $sent_folder, $old_folder)) {
                        /* Rebuild folder hash. */
                        $imp_folder->exists($imp['stream'], $sent_folder, true);
                        if (!$imp_folder->create($imp['stream'], $sent_folder, $prefs->getValue('subscribe'))) {
                            $success = false;
                        }
                    } else {
                        $success = false;
                    }
                }
            }

        }

        return $success;
    }

    /**
     * Returns information for the maintenance function.
     *
     * @access public
     *
     * @return string  Description of what the operation is going to do during
     *                 this login.
     */
    function describeMaintenance()
    {
        global $imp;

        include_once 'Horde/Identity.php';
        $identity = &Identity::singleton(array('imp', 'imp'));
        $folder_list = $identity->getAllSentmailfolders();

        /* Get rid of duplicates. */
        $folder_list = array_unique($folder_list);

        $new_folders = $old_folders = array();
        foreach ($folder_list as $folder) {
            if (!empty($folder)) {
                $old_folders[] = IMP::displayFolder($folder);
                $new_folders[] = IMP::displayFolder(Maintenance_Task_rename_sentmail_monthly::_renameSentmailMonthlyName($folder));
            }
        }

        return sprintf(_("The current folder(s) \"%s\" will be renamed to \"%s\"."), implode(', ', $old_folders), implode(', ', $new_folders));
    }

    /**
     * Determines the name the sent-mail folder will be renamed to.
     * <pre>
     * Folder name: sent-mail-month-year
     *   month = English:         3 letter abbreviation
     *           Other Languages: Month value (01-12)
     *   year  = 4 digit year
     * The folder name needs to be in this specific format (as opposed to a
     *   user-defined one) to ensure that 'delete_sentmail_monthly' processing
     *   can accurately find all the old sent-mail folders.
     * </pre>
     *
     * @access private
     *
     * @param string $folder  The name of the sent-mail folder to rename.
     *
     * @return string  New sent-mail folder name.
     */
    function _renameSentmailMonthlyName($folder)
    {
        global $language, $prefs;

        $last_maintenance = $prefs->getValue('last_maintenance');
        $last_maintenance = empty($last_maintenance) ? mktime(0, 0, 0, date('m') - 1, 1) : $last_maintenance;

        if (substr($language, 0, 2) == 'en') {
            return $folder . String::convertCharset(strtolower(strftime('-%b-%Y', $last_maintenance)), NLS::getExternalCharset(), 'UTF7-IMAP');
        } else {
            return $folder . String::convertCharset(strftime('-%m-%Y', $last_maintenance), NLS::getExternalCharset(), 'UTF7-IMAP');
        }
    }

}
