/* GNOME DB libary
 * Copyright (C) 1998,1999 Michael Lausch
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gnome-db-error-dlg.h"
#include "gnome-db-util.h"
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-stock.h>

struct _GnomeDbErrorDialogPrivate {
	GnomeDbError *error_widget;
	gchar *title;
};

static void gnome_db_error_dialog_class_init (GnomeDbErrorDialogClass* klass);
static void gnome_db_error_dialog_init       (GnomeDbErrorDialog* error);
static void gnome_db_error_dialog_destroy    (GtkObject *object);

/*
 * Private functions
 */
static void
cb_prev_error (GtkWidget* button, gpointer data)
{
	GnomeDbErrorDialog *dialog = (GnomeDbErrorDialog *) data;

	g_return_if_fail (GNOME_DB_IS_ERROR_DIALOG (dialog));
	gnome_db_error_prev (GNOME_DB_ERROR (dialog->priv->error_widget));
}

static void
cb_next_error (GtkWidget* button, gpointer data)
{
	GnomeDbErrorDialog *dialog = (GnomeDbErrorDialog *) data;

	g_return_if_fail (GNOME_DB_IS_ERROR_DIALOG (dialog));
	gnome_db_error_next (GNOME_DB_ERROR(dialog->priv->error_widget));
}

static void
cb_dismiss_error (GtkWidget* button, gpointer data)
{
	GnomeDbErrorDialog *dialog = GNOME_DB_ERROR_DIALOG(data);
	gnome_dialog_close(GNOME_DIALOG(data));
}

/*
 * GnomeDbErrorDialog class implementation
 */
GtkType
gnome_db_error_dialog_get_type (void)
{
	static GtkType type = 0;

	if (!type) {
		GtkTypeInfo info = {
			"GnomeDbErrorDialog",
			sizeof (GnomeDbErrorDialog),
			sizeof (GnomeDbErrorDialogClass),
			(GtkClassInitFunc) gnome_db_error_dialog_class_init,
			(GtkObjectInitFunc) gnome_db_error_dialog_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL,
		};
		type = gtk_type_unique (gnome_dialog_get_type (), &info);
	}
	return type;
}

static void
gnome_db_error_dialog_class_init (GnomeDbErrorDialogClass* klass)
{
	GtkObjectClass *object_class = GTK_OBJECT_CLASS (klass);

	object_class->destroy = gnome_db_error_dialog_destroy;
}
      
static void
gnome_db_error_dialog_init (GnomeDbErrorDialog *dialog)
{
	const gchar *buttons[4];

	buttons[0] = GNOME_STOCK_BUTTON_PREV;
	buttons[1] = GNOME_STOCK_BUTTON_NEXT;
	buttons[2] = GNOME_STOCK_BUTTON_OK;
	buttons[3] = 0;

	dialog->priv = g_new (GnomeDbErrorDialogPrivate, 1);
	dialog->priv->title = NULL;

	gnome_dialog_close_hides (GNOME_DIALOG (dialog), TRUE);

	/* create the error viewer */
	dialog->priv->error_widget = GNOME_DB_ERROR (gnome_db_error_new ());
	gtk_widget_show (GTK_WIDGET (dialog->priv->error_widget));

	gnome_dialog_constructv (GNOME_DIALOG(dialog),
				 _("Error viewer"),
				 buttons);
	gnome_dialog_button_connect (GNOME_DIALOG (dialog), 0,
				     GTK_SIGNAL_FUNC (cb_prev_error), dialog);
	gnome_dialog_button_connect (GNOME_DIALOG (dialog), 1,
				     GTK_SIGNAL_FUNC (cb_next_error), dialog);
	gnome_dialog_button_connect (GNOME_DIALOG (dialog), 2,
				     GTK_SIGNAL_FUNC (cb_dismiss_error), dialog);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox),
			    dialog->priv->error_widget,
			    TRUE, TRUE, 0);
}

static void
gnome_db_error_dialog_destroy (GtkObject *object)
{
	GtkObjectClass *parent_class;
	GnomeDbErrorDialog *dialog = (GnomeDbErrorDialog *) object;

	g_return_if_fail (GNOME_DB_IS_ERROR_DIALOG (dialog));

	/* free memory */
	if (dialog->priv->title)
		g_free (dialog->priv->title);
	g_free (dialog->priv);
	dialog->priv = NULL;

	parent_class = gtk_type_class (gnome_dialog_get_type ());
	if (parent_class && parent_class->destroy)
		parent_class->destroy (object);
}

/**
 * gnome_db_error_dialog_new
 * title: title of the dialog box
 *
 * Create a new GnomeDbErrorDialog, which is a subclass of the GnomeDialog
 * widget, just acting as a container for a GnomeDbError widget
 *
 * Returns: a pointer to the new widget, or NULL on error
 */
GtkWidget*
gnome_db_error_dialog_new (gchar* title)
{
	GnomeDbErrorDialog *dialog;

	dialog = GNOME_DB_ERROR_DIALOG (gtk_type_new (gnome_db_error_dialog_get_type ()));  
	if (title)
		gnome_db_error_dialog_set_title (dialog, title);

	return GTK_WIDGET (dialog);
}

/**
 * gnome_db_error_dialog_show_errors
 * @dialog: the GnomeDbErrorDialog widget
 * @error_list: list of errors to show
 *
 * Pops up the given GnomeDbErrorDialog widget, displaying any errors that
 * may be reported for the connection being used by this widget
 */
void
gnome_db_error_dialog_show_errors (GnomeDbErrorDialog *dialog,
                                   GList *error_list)
{
	g_return_if_fail (GNOME_DB_IS_ERROR_DIALOG (dialog));

	gnome_db_error_show (GNOME_DB_ERROR (dialog->priv->error_widget),
	                     error_list);
	gtk_widget_show (GTK_WIDGET (dialog));
}

/**
 * gnome_db_error_dialog_set_title
 */
const gchar *
gnome_db_error_dialog_get_title (GnomeDbErrorDialog *dialog)
{
	g_return_val_if_fail (GNOME_DB_IS_ERROR_DIALOG (dialog), NULL);
	return (const gchar *) dialog->priv->title;
}

/**
 * gnome_db_error_dialog_set_title
 * @dialog: the GnomeDbErrorDialog widget
 * @title: title to be shown
 *
 * Change the title for the given GnomeDbErrorDialog widget
 */
void
gnome_db_error_dialog_set_title (GnomeDbErrorDialog *dialog, const gchar *title)
{
	g_return_if_fail(GNOME_DB_IS_ERROR_DIALOG(dialog));
	g_return_if_fail(title != 0);

	if (dialog->priv->title != 0)
		g_free((gpointer) dialog->priv->title);
	dialog->priv->title = g_strdup(title);
}
