/* C Mode */

/* cgiparse.c
   Implements CGI (and FastCGI intepretation) for FDScript
   Originally implemented by Ken Haase in the Machine Understanding Group
     at the MIT Media Laboratory.

   Copyright (C) 1994-2001 Massachusetts Institute of Technology
   Copyright (C) 2001-2002 beingmeta, inc. (A Delaware Corporation)

   This program comes with absolutely NO WARRANTY, including implied
   warranties of merchantability or fitness for any particular purpose.

    Use, modification, and redistribution of this program is permitted
    under the terms of either (at the developer's discretion) the GNU
    General Public License (GPL) Version 2, the GNU Lesser General Public
    License.

    This program is based on the FramerD library released in Fall 2001 by
    MIT under both the GPL and the LGPL licenses, both of which accompany
    this distribution.  Subsequent modifications by beingmeta, inc. are
    also released under both the GPL and LGPL licenses (at the developer's
    discretion).
*/ 

static char vcid[] = "$Id: cgiparse.c,v 1.15 2002/04/29 13:48:57 haase Exp $";

#define FD_SOURCE 1
#include "framerd/fdscript.h"
#include "framerd/fdwww.h"

static struct FD_TEXT_ENCODING *iso_latin1;
static lisp cgi_data_symbol, content_symbol, content_disposition_symbol;
EXPORTED fd_lispenv fd_cgiparse_env;
fd_lispenv fd_cgiparse_env;
static int cgi_init_done=0;

/* Parsing CGI args */

static void init_uri_encoded_form_data(fd_slotmap cgi_frame,char *data)
{
  /* Initialize any variables bound in the query string */
  char *scan=data; 
  while (scan && (*scan)) {
    char *equals=strchr(scan,'='), *end;
    fd_u8char *varname, *value;
    if (equals) end=strchr(equals,'&'); else break;
    varname=convert_string(scan,equals);
    value=convert_string(equals+1,end);
    init_cgi_var(cgi_frame,varname,value);
    fd_xfree(varname); fd_xfree(value);
    if (end) scan=end+1; else scan=end;}
}

static void init_cgi_var(fd_slotmap smap,char *var,char *string)
{
  lisp symbol=fd_parse_string(var);
  if (!(SYMBOLP(symbol))) symbol=fd_make_symbol(var);
  if (string == NULL) fd_slotmap_add(smap,symbol,FD_EMPTY_CHOICE);
  else {
    lisp current=fd_slotmap_get(smap,symbol,FD_VOID);
    lisp new_string=fd_copy_string(string);
    if (FD_VOIDP(current)) {
      fd_slotmap_set(smap,symbol,new_string);
      fd_decref(new_string);}
    else if (FD_STRINGP(current)) {
      lisp new_list=FD_MAKE_LIST(2,new_string,incref(current));
      fd_slotmap_set(smap,symbol,new_list);
      fd_decref(new_list);}
    else if (FD_PAIRP(current)) {
      lisp new_list=FD_MAKE_PAIR(new_string,current);
      fd_slotmap_set(smap,symbol,new_list);
      fd_decref(new_list);}
    else fd_type_error("not a string or pair",current);}
}

static int parse_hex(char *buf)
{
  char *scan=buf; int c;
  while (*scan)
    if (isxdigit(*scan)) scan++;
    else fd_raise_exception("Invalid unicode escape");
  c=strtol(buf,NULL,16);
  return c;
}

static fd_u8char *convert_string(char *start,char *end)
{
  unsigned char *string=start, *scan=string;
  unsigned char *string_end=((end == NULL) ? (start+strlen(start)) : (end));
  struct FD_STRING_STREAM os;
  FD_INITIALIZE_STRING_STREAM(&os,(string_end-string)+3);
  while (scan < string_end)
    if (*scan == '+') {fd_sputc(&os,' '); scan++;}
    else if (*scan > 0x7f) {fd_sputc(&os,*scan); scan++;}  
    else if (*scan == '%') {
      char buf[3]; long charcode;
      scan++; buf[0]=*scan++; buf[1]=*scan++; buf[2]='\0';
      charcode=strtol(buf,NULL,16);
      if ((charcode == '\\') && (scan[0] == 'u')) {
	char buf[5]; long c; scan=scan+1;
	buf[0]=*scan++; buf[1]=*scan++; buf[2]=*scan++; buf[3]=*scan++;
	buf[4]=0; c=parse_hex(buf);
	fd_sputc(&os,c);}
      else if ((charcode == '\\') && (scan[0] == 'U')) {
	char buf[9]; long c; scan=scan+1;
	buf[0]=*scan++; buf[1]=*scan++; buf[2]=*scan++; buf[3]=*scan++;
	buf[4]=*scan++; buf[5]=*scan++; buf[6]=*scan++; buf[7]=*scan++;	
	buf[8]=0; c=parse_hex(buf);
	fd_sputc(&os,c);}
      else fd_sputc(&os,charcode);}
    else if ((*scan == '\\') && (scan[1] == 'u')) {
      char buf[5]; long c; scan=scan+2;
      buf[0]=*scan++; buf[1]=*scan++; buf[2]=*scan++; buf[3]=*scan++;
      buf[4]=0; c=parse_hex(buf);
      fd_sputc(&os,c);}
    else if ((*scan == '\\') && (scan[1] == 'U')) {
      char buf[9]; long c; scan=scan+1;
      buf[0]=*scan++; buf[1]=*scan++; buf[2]=*scan++; buf[3]=*scan++;
      buf[4]=*scan++; buf[5]=*scan++; buf[6]=*scan++; buf[7]=*scan++;	
      buf[8]=0; c=parse_hex(buf);
      fd_sputc(&os,c);}
    else {int c=*scan++; fd_sputc(&os,c);}
  return os.ptr;
}

/* Handling some special fields */

static void parse_cookie(fd_slotmap cgi_frame,char *cookie)
{
  char *binding=cookie;
  while (isspace(*binding)) binding++;
  while (1) {
    char *equals=strchr(binding,'='), *varname, *varval, *scan, *end=NULL;
    if (equals == NULL) {
      fd_warn("Bad cookie spec: `%s'",cookie);
      return;}
    else end=strchr(equals,';');
    /* Extract the varname and valname */
    varname=convert_string(binding,equals);
    varval=convert_string(equals+1,end);
    /* Uppercase the variable name */
    scan=varname; while (*scan) {*scan=toupper(*scan); scan++;}
    /* Do the init and then free the strings you created */
    init_cgi_var(cgi_frame,varname,varval);
    fd_xfree(varname); fd_xfree(varval);
    /* Advance to the next binding */
    if (end) binding=end+1; else break;
    while (isspace(*binding)) binding++;}
}

static void check_secure_connection(fd_slotmap cgi_frame,char *env_val)
{
  if (env_val == NULL)
    fd_slotmap_add(cgi_frame,fd_make_symbol("HTTP-SECURE"),FD_FALSE);
  else if ((strcasecmp(env_val,"NO") == 0) || (strcasecmp(env_val,"OFF") == 0))
    fd_slotmap_add(cgi_frame,fd_make_symbol("HTTP-SECURE"),FD_FALSE);
  else fd_slotmap_add(cgi_frame,fd_make_symbol("HTTP-SECURE"),FD_TRUE);
}

/* Handling client preferences */

struct _FD_CLIENT_PREF {fd_lisp pref; float weight; int init_pos;};

static int compare_client_pref(const void *v1,const void *v2)
{
  const struct _FD_CLIENT_PREF *p1=v1, *p2=v2;
  if (p1->weight > p2->weight) return 1;
  else if (p2->weight > p1->weight) return -1;
  else if (p1->init_pos < p2->init_pos) return -1;
  else if (p1->init_pos > p2->init_pos) return 1;
  else return 0;
}

static void bind_client_preferences
  (fd_slotmap cgi_data,char *symbol_name,char *env_val)
{
  if (env_val == NULL) {
    /* fprintf(stderr,"Binding for %s is NULL\n",symbol_name); */
    return;}
  else {
    struct _FD_CLIENT_PREF *prefs=
      fd_xmalloc(sizeof(struct _FD_CLIENT_PREF)*64);
    char *scan=env_val;
    fd_lisp results=FD_EMPTY_LIST;
    int i=0, n_prefs=0, max_prefs=64;
    /* fprintf(stderr,"BINDING %s based on %s\n",symbol_name,env_val); */
    while ((*scan) && (isspace(*scan))) scan++;
    while ((scan) && (*scan)) {
      char *equals=strstr(scan,";q="), *end=strchr(scan,',');
      int pref_len, weight_len;
      if (equals == NULL) equals=strstr(scan," q=");
      /* fprintf(stderr,"scan=%s\nequals=%s\nend=%s\n",scan,equals,end); */
      if (n_prefs >= max_prefs) {
	prefs=fd_xrealloc(prefs,sizeof(struct _FD_CLIENT_PREF)*(64+max_prefs));
	max_prefs=max_prefs+64;}
      if ((end == NULL) && (equals == NULL)) {
	pref_len=strlen(scan); weight_len=0;}
      else if ((end == NULL) && (equals)) {
	pref_len=equals-scan; weight_len=strlen(equals+3);}
      else if ((equals) && (equals < end)) {
	pref_len=equals-scan; weight_len=end-(equals+3);}
      else {pref_len=end-scan; weight_len=0;}
      /* fprintf(stderr,"weight_len=%d\npref_len=%d\n",weight_len,pref_len); */
      if (weight_len > 64) {
	fd_warn(_("Strange value to init %s: %s"),symbol_name,env_val);
	if (end) scan=end+1; else scan=end;}
      else if (weight_len == 0) {
	prefs[n_prefs].pref=fd_make_substring(scan,scan+pref_len);
	prefs[n_prefs].weight=1.0; prefs[n_prefs].init_pos=n_prefs;
	n_prefs++;}
      else {
	char buf[64];
	prefs[n_prefs].pref=fd_make_substring(scan,scan+pref_len);
	strncpy(buf,equals+3,weight_len); buf[weight_len]=NUL;
	sscanf(buf,"%f",&(prefs[n_prefs].weight));
	prefs[n_prefs].init_pos=n_prefs;
	n_prefs++;}
      if (end) {
	scan=end+1; while ((*scan) && (isspace(*scan))) scan++;}
      else scan=end;}
    /* fprintf(stderr,"About to start sorting %d prefs\n"); */
    qsort(prefs,n_prefs,sizeof(struct _FD_CLIENT_PREF),compare_client_pref);
    /* fprintf(stderr,"Done sorting %d prefs\n"); */
    i=0; while (i < n_prefs) {
      if (prefs[i].weight == 1.0)
	results=FD_MAKE_PAIR(FD_MAKE_PAIR(prefs[i].pref,FD_LISPFIX(1)),results);
      else results=
	     FD_MAKE_PAIR(FD_MAKE_PAIR(prefs[i].pref,
				       FD_LISPFLOAT(prefs[i].weight)),
			  results);
      i++;}
    /* fd_fprintf(stderr,"Results is %q\n",results); */
    fd_slotmap_set(cgi_data,fd_make_symbol(symbol_name),results);
    fd_decref(results);}
}

/* Actually doing the CGI init */

static char *overriden_query_string=NULL;

EXPORTED fd_override_query_string(char *s)
{
  overriden_query_string=fd_strdup(s);
}

EXPORTED lisp fd_stdcgi_init(fd_lispenv env)
{
  char *cookie, *rmethod, *query_string;
  lisp cgi_data; fd_slotmap cgi_frame;
  enum {get, post} method;
  if (cgi_init_done)
    return fd_symeval(cgi_data_symbol,env);
  else cgi_init_done=1;
  cgi_data=fd_make_slotmap(16);
  cgi_frame=SLOTMAP_PTR(cgi_data);
  fd_set_value(cgi_data_symbol,cgi_data,env);

  init_cgi_var(cgi_frame,"SERVER-SOFTWARE",getenv("SERVER_SOFTWARE"));
  init_cgi_var(cgi_frame,"SERVER-NAME",getenv("SERVER_NAME"));
  init_cgi_var(cgi_frame,"SERVER-PORT",getenv("SERVER_PORT"));
  init_cgi_var(cgi_frame,"SCRIPT-NAME",getenv("SCRIPT_NAME"));

  init_cgi_var(cgi_frame,"REFERER",getenv("HTTP_REFERER"));
  init_cgi_var(cgi_frame,"USER-AGENT",getenv("HTTP_USER_AGENT"));
  init_cgi_var(cgi_frame,"REMOTE-IDENT",getenv("REMOTE_IDENT"));
  init_cgi_var(cgi_frame,"REMOTE-HOST",getenv("REMOTE_HOST"));
  init_cgi_var(cgi_frame,"REMOTE-ADDRESS",getenv("REMOTE_ADDR"));

  init_cgi_var(cgi_frame,"PATH-INFO",getenv("PATH_INFO"));
  init_cgi_var(cgi_frame,"DOCUMENT-ROOT",getenv("DOCUMENT_ROOT"));
  init_cgi_var(cgi_frame,"PATH-TRANSLATED",getenv("PATH_TRANSLATED"));

  init_cgi_var(cgi_frame,"AUTH-TYPE",getenv("AUTH_TYPE"));
  init_cgi_var(cgi_frame,"REMOTE-USER",getenv("REMOTE_USER"));
  init_cgi_var(cgi_frame,"HTTP-COOKIE",getenv("HTTP_COOKIE"));
  init_cgi_var(cgi_frame,"SERVER-HOST-NAME",getenv("SERVER_HOST"));

  /* Parse the cookies */
  cookie=getenv("HTTP_COOKIE");
  if (cookie) parse_cookie(cgi_frame,cookie);

  /* Check if we have a secure connection */
  check_secure_connection(cgi_frame,getenv("HTTPS"));
    
  /* Get client preferences (accepted mime types, charsets, languages, etc) */
  bind_client_preferences
    (cgi_frame,"ACCEPTED-MIME-TYPES",getenv("HTTP_ACCEPT"));
  bind_client_preferences
    (cgi_frame,"ACCEPTED-CHARSETS",getenv("HTTP_ACCEPT_CHARSET"));
  bind_client_preferences
    (cgi_frame,"ACCEPTED-ENCODINGS",getenv("HTTP_ACCEPT_ENCODING"));
  bind_client_preferences
    (cgi_frame,"ACCEPTED-LANGUAGES",getenv("HTTP_ACCEPT_LANGUAGE"));

  /* Get the request method being used */
  rmethod=getenv("REQUEST_METHOD");
  if (rmethod) {
    init_cgi_var(cgi_frame,"REQUEST-METHOD",rmethod);
    if (strcmp(rmethod,"POST") == 0) method=post;
    else method=get;}
  else method=get;

  /* Get the query string */
  if (overriden_query_string)
    query_string=overriden_query_string;
  else query_string=getenv("QUERY_STRING");
  if (query_string == NULL)
    return fd_symeval(cgi_data_symbol,env);
  else {
    char *qstring=convert_string(query_string,NULL);
    init_cgi_var(cgi_frame,"QUERY",qstring);
    init_cgi_var(cgi_frame,"QUERY-STRING",qstring);
    init_cgi_var(cgi_frame,"QUERY_STRING",qstring);
    fd_xfree(qstring);}

  /* If we're posting, get the content length and read that as the query string,
     otherwise just use the environment variable. */
  if (method == post) {
    char *size_string=getenv("CONTENT_LENGTH");
    int size=strtol(size_string,NULL,10);
    int actual_size;
    query_string=fd_xmalloc(size+1);
    actual_size=fread(query_string,sizeof(char),size,stdin);
    query_string[actual_size]='\0';
    init_cgi_var(cgi_frame,"POST",query_string);}
  else query_string=fd_strdup(query_string);

  /* Initialize any variables bound in the query string */
  if (strchr(query_string,'='))
    init_uri_encoded_form_data(query_string);
  
  fd_xfree(query_string);
  return cgi_data;
}

#if (HAVE_FASTCGI)
#define fgetenv(x) FCGX_GetParam(x,fenv)

EXPORTED lisp fd_fastcgi_init
  (fd_lispenv env,FCGX_ParamArray fenv,FCGX_Stream *in)
{
  char *cookie, *rmethod, *ctype, *query_data;
  lisp cgi_data; fd_slotmap cgi_frame;
  enum {get, post} method;
  int mime_encoded=0, post_size;
  cgi_init_done=1; /* Housecleaning */
  cgi_data=fd_make_slotmap(16);
  cgi_frame=SLOTMAP_PTR(cgi_data);
  fd_set_value(cgi_data_symbol,cgi_data,env);

  init_cgi_var(cgi_frame,"SERVER-SOFTWARE",fgetenv("SERVER_SOFTWARE"));
  init_cgi_var(cgi_frame,"SERVER-NAME",fgetenv("SERVER_NAME"));
  init_cgi_var(cgi_frame,"SERVER-PORT",fgetenv("SERVER_PORT"));
  init_cgi_var(cgi_frame,"SCRIPT-NAME",fgetenv("SCRIPT_NAME"));

  init_cgi_var(cgi_frame,"REFERER",fgetenv("HTTP_REFERER"));
  init_cgi_var(cgi_frame,"USER-AGENT",fgetenv("HTTP_USER_AGENT"));
  init_cgi_var(cgi_frame,"REMOTE-IDENT",fgetenv("REMOTE_IDENT"));
  init_cgi_var(cgi_frame,"REMOTE-HOST",fgetenv("REMOTE_HOST"));
  init_cgi_var(cgi_frame,"REMOTE-ADDRESS",fgetenv("REMOTE_ADDR"));

  init_cgi_var(cgi_frame,"PATH-INFO",fgetenv("PATH_INFO"));
  init_cgi_var(cgi_frame,"DOCUMENT-ROOT",fgetenv("DOCUMENT_ROOT"));
  init_cgi_var(cgi_frame,"PATH-TRANSLATED",fgetenv("PATH_TRANSLATED"));

  init_cgi_var(cgi_frame,"AUTH-TYPE",fgetenv("AUTH_TYPE"));
  init_cgi_var(cgi_frame,"REMOTE-USER",fgetenv("REMOTE_USER"));
  init_cgi_var(cgi_frame,"HTTP-COOKIE",fgetenv("HTTP_COOKIE"));
  init_cgi_var(cgi_frame,"SERVER-HOST-NAME",fgetenv("SERVER_HOST"));


  /* Parse the cookies */
  cookie=fgetenv("HTTP_COOKIE");
  if (cookie) parse_cookie(cgi_frame,cookie);

  /* Check if we have a secure connection */
  check_secure_connection(cgi_frame,fgetenv("HTTPS"));
    
  /* Get client preferences (accepted mime types, charsets, languages, etc) */
  bind_client_preferences
    (cgi_frame,"ACCEPTED-MIME-TYPES",fgetenv("HTTP_ACCEPT"));
  bind_client_preferences
    (cgi_frame,"ACCEPTED-CHARSETS",fgetenv("HTTP_ACCEPT_CHARSET"));
  bind_client_preferences
    (cgi_frame,"ACCEPTED-ENCODINGS",fgetenv("HTTP_ACCEPT_ENCODING"));
  bind_client_preferences
    (cgi_frame,"ACCEPTED-LANGUAGES",fgetenv("HTTP_ACCEPT_LANGUAGE"));

  /* Get the request method being used */
  rmethod=fgetenv("REQUEST_METHOD");
  if (rmethod) {
    init_cgi_var(cgi_frame,"REQUEST-METHOD",rmethod);
    if (strcmp(rmethod,"POST") == 0) method=post;
    else method=get;}
  else method=get;

  /* Get the request method being used */
  ctype=fgetenv("CONTENT_TYPE");
  if (rencoding) {
    init_cgi_var(cgi_frame,"CONTENT-TYPE",ctype);
    if (strcmp(ctype,"multipart/form-encoded") == 0) mime_encoded=1;
    else method=get;}
  else mime_encoded=0;

  /* Get the query string */
  if (method == get) {
    query_data=fgetenv("QUERY_STRING");
    if (query_data == NULL)
      return fd_symeval(cgi_data_symbol,env);
    else {
      char *qstring=convert_string(query_data,NULL);
      init_cgi_var(cgi_frame,"QUERY",qstring);
      init_cgi_var(cgi_frame,"QUERY-STRING",qstring);
      init_cgi_var(cgi_frame,"QUERY_STRING",qstring);
      free(qstring);}}
  /* If we're posting, get the content length and read that as the query string,
     otherwise just use the environment variable. */
  else if (method == post) {
    char *size_string=fgetenv("CONTENT_LENGTH");
    post_size=strtol(size_string,NULL,10);
    query_data=fd_xmalloc(post_size+1);
    FCGX_GetStr(query_data,post_size,in);
    query_data[post_size]='\0';
    init_cgi_var(cgi_frame,"POST",query_data);}
  else query_string=fd_strdup(query_data);

  /* Initialize any variables bound in the query string */
  if (mime_encoded) {
    fd_lisp entity=fd_parse_mime(query_data,post_size);
    fd_lisp contents=fd_prim_get(entity,content_symbol);
    DOLIST(item,contents) {
      if (FD_SLOTMAPP(item)) {
	fd_lisp disp=fd_prim_get(entity,content_disposition_symbol);
	if (FD_STRINGP(disp)) {
	  fd_u8char *data=FD_STRING_DATA(disp);
	  fd_u8char *namestart=strstr(" name=",data);
	  if (namestart) {
	    fd_prim_add(cgi_data,fd_make_symbol(),fd_prim_get(entity,content_symbol));}}}}
    fd_slotmap_add(cgi_frame,fd_make_symbol("PARSED-POST"),entity);}
  else if (strchr(query_data,'='))
    init_uri_encoded_form_data(query_string);

  fd_xfree(query_data);
  return cgi_data;
}
#endif

static lisp reverse_list(lisp lst)
{
  if (!(PAIRP(lst))) return lst;
  else {
    lisp answer=FD_EMPTY_LIST;
    DOLIST(elt,lst)
      answer=FD_MAKE_PAIR(incref(elt),answer);
    decref(lst);
    return answer;}
}

static lisp cgi_init_handler(lisp expr,fd_lispenv env)
{
  lisp vars=fd_get_body(expr,1);
  lisp table=fd_symeval(cgi_data_symbol,env);
  {DOLIST(var,vars)
     if (SYMBOLP(var)) {
       lisp v=fd_prim_get(table,var);
       if (FD_PAIRP(v)) {
	 fd_lisp rv=reverse_list(v);
	 fd_bind_value(var,rv,env);
	 fd_decref(rv);}
       else if (FD_EMPTYP(v))
	 fd_bind_value(var,FD_EMPTY_LIST,env);
       else fd_bind_value(var,v,env);}}
  fd_decref(table);
  return FD_VOID;
}

static lisp cgi_var_handler(lisp expr,fd_lispenv env)
{
  lisp vars=fd_get_body(expr,1);
  lisp table=fd_symeval(cgi_data_symbol,env);
  {DOLIST(var,vars)
     if (SYMBOLP(var)) {
       lisp vals=fd_prim_get(table,var);
       if (FD_STRINGP(vals)) {
	 lisp nv=fd_parse_string(FD_STRING_DATA(vals));
	 fd_bind_value(var,nv,env);
	 fd_decref(nv);}
       else if (FD_PAIRP(vals)) {
	 lisp nv=FD_EMPTY_CHOICE;
	 DOLIST(v,vals) {
	   if (FD_STRINGP(v)) {
	     ADD_TO_CHOICE(nv,fd_parse_string(FD_STRING_DATA(v)));}
	   else {ADD_TO_CHOICE(nv,incref(v));}}
	 fd_bind_value(var,nv,env);
	 fd_decref(nv);}
       else fd_bind_value(var,vals,env);}}
  fd_decref(table);
  return FD_VOID;
}

#if 0 /* Removed since functionality based on (GET *CGI-DATA* 'x) is almost as clear... */
static lisp cgi_get_strings_cproc(lisp slotid)
{
  lisp table=fd_symeval(cgi_data_symbol,env);
  lisp result=fd_prim_get(table,slotid);
  if (FD_STRINGP(result)) return FD_MAKE_LIST1(result);
  else return result;
}

static lisp cgi_get_cproc(lisp slotid)
{
  lisp table=fd_symeval(cgi_data_symbol,env);
  lisp strings=fd_prim_get(table,slotid);
  lisp result=(FD_EMPTY_CHOICE);
  if (FD_STRINGP(strings)) {
    FD_ADD_TO_CHOICE(result,fd_parse_string(FD_STRING_DATA(strings)));}
  else if (FD_PAIRP(strings)) {
    DOLIST(string,strings) {
      if (FD_STRINGP(string)) {
	FD_ADD_TO_CHOICE(results,fd_parse_string(FD_STRING_DATA(string)));}
      else FD_ADD_TO_CHOICE(results,fd_incref(string));}}
  else result=fd_incref(strings);
  fd_decref(strings); fd_decref(table);
  return result;
}
#endif

/* Fast CGI output extensions */

#if (HAVE_FASTCGI)
static void fcgi_puts(char *s,fd_htstream *f)
{
  if (f->stream_type == fcgi) {
    if (FCGX_PutS(s,f->stream.fcgi) < 0) 
      fd_raise_exception("FCGI error");}
  else fd_raise_exception("Weird HTTP stream");
}
static void fcgi_putc(int ch,fd_htstream *f)
{
  if (f->stream_type == fcgi) {
    if (FCGX_PutChar(ch,f->stream.fcgi) < 0)
      fd_raise_exception("FCGI error");}
  else fd_raise_exception("Weird HTTP stream");
}
static void fcgi_putn(char *s,int n,fd_htstream *f)
{
  if (f->stream_type == fcgi) {
    if (FCGX_PutStr(s,n,f->stream.fcgi) < 0)
      fd_raise_exception("FCGI error");}
  else fd_raise_exception("Weird HTTP stream");
}
#endif

void fd_init_cgiparse_c()
{
  fd_lispenv menv=fd_make_module();
  fd_cgiparse_env=menv;

  cgi_data_symbol=fd_make_symbol("CGI-DATA");
  content_symbol=fd_make_symbol("CONTENT");
  content_disposition_symbol=fd_make_symbol("CONTENT-DISPOSITION");
  iso_latin1=fd_get_encoding("LATIN-1");
  fd_add_special_form(menv,"CGI-INIT",cgi_init_handler);
  fd_add_special_form(menv,"CGI-VAR",cgi_var_handler);

#if (HAVE_FASTCGI)
  fd_set_http_output_methods
    ((void (*)(char *,void *))fcgi_puts,
     (void (*)(int,void *))fcgi_putc,
     (void (*)(char *,int,void *))fcgi_putn);
#endif

  fd_register_module("CGITOOLS",menv);

  fd_register_source_file("cgiparse",__DATE__,vcid);
}



/* File specific stuff */

/* The CVS log for this file
   $Log: cgiparse.c,v $
   Revision 1.15  2002/04/29 13:48:57  haase
   Fixed leak in port argument to PRINTOUT-TO

   Revision 1.14  2002/04/16 16:14:39  haase
   Fixed some inconsistent returns

   Revision 1.13  2002/04/11 00:23:21  haase
   Register the CGI module for external access

   Revision 1.12  2002/04/02 21:39:33  haase
   Added log and emacs init entries to C source files

*/

/* Emacs local variables
;;;  Local variables: ***
;;;  compile-command: "cd ../..; make" ***
;;;  End: ***
*/
