/*
 * FISG - Fast IRC Statistic Generator
 * Programmed and designed by Matti 'ccr' Hamalainen
 * (C) Copyright 2003 Tecnic Software productions (TNSP)
 *
 * Please read file 'COPYING' for information on license and distribution.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "fisg.h"
#include "th_util.h"

/*
 * Handling of userlists
 */
t_user_entry *user_new(char *userHandle)
{
 t_user_entry *pResult;

 /* Allocate memory for new node */
 pResult = (t_user_entry *) calloc(1, sizeof(t_user_entry));
 if (pResult == NULL) return NULL;

 /* Set fields */
 th_strcpy(&pResult->userHandle, userHandle);

 return pResult;
}


void user_free(t_user_entry *pUser)
{
 assert(pUser);

 if (pUser->userHandle) free(pUser->userHandle);
 if (pUser->picPath) free(pUser->picPath);
 if (pUser->linkURL) free(pUser->linkURL);
 if (pUser->sComment) free(pUser->sComment);

 free(pUser);
}


t_nick_entry *nick_new(char *newNick)
{
 t_nick_entry *pResult;

 /* Allocate memory for new node */
 pResult = (t_nick_entry *) calloc(1, sizeof(t_nick_entry));
 if (pResult == NULL) return NULL;

 /* Set fields */
 strncpy(pResult->nick, newNick, SET_MAX_NICKLEN);
 pResult->nick[SET_MAX_NICKLEN - 1] = 0;
 
 return pResult;
}


void nick_free(t_nick_entry *pNick)
{
 assert(pNick);

 free(pNick);
}



/*
 * Insert given node to linked list
 */
#define LPREV	(pNode->pPrev)
#define LTHIS	(pNode)
#define LNEXT	(pNode->pNext)

void user_insert(t_user_entry **userList, t_user_entry *pNode)
{
 assert(userList);
 
 if (*userList)
	{
	/* The first node's pPrev points to last node */
	LPREV = (*userList)->pPrev;	/* New node's prev = Previous last node */
	(*userList)->pPrev->pNext = pNode;	/* Previous last node's next = New node */
	(*userList)->pPrev = pNode;		/* New last node = New node */
	LNEXT = NULL;				/* But next is NULL! */
 	} else {
	*userList = pNode;			/* First node ... */
	LPREV = pNode;				/* ... it's also last */
	LNEXT = NULL;				/* But next is NULL! */
 	}
}


/*
 * Delete given node from linked list
 */
void user_delete(t_user_entry **userList, t_user_entry *pNode)
{
 assert(userList);
 assert(*userList);
 assert(pNode);

 if (LPREV) LPREV->pNext = LNEXT;
 if (LNEXT)
	LNEXT->pPrev = LPREV;
	else
	(*userList)->pPrev = LPREV;

 LPREV = LNEXT = NULL;
}


/*
 * Insert given node into the nicklist hashtable
 */
int nick_insert(t_nick_entry *nickList[], t_nick_entry *pNode)
{
 int i;
 char *tmpNick;

 assert(nickList);
 
 /* If the first character is patternmatcher, find the first alphanum */
 tmpNick = pNode->nick;
 while ((*tmpNick == '*') || (*tmpNick == '?')) tmpNick++;
 i = th_tolower(*tmpNick);
 
 /* Check the hashcode */
 if ((i < 0) && (i >= SET_HASH_MAXINDEX))
 	return -1;
 
 if (nickList[i])
	{
	/* The first node's pPrev points to last node */
	pNode->pPrev = nickList[i]->pPrev;	/* New node's prev = Previous last node */
	nickList[i]->pPrev->pNext = pNode;	/* Previous last node's next = New node */
	nickList[i]->pPrev = pNode;		/* New last node = New node */
	pNode->pNext = NULL;			/* But next is NULL! */
 	} else {
	nickList[i] = pNode;			/* First node */
	pNode->pPrev = pNode;			/* But also last */
	pNode->pNext = NULL;			/* But next is NULL! */
 	}
 
 return 0; 
}


/*
 * Check if given nick is in the userlist
 */
t_user_entry *user_search(t_nick_entry *nickList[], char *findNick)
{
 t_nick_entry *pCurr;
 int i, j;
 BOOL isFound;

 assert(nickList);

 isFound = FALSE;
 pCurr = NULL;
 j = 0;
 
 /* Check hash */
 while ((!isFound) && (findNick[j]))
 {
 i = th_tolower(findNick[j]);
 if ((i >= 0) && (i < SET_HASH_MAXINDEX))
 	{
 	/* Find from linked list */
	pCurr = nickList[i];
	while (pCurr && (!isFound))
		{
		if (th_strcasematch(findNick, pCurr->nick))
			isFound = TRUE;
			else
			pCurr = pCurr->pNext;
		}
	}
 j++;
 }
 
 /* Return result */
 if (isFound)
	return pCurr->user;
	else
	return NULL;
}


/*
 * Update nicks for given user to another user
 */
void nick_change(t_nick_entry *nickList[], t_user_entry *pFind, t_user_entry *pNew)
{
 t_nick_entry *pCurr;
 int i;

 assert(nickList);
 
 for (i = 0; i < SET_HASH_MAXINDEX; i++)
	{
 	/* Find from linked list */
	pCurr = nickList[i];
	while (pCurr)
		{
		if (pCurr->user == pFind)
			pCurr->user = pNew;

		pCurr = pCurr->pNext;
		}
	}
}


