/*
 *
 *   Copyright (C) International Business Machines  Corp., 2003, 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <plugin.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/vfs.h>

#include "osi_endian.h"
#include "ogfs_ondisk.h"
#include "ogfsfsim.h"
#include "cluster_config.h"

plugin_record_t *my_plugin_record = &ogfs_plugin_record;
engine_functions_t *EngFncs = NULL;
static u_int32_t ogfsutils_support = 0;

/**
 *	ogfs_get_volume_type - identify the type of volume from metadata in vol->private_data
 *	@volume: address of logical volume structure corresponding to OGFS volume
 *
 *	This routine examines the metadata structure pointed to by the given
 *	volume structure's private_data field. We return a enum type corresponding
 *	to the type of ogfs volume with think it represents.
 */
static ogfs_volume_type_t ogfs_get_volume_type(logical_volume_t *volume)
{
	ogfs_volume_type_t type = OGFS_UNKNOWN_VOL;
	
	if (volume->private_data != NULL) {
		uint32 magic = *((uint32 *)volume->private_data);
		
		if (magic == OGFS_MAGIC) {
			ogfs_meta_header_t *ogfs_mh;
	
			ogfs_mh = (ogfs_meta_header_t *)volume->private_data;

			if (ogfs_mh->mh_type == OGFS_METATYPE_SB) {
				type = OGFS_FS_VOL;
			} else if (ogfs_mh->mh_type == OGFS_METATYPE_LH) {
				type = OGFS_JOURNAL_VOL;
			}
		} else if (magic == OGFS_CLUSTER_GLOBAL) {
			type = OGFS_CI_VOL;
		}
	}
	
	return type;
}

/**
 *	ogfs_exec_utility - fork and exec some FSIM utility
 *
 *	This routine basically does what pretty much all the other FSIMs do when
 *	exec'ing a filesystem utility so *cut* & *paste*, here it is! I made it
 *	slightly more generic though.
 */
int ogfs_exec_utility(logical_volume_t *volume, char **argv)
{
	pid_t pidm;
	char *buffer = NULL;
	boolean display_to_user = FALSE;
	int rc = 0, status, fds2[2], bytes_read = 0;

	if (!(buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN))) {
		return ENOMEM;
	}

	/* open pipe, alloc buffer for collecting mkfs output */
	if ( (rc = pipe(fds2)) ) {
		return rc;
	}

	/* determine if output should be shown to user besides logged */
	if (strcasecmp(argv[0], "ogfsck") == 0) {
		display_to_user = TRUE;
	}

	pidm = EngFncs->fork_and_execvp(volume, argv, NULL, fds2, fds2);
	if (pidm != -1) {
		fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);
		while (!(waitpid( pidm, &status, WNOHANG ))) {
			bytes_read = read(fds2[0], buffer, MAX_USER_MESSAGE_LEN);
			if (bytes_read > 0) {
				if (display_to_user) {
					MESSAGE(_("%s output: \n%s\n"), argv[0], buffer);
				} else {
					LOG_DETAILS("%s output: \n%s\n", argv[0], buffer);
				}
				/* clear out buffer */
				memset(buffer, 0, bytes_read);
			}
			/* don't hog all the cpu */
			usleep(10000);
		}

		if (WIFEXITED(status)) {
			do {
				bytes_read = read(fds2[0], buffer, MAX_USER_MESSAGE_LEN);
				if (bytes_read > 0) {
					if (display_to_user) {
						MESSAGE(_("%s output: \n%s\n"), argv[0], buffer);
					} else {
						LOG_DETAILS("%s output: \n%s\n", argv[0], buffer);
					}
				}
			} while (bytes_read > 0);
			rc = WEXITSTATUS(status);
			if (rc == 0) {
				LOG_DETAILS("%s completed with exit code %d \n", argv[0], rc);
			} else {
				LOG_ERROR("%s completed with exit code %d \n", argv[0], rc);
			}
		} else {
			rc = EINTR;
		}
	} else {
		rc = EIO;
	}

	EngFncs->engine_free(buffer);
	close(fds2[0]);
	close(fds2[1]);
	
	return rc;
}

/**
 *	ogfs_setup - plug-in initialization routine
 *	@engine_function_table: address of function table of engine services
 *
 *	This routine is called by the engine when the FSIM is loaded.
 *	We initialize some necessary globals and do any other necessary
 *	initialization.
 */
static int ogfs_setup(engine_functions_t *engine_function_table)
{
	int rc = 0;
	char *argv[3];
	
	EngFncs = engine_function_table;	
	LOG_ENTRY();
	
	/*
	 * BUGBUG: for now just to see if mkfs.ogfs is installed. this
	 * test needs to be improved to test for package releases.
	 */
	argv[0] = "mkfs.ogfs";
	argv[1] = "-V";
	argv[2] = NULL;
	
	if (ogfs_exec_utility(NULL, argv) == 0) {
		ogfsutils_support = 1;
	} else {
		ogfsutils_support = 0;
	}		
	
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_free_associated_vols_list - free list of volumes associated with fs
 *	@ogfs_sb: pointer to ogfs superblock structure
 *
 *	This routine determines if we need additional cleanup of a
 *	ogfs superblock as we used the reserved area to anchor a
 *	list of volumes (external journals and cidev volume) associated
 *	with the filesystem volume. If there is such a list we delete
 *	it.
 */
static void ogfs_free_associated_vols_list(struct ogfs_sb *ogfs_sb)
{
	list_anchor_t assoc_vols;
				
	assoc_vols = *((list_anchor_t *)(ogfs_sb->sb_reserved));
				
	if (assoc_vols != NULL) {
		EngFncs->delete_all_elements(assoc_vols);
	}
}

/**
 *	ogfs_free_private_data - free struct attached to volume private data
 *	@volume: pointer to logical volume structure
 *
 *	This routine simply frees the memory allocated and attached to
 *	the volume structure through the private_data pointer and clears
 *	the pointer value afterwards.
 */
static void ogfs_free_private_data(logical_volume_t *volume)
{
	if (volume->private_data != NULL) {
		if (ogfs_get_volume_type(volume) == OGFS_FS_VOL) {
			ogfs_free_associated_vols_list((struct ogfs_sb *)volume->private_data);
		}
		EngFncs->engine_free(volume->private_data);
		volume->private_data = NULL;
	}
}

/**
 *	ogfs_cleanup - allows for cleanup when plugin is being unloaded
 *	@void
 *
 *	This routine retrieves all volumes we have claimed and frees any
 *	memory we may have allocated and hung off each volume->private_data
 *	pointer.
 */
static void ogfs_cleanup(void)
{
	int rc;
	logical_volume_t *volume;
	list_anchor_t global_volumes;
	list_element_t vol_list_iter;

	LOG_ENTRY();

	rc = EngFncs->get_volume_list(my_plugin_record, NULL, 0, &global_volumes);
	if (rc == 0) {
		LIST_FOR_EACH(global_volumes, vol_list_iter, volume) {
			ogfs_free_private_data(volume);
		}
		EngFncs->destroy_list(global_volumes);
	}

	LOG_EXIT_VOID();
	return;
}

/**
 *	ogfs_discard - cleanup data structures associated with an ogfs volume
 *	@volume: pointer to logical volume structure
 *
 *	This routine is invoked by the engine when it wants us to forget
 *	about a volume. We free our in-memory metadata structure pointed
 *	to by the volume->private_data field.
 */
static int ogfs_discard(logical_volume_t *volume)
{
	int rc = 0;
	
	LOG_ENTRY();

	ogfs_free_private_data(volume);	
		
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	fsim_read_bytes - read specific number of bytes from a volume
 *
 *	volume: pointer to logical volume structure
 *	fd: file handle of an opened device to read from
 *	offset: byte offset from beginning of volume for start of read
 *	bytes_to_read: number of bytes to read
 *	buffer: buffer to transfer data in to
 */
static int fsim_read_bytes(logical_volume_t *volume,
			   int     fd,
			   int64_t offset,
			   int32_t bytes_to_read,
			   void   *buffer)
{
	int rc = 0;
	size_t bytes_read;
	
	LOG_ENTRY();
	
	bytes_read = EngFncs->read_volume(volume, fd, buffer,
					  bytes_to_read, offset);
	if (bytes_read != bytes_to_read)
		rc = EIO;
	
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	fsim_write_bytes - write specific number of bytes to a volume
 *
 *	volume: pointer to logical volume structure
 *	fd: file descriptor of an opened device to write to
 *	offset: byte offset from beginning of volume for start of write
 *	bytes_to_read: number of bytes to write
 *	buffer: buffer to transfer data out of
 */
static int fsim_write_bytes(logical_volume_t *volume,
			   int     fd,
			   int64_t offset,
			   int32_t bytes_to_write,
			   void   *buffer)
{
	int rc = 0;
	size_t bytes_written;
	
	LOG_ENTRY();
	
	bytes_written = EngFncs->write_volume(volume, fd, buffer,
					      bytes_to_write, offset);
	if (bytes_written != bytes_to_write)
		rc = EIO;
	
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_statfs - return file system statistics
 *	@volume: the address of the logical volume structure
 *	@fs_size: address to return current filesystem size in sectors
 *	@min_fs_size: address to return the minimum filesystem size in sectors
 *	@max_fs_size: address to return the maximum filesystem size in sectors
 *	@max_object_size: address to return the maximum size for the volume
 *
 *	This routine returns some filesystem statistics. At the moment, it uses
 *	statfs() to gather the information. This means the volume must be mounted.
 *	This is OK as we typically gather this information for the expand command
 *	which ogfs requires be done online.
 */
static int ogfs_statfs(logical_volume_t *volume,
		       sector_count_t *fs_size,
		       sector_count_t *min_fs_size,
		       sector_count_t *max_fs_size,
		       sector_count_t *max_object_size)
{
	int rc;
	
	if (EngFncs->is_mounted(volume->dev_node, NULL) &&
		ogfs_get_volume_type(volume) == OGFS_FS_VOL) {
		struct statfs stats;
		
		rc = statfs(volume->mount_point, &stats);
		if (rc == 0) {
			/* current filesystem size */
			if (fs_size != NULL) {
				*fs_size = stats.f_blocks *
					(stats.f_bsize >> EVMS_VSECTOR_SIZE_SHIFT);
			}
			
			/* minimum filesystem size */
			if (min_fs_size != NULL) {
				*min_fs_size = stats.f_blocks *
					(stats.f_bsize >> EVMS_VSECTOR_SIZE_SHIFT);
			}
			
			/* maximum filesystem size */
			if (max_fs_size != NULL) {
				*max_fs_size = (u_int64_t)1 << 63;
			}
			
			/* maximum volume size */
			if (max_object_size != NULL) {
				*max_fs_size = (u_int64_t)1 << 63;
			}
		} else {
			rc = errno;
		}
	} else {
		rc = EPERM;
	}
	
	return rc;
}

/**
 *	ogfs_get_fs_limits - obtain minimum and maximum filesystem limits
 *	@volume: the address of the logical volume structure
 *	@min_fs_size: address to return the minimum filesystem size in sectors
 *	@max_fs_size: address to return the maximum filesystem size in sectors
 *	@max_object_size: address to return the maximum size for the volume
 *
 *	This routine returns limit information for only a type OGFS_FS_TYPE volume.
 */
static int ogfs_get_fs_limits(logical_volume_t *volume,
			      sector_count_t *min_fs_size,
			      sector_count_t *max_fs_size,
			      sector_count_t *max_object_size)
{
	int rc;
	
	LOG_ENTRY();
	
	rc = ogfs_statfs(volume, NULL, min_fs_size, max_fs_size, max_object_size);
	
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_copy_metadata - copy input disk metadata to host order
 *	@buf: buffer containing metadata in disk-order as input
 *	@ogfs_mh: buffer containing metadata in cpu-order as output
 *
 *	This routine copies metadata in disk-order in the given buffer
 *	over to the ogfs_mh buffer in cpu-order. We only care about
 *	certain metatypes and others are summarily ignored/rejected.
 */
static int ogfs_copy_metadata(char *buf, ogfs_meta_header_t *ogfs_mh)
{
	int rc = 0;
	
	LOG_ENTRY();
		
	switch (ogfs_mh->mh_type) {
	case OGFS_METATYPE_SB:
		ogfs_sb_in((struct ogfs_sb *)ogfs_mh, buf);
		break;
	case OGFS_METATYPE_LH:
		ogfs_log_header_in((struct ogfs_log_header *)ogfs_mh, buf);
		break;
	case OGFS_METATYPE_DI:
		ogfs_dinode_in((ogfs_dinode_t *)ogfs_mh, buf);
		/* if the dinode has some stuffed data. copy it as well */
		if (((ogfs_dinode_t *)ogfs_mh)->di_height == 0) {
			memcpy((char *)ogfs_mh + sizeof(ogfs_dinode_t),
			       buf + sizeof(ogfs_dinode_t),
			       ((ogfs_dinode_t *)ogfs_mh)->di_size);
		}
		break;
	default:
		rc = EINVAL;
	}
	
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_read_meta_header - read a block containing ogfs_meta_header
 *	@volume: pointer to evms volume structure
 *	@ogfs_mh: pointer to a buffer to copy ogfs meta data
 *	@fd: the file descriptor of an opened volume device
 *	@offset: offset from start of volume to read an ogfs disk structure
 *	@size: block size in bytes
 *
 *	This routine reads an ogfs basic block from the given volume
 *	device and checks for the magic number. It converts the ogfs
 *	meta_header to cpu-order and makes additional checks to restrict what
 *	sort of meta-data we currently care about.
 */
static int ogfs_read_meta_header(logical_volume_t *volume,
				 ogfs_meta_header_t *ogfs_mh,
				 int fd,
				 int64_t offset,
				 int32_t size)
{
	int rc;
	char *buf;

	LOG_ENTRY();
	
	/*
	 * Allocate a buffer for the read which will contain the metadata
	 * in disk-order. On success, the ogfs_mh will point to the copy
	 * of this metadata in cpu-order.
	 */
	buf = EngFncs->engine_alloc(size);
	if (buf != NULL) {	
		/*
		 * Read a block to contain at least an ogfs_meta_header and a
		 * bit more (like superblock info).
		 */
		rc = fsim_read_bytes(volume, fd, offset, size, buf);
		if (rc == 0) {
			ogfs_meta_header_in(ogfs_mh, buf);
			if (ogfs_mh->mh_magic != OGFS_MAGIC) {
				rc = EINVAL;
			} else {
				rc = ogfs_copy_metadata(buf, ogfs_mh);
			}
		}
		EngFncs->engine_free(buf);
	} else {
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_get_super_block - get an ogfs filesystem superblock
 *	@volume: pointer to evms volume structure
 *	@ogfs_mh: pointer to a buffer to copy ogfs meta data
 *	@fd: the file descriptor of an opened volume device
 */
inline static int ogfs_get_super_block(logical_volume_t *volume,
				       ogfs_meta_header_t *ogfs_mh,
				       int fd)
{
	int rc;
	
	rc = ogfs_read_meta_header(volume, ogfs_mh, fd,
				   OGFS_SB_OFFSET, OGFS_BASIC_BLOCK);
	if (rc == 0 && ogfs_mh->mh_type != OGFS_METATYPE_SB)
		rc = EINVAL;
		
	return rc;
}

/**
 *	ogfs_get_log_header - get an ogfs external journal log header
 *	@volume: pointer to evms volume structure
 *	@ogfs_mh: pointer to a buffer to copy ogfs meta data
 *	@fd: the file descriptor of an opened volume device
 */
inline static int ogfs_get_log_header(logical_volume_t *volume,
				      ogfs_meta_header_t *ogfs_mh,
				      int fd)
{
	int rc;
	
	rc = ogfs_read_meta_header(volume, ogfs_mh, fd,
				   OGFS_LH_OFFSET, OGFS_BASIC_BLOCK);
	if (rc == 0 && ogfs_mh->mh_type != OGFS_METATYPE_LH)
		rc = EINVAL;
		
	return rc;
}

/**
 *	ogfs_get_meta_header - returns ogfs metadata from given volume
 *	@volume: the volume to read from
 *	@ogfs_mh: the buffer to place the ogfs metadata within
 *
 *	This routine opens the given volume device and attempts to read
 *	certain types of ogfs metadata from the the volume. On success
 *	the return buffer will contain either an ogfs superblock or ogfs
 *	journal header.
 */
static int ogfs_get_meta_header(logical_volume_t *volume,
				ogfs_meta_header_t *ogfs_mh)
{
	int rc, fd;
	
	LOG_ENTRY();

	fd = EngFncs->open_volume(volume, O_RDONLY, 0);
	if (fd < 0) {
		rc = -fd;
	} else {
		/* Try getting an ogfs superblock */
		rc = ogfs_get_super_block(volume, ogfs_mh, fd);
		if (rc != 0) {
			/* Try getting an ogfs journal header */
			rc = ogfs_get_log_header(volume, ogfs_mh, fd);
			if (rc == 0) {
				/* Mark the journal volume as unmountable */
				volume->flags |= VOLFLAG_NOT_MOUNTABLE;
			}
		}
		EngFncs->close_volume(volume, fd);
	}
	
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_get_filesys_metadata - get ogfs superblock or journal metadata
 *	@volume: pointer to evms volume structure
 *	@metadata: address of pointer to have point at filesys related metadata
 *
 *	This routine attempts to read a ogfs block containing data that is not
 *	cluster related such as an ogfs filesystem superblock or external
 *	journal log header.
 */
static int ogfs_get_filesys_metadata(logical_volume_t *volume, char **metadata)
{
	int rc;
	ogfs_meta_header_t *ogfs_mh;

	LOG_ENTRY();

	/*
	 * Allocate space of OGFS_BASIC_BLOCK size which should be sufficient
	 * to read various meta types (ogfs_sb, ogfs_log_header, etc.)
	 */
	ogfs_mh = EngFncs->engine_alloc(OGFS_BASIC_BLOCK);

	if (ogfs_mh) {
		/*
		 * See if we get an ogfs_meta_header plus additional type
		 * specific data, e.g. superblock or log header.
		 */
		rc = ogfs_get_meta_header(volume, ogfs_mh);

		if (rc == 0) {
			/* return pointer to ogfs metadata */
			*metadata = (char *)ogfs_mh;
		} else {
			/* could not get ogfs metadata */
			EngFncs->engine_free(ogfs_mh);
		}
	} else {
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_read_cluster_info - read a block containing cluster configuration
 *	@volume: pointer to evms volume structure
 *	@ogfs_mh: pointer to a buffer to copy ogfs global cluster info
 *	@fd: the file descriptor of an opened volume device
 *
 *	This routine reads an cluster information block from the given volume
 *	device and checks for the magic number. It converts the ogfs cluster
 *	information read from disk to cpu-order.
 */
static int ogfs_read_cluster_info(logical_volume_t *volume,
				  cluster_global_t *ogfs_cg,
				  int fd)
{
	int rc;
	char *buf;

	LOG_ENTRY();
	
	buf = EngFncs->engine_alloc(CIDEV_BLOCKSIZE);
	if (buf != NULL) {	
		rc = fsim_read_bytes(volume, fd, OGFS_CG_OFFSET, CIDEV_BLOCKSIZE, buf);
		if (rc == 0) {
			cluster_global_in(ogfs_cg, buf);
			if (ogfs_cg->cg_magic != OGFS_CLUSTER_GLOBAL) {
				/* bad magic */
				rc = EINVAL;
			} else if ((((ogfs_cg->cg_version >> 16) & 0xff) !=
					((OGFS_CLUSTER_VERSION >> 16) & 0xff))
					|| (((ogfs_cg->cg_version >> 8) & 0xff) !=
					((OGFS_CLUSTER_VERSION >> 8) & 0xff))) {
				/* version mismatch */
				rc = EINVAL;
			}
		}
		EngFncs->engine_free(buf);
	} else {
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_get_cluster_info - returns ogfs cluster info from given volume
 *	@volume: the volume to read from
 *	@ogfs_cg: the buffer to place the ogfs cluster information in
 *
 *	This routine opens the given volume device and attempts to read
 *	certain types of ogfs cluster config data from the the volume.
 */
static int ogfs_get_cluster_info(logical_volume_t *volume,
				 cluster_global_t *ogfs_cg)
{
	int rc, fd;
	
	LOG_ENTRY();

	fd = EngFncs->open_volume(volume, O_RDONLY, 0);
	if (fd < 0) {
		rc = -fd;
	} else {
		rc = ogfs_read_cluster_info(volume, ogfs_cg, fd);
		EngFncs->close_volume(volume, fd);
	}
	
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_get_cluster_metadata - get ogfs cluster configuration metadata
 *	@volume: pointer to evms volume structure
 *	@metadata: address of pointer to have point at cluster related metadata
 *
 *	This routine attempts to read an ogfs cluster block containing data
 *	that contains cluster configuration information.
 */
static int ogfs_get_cluster_metadata(logical_volume_t *volume, char **metadata)
{
	int rc;
	cluster_global_t *ogfs_cg;

	LOG_ENTRY();

	ogfs_cg = EngFncs->engine_alloc(CIDEV_BLOCKSIZE);

	if (ogfs_cg) {
		rc = ogfs_get_cluster_info(volume, ogfs_cg);

		if (rc == 0) {
			/* return pointer to ogfs metadata */
			*metadata = (char *)ogfs_cg;
			/* mark this volume as unmountable */
			volume->flags |= VOLFLAG_NOT_MOUNTABLE;
		} else {
			/* could not get ogfs metadata */
			EngFncs->engine_free(ogfs_cg);
		}
	} else {
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_probe - determine if this volume contains ogfs metadata
 *	@volume: the volume we are probing
 *
 *	This routine attempts to read a block from the given volume to
 *	determine if it should be claimed by this fsim if it contains the
 *	expected ogfs filesystem or cluster configuration metadata.
 */
static int ogfs_probe(logical_volume_t *volume)
{
	int rc;
	char *metadata;

	LOG_ENTRY();

	rc = ogfs_get_filesys_metadata(volume, &metadata);
	if (rc != 0)
		rc = ogfs_get_cluster_metadata(volume, &metadata);
		
	if (rc == 0) {
		volume->private_data = metadata;
	} else {
		volume->private_data = NULL;
	}
	
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 *	ogfs_can_mkfs - can we install a filesystem on a given volume
 *	@volume: the address of the logical volume structure
 *
 *	This routine is invoked to determine if we can make a filesystem
 *	on the given volume.
 */
static int ogfs_can_mkfs(logical_volume_t *volume)
{
	int rc = 0;

	LOG_ENTRY();

	if (!ogfsutils_support) {
		rc = EINVAL;
	} else if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't format. */
		rc = EBUSY;
	} else if (!(volume->flags & VOLFLAG_CLUSTER_SHARED) ||
			(volume->vol_size * EVMS_VSECTOR_SIZE) < MINOGFS) {
		/* Not a cluster shared volume of the minimum size */
		rc = EPERM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_vol_is_orphan - determine if journal or cidev is associated with a filesystem
 *	@volume: the address of the logical volume structure
 *
 *	This routine scans all ogfs filesystem volumes looking to see if the given
 *	volume is associated with one of them. If it isn't claimed, we return TRUE.
 */
static boolean ogfs_vol_is_orphan(logical_volume_t *volume)
{
	int rc;
	boolean is_orphan = TRUE;
	list_anchor_t ogfs_volumes;

	LOG_ENTRY();

	rc = EngFncs->get_volume_list(my_plugin_record, NULL, 0, &ogfs_volumes);
	if (rc == 0) {
		logical_volume_t *fs_vol;
		list_element_t vol_list_iter;

		LIST_FOR_EACH(ogfs_volumes, vol_list_iter, fs_vol) {
			if (ogfs_get_volume_type(fs_vol) == OGFS_FS_VOL) {
				struct ogfs_sb *ogfs_sb;
				list_anchor_t assoc_vols;
				
				ogfs_sb = (struct ogfs_sb *)fs_vol->private_data;
				assoc_vols = *((list_anchor_t *)(ogfs_sb->sb_reserved));

				if (assoc_vols) {
					logical_volume_t *assoc_vol;
					list_element_t assoc_vol_iter;

					/*
					 * scan each associated volume to determine if
					 * the filesystem volume claims to be using the
					 * volume we are searching for.
					 */
					 LIST_FOR_EACH(assoc_vols, assoc_vol_iter, assoc_vol) {
					 	if (!strcasecmp(assoc_vol->dev_node, volume->dev_node)) {
					 		is_orphan = FALSE;
					 		break;
					 	}
					 }
					
					 if (is_orphan == FALSE) {
					 	/* volume is in-use so break out of loop */
					 	break;
					 }
				}				
			}
		}
		EngFncs->destroy_list(ogfs_volumes);
	}

	LOG_EXIT_BOOL(is_orphan);
	return is_orphan;
}

/*
 *	ogfs_can_unmkfs - can we remove out filesystem or metadata from this volume
 *	@volume: the address of the logical volume structure
 *
 *	This routine is invoked to determine if we can nuke enough metadata
 *	so that we no longer recognize this volume as belonging to ogfs.
 */
static int ogfs_can_unmkfs(logical_volume_t *volume)
{
	int rc;
	
	LOG_ENTRY();

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't unmkfs. */
		rc = EBUSY;
	} else {
		/*
		 * If it's a filesystem type vol, then it's unmkfsable. If it's
		 * either a cidev or journal volume then we must check to see if
		 * any ogfs fs volume claims it. If it turns out to be an orphan
		 * then we allow it to be nuked.
		 */
		switch (ogfs_get_volume_type(volume)) {
		case OGFS_FS_VOL:
			rc = 0;
			break;
		case OGFS_CI_VOL:
		case OGFS_JOURNAL_VOL:
			if (ogfs_vol_is_orphan(volume)) {
				rc = 0;
			} else {
				rc = EPERM;
			}
			break;
		default:
			rc = EINVAL;
			break;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/**
 *	dupstr - similar to strdup() except using engine allocated memory
 *	@original: the original string to make a copy of
 */
static inline char *dupstr(char *original)
{
	char *new = NULL;
	
	if (original) {
		int len = strlen(original);
		
		new = EngFncs->engine_alloc(len + 1);
		if (new) {
			memcpy(new, original, len);
		}
	}
	return new;
}

/**
 *	ogfs_read_dinode - simple read of a dinode
 *	@volume: the logical volume structure for the filesystem
 *	@i_num: the dinode block address
 *	@dinode: address of where the return the dinode contents
 *
 *	This routine handles a simple read of a dinode at the
 *	given block address for the size of one block.
 */
static int ogfs_read_dinode(logical_volume_t *volume, ogfs_inum_t i_num, ogfs_dinode_t *dinode)
{
	int rc, fd;
	struct ogfs_sb *ogfs_sb;
		
	LOG_ENTRY();
	
	ogfs_sb = (struct ogfs_sb *)volume->private_data;

	fd = EngFncs->open_volume(volume, O_RDONLY, 0);
	if (fd < 0) {
		rc = -fd;
	} else {
		ogfs_meta_header_t *ogfs_mh;
		
		ogfs_mh = (ogfs_meta_header_t *)dinode;
		
		rc = ogfs_read_meta_header(volume,
					   ogfs_mh,
					   fd,
					   i_num.no_addr * ogfs_sb->sb_bsize,
					   ogfs_sb->sb_bsize);
		if (rc == 0 && ogfs_mh->mh_type != OGFS_METATYPE_DI) {
			rc = EINVAL;
		}
		EngFncs->close_volume(volume, fd);
	}
	
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_read_jindex - read the jindex data
 *	@volume: the logical volume
 *	@jindex: address to store jindex data
 *	@count: address to store count of jindex entries
 *
 *	This routine reads the journal index entries that should contain
 *	information such as the names of the external journal volumes.
 */
static int ogfs_read_jindex(logical_volume_t *volume, ogfs_jindex_t *jindex, u_int32_t *count)
{
	int rc;
	ogfs_dinode_t *dinode;	
	struct ogfs_sb *ogfs_sb;

	LOG_ENTRY();
	
	ogfs_sb = (struct ogfs_sb *)volume->private_data;
	
	dinode = EngFncs->engine_alloc(ogfs_sb->sb_bsize);
	if (dinode) {
		rc = ogfs_read_dinode(volume, ogfs_sb->sb_jindex_di, dinode);
		if (rc == 0) {
			if (!(dinode->di_flags & OGFS_DIF_JDATA) ||
			    dinode->di_payload_format != OGFS_FORMAT_JI ||
			    dinode->di_size == 0 ||
			    (dinode->di_size % sizeof(ogfs_jindex_t)) != 0) {
				/* doesn't smell like a valid journal index */
				rc = EINVAL;
			} else if (dinode->di_height == 0) {
				int i;
				char *buf;
				
				/*
				 * The jindex data is stuffed in the dinode (inline).
				 * Copy the pertinent journal index entries to the
				 * output buffer while skipping the dinode header.
				 */
				buf = (char *)dinode + sizeof(ogfs_dinode_t);
				*count = dinode->di_size / sizeof(ogfs_jindex_t);
				
				for (i = 0; i < *count; i++) {
					ogfs_jindex_in(&jindex[i], buf);
					buf += sizeof(ogfs_jindex_t);
				}
			} else {
				/*
				 * BUGBUG: We currently can't handle an unstuffed
				 * jindex as this requires more complex code. For
				 * now we must return an error code.
				 */
				 rc = EIO;
			}
		}
		EngFncs->engine_free(dinode);
	} else {
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	set_arg - duplicate a given string and place it in a pointer array
 *	@argv: the array of string pointers
 *	@index: the index in the array to insert the duplicate string
 *	@str: the string to duplicate
 *
 *	This routine uses the SET_STRING() plugin macro to duplicate
 *	a given string an assign it to the proper index in the pointer
 *	array.
 */
static inline int set_arg(char **argv, int index, char *str)
{
	argv[index] = NULL;
	argv[index] = EngFncs->engine_strdup(str);
	return 0;
}

/**
 *	ogfs_can_fsck - can we fsck the given volume
 *	@volume: the address of the logical volume structure
 *
 *	This routine is invoked to determine if the volume can be checked
 *	for filesystem inconsistencies.
 */
static int ogfs_can_fsck(logical_volume_t *volume)
{
	int rc;

	LOG_ENTRY();
	
	if (!ogfsutils_support) {
		rc = EINVAL;
	} else if (EngFncs->is_mounted(volume->dev_node, NULL) &&
		   ogfs_get_volume_type(volume) == OGFS_FS_VOL) {
		/*
		 * If it's mounted and it's got an ogfs filesystem then
		 * that's good enough for running ogfsck on the volume.
		 */
		rc = 0;
	} else {
		rc = EPERM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_get_fs_size - get filesystem size
 *	@volume: pointer to volume structure
 *	@size: the address of where to store the fssize (in sectors)
 *
 *	This routine returns the current filesystem size.
 */
static int ogfs_get_fs_size(logical_volume_t *volume, sector_count_t *size)
{
	int rc;

	LOG_ENTRY();

	rc = ogfs_statfs(volume, size, NULL, NULL, NULL);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_expand - execute the ogfs_expand utility to expand filesystem
 *	@volume: the volume being expanded
 *	@new_size: the new volume size in sectors
 *
 *	This routine is called to expand the ogfs filesystem after a volume
 *	has been expanded. The volume must be online for the expand to
 *	succeed (the ogfs_expand accepts the parm to be the mount point,
 *	not the device).
 */
static int ogfs_expand(logical_volume_t *volume, sector_count_t *new_size)
{
	int rc;

	LOG_ENTRY();
	
	if (!ogfsutils_support) {
		rc = EINVAL;
	} else if (EngFncs->is_mounted(volume->dev_node, NULL) &&
		   ogfs_get_volume_type(volume) == OGFS_FS_VOL) {
		char *argv[] = {"ogfs_expand", NULL, NULL};

		argv[1] = volume->mount_point;
		rc = ogfs_exec_utility(volume, argv);
	} else {
		rc = EPERM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/**
 *	ogfs_enumerate_journal_names - return list of external log volume names for a volume
 *	@volume: the address of the filesystem's logical volume structure
 *	@names: list to add journal volume names to
 *
 *	This routine digs through the journal index info to find the device names
 *	for the external log volumes (if any) and returns a list containing the
 *	device names.
 */
static void ogfs_enumerate_journal_names(logical_volume_t *volume, list_anchor_t names)
{
	u_int32_t count;
	ogfs_jindex_t *ji;
	struct ogfs_sb *ogfs_sb;

	ogfs_sb = (struct ogfs_sb *)volume->private_data;

	ji = EngFncs->engine_alloc(ogfs_sb->sb_bsize);
	
	if (ji) {
		int rc;

		rc = ogfs_read_jindex(volume, ji, &count);
		if (rc == 0) {
			int i;
		
			for (i = 0; i < count; i++) {
				/*
				 * if external journal detected,
				 * add device name to list
				 */
				if (ji[i].ji_addr >= (UINT64_MAX / 2)) {
					EngFncs->insert_thing(names,
							      dupstr(ji[i].ji_device),
							      INSERT_BEFORE, NULL);
				}			
			}
		}
		EngFncs->engine_free(ji);
	}
}

/**
 *	ogfs_unclaim_volumes - unclaim volumes in volume list for the given names
 *	@volumes: list of logical volume structures to examine
 *	@volnames: list of volume names that should be unclaimed
 *
 *	This routine loops through the list of logical volumes trying to find the
 *	logical volume structures matching either the cidev and/or external journal
 *	volume names given (sorry, ogfs doesn't track them by uuid) associated with
 *	a filesystem volume. It returns a list of the volumes it was able to unclaim.
 *
 */
static list_anchor_t ogfs_unclaim_volumes(list_anchor_t volumes, list_anchor_t volnames)
{
	logical_volume_t *volume;
	list_element_t iter1, iter2, iter3;
	list_anchor_t unclaimed_volumes;

	unclaimed_volumes = EngFncs->allocate_list();
	
	if (unclaimed_volumes) {
		LIST_FOR_EACH(volumes, iter1, volume) {
			char *volname;
		
			LIST_FOR_EACH_SAFE(volnames, iter2, iter3, volname) {
				if (strcasecmp((const char *)volname,
						(const char *)volume->dev_node) == 0) {
					/*
					 * Unclaim the volume; add it to the return
					 * unclaimed volumes list and then remove the
					 * search volume name from the search list.
					 */
					EngFncs->unassign_fsim_from_volume(volume);
					EngFncs->insert_thing(unclaimed_volumes, volume,
							INSERT_AFTER, NULL);
					EngFncs->delete_element(iter2);
					break;
				}
			}
		}
	}
	
	return unclaimed_volumes;
}

/**
 *	ogfs_full_unmkfs_setup - handle unassignment from volume
 *	@volume: the address of the logical volume structure
 *
 *	This routine handles the ogfs fsim being removed from a volume.
 *	We find the external journals and the cidev and call the proper
 *	engine service to unclaim them as well.
 *
 *	We then create a list of the volumes associated with the filesystem
 *	volume and place the list anchor in the reserved field of the
 *	superblock for access later from the ogfs_remove_filesystem() routine.
 */
static int ogfs_full_unmkfs_setup(logical_volume_t *volume)
{
	int rc;
	struct ogfs_sb *ogfs_sb;
	list_anchor_t ogfs_vols, claimed_volumes, unclaimed_volumes = NULL;

	LOG_ENTRY();
		
	ogfs_sb = (struct ogfs_sb *)volume->private_data;
		
	claimed_volumes = EngFncs->allocate_list();
		
	/* append to list external journal volume names */
	ogfs_enumerate_journal_names(volume, claimed_volumes);
		
	/* insert cidev volume name */
	EngFncs->insert_thing(claimed_volumes, dupstr(ogfs_sb->sb_locktable),
			      INSERT_BEFORE, NULL);
		
	/* get list of ogfs claimed volumes */
	rc = EngFncs->get_volume_list(my_plugin_record, NULL, 0, &ogfs_vols);		
	if (rc == 0) {
		/*
		 * Unclaim the named volumes and return a list of the logical
		 * volume structures that were actually unclaimed
		 */
		unclaimed_volumes = ogfs_unclaim_volumes(ogfs_vols, claimed_volumes);
		EngFncs->destroy_list(ogfs_vols);
	}
	EngFncs->destroy_list(claimed_volumes);
		
	/*
	 * Anchor the list of volumes structures associated with our
	 * filesystem volume that we just unclaimed. Place the anchor
	 * pointer within the reserved area of the superblock for use
	 * later at actual ogfs_unmkfs() time.
	 */
	*((list_anchor_t *)(ogfs_sb->sb_reserved)) = unclaimed_volumes;

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_unmkfs_setup - handle unassignment from volume
 *	@volume: the address of the logical volume structure
 *
 *	This routine handles the ogfs fsim being removed from a volume.
 *	A filesystem will have it's associated journals and cidev
 *	unclaimed. We do nothing much if it's just a single orhpan
 *	cidev or journal being nuked.
 */
static int ogfs_unmkfs_setup(logical_volume_t *volume)
{
	int rc;
		
	LOG_ENTRY();
	
	switch (ogfs_get_volume_type(volume)) {
	case OGFS_FS_VOL:
		rc = ogfs_full_unmkfs_setup(volume);
		break;
	case OGFS_CI_VOL:
	case OGFS_JOURNAL_VOL:
		if (ogfs_vol_is_orphan(volume)) {
			rc = 0;
		} else {
			rc = EPERM;
		}
		break;
	default:
		rc = EINVAL;
		break;	
	}
	
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_remove_cidev - remove the cluster information global structure
 *	@volume: the address of the logical volume structure
 *
 *	This routine wipes out the cluster information global on-disk structure
 *	so that we don't claim this on a re-discover.
 */
static int ogfs_remove_cidev(logical_volume_t *volume)
{
	int fd, rc;
	
	LOG_ENTRY();
	
	fd = EngFncs->open_volume(volume, O_RDWR | O_EXCL, 0);
	
	if (fd >= 0) {
		char *zeroblock;

		zeroblock = EngFncs->engine_alloc(CIDEV_BLOCKSIZE);
		if (zeroblock) {
			/*
			 * We expect a successful engine_alloc() to always
			 * returns zero initialized memory so we can just
			 * go ahead a send it on down to wipe out the
			 * cluster global data.
			 */
			rc = fsim_write_bytes(volume, fd, OGFS_CG_OFFSET,
					      CIDEV_BLOCKSIZE, zeroblock);
			if (rc == 0) {
				/* free cluster info in private data */
				ogfs_free_private_data(volume);
			}
			EngFncs->engine_free(zeroblock);
		} else {
			rc = ENOMEM;
		}
		EngFncs->close_volume(volume, fd);
	} else {
		rc = -fd;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_remove_external_journal - remove first log header at start of volume
 *	@volume: the address of the logical volume structure
 *
 *	This routine nukes the log header at the start of an external journal
 *	volume.
 */
static int ogfs_remove_external_journal(logical_volume_t *volume)
{
	int fd, rc;
	
	LOG_ENTRY();
	
	fd = EngFncs->open_volume(volume, O_RDWR | O_EXCL, 0);
	
	if (fd >= 0) {	
		char *zeroblock;
		
		zeroblock = EngFncs->engine_alloc(OGFS_BASIC_BLOCK);
		if (zeroblock) {
			/*
			 * We expect a successful engine_alloc() to return
			 * zero initialized memory so we can safely issue
			 * the write to zero out the journal log header.
			 */
			rc = fsim_write_bytes(volume, fd, OGFS_LH_OFFSET,
					      OGFS_BASIC_BLOCK, zeroblock);
			if (rc == 0) {
				/* free log header in private data */
				ogfs_free_private_data(volume);
			}
			EngFncs->engine_free(zeroblock);
		} else {
			rc = ENOMEM;
		}
		EngFncs->close_volume(volume, fd);
	} else {
		rc = -fd;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_remove_associated_volumes - remove journals and cluster information
 *	@volumes: list of volumes associated with filesystem volume to remove
 *
 *	This routine walks the list and invokes either ogfs_remove_cidev() or
 *	ogfs_remove_journal() to remove the cluster information volume and
 *	any external journal volumes respectively.
 */
static void ogfs_remove_associated_volumes(list_anchor_t volumes)
{
	list_element_t iter1;
	logical_volume_t *volume;

	LIST_FOR_EACH(volumes, iter1, volume) {
		ogfs_volume_type_t type;
		void *saved_private_data;
		
		/*
		 * Ensure that the original private_data is being used as it
		 * is possible that a mkfs was done when we unclaimed the
		 * volume in the unmkfs_setup phase.
		 */
		saved_private_data = volume->private_data;
		volume->private_data = volume->original_fsim_private_data;
		
		/* Identify the type of volume to remove */
		type = ogfs_get_volume_type(volume);
		
		if (type == OGFS_CI_VOL)
			ogfs_remove_cidev(volume);
		else if (type == OGFS_JOURNAL_VOL)
			ogfs_remove_external_journal(volume);
		
		/* Restore the saved private_data pointer before continuing */
		volume->private_data = saved_private_data;		
	}
}

/**
 *	ogfs_remove_filesystem - nuke filesystem and associated external journals
 *	@volume: the address of the logical volume structure
 *
 *	This routine removes the superblock from the given filesystem and wipes
 *	out the first log header for each external journal the filesystem was
 *	using as well as the volume that contains the associated cluster
 *	configuration.
 */
static int ogfs_remove_filesystem(logical_volume_t *volume)
{
	int fd, rc;
	
	LOG_ENTRY();
	
	fd = EngFncs->open_volume(volume, O_RDWR | O_EXCL, 0);
	
	if (fd >= 0) {
		char *zeroblock;
		
		zeroblock = EngFncs->engine_alloc(OGFS_BASIC_BLOCK);
		if (zeroblock) {
			/*
			 * We expect a successful engine_alloc() to return
			 * zero initialized memory that we can turn around
			 * and write to disk to zero out the superblock.
			 */
			rc = fsim_write_bytes(volume, fd, OGFS_SB_OFFSET,
					      OGFS_BASIC_BLOCK, zeroblock);
			if (rc == 0) {
				struct ogfs_sb *ogfs_sb;
				list_anchor_t assoc_vols;

				ogfs_sb = (struct ogfs_sb *)volume->private_data;
				assoc_vols = *((list_anchor_t *)(ogfs_sb->sb_reserved));
				
				/* nuke any external journals and cidev */
				if (assoc_vols != NULL) {
					ogfs_remove_associated_volumes(assoc_vols);
				}

				/* free superblock in private data */
				ogfs_free_private_data(volume);
			}
			EngFncs->engine_free(zeroblock);
		} else {
			rc = ENOMEM;
		}
		EngFncs->close_volume(volume, fd);
	} else {
		rc = -fd;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_unmkfs - remove a filesystem or ogfs meta-data from a volume
 *	@volume: the address of the logical volume structure
 *
 *	This routine zeroes out enough meta-data from the ogfs volume to avoid
 *	it being recognized.
 */
static int ogfs_unmkfs(logical_volume_t *volume)
{
	int rc;

	LOG_ENTRY();

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't unmkfs. */
		rc = EBUSY;
	} else {
		ogfs_volume_type_t type;
		
		type = ogfs_get_volume_type(volume);
		
		switch (type) {
		case OGFS_FS_VOL:
			rc = ogfs_remove_filesystem(volume);
			break;
		case OGFS_CI_VOL:
		case OGFS_JOURNAL_VOL:
			if (ogfs_vol_is_orphan(volume)) {
				if (type == OGFS_JOURNAL_VOL) {
					rc = ogfs_remove_external_journal(volume);
				} else {
					rc = ogfs_remove_cidev(volume);
				}
			} else {
				rc = EPERM;
			}
			break;
		default:
			rc = EINVAL;
			break;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_shrink - shrink a volume
 *	@volume: address of logical volume structure
 *	@requested_size: size in sectors to shrink by
 *	@new_size: address to update new volume size
 *
 *	OpenGFS currently does not support shrinking any of its volume types.
 */
static int ogfs_shrink(logical_volume_t *volume,
		       sector_count_t requested_size,
		       sector_count_t *new_size)
{
	int rc = ENOSYS;
	
	LOG_ENTRY();
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_check_fs_blocksize - validate a block size for mkfs supplied by application
 *	@size: address of block size
 *
 *	This routine validates a block size as supplied by an application program for
 *	the filesystem block size option for mkfs. I lifted this code from
 *	lvm_check_pe_size() as it did pretty much what I needed.
 *
 *	It returns 0 if the block size was fine as is or -1 if it was fudged to make
 *	it acceptable.
 */
static int ogfs_check_fs_blocksize(u_int32_t * size)
{
	unsigned long mask = 1;
	int rc = 0;

	LOG_ENTRY();

	if (*size < OGFS_MIN_FS_BLOCKSIZE) {
		LOG_WARNING("Block size %d below lower limit.\n", *size);
		LOG_WARNING("Resetting block size to %d.\n", OGFS_MIN_FS_BLOCKSIZE);
		*size = OGFS_MIN_FS_BLOCKSIZE;
		rc = -1;
	} else if (*size > OGFS_MAX_FS_BLOCKSIZE) {
		LOG_WARNING("Block size %d above upper limit.\n", *size);
		LOG_WARNING("Resetting block size to %d.\n", OGFS_MAX_FS_BLOCKSIZE);
		*size = OGFS_MAX_FS_BLOCKSIZE;
		rc = -1;
	} else if ((*size & (*size - 1)) != 0) {
		LOG_WARNING("Block size %d not a power of 2.\n", *size);
		while ((*size & (*size - 1)) != 0) {
			*size =  *size & ~mask;
			mask = mask << 1;
		}
		LOG_WARNING("Rounding block size down to %d.\n", *size);
		rc = -1;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	init_fake_sb - initialize certain fields in fake superblock from options
 *	@sb: the fake superblock
 *	@options: array of option key/value pairs
 *
 *	This routine scans the option key/value pairs to assign credible values
 *	to fields in the fake superblock that can be viewed when displaying
 *	volume details.
 */
static void init_fake_sb(struct ogfs_sb *sb, option_array_t *options)
{
	int i;

	for (i = 0; i < options->count; i++) {
		/*
		 * If option is name based, set its option number to allow having
		 * code be process by one common code section.
		 */
		if (options->option[i].is_number_based == FALSE) {
			if (!strcmp(options->option[i].name, OGFS_MKFS_LOCKDEV_NAME)) {
				options->option[i].number = OGFS_MKFS_LOCKDEV_INDEX;
			} else if (!strcmp(options->option[i].name, OGFS_MKFS_PROTOCOL_NAME)) {
				options->option[i].number = OGFS_MKFS_PROTOCOL_INDEX;
			} else if (!strcmp(options->option[i].name, OGFS_MKFS_BLOCKSIZE_NAME)) {
				options->option[i].number = OGFS_MKFS_BLOCKSIZE_INDEX;
			} else {
				/* unknown option name. ignore */
				continue;
			}
		}

		switch (options->option[i].number) {
		case OGFS_MKFS_LOCKDEV_INDEX:
			if (options->option[i].value.s) {
				strncpy(sb->sb_locktable,
					options->option[i].value.s,
					OGFS_LOCKNAME_LEN);
			}
			break;
		case OGFS_MKFS_PROTOCOL_INDEX:
			if (options->option[i].value.s) {
				strncpy(sb->sb_lockproto,
					options->option[i].value.s,
					OGFS_LOCKNAME_LEN);
			}
			break;
		case OGFS_MKFS_BLOCKSIZE_INDEX:
			ogfs_check_fs_blocksize(&options->option[i].value.ui32);
			sb->sb_bsize = options->option[i].value.ui32;
			break;
		default:
			break;
		}
	}
}

/**
 *	ogfs_create_fake_sb - allocate an ogfs_sb that can pass minimal checks
 *	@options: array of option key/value pairs
 *
 *	This routine allocates and minimally initializes a filesystem super block
 *	we can use to pass off any minimal checks.
 */
static struct ogfs_sb *ogfs_create_fake_sb(option_array_t *options)
{
	struct ogfs_sb *sb;

	sb = EngFncs->engine_alloc(sizeof(struct ogfs_sb));
	if (sb) {
		sb->sb_header.mh_magic = OGFS_MAGIC;
		sb->sb_header.mh_type = OGFS_METATYPE_SB;
		sb->sb_header.mh_format = OGFS_FORMAT_SB;
		sb->sb_fs_format = OGFS_FORMAT_FS;
		sb->sb_multihost_format = OGFS_FORMAT_MULTI;
		sb->sb_bsize = OGFS_DEFAULT_FS_BLOCKSIZE;
		init_fake_sb(sb, options);
	}
	return sb;
}

/**
 *	ogfs_create_fake_lh - allocate an ogfs_log_header that can pass checks
 *	@void
 *
 *	This routine allocates and minimally initializes a log header we can use
 *	to pass off as the real thing.
 */
static struct ogfs_log_header *ogfs_create_fake_lh(void)
{
	struct ogfs_log_header *lh;

	lh = EngFncs->engine_alloc(sizeof(struct ogfs_log_header));
	if (lh) {
		lh->lh_header.mh_magic = OGFS_MAGIC;
		lh->lh_header.mh_type = OGFS_METATYPE_LH;
		lh->lh_header.mh_format = OGFS_FORMAT_LH;
		lh->lh_flags = OGFS_LOG_HEAD_UNMOUNT;
	}
	return lh;
}

/**
 *	ogfs_claim_external_journal - claim an external log for a new filesystem
 *	@volume: the external journal logical volume structure
 *
 *	This routine claims the given volume as an external journal. We simply
 *	assign our fsim to it a build a dummy log header to place in the volume
 *	private data.
 */
static int ogfs_claim_external_journal(logical_volume_t *volume)
{
	int rc = 0;
	struct ogfs_log_header *lh;

	LOG_ENTRY();

	lh = ogfs_create_fake_lh();

	if (lh) {
		EngFncs->assign_fsim_to_volume(my_plugin_record, volume);
		volume->private_data = lh;
	} else {
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_claim_external_journal_on_match - check if volume in list of journals
 *	@volume: a logical volume structure
 *	@journals: the list of names of volumes to be used as external journals
 *
 *	This routine scans the given list of journal volume names and compares the
 *	name of the given volume for a match. If a match is made, it claims that
 *	volume and returns.
 */
static int ogfs_claim_external_journal_on_match(logical_volume_t *volume,
						struct value_list_s *journals)
{
	int rc = 0, i;

	for (i = 0; i < journals->count; i++) {
		if (strcasecmp((const char *)journals->value[i].s,
				(const char *)volume->dev_node) == 0) {
			rc = ogfs_claim_external_journal(volume);
			break;
		}
	}

	return rc;
}

/**
 *	ogfs_claim_external_journals - claim external journals from mkfs
 *	@journals: list containing log volumes from mkfs option
 *
 *	This routine checks the given value list containing the names of
 *	the volumes selected for external journals from the mkfs options,
 *	searches the engine's volume list and claims those volumes in the
 *	name of ogfs.
 */
static int ogfs_claim_external_journals(struct value_list_s *journals)
{
	int rc = 0;

	LOG_ENTRY();

	if (journals->count > 0) {
		list_anchor_t volumes;

		rc = EngFncs->get_volume_list(NULL, NULL, 0, &volumes);
		if (rc == 0) {
			logical_volume_t *volume;
			list_element_t vol_list_iter;

			LIST_FOR_EACH(volumes, vol_list_iter, volume) {
				rc = ogfs_claim_external_journal_on_match(volume, journals);
				if (rc) break;
			}
			EngFncs->destroy_list(volumes);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_mkfs_setup - attach superblock to volume and claim journal volumes
 *	@volume: logical volume structure for filesystem volume
 *	@options: array of mkfs option values
 *
 *	This routine handles some initial setup for a mkfs. We create a fake sb
 *	that we can use to identify the volume as a ogfs filesystem volume and
 *	we claim the external journal volumes as well.
 */
static int ogfs_mkfs_setup(logical_volume_t *volume, option_array_t *options)
{
	int rc, i;
	struct value_list_s *journals = NULL;

	LOG_ENTRY();

	for (i = 0; i < options->count; i++) {
		if (!options->option[i].is_number_based) {
			if (!strcmp(options->option[i].name, OGFS_MKFS_JOURNALS_NAME)) {
				options->option[i].number = OGFS_MKFS_JOURNALS_INDEX;
			}
		}

		if (options->option[i].number == OGFS_MKFS_JOURNALS_INDEX &&
			options->option[i].value.list) {
			journals = options->option[i].value.list;
			break;
		}
	}

	if (journals) {
		struct ogfs_sb *sb;

		sb = ogfs_create_fake_sb(options);
		if (sb) {
			rc = ogfs_claim_external_journals(journals);
			if (rc == 0) {
				volume->private_data = sb;
			} else {
				EngFncs->engine_free(sb);
			}
		} else {
			rc = ENOMEM;
		}
	} else {
		/* we expect external journals since we do non-pool support only */
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	write_journal_cf_entries - write entries to journals config input file used by mkfs
 *	@fd: file handle to temp config file
 *	@journals: the value list containing external journal volume names
 *
 *	This routine writes the entries that describe the external journal volumes
 *	to be used when a mkfs.ogfs command is executed.
 */
static int write_journal_cf_entries(int fd, value_list_t *journals)
{
	int rc = 0;
	char *buffer;

	buffer = EngFncs->engine_alloc(OGFS_JOURNAL_CF_BUFSIZE);
	if (buffer) {
		int i;
		
		snprintf(buffer, OGFS_JOURNAL_CF_BUFSIZE, "journals %d\n", journals->count);
		write(fd, buffer, strlen(buffer));
		
		for (i = 0; i < journals->count; i++) {
			snprintf(buffer, OGFS_JOURNAL_CF_BUFSIZE, "journal %d ext %s\n",
					i, journals->value[i].s);
			write(fd, buffer, strlen(buffer));
		}
		EngFncs->engine_free(buffer);
	} else {
		rc = ENOMEM;
	}
	return rc;
}

/**
 *	create_journal_config_file - create file containing external journal for mkfs
 *	@journals: pointer to value list containing strings with journal volume names
 *	@tmpfile: address to store config filename
 *
 *	This routine creates a unique temporary file containing the
 *	list of external journals to be used by the mkfs.ogfs command.
 */
static int create_journal_config_file(value_list_t *journals, char **tmpfile)
{
	int rc = 0, fd;
	char filename[] = "/tmp/evmsXXXXXX";
	
	fd = mkstemp(filename);
	if (fd >= 0) {
		rc = write_journal_cf_entries(fd, journals);
		close(fd);
		
		if (rc == 0) {
			*tmpfile = dupstr(filename);
		} else {
			unlink(filename);
		}
	} else {
		rc = EEXIST;
	}
	return rc;
}

/**
 *	free_argv_strings - free argument arrays strings
 *	@argv: NULL terminated array of string pointers
 *
 *	This routine simply frees the memory allocated for all the argv
 *	strings.
 */
static inline void free_argv_strings(char **argv)
{
	int i;

	for (i = 0; argv[i] != NULL; i++) {
		EngFncs->engine_free(argv[i]);
	}
}

/**
 *	set_mkfs_args - convert option values to mkfs command line arguments
 *	@options: the mkfs key/value pairs
 *	@argv: address of array of arguments strings
 *	@argc: address of argument count so far
 *	@tmpfile: address to write string pointer with name of temp journals config file
 *
 *	This routine converts option values for the mkfs task into command
 *	line arguments and adds then to the mkfs argument pointer array.
 */
static int set_mkfs_args(option_array_t *options, char **argv, int *argc, char **tmpfile)
{
	int i, rc = 0, index;

	for (i = 0, index = *argc; i < options->count && rc == 0; i++) {
		/*
		 * If option is name based, set its option number to allow having
		 * code be process by one common code section.
		 */
		if (options->option[i].is_number_based == FALSE) {
			if (!strcmp(options->option[i].name, OGFS_MKFS_LOCKDEV_NAME)) {
				options->option[i].number = OGFS_MKFS_LOCKDEV_INDEX;
			} else if (!strcmp(options->option[i].name, OGFS_MKFS_PROTOCOL_NAME)) {
				options->option[i].number = OGFS_MKFS_PROTOCOL_INDEX;
			} else if (!strcmp(options->option[i].name, OGFS_MKFS_BLOCKSIZE_NAME)) {
				options->option[i].number = OGFS_MKFS_BLOCKSIZE_INDEX;
			} else if (!strcmp(options->option[i].name, OGFS_MKFS_JOURNALS_NAME)) {
				options->option[i].number = OGFS_MKFS_JOURNALS_INDEX;
			} else {
				/* unknown option name. ignore */
				continue;
			}
		}

		switch (options->option[i].number) {
		case OGFS_MKFS_LOCKDEV_INDEX:
			if (options->option[i].value.s) {
				rc = set_arg(argv, index++, "-t");
				if (rc == 0) {
					rc = set_arg(argv, index++,
						     options->option[i].value.s);
					if (rc == 0) {
						/*
						 * Being that we don't run a cluster
						 * config tool such as ogfsconf on the
						 * cidev or claim it, remind the user
						 * that (s)he needs to do that themselves.
						 */
						 MESSAGE(_("Please configure lock table device "
							   "%s with ogfsconf before mounting "
							   "the new file system"),
						 	 options->option[i].value.s);
					}
				}
			}
			break;
		case OGFS_MKFS_PROTOCOL_INDEX:
			if (options->option[i].value.s) {
				rc = set_arg(argv, index++, "-p");
				if (rc == 0) {
					rc = set_arg(argv, index++,
						     options->option[i].value.s);
				}
			}
			break;
		case OGFS_MKFS_BLOCKSIZE_INDEX:
			/* ensure size is a power of 2 from 512 through 65536*/
			ogfs_check_fs_blocksize(&options->option[i].value.ui32);
			{
				char tmp[8];
				sprintf(tmp, "%u", options->option[i].value.ui32);
				rc = set_arg(argv, index++, "-b");
				if (rc == 0) {
					rc = set_arg(argv, index++, tmp);
				}
			}
			break;
		case OGFS_MKFS_JOURNALS_INDEX:
			rc = create_journal_config_file(options->option[i].value.list, tmpfile);
			if (rc == 0) {
				rc = set_arg(argv, index++, "-c");
				if (rc == 0) {
					rc = set_arg(argv, index++, *tmpfile);
				}
			}
			break;
		default:
			break;
		}
	}
	
	*argc = index;
	
	return rc;
}

/**
 *	build_mkfs_exec_args - build the argv for exec of mkfs.ogfs
 *	@options: array of option key/value pairs
 *	@volume: the logical volume to run mkfs on
 *	@argv: the argument array to fill
 *	@tmpname: the temporary file name containing journal configuration info
 *
 *	This routine builds the array of string pointers used as input to the
 *	fork_and_execvp() engine service in order to execute the mkfs.ogfs
 *	command. We also return the name of the tmpfile listing the external
 *	journals to be deleted upon completion of the execution of the command.
 */
static int build_mkfs_exec_args(logical_volume_t *volume, option_array_t *options,
				char **argv, char **tmpfile)
{
	int rc, argc = 0;

	rc = set_arg(argv, argc++, "mkfs.ogfs");
	if (rc == 0) {
		rc = set_mkfs_args(options, argv, &argc, tmpfile);
		if (rc == 0) {
			rc = set_arg(argv, argc++, volume->dev_node);
			if (rc == 0) {
				argv[argc] = NULL;
			}
		}
	}
	return rc;
}

/**
 *	ogfs_create - create an ogfs filesystem
 *	@volume: the logical volume to create the filesystem on
 *	@options: the mkfs option values
 *
 *	This routine handles the creation of an ogfs filesystem on the given
 *	volume with the given options.
 */
static int ogfs_create(logical_volume_t *volume, option_array_t *options)
{
	int rc;
	char *tmpfile, *argv[OGFS_MKFS_MAX_ARG_COUNT+1];

	LOG_ENTRY();
	
	rc = build_mkfs_exec_args(volume, options, argv, &tmpfile);
	if (rc == 0) {
		rc = ogfs_exec_utility(volume, argv);
		if (tmpfile) {
			unlink(tmpfile);
			EngFncs->engine_free(tmpfile);
		}
	}
	free_argv_strings(argv);
	
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_mkfs - create an ogfs filesystem
 *	@volume: the logical volume to create the filesystem on
 *	@options: the mkfs option values
 *
 *	This routine handles the creation of an ogfs filesystem on the given
 *	volume with the given options.
 */
static int ogfs_mkfs(logical_volume_t *volume, option_array_t *options)
{
	int rc;

	LOG_ENTRY();

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* Don't format if mounted */
		rc = EBUSY;
	} else {
		rc = ogfs_create(volume, options);
		if (rc == 0) {
			/* Re-probe to update private data */
			rc = ogfs_probe(volume);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_fsck - check a mounted ogfs filesystem
 *	@volume: the volume to fsck
 *	@options: option array (should be empty as we have none)
 *
 *	This routine simply executes the ogfsck utility on the given
 *	volume as long as it's got an ogfs filesystem on it and the
 *	volume is mounted.
 */
static int ogfs_fsck(logical_volume_t *volume, option_array_t *options)
{
	int rc;

	LOG_ENTRY();
	
	rc = ogfs_can_fsck(volume);
	if (rc == 0) {
		char *argv[] = {"ogfsck", NULL, NULL};

		argv[1] = volume->dev_node;
		rc = ogfs_exec_utility(volume, argv);		
	}
	
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_get_option_count - return max option count for a given task action
 *	@context: pointer to task context
 *
 *	This routine simply returns the maximum count of possible options
 *	for a given task action such as mkfs, etc.
 */
static int ogfs_get_option_count(task_context_t *context)
{
	int count = 0;

	LOG_ENTRY();

	switch (context->action) {
	case EVMS_Task_mkfs:
		count = OGFS_MKFS_OPTIONS_COUNT;
		break;
	case EVMS_Task_fsck:
		break;
	case EVMS_Task_Expand:
		break;
	default:
		break;
	}

	LOG_EXIT_INT(count);
	return count;
}

/**
 *	get_constraint_volumes - return a list of potential volumes for constraint lists
 *	@context: pointer to task context
 *	@minsize: volume minimum size in bytes
 *
 *	This routine returns a list of volumes that could be used by either the
 *	option constraint lists for journal or lock table volumes. Candidate volumes
 *	must not have an FSIM already claiming them; not the volume we intend to mkfs;
 *	and must be a shared volume.
 */
static list_anchor_t get_constraint_volumes(task_context_t *context, u_int32_t minsize)
{
	list_anchor_t volumes = NULL;

	LOG_ENTRY();

	if (context->volume != NULL) {
		int rc;

		if (context->volume->disk_group != NULL) {
			rc = EngFncs->get_volume_list(NULL, context->volume->disk_group,
						      0, &volumes);
		} else {
			rc = EngFncs->get_volume_list(NULL, NULL, VOL_NO_DISK_GROUP,
						      &volumes);
		}

		if (!rc) {
			logical_volume_t *volume;
			list_element_t iter1, iter2;

			LIST_FOR_EACH_SAFE(volumes, iter1, iter2, volume) {
				if (volume->file_system_manager ||
				    !(volume->flags & VOLFLAG_CLUSTER_SHARED) ||
				    !strcasecmp(context->volume->dev_node, volume->dev_node) ||
				    volume->vol_size * EVMS_VSECTOR_SIZE < minsize ||
				    EngFncs->is_mounted(volume->dev_node, NULL)) {
				    	/* Criteria not met. Remove this volume from list. */
				    	EngFncs->delete_element(iter1);
				}
			}
		}
	} else {
		LOG_ERROR("There is no volume in the task context!!!\n");
	}
	
	return volumes;
}

/**
 *	allocate_protocol_constraint_list - allocate the protocol option constraint list
 *	@void
 *
 *	This routine simply allocates a value constraint list with the locking protocols
 *	that could be used by mkfs task.
 */
static value_list_t *allocate_protocol_constraint_list(void)
{
	value_list_t *constraints;
	
	constraints = EngFncs->engine_alloc(sizeof(value_list_t) +
				(sizeof(value_t) * OGFS_LOCKING_PROTOCOL_COUNT));
	
	if (constraints) {
		/*
		 * NOTE: "nolock" protocol is not offered on purpose since we
		 * don't want to confuse users into thinking they don't need
		 * locking when they set up a shared volume with multiple nodes.
		 */
		constraints->value[constraints->count].s = dupstr("memexp");
		if (constraints->value[constraints->count].s) {
			constraints->count++;
			constraints->value[constraints->count].s = dupstr("opendlm");
			if (constraints->value[constraints->count].s) {
				constraints->count++;
			}
		}
	}
	
	return constraints;
}

/**
 *	allocate_journals_value_list - allocate value list to hold journal selections
 *	@context: pointer to task context record
 *
 *	This routine allocates the initial value_list that will contain the selections
 *	the user made from the journals constraint list. We size it to match the size
 *	of the constraint list.
 */
static value_list_t *allocate_journals_value_list(task_context_t *context)
{
	value_list_t *list = NULL, *constraints;
	
	constraints = context->option_descriptors->option[OGFS_MKFS_JOURNALS_INDEX].constraint.list;
	if (constraints) {
		list = EngFncs->engine_alloc(sizeof(value_list_t) +
					(sizeof(value_t) * constraints->count));
	}
	
	return list;
}

/**
 *	update_constraint_list - update given constraint list from given volume list
 *	@volumes: input list of volumes to transfer names to to constraint list
 *	@list: address of pointer to constraint list to update
 *
 *	This routine adds the volume names in the supplies volume list to the
 *	constraint value list to update.
 */
static void update_constraint_list(list_anchor_t volumes, value_list_t **list)
{
	int i = 0, size;

	/* clear out and free any previous constraint list */
	if (*list) {
		for (i = 0; i < (*list)->count; i++) {
			if ((*list)->value[i].s) {
				EngFncs->engine_free((*list)->value[i].s);
			}
		}
		EngFncs->engine_free(*list);
	}

	/* allocate new list and fill it with volume names from candidate volume list */
	size = sizeof(value_list_t) + (EngFncs->list_count(volumes) * sizeof(value_t));
	*list = EngFncs->engine_alloc(size);
		
	if (*list) {
		list_element_t iter1;
		logical_volume_t *volume;

		i = 0;
			
		LIST_FOR_EACH(volumes, iter1, volume) {
			(*list)->value[i].s = dupstr(volume->dev_node);
			if ((*list)->value[i].s) {
				i++;
			}
		}
		(*list)->count = i;
	}		
}

/**
 *	update_lockdev_constraint_list - ensure lock table constraints sane
 *	context: pointer to task context record
 *
 *	This routine builds the list of candidate volumes for the lock table
 *	device and ensures that we don't place volumes in the list that are
 *	volumes selected already as external journals.
 */
static void update_lockdev_constraint_list(task_context_t *context)
{
	list_anchor_t volumes;
	option_desc_array_t *od = context->option_descriptors;
	
	volumes = get_constraint_volumes(context, MINOGFS_TABLE);
	if (volumes) {
		/* filter volumes selected as external journals from list as necessary */
		if (!(od->option[OGFS_MKFS_JOURNALS_INDEX].flags & EVMS_OPTION_FLAGS_NO_INITIAL_VALUE)) {
			int i = 0;
			logical_volume_t *volume;
			list_element_t iter1, iter2;
			value_list_t *journals = od->option[OGFS_MKFS_JOURNALS_INDEX].value.list;

			LIST_FOR_EACH_SAFE(volumes, iter1, iter2, volume) {
				for (i = 0; i < journals->count; i++) {
					if (!strcasecmp(journals->value[i].s, volume->dev_node)) {
					    	EngFncs->delete_element(iter1);
					}
				}
			}
		}		
		update_constraint_list(volumes, &(od->option[OGFS_MKFS_LOCKDEV_INDEX].constraint.list));
		EngFncs->destroy_list(volumes);
	}
}

/**
 *	update_journals_constraint_list - ensure journal volumes constraint sane
 *	context: pointer to task context record
 *
 *	This routine builds the list of candidate volumes for the external
 *	journals and ensures that we don't place a volume in the constraint
 *	list that has already been selected as the lock table volume.
 */
static void update_journals_constraint_list(task_context_t *context)
{
	list_anchor_t volumes;
	option_desc_array_t *od = context->option_descriptors;
	
	volumes = get_constraint_volumes(context, MINOGFS);
	if (volumes) {
		/* filter the lock device from the list if necessary */		
		if (!(od->option[OGFS_MKFS_LOCKDEV_INDEX].flags & EVMS_OPTION_FLAGS_INACTIVE)) {
			if (!(od->option[OGFS_MKFS_LOCKDEV_INDEX].flags & EVMS_OPTION_FLAGS_NO_INITIAL_VALUE)) {
				logical_volume_t *volume;
				list_element_t iter1, iter2;

				LIST_FOR_EACH_SAFE(volumes, iter1, iter2, volume) {
					if (!strcasecmp(od->option[OGFS_MKFS_LOCKDEV_INDEX].value.s,
							volume->dev_node)) {
					    	EngFncs->delete_element(iter1);
					}
				}
			}
		}
		update_constraint_list(volumes, &(od->option[OGFS_MKFS_JOURNALS_INDEX].constraint.list));
		EngFncs->destroy_list(volumes);
	}	
}

/*
 *	ogfs_init_mkfs_acceptable_objects - initialize mkfs acceptable objects
 *	@context: pointer to task context
 *
 *	This routine initializes the mkfs task acceptable objects by enumerating
 *	volumes, finding those that have no FSIM claiming them and are of the
 *	proper size and adding them to the acceptable objects list.
 */
static int ogfs_init_mkfs_acceptable_objects(task_context_t *context)
{
	int rc;
	list_anchor_t global_volumes;
	list_element_t vol_list_iter;
	logical_volume_t * volume;

	LOG_ENTRY();

	rc = EngFncs->get_volume_list(NULL, NULL, 0, &global_volumes);

	if (!rc) {
		LIST_FOR_EACH(global_volumes, vol_list_iter, volume) {
			/* Only mkfs unformatted, unmounted, shared, 'large enough' volumes */
			if (!volume->file_system_manager &&
			    volume->flags & VOLFLAG_CLUSTER_SHARED &&
			    volume->vol_size * EVMS_VSECTOR_SIZE >= MINOGFS &&
			    !EngFncs->is_mounted(volume->dev_node, NULL)) {
				EngFncs->insert_thing(context->acceptable_objects,
						      volume,
						      INSERT_BEFORE,
						      NULL);
			}
		}
		EngFncs->destroy_list(global_volumes);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_init_mkfs_option_descriptors - initialize mkfs option descriptors
 *	@context: pointer to task context
 *
 *	This routine initializes the option descriptors for a mkfs task. The
 *	constraints are subject to change as option selections are made.
 */
static int ogfs_init_mkfs_option_descriptors(task_context_t *context)
{
	int rc = 0;
	option_desc_array_t *od = context->option_descriptors;
	
	LOG_ENTRY();
	
	/* set blocksize */
	od->option[OGFS_MKFS_BLOCKSIZE_INDEX].name = EngFncs->engine_strdup(OGFS_MKFS_BLOCKSIZE_NAME);
	od->option[OGFS_MKFS_BLOCKSIZE_INDEX].title = EngFncs->engine_strdup(_("Block size"));
	od->option[OGFS_MKFS_BLOCKSIZE_INDEX].tip =
		EngFncs->engine_strdup(_("Acceptable range: 512 to 65536 bytes. Must be a power of 2."));
	od->option[OGFS_MKFS_BLOCKSIZE_INDEX].type = EVMS_Type_Unsigned_Int32;
	od->option[OGFS_MKFS_BLOCKSIZE_INDEX].unit = EVMS_Unit_Bytes;
	od->option[OGFS_MKFS_BLOCKSIZE_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED | EVMS_OPTION_FLAGS_AUTOMATIC;
	od->option[OGFS_MKFS_BLOCKSIZE_INDEX].constraint_type = EVMS_Collection_List;
	SET_POWER2_LIST(od->option[OGFS_MKFS_BLOCKSIZE_INDEX].constraint.list,
			OGFS_MIN_FS_BLOCKSIZE, OGFS_MAX_FS_BLOCKSIZE);
	od->option[OGFS_MKFS_BLOCKSIZE_INDEX].value.ui32 = OGFS_DEFAULT_FS_BLOCKSIZE;
			
	/* set locking protocol*/
	od->option[OGFS_MKFS_PROTOCOL_INDEX].name = EngFncs->engine_strdup(OGFS_MKFS_PROTOCOL_NAME);
	od->option[OGFS_MKFS_PROTOCOL_INDEX].title = EngFncs->engine_strdup(_("Locking Protocol"));
	od->option[OGFS_MKFS_PROTOCOL_INDEX].tip = EngFncs->engine_strdup(_("Name of the locking protocol"));
	od->option[OGFS_MKFS_PROTOCOL_INDEX].type = EVMS_Type_String;
	od->option[OGFS_MKFS_PROTOCOL_INDEX].min_len = 1;
	od->option[OGFS_MKFS_PROTOCOL_INDEX].max_len = EVMS_VOLUME_NAME_SIZE;
	od->option[OGFS_MKFS_PROTOCOL_INDEX].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
	od->option[OGFS_MKFS_PROTOCOL_INDEX].constraint_type = EVMS_Collection_List;
	od->option[OGFS_MKFS_PROTOCOL_INDEX].constraint.list = allocate_protocol_constraint_list();
	od->option[OGFS_MKFS_PROTOCOL_INDEX].value.s = EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE + 1);
	if (od->option[OGFS_MKFS_PROTOCOL_INDEX].value.s == NULL) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;		
	}
	
	/* set lock table device */
	od->option[OGFS_MKFS_LOCKDEV_INDEX].name = EngFncs->engine_strdup(OGFS_MKFS_LOCKDEV_NAME);
	od->option[OGFS_MKFS_LOCKDEV_INDEX].title = EngFncs->engine_strdup(_("Lock Table Volume"));
	od->option[OGFS_MKFS_LOCKDEV_INDEX].tip = EngFncs->engine_strdup(_("Shared volume containing locking metadata"));
	od->option[OGFS_MKFS_LOCKDEV_INDEX].type = EVMS_Type_String;
	od->option[OGFS_MKFS_LOCKDEV_INDEX].min_len = 1;
	od->option[OGFS_MKFS_LOCKDEV_INDEX].max_len = EVMS_VOLUME_NAME_SIZE;
	od->option[OGFS_MKFS_LOCKDEV_INDEX].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE | EVMS_OPTION_FLAGS_INACTIVE;
	od->option[OGFS_MKFS_LOCKDEV_INDEX].constraint_type = EVMS_Collection_List;
	od->option[OGFS_MKFS_LOCKDEV_INDEX].value.s = EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE + 1);
	if (od->option[OGFS_MKFS_LOCKDEV_INDEX].value.s == NULL) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;		
	}
	/*
	 * we defer population of constraint list until locking protocol option
	 * is set which also activates this option
	 */
	
	/* set external journals multi-selection list */
	od->option[OGFS_MKFS_JOURNALS_INDEX].name = EngFncs->engine_strdup(OGFS_MKFS_JOURNALS_NAME);
	od->option[OGFS_MKFS_JOURNALS_INDEX].title = EngFncs->engine_strdup(_("External Journals"));
	od->option[OGFS_MKFS_JOURNALS_INDEX].tip = EngFncs->engine_strdup(_("Journal volumes (one for each node)."));
	od->option[OGFS_MKFS_JOURNALS_INDEX].type = EVMS_Type_String;
	od->option[OGFS_MKFS_JOURNALS_INDEX].min_len = 1;
	od->option[OGFS_MKFS_JOURNALS_INDEX].max_len = EVMS_VOLUME_NAME_SIZE;
	od->option[OGFS_MKFS_JOURNALS_INDEX].flags = EVMS_OPTION_FLAGS_VALUE_IS_LIST | EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
	od->option[OGFS_MKFS_JOURNALS_INDEX].constraint_type = EVMS_Collection_List;
	update_journals_constraint_list(context);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_init_mkfs_task - initialize acceptable objects and options for mkfs
 *	@context: pointer to task context
 *
 *	This routine initializes the option descriptors and acceptable objects
 *	list for a mkfs task action.
 */
static int ogfs_init_mkfs_task(task_context_t *context)
{
	int rc;
	
	LOG_ENTRY();

	context->min_selected_objects = 1;
	context->max_selected_objects = 1;
	context->option_descriptors->count = OGFS_MKFS_OPTIONS_COUNT;
	
	rc = ogfs_init_mkfs_acceptable_objects(context);
	if (rc == 0) {
		rc = ogfs_init_mkfs_option_descriptors(context);
	}
	
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_init_task- initial a given task context
 *	@context: pointer to task context
 *
 *	This routine initializes a new task action with the list of
 *	acceptable objects, the miniumum and maximum number of objects
 *	to select, and the initial values for option descriptors.
 */
static int ogfs_init_task(task_context_t *context)
{
	int rc;

	LOG_ENTRY();
	
	switch (context->action) {
	case EVMS_Task_mkfs:
		rc = ogfs_init_mkfs_task(context);
		break;
	case EVMS_Task_fsck:
		/* Surprisingly, the ogfsck utility has no real command line options */
		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		context->option_descriptors->count = 0;

		rc = 0;
		break;	

	case EVMS_Task_Expand:
		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		context->option_descriptors->count = 0;

		rc = 0;
		break;

	default:
		rc = EINVAL;
		break;
	}
	
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_set_mkfs_option - validate/set an option value for mkfs task
 *	@context: pointer to task context
 *	@index: option descriptor array index
 *	@value: address of given/modified value
 *	@effect: address of flags of side-effects
 *
 *	This routine handles the validation of updated option values
 *	for a mkfs task.	
 */
static int ogfs_set_mkfs_option(task_context_t *context,
				u_int32_t       index,
				value_t        *value,
				task_effect_t  *effect)
{
	int rc = 0, i = 0;
	option_desc_array_t *od = context->option_descriptors;
	
	LOG_ENTRY();
	
	switch (index) {
	case OGFS_MKFS_BLOCKSIZE_INDEX:
		if (ogfs_check_fs_blocksize(&(value->ui32))) {
			*effect |= EVMS_Effect_Inexact;
		}
		od->option[index].value.i32 = value->i32;
		break;
	case OGFS_MKFS_JOURNALS_INDEX:
		for (i = 0; i < value->list->count; i++) {
			if (od->option[index].value.list->value[i].s) {
				EngFncs->engine_free(od->option[index].value.list->value[i].s);
				od->option[index].value.list->value[i].s = NULL;
			}
			od->option[index].value.list->value[i].s = EngFncs->engine_strdup(value->list->value[i].s);
		}
		for (; i < od->option[index].value.list->count; i++) {
			if (od->option[index].value.list->value[i].s) {
				EngFncs->engine_free(od->option[index].value.list->value[i].s);
				od->option[index].value.list->value[i].s = NULL;
			}
		}
		od->option[index].value.list->count = value->list->count;
		od->option[index].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
		
		if (!(od->option[OGFS_MKFS_LOCKDEV_INDEX].flags & EVMS_OPTION_FLAGS_INACTIVE)) {
			update_lockdev_constraint_list(context);
			*effect |= EVMS_Effect_Reload_Options;
		}
		break;
	case OGFS_MKFS_PROTOCOL_INDEX:
		strncpy(od->option[index].value.s, value->s, EVMS_VOLUME_NAME_SIZE);
		if (od->option[index].flags & EVMS_OPTION_FLAGS_NO_INITIAL_VALUE) {
			/*
			 * Now that we have a value, ensure that lock device option
			 * becomes active, populated with acceptable selections, and
			 * ask application to reload options to see it.
			 */
			od->option[OGFS_MKFS_LOCKDEV_INDEX].flags &= ~EVMS_OPTION_FLAGS_INACTIVE;
			update_lockdev_constraint_list(context);
			*effect |= EVMS_Effect_Reload_Options;
		}
		break;
	case OGFS_MKFS_LOCKDEV_INDEX:
		strncpy(od->option[index].value.s, value->s, EVMS_VOLUME_NAME_SIZE);
		od->option[index].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
		update_journals_constraint_list(context);
		*effect |= EVMS_Effect_Reload_Options;
		break;
	default:
		rc = EINVAL;
	}
	
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_set_option - validate and set value for option descriptor
 *	@context: pointer to task context
 *	@index: option descriptor array index
 *	@value: address of given/modified value
 *	@effect: address of flags of side-effects
 *
 *	This routine dispatches the handling of updates to option
 *	descriptor values to their respective task handlers.
 */
static int ogfs_set_option(task_context_t *context,
			   u_int32_t       index,
			   value_t        *value,
			   task_effect_t  *effect)
{
	int rc;

	LOG_ENTRY();


	switch (context->action) {
	case EVMS_Task_mkfs:
		rc = ogfs_set_mkfs_option(context, index, value, effect);
		break;
	case EVMS_Task_fsck:
		/* valid task but no options possible. just ignore. */
		rc = 0;
		break;
	default:
		rc = EINVAL;
		break;
	}
	
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_set_volumes - validate a selected volume for a task
 *	@context: pointer to task context structure
 *	@declined_volumes: we add the volume to this list if we don't like it
 *	@effect: address of side effect flag variable
 *
 *	This routine validates the selected volume(s) for a task. It
 *	may place the volume on the declined list if it doesn't like it.
 *	It may also set the side effect flags if something, such as
 *	reloading option descriptors, is necessary by the application.
 */
static int ogfs_set_volumes(task_context_t *context,
			    list_anchor_t declined_volumes,
			    task_effect_t *effect)
{
	int rc = 0;

	LOG_ENTRY();

	if (context->action == EVMS_Task_mkfs) {
		logical_volume_t *vol;
		option_desc_array_t *od = context->option_descriptors;
		
		/* get the selected volume */
		vol = EngFncs->first_thing(context->selected_objects, NULL);

		if (vol) {
			if (EngFncs->is_mounted(vol->dev_node, NULL)) {
				/* If mounted, can't mkfs */
				rc = EBUSY;
			} else {
				if ((vol->vol_size * EVMS_VSECTOR_SIZE) < MINOGFS) {
					rc = ENOSPC;
				} else {
					/* set the selected volume as our context volume */
					context->volume = vol;

					/* re-initialize the journals and lockdev options */
					od->option[OGFS_MKFS_LOCKDEV_INDEX].flags |= EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
					od->option[OGFS_MKFS_JOURNALS_INDEX].flags |= EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
					update_journals_constraint_list(context);
					update_lockdev_constraint_list(context);

					/* allocate the value_list to hold journal selections if necessary */
					if (od->option[OGFS_MKFS_JOURNALS_INDEX].value.list == NULL) {
						od->option[OGFS_MKFS_JOURNALS_INDEX].value.list =
								allocate_journals_value_list(context);
					}
					*effect |= EVMS_Effect_Reload_Options;
				}
			}
			
			/*
			 * If an error occurred, place volume on declined list so it
			 * gets removed from selected_objects by the engine.
			 */
			if (rc) {
				declined_object_t *declined_vol;
				
				declined_vol = EngFncs->engine_alloc(sizeof(declined_object_t));
				if (declined_vol) {
					declined_vol->object = vol;
					declined_vol->reason = rc;
					EngFncs->insert_thing(declined_volumes,
							declined_vol, INSERT_BEFORE, NULL);
				}
			}
		} else {
			rc = ENODATA;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_get_cluster_extended_info - return info from the cluster config header
 *	@volume: the volume to retrieve opengfs info from
 *	@info_name: name of field to retrieve additional info. NULL for all.
 *	@info: address to return pointer to fsim extended info
 *
 *	This routine is called to return additional information related to
 *	opengfs information about the existing volume. We allocate, initialize,
 *	and return an extended_info_array_t with selected information.
 */
static int ogfs_get_cluster_extended_info(logical_volume_t *volume,
					  char *info_name,
					  extended_info_array_t **info)
{
	int rc = EINVAL;
	cluster_global_t      *ogfs_cg;
	extended_info_array_t *einfo;
	
	LOG_ENTRY();

	ogfs_cg = (cluster_global_t *)volume->private_data;
	einfo = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     (OGFS_CG_INFO_COUNT * sizeof(extended_info_t)));
	
	if (einfo != NULL) {
		einfo->count = OGFS_CG_INFO_COUNT;

		/* version */
		einfo->info[0].name = EngFncs->engine_strdup("Version");
		einfo->info[0].title = EngFncs->engine_strdup(_("Version Number"));
		einfo->info[0].desc = EngFncs->engine_strdup(_("Version number of cluster configuration data"));
		einfo->info[0].type              = EVMS_Type_Unsigned_Int32;
		einfo->info[0].unit              = EVMS_Unit_None;
		einfo->info[0].value.ui32        = ogfs_cg->cg_version;
		einfo->info[0].collection_type   = EVMS_Collection_None;
		memset(&einfo->info[0].group, 0, sizeof(group_info_t));
		
		/* lock device */
		einfo->info[1].name = EngFncs->engine_strdup("LockDev");
		einfo->info[1].title = EngFncs->engine_strdup(_("Lock Device"));
		einfo->info[1].desc = EngFncs->engine_strdup(_("Name of lock device"));
		einfo->info[1].type              = EVMS_Type_String;
		einfo->info[1].unit              = EVMS_Unit_None;
		einfo->info[1].value.s = EngFncs->engine_strdup(ogfs_cg->cg_lockdev);
		einfo->info[1].collection_type   = EVMS_Collection_None;
		memset(&einfo->info[1].group, 0, sizeof(group_info_t));

		/* data device */
		einfo->info[2].name = EngFncs->engine_strdup("DataDev");
		einfo->info[2].title = EngFncs->engine_strdup(_("Data Device"));
		einfo->info[2].desc = EngFncs->engine_strdup(_("Name of data device"));
		einfo->info[2].type              = EVMS_Type_String;
		einfo->info[2].unit              = EVMS_Unit_None;
		einfo->info[2].value.s = EngFncs->engine_strdup(ogfs_cg->cg_datadev);
		einfo->info[2].collection_type   = EVMS_Collection_None;
		memset(&einfo->info[2].group, 0, sizeof(group_info_t));

		/* callback port */
		einfo->info[3].name = EngFncs->engine_strdup("Port");
		einfo->info[3].title = EngFncs->engine_strdup(_("Callback Port"));
		einfo->info[3].desc = EngFncs->engine_strdup(_("Port number used for callbacks between nodes"));
		einfo->info[3].type              = EVMS_Type_Unsigned_Int32;
		einfo->info[3].unit              = EVMS_Unit_None;
		einfo->info[3].value.ui32        = ogfs_cg->cg_cbport;
		einfo->info[3].collection_type   = EVMS_Collection_None;
		memset(&einfo->info[3].group, 0, sizeof(group_info_t));
		
		/* node timeout */
		einfo->info[4].name = EngFncs->engine_strdup("Timeout");
		einfo->info[4].title = EngFncs->engine_strdup(_("Node Timeout"));
		einfo->info[4].desc = EngFncs->engine_strdup(_("Time in seconds before node is considered \"dead\""));
		einfo->info[4].type              = EVMS_Type_Unsigned_Int32;
		einfo->info[4].unit              = EVMS_Unit_None;
		einfo->info[4].value.ui32        = ogfs_cg->cg_node_timeout;
		einfo->info[4].collection_type   = EVMS_Collection_None;
		memset(&einfo->info[4].group, 0, sizeof(group_info_t));

		*info = einfo;
		rc = 0;
	} else {
		rc = ENOMEM;
	}
		
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_get_sb_extended_info - return info from the filesystem superblock
 *	@volume: the volume to retrieve opengfs info from
 *	@info_name: name of field to retrieve additional info. NULL for all.
 *	@info: address to return pointer to fsim extended info
 *
 *	This routine is called to return additional information related to
 *	opengfs information about the existing volume. We allocate, initialize,
 *	and return an extended_info_array_t with selected information.
 */
static int ogfs_get_sb_extended_info(logical_volume_t *volume,
				     char *info_name,
				     extended_info_array_t **info)
{
	int rc = EINVAL;
	struct ogfs_sb        *ogfs_sb;
	extended_info_array_t *einfo;
	
	LOG_ENTRY();
	
	ogfs_sb = (struct ogfs_sb *)volume->private_data;
	einfo = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     (OGFS_SB_INFO_COUNT * sizeof(extended_info_t)));
	
	if (einfo != NULL) {
		einfo->count = OGFS_SB_INFO_COUNT;

		/* generation number */
		einfo->info[0].name = EngFncs->engine_strdup("Generation");
		einfo->info[0].title = EngFncs->engine_strdup(_("Generation Number"));
		einfo->info[0].desc = EngFncs->engine_strdup(_("Number incremented each time a change occurs"));
		einfo->info[0].type              = EVMS_Type_Unsigned_Int64;
		einfo->info[0].unit              = EVMS_Unit_None;
		einfo->info[0].value.ui64        = ogfs_sb->sb_header.mh_generation;
		einfo->info[0].collection_type   = EVMS_Collection_None;
		memset(&einfo->info[0].group, 0, sizeof(group_info_t));

		/* filesystem block size */
		einfo->info[1].name = EngFncs->engine_strdup("BlockSize");
		einfo->info[1].title = EngFncs->engine_strdup(_("Block Size"));
		einfo->info[1].desc = EngFncs->engine_strdup(_("File system block size in bytes"));
		einfo->info[1].type              = EVMS_Type_Unsigned_Int32;
		einfo->info[1].unit              = EVMS_Unit_Bytes;
		einfo->info[1].value.ui32        = ogfs_sb->sb_bsize;
		einfo->info[1].collection_type   = EVMS_Collection_None;
		memset(&einfo->info[1].group, 0, sizeof(group_info_t));
		
		/* locking protocol name */
		einfo->info[2].name = EngFncs->engine_strdup("LockProtocol");
		einfo->info[2].title = EngFncs->engine_strdup(_("Locking Protocol"));
		einfo->info[2].desc = EngFncs->engine_strdup(_("Name of locking protocol filesystem is using"));
		einfo->info[2].type              = EVMS_Type_String;
		einfo->info[2].unit              = EVMS_Unit_None;
		einfo->info[2].value.s = EngFncs->engine_strdup(ogfs_sb->sb_lockproto);
		einfo->info[2].collection_type   = EVMS_Collection_None;
		memset(&einfo->info[2].group, 0, sizeof(group_info_t));

		/* lock table name */
		einfo->info[3].name = EngFncs->engine_strdup("LockTable");
		einfo->info[3].title = EngFncs->engine_strdup(_("Lock Table"));
		einfo->info[3].desc = EngFncs->engine_strdup(_("Name of lock table for this filesystem"));
		einfo->info[3].type              = EVMS_Type_String;
		einfo->info[3].unit              = EVMS_Unit_None;
		einfo->info[3].value.s = EngFncs->engine_strdup(ogfs_sb->sb_locktable);
		einfo->info[3].collection_type   = EVMS_Collection_None;
		memset(&einfo->info[3].group, 0, sizeof(group_info_t));
		
		*info = einfo;
		rc = 0;
	} else {
		rc = ENOMEM;
	}
	
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_get_volume_info - return FSIM specific information about a volume
 *	@volume: the volume to retrieve opengfs info from
 *	@info_name: name of field to retrieve additional info. NULL for all.
 *	@info: address to return pointer to fsim extended info
 *
 *	This routine is called to return additional information related to
 *	opengfs information about the existing volume. We allocate, initialize,
 *	and return an extended_info_array_t with selected information.
 */
static int ogfs_get_volume_info(logical_volume_t *volume,
				char *info_name,
				extended_info_array_t **info)
{
	int rc;

	LOG_ENTRY();
	
	switch (ogfs_get_volume_type(volume)) {
	case OGFS_FS_VOL:
		rc = ogfs_get_sb_extended_info(volume, info_name, info);
		break;
	case OGFS_CI_VOL:
		rc = ogfs_get_cluster_extended_info(volume, info_name, info);
		break;
	default:
		rc = ENOSYS;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_get_plugin_info - return standard information on plugin
 *	@descriptor_name: return information on a particular field
 *	@info: address to return pointer to plugin information
 *
 *	This routine returns an extended_info_array_t contains some
 *	standard information on the plugin.
 */
static int ogfs_get_plugin_info(char *descriptor_name,
				extended_info_array_t **info)
{
	int rc = EINVAL;
	extended_info_array_t *einfo;
	char                  version_string[64];
	char                  required_engine_api_version_string[64];
	char                  required_fsim_api_version_string[64];

	LOG_ENTRY();

	if (info) {
		if (descriptor_name == NULL) {
			*info = NULL;

			einfo = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
						      (OGFS_PLUGIN_INFO_COUNT *
						      sizeof(extended_info_t)));
			if (einfo) {
				einfo->count = OGFS_PLUGIN_INFO_COUNT;

				sprintf(version_string, "%d.%d.%d",
					MAJOR_VERSION,
					MINOR_VERSION,
					PATCH_LEVEL);

				sprintf(required_engine_api_version_string, "%d.%d.%d",
					my_plugin_record->required_engine_api_version.major,
					my_plugin_record->required_engine_api_version.minor,
					my_plugin_record->required_engine_api_version.patchlevel);

				sprintf(required_fsim_api_version_string, "%d.%d.%d",
					my_plugin_record->required_plugin_api_version.fsim.major,
					my_plugin_record->required_plugin_api_version.fsim.minor,
					my_plugin_record->required_plugin_api_version.fsim.patchlevel);

				einfo->info[0].name = EngFncs->engine_strdup("Short Name");
				einfo->info[0].title = EngFncs->engine_strdup(_("Short Name"));
				einfo->info[0].desc = EngFncs->engine_strdup(_("A short name given to this plug-in"));
				einfo->info[0].type              = EVMS_Type_String;
				einfo->info[0].unit              = EVMS_Unit_None;
				einfo->info[0].value.s = EngFncs->engine_strdup(my_plugin_record->short_name);
				einfo->info[0].collection_type    = EVMS_Collection_None;
				memset(&einfo->info[0].group, 0, sizeof(group_info_t));

				einfo->info[1].name = EngFncs->engine_strdup("Long Name");
				einfo->info[1].title = EngFncs->engine_strdup(_("Long Name"));
				einfo->info[1].desc = EngFncs->engine_strdup(_("A longer, more descriptive name for this plug-in"));
				einfo->info[1].type              = EVMS_Type_String;
				einfo->info[1].unit              = EVMS_Unit_None;
				einfo->info[1].value.s = EngFncs->engine_strdup(my_plugin_record->long_name);
				einfo->info[1].collection_type   = EVMS_Collection_None;
				memset(&einfo->info[1].group, 0, sizeof(group_info_t));

				einfo->info[2].name = EngFncs->engine_strdup("Type");
				einfo->info[2].title = EngFncs->engine_strdup(_("Plug-in Type"));
				einfo->info[2].desc = EngFncs->engine_strdup(_("There are various types of plug-ins, each responsible for some kind of storage object or logical volume."));
				einfo->info[2].type              = EVMS_Type_String;
				einfo->info[2].unit              = EVMS_Unit_None;
				einfo->info[2].value.s = EngFncs->engine_strdup(_("File System Interface Module"));
				einfo->info[2].collection_type   = EVMS_Collection_None;
				memset(&einfo->info[2].group, 0, sizeof(group_info_t));

				einfo->info[3].name = EngFncs->engine_strdup("Version");
				einfo->info[3].title = EngFncs->engine_strdup(_("Plug-in Version"));
				einfo->info[3].desc = EngFncs->engine_strdup(_("This is the version number of the plug-in."));
				einfo->info[3].type              = EVMS_Type_String;
				einfo->info[3].unit              = EVMS_Unit_None;
				einfo->info[3].value.s = EngFncs->engine_strdup(version_string);
				einfo->info[3].collection_type   = EVMS_Collection_None;
				memset(&einfo->info[3].group, 0, sizeof(group_info_t));

				einfo->info[4].name = EngFncs->engine_strdup("Required Engine Services Version");
				einfo->info[4].title = EngFncs->engine_strdup(_("Required Engine Services Version"));
				einfo->info[4].desc = EngFncs->engine_strdup(_("This is the version of the Engine services that this plug-in requires.  "
									       "It will not run on older versions of the Engine services."));
				einfo->info[4].type              = EVMS_Type_String;
				einfo->info[4].unit              = EVMS_Unit_None;
				einfo->info[4].value.s = EngFncs->engine_strdup(required_engine_api_version_string);
				einfo->info[4].collection_type   = EVMS_Collection_None;
				memset(&einfo->info[4].group, 0, sizeof(group_info_t));

				einfo->info[5].name = EngFncs->engine_strdup("Required Engine FSIM API Version");
				einfo->info[5].title = EngFncs->engine_strdup(_("Required Engine FSIM API Version"));
				einfo->info[5].desc = EngFncs->engine_strdup(_("This is the version of the Engine FSIM API that this plug-in requires.  "
									       "It will not run on older versions of the Engine FSIM API."));
				einfo->info[5].type              = EVMS_Type_String;
				einfo->info[5].unit              = EVMS_Unit_None;
				einfo->info[5].value.s = EngFncs->engine_strdup(required_fsim_api_version_string);
				einfo->info[5].collection_type   = EVMS_Collection_None;
				memset(&einfo->info[5].group, 0, sizeof(group_info_t));

				*info = einfo;

				rc = 0;
			} else {
				rc = ENOMEM;
			}
		} else {
			/* There is no more information on any of the descriptors. */
			rc = EINVAL;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_can_expand_by - check to see if volume and filesystem can expand
 *	@volume: the volume being asked to expand
 *	@delta: amount in sectors that are possible to expand
 *
 *	This routine ensures the volume to expand is mounted and that the
 *	amount by which to expand looks cool.
 */
static int ogfs_can_expand_by(logical_volume_t *volume, sector_count_t *delta)
{
	int rc = EPERM;

	LOG_ENTRY();
	
	if (ogfs_get_volume_type(volume) == OGFS_FS_VOL &&
	    EngFncs->is_mounted(volume->dev_node, NULL) &&
	    ogfsutils_support) {
		struct ogfs_sb *ogfs_sb;
		
		ogfs_sb = (struct ogfs_sb *)volume->private_data;

		/*
		 * Check to see that delta is at least 100 ogfs blocks in size. The
		 * ogfs_expand tool won't consider it worth the effort to expand if
		 * the delta isn't at least 100 ogfs blocks in size.
		 */
		if (*delta >= (100 * (ogfs_sb->sb_bsize >> EVMS_VSECTOR_SIZE_SHIFT))) {
			if (*delta > volume->max_fs_size - volume->fs_size) {
				*delta = volume->max_fs_size - volume->fs_size;
			}
			rc = 0;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 *	ogfs_can_shrink_by - check to see if volume and filesystem can shrink
 *	@volume: the volume being asked to shrink
 *	@delta: amount in sectors by which we can shrink
 *
 *	Currently OpenGFS does not support shrinking a file system.
 */
static int ogfs_can_shrink_by(logical_volume_t *volume, sector_count_t *delta)
{
	int rc = ENOSYS;
	
	LOG_ENTRY();
	LOG_EXIT_INT(rc);
	return rc;
}

static fsim_functions_t ogfs_fsim_ops = {
	.setup_evms_plugin	= ogfs_setup,
	.cleanup_evms_plugin	= ogfs_cleanup,
	.probe			= ogfs_probe,
	.can_mkfs		= ogfs_can_mkfs,
	.can_unmkfs		= ogfs_can_unmkfs,
	.can_fsck		= ogfs_can_fsck,
	.get_fs_size		= ogfs_get_fs_size,
	.get_fs_limits		= ogfs_get_fs_limits,
	.can_expand_by		= ogfs_can_expand_by,
	.can_shrink_by		= ogfs_can_shrink_by,
	.expand			= ogfs_expand,
	.shrink			= ogfs_shrink,
	.mkfs			= ogfs_mkfs,
	.mkfs_setup		= ogfs_mkfs_setup,
	.fsck			= ogfs_fsck,
	.unmkfs			= ogfs_unmkfs,
	.discard		= ogfs_discard,
	.unmkfs_setup		= ogfs_unmkfs_setup,
	.get_option_count	= ogfs_get_option_count,
	.init_task		= ogfs_init_task,
	.set_option		= ogfs_set_option,
	.set_volumes		= ogfs_set_volumes,
	.get_volume_info	= ogfs_get_volume_info,
	.get_plugin_info	= ogfs_get_plugin_info
};

plugin_record_t ogfs_plugin_record = {
	.id			= SetPluginID(EVMS_OEM_IBM,
				    EVMS_FILESYSTEM_INTERFACE_MODULE,
				    FS_TYPE_OGFS),
	.version = {
		.major		= MAJOR_VERSION,
		.minor		= MINOR_VERSION,
		.patchlevel	= PATCH_LEVEL
	},
	.required_engine_api_version = {
		.major		= 15,
		.minor		= 0,
		.patchlevel	= 0
	},
	.required_plugin_api_version = {
		.fsim = {
			.major	= 11,
			.minor	= 0,
			.patchlevel = 0
		}
	},
	.short_name		= "OGFS",
	.long_name		= "OpenGFS File System Interface Module",
	.oem_name		= "IBM",
	.functions  = {
		.fsim = &ogfs_fsim_ops
	},
	.container_functions	= NULL
};

plugin_record_t *evms_plugin_records[] = {
	&ogfs_plugin_record,
	NULL
};
