# -*- coding: utf-8 -*-

# Copyright (c) 2003 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the message box wizard dialog.
"""

import os

from qt import *

from MessageBoxWizardForm import MessageBoxWizardForm

class MessageBoxWizardDialog(MessageBoxWizardForm):
    """
    Module implementing the message box wizard dialog.
    
    It displays a dialog for entering the parameters
    for the QMessageBox code generator.
    
    """
    def __init__(self, parent=None):
        """
        Constructor
        
        @param parent parent widget (QWidget)
        """
        MessageBoxWizardForm.__init__(self, parent, None, 1)
        
        # to get around a quirk in PyQt if the subclass has no translatable strings
        dummy = self.trUtf8('Dummy')
        
        if qVersion() < "3.2":
            self.rQuestion.hide()
        
    def testMsg(self):
        """
        Private method to test the selected options.
        """
        if self.rAbout.isOn():
            QMessageBox.about(None,
                self.eCaption.text(),
                self.eMessage.text()
            )
        elif self.rAboutQt.isOn():
            QMessageBox.aboutQt(None,
                self.eCaption.text()
            )
        else:
            b1 = None
            b2 = None
            b3 = None
            if not self.cButton0.currentText().isEmpty():
                b1 = self.cButton0.currentText()
                if not self.cButton1.currentText().isEmpty():
                    b2 = self.cButton1.currentText()
                    if not self.cButton2.currentText().isEmpty():
                        b3 = self.cButton2.currentText()
                
            if self.rInformation.isOn():
                QMessageBox.information(None,
                    self.eCaption.text(),
                    self.eMessage.text(),
                    b1, b2, b3, 
                    self.sDefault.value(),
                    self.sEscape.value()
                )
            elif self.rQuestion.isOn():
                QMessageBox.question(None,
                    self.eCaption.text(),
                    self.eMessage.text(),
                    b1, b2, b3, 
                    self.sDefault.value(),
                    self.sEscape.value()
                )
            elif self.rWarning.isOn(): 
                QMessageBox.warning(None,
                    self.eCaption.text(),
                    self.eMessage.text(),
                    b1, b2, b3, 
                    self.sDefault.value(),
                    self.sEscape.value()
                )
            elif self.rCritical.isOn():
                QMessageBox.critical(None,
                    self.eCaption.text(),
                    self.eMessage.text(),
                    b1, b2, b3, 
                    self.sDefault.value(),
                    self.sEscape.value()
                )

    def getCode(self, indLevel, indString):
        """
        Public method to get the source code.
        
        @param indLevel indentation level (int)
        @param indString string used for indentation (space or tab) (string)
        @return generated code (string)
        """
        # calculate our indentation level and the indentation string
        il = indLevel + 1
        istring = il * indString
        
        # now generate the code
        msgdlg = 'QMessageBox.'
        if self.rAbout.isOn():
            msgdlg += "about(None,%s" % os.linesep
        elif self.rAboutQt.isOn():
            msgdlg += "aboutQt(None, %s" % os.linesep
        elif self.rInformation.isOn():
            msgdlg += "information(None,%s" % os.linesep
        elif self.rQuestion.isOn():
            msgdlg += "question(None,%s" % os.linesep
        elif self.rWarning.isOn():
            msgdlg += "warning(None,%s" % os.linesep
        else:
            msgdlg +="critical(None,%s" % os.linesep
        msgdlg += '%sself.trUtf8("%s")' % (istring, unicode(self.eCaption.text()))
        if not self.rAboutQt.isOn():
            msgdlg += ',%s%sself.trUtf8("""%s""")' % \
                (os.linesep, istring, unicode(self.eMessage.text()))
        if self.rAbout.isOn() or self.rAboutQt.isOn():
            pass
        else:
            b1 = None
            b2 = None
            b3 = None
            if not self.cButton0.currentText().isEmpty():
                b1 = self.cButton0.currentText()
                if not self.cButton1.currentText().isEmpty():
                    b2 = self.cButton1.currentText()
                    if not self.cButton2.currentText().isEmpty():
                        b3 = self.cButton2.currentText()
            for button in [b1, b2, b3]:
                if button is None:
                    fmt = ',%s%s%s'
                else:
                    fmt = ',%s%sself.trUtf8("%s")'
                msgdlg += fmt % (os.linesep, istring, unicode(button))
            msgdlg += ',%s%s%d, %d' % \
                (os.linesep, istring, self.sDefault.value(), self.sEscape.value())
        msgdlg +=')%s' % os.linesep
        return msgdlg
        
    def button0Text(self, text):
        """
        Private slot to enable/disable the other button combos depending on its contents.
        
        @param text the new text (QString)
        """
        if text.isEmpty():
            self.cButton1.setEnabled(0)
            self.cButton2.setEnabled(0)
        else:
            self.cButton1.setEnabled(1)
            if self.cButton1.currentText().isEmpty():
                self.cButton2.setEnabled(0)
            else:
                self.cButton2.setEnabled(1)
        self.sDefault.setMaxValue(0)
        self.sEscape.setMaxValue(0)

    def button1Text(self, text):
        """
        Private slot to enable/disable the other button combos depending on its contents.
        
        @param text the new text (QString)
        """
        if text.isEmpty():
            self.cButton2.setEnabled(0)
            self.sDefault.setMaxValue(0)
            self.sEscape.setMaxValue(0)
        else:
            self.cButton2.setEnabled(1)
            self.sDefault.setMaxValue(1)
            self.sEscape.setMaxValue(1)

    def button2Text(self, text):
        """
        Private slot to enable/disable the other button combos depending on its contents.
        
        @param text the new text (QString)
        """
        if text.isEmpty():
            self.sDefault.setMaxValue(1)
            self.sEscape.setMaxValue(1)
        else:
            self.sDefault.setMaxValue(2)
            self.sEscape.setMaxValue(2)

    def typeSelected(self, id):
        """
        Private slot to enable/disable certain fields depending on index.
        
        @param id index of selected radio button
        """
        if id == 4:
            self.cButton0.setEnabled(0)
            self.cButton1.setEnabled(0)
            self.cButton2.setEnabled(0)
            self.sDefault.setEnabled(0)
            self.sEscape.setEnabled(0)
            self.eMessage.setEnabled(1)
        elif id == 5:
            self.cButton0.setEnabled(0)
            self.cButton1.setEnabled(0)
            self.cButton2.setEnabled(0)
            self.sDefault.setEnabled(0)
            self.sEscape.setEnabled(0)
            self.eMessage.setEnabled(0)
        else:
            max = 0
            self.cButton0.setEnabled(1)
            if self.cButton0.currentText().isEmpty():
                self.cButton1.setEnabled(0)
            else:
                self.cButton1.setEnabled(1)
            if self.cButton1.currentText().isEmpty():
                self.cButton2.setEnabled(0)
            else:
                self.cButton2.setEnabled(1)
                max = 1
            if not self.cButton2.currentText().isEmpty():
                max = 2
            self.sDefault.setEnabled(1)
            self.sDefault.setMaxValue(max)
            self.sEscape.setEnabled(1)
            self.sEscape.setMaxValue(max)
            self.eMessage.setEnabled(1)
