#include <gtk/gtk.h>
#include <stdlib.h>
#include <stdio.h>
#include "entity.h"
#include "gtk-common.h"
#include "gtk-widget-attr.h"
#include <gtkdatabox.h>
#include <glib.h>

#define REDRAW_NOTIFY(point) if(! point ->databox->notify_changed){ point ->databox->notify_changed++;}

typedef struct {
    GtkWidget *widget;		/* the parent widget of a set */
    guint notify_changed;	/* if non-null, the databox will redraw
				 * itself */
} Databox;

typedef struct {
    Databox *databox;
    gint index;
} DataLine;

typedef struct {
    Databox *databox;
    gint index;
    gfloat x;
    gfloat y;
} DataPoint;


void
rendgtk_graph_redraw (GtkDatabox * box)
{
    gtk_databox_rescale (box);
    gtk_databox_redraw (box);
}

/* define the four functions which will be passed to g_source_add() */

static gboolean
rendgtk_graph_event_source_prepare (gpointer source_data,
				    GTimeVal * current_time, gint * timeout,
				    gpointer user_data)
{
    guint *notify_changed = (guint *) source_data;
    *timeout = -1;

    if (*notify_changed) {
	return TRUE;
    } else {
	return FALSE;
    }
}

static gboolean
rendgtk_graph_event_source_check (gpointer source_data, GTimeVal * current_time,
				  gpointer user_data)
{
    return FALSE;
}

static gboolean
rendgtk_graph_event_source_dispatch (gpointer source_data,
				     GTimeVal * current_time,
				     gpointer user_data)
{
    GtkWidget *widget = (GtkWidget *) user_data;
    guint *notify_changed = (guint *) source_data;
    *notify_changed = 0;

    EDEBUG (("databox-rendrerer-1", "in rendgtk_graph_event_source_dispatch"));
    rendgtk_graph_redraw (GTK_DATABOX (widget));
    return TRUE;
}

static void
rendgtk_graph_event_source_destroy (gpointer user_data)
{
    /* g_message ("(rendgtk_graph_event_source_destroy)"); */
    EDEBUG (("databox-rendrerer", "in rendgtk_graph_event_source_destroy"));
}

static void
rendgtk_databox_intersecting_selection (ENode * node,
					GtkDataboxValue * val1,
					GtkDataboxValue * val2)
{
    GSList *children;
    GSList *tmp;

    children = enode_children (node, "graph-point");
    tmp = children;

    while (tmp) {
	ENode *node;
	EBuf *x,
	*y;
	gfloat x1,
	 y1;

	node = tmp->data;

	x = enode_attrib (node, "x", NULL);
	y = enode_attrib (node, "y", NULL);
	if (ebuf_not_empty (x) && ebuf_not_empty (y)) {
	    gfloat larger;
	    gfloat smaller;

	    x1 = erend_get_float (x);
	    y1 = erend_get_float (y);

	    if (val1->x > val2->x) {
		larger = val1->x;
		smaller = val2->x;
	    } else {		/* Else we know that val2 is larger... */

		larger = val2->x;
		smaller = val1->x;
	    }

	    if ((x1 >= smaller) && (x1 <= larger)) {
		EBuf *path;

		if (val1->y > val2->y) {
		    larger = val1->y;
		    smaller = val2->y;
		} else {	/* We know that val2 is larger. */

		    larger = val2->y;
		    smaller = val1->y;
		}


		if ((y1 >= smaller) && (y1 <= larger)) {
		    path = enode_path (node);
		    ebuf_free (path);
		    enode_attrib (node, "selected", ebuf_new_with_true ());
		}
	    }
	}

	tmp = tmp->next;
    }
    g_slist_free (children);
}


static gint
rendgtk_databox_selection_stopped (GtkDatabox * box, GtkDataboxCoord * marked,
				   GtkDataboxCoord * select, gpointer data)
{
    ENode *node = data;
    EBuf *value;
    GtkDataboxValue value1;
    GtkDataboxValue value2;
    gchar x[50],
     y[50],
     x2[50],
     y2[50];
    gtk_databox_data_get_value (box, *marked, &value1);
    gtk_databox_data_get_value (box, *select, &value2);

    value = enode_attrib (node, "onselect", NULL);

    EDEBUG (("databox-renderer", "onselection callback - %g,%g - %g,%g",
	     value1.x, value1.y, value2.x, value2.y));

    if (ebuf_not_empty (value)) {
	g_snprintf (x, 50, "%f", value1.x);
	g_snprintf (y, 50, "%f", value1.y);
	g_snprintf (x2, 50, "%f", value2.x);
	g_snprintf (y2, 50, "%f", value2.y);

	rendgtk_databox_intersecting_selection (node, &value1, &value2);

	enode_call_ignore_return (node, value->str, "ssss", x, y, x2, y2);

    }

    return (TRUE);
}


static gint
rendgtk_databox_selection_canceled (GtkDatabox * box, gpointer data)
{
    GSList *children;
    GSList *tmp;
    ENode *node = data;

    if (!node)
	return (0);

    EDEBUG (("databox-renderer", "in rendgtk_databox_selection_canceled"));

    children = enode_children (node, "graph-point");
    tmp = children;

    while (tmp) {
	EBuf *selected;
	node = tmp->data;

	selected = enode_attrib (node, "selected", NULL);

	if (ebuf_not_empty (selected)) {
	    enode_attrib (node, "selected", ebuf_new_with_str (""));
	}

	tmp = tmp->next;
    }
    g_slist_free (children);

    return (0);
}


static gint
rendgtk_databox_zoomable_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *databox;

    databox = enode_get_kv (node, "top-widget");
    if (!databox)
	return (TRUE);

    if (erend_value_is_true (value))
	gtk_databox_enable_zoom (GTK_DATABOX (databox));
    else
	gtk_databox_disable_zoom (GTK_DATABOX (databox));

    return (TRUE);
}

static gint
rendgtk_databox_selectable_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *databox;

    databox = enode_get_kv (node, "top-widget");
    if (!databox)
	return TRUE;

    if (erend_value_is_true (value))
	gtk_databox_enable_selection (GTK_DATABOX (databox));
    else
	gtk_databox_disable_selection (GTK_DATABOX (databox));

    return TRUE;
}

static gint
rendgtk_databox_rulers_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *databox;

    databox = enode_get_kv (node, "top-widget");
    if (!databox)
	return TRUE;

    if (erend_value_is_true (value))
	gtk_databox_show_rulers (GTK_DATABOX (databox));
    else
	gtk_databox_hide_rulers (GTK_DATABOX (databox));

    return TRUE;
}

static gint
rendgtk_databox_cross_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *databox;

    databox = enode_get_kv (node, "top-widget");
    if (!databox)
	return TRUE;

    if (erend_value_is_true (value))
	gtk_databox_show_cross (GTK_DATABOX (databox));
    else
	gtk_databox_hide_cross (GTK_DATABOX (databox));

    return TRUE;
}

#define DATABOX_BOUNDS(bound)                                       \
  (ebuf_equal_str (attr, #bound)) {                                 \
      gfloat x;                                                     \
      EBuf *val = enode_attrib (node, #bound, NULL);		    \
      if (ebuf_not_empty (val)) {				    \
	  x = erend_get_float (val);				    \
	  gtk_databox_set_##bound(GTK_DATABOX(databox->widget), x); \
      } else {                                                      \
	gtk_databox_unset_##bound(GTK_DATABOX(databox->widget));    \
      }                                                             \
      return TRUE;                                                  \
    }


static gint
rendgtk_databox_graph_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    Databox *databox = (Databox *) enode_get_kv (node, "rendgtk-databox");

    if (!databox || !databox->widget)
	return (FALSE);

    if (ebuf_equal_str (attr, "frozen")) {
	if (erend_value_is_true (value)) {
	    gtk_databox_redraw (GTK_DATABOX (databox->widget));
	}
	return (TRUE);
    } else if DATABOX_BOUNDS (xmin)
    else if DATABOX_BOUNDS (xmax)
    else if DATABOX_BOUNDS (ymin)
    else if DATABOX_BOUNDS (ymax)
    else
	g_warning ("rendgtk_databox_graph_attr_set: unknown attribute `%s'",
	    attr->str);

    return (TRUE);
}


static void
rendgtk_databox_graph_destroy (ENode * node)
{
    gint id;
    DataPoint *point;

    point = enode_get_kv (node, "rendgtk-databox-point");
    if (point)
	g_free (point);

    id = GPOINTER_TO_INT (enode_get_kv (node, "rendgtk-graph-source"));
    if (id > 0)
	g_source_remove (id);

    rendgtk_element_destroy (node);
}

static void
rendgtk_databox_graph_render (ENode * node)
{
    gint id;
    Databox *databox = g_new0 (Databox, 1);
    GSourceFuncs *event_source_funcs = g_new0 (GSourceFuncs, 1);

    databox->widget = gtk_databox_new ();
    /* 
     * gtk_signal_connect (GTK_OBJECT (databox->widget), "destroy",
     * GTK_SIGNAL_FUNC (gtk_databox_data_destroy_all), NULL); */

    gtk_signal_connect (GTK_OBJECT (databox->widget),
			"gtk_databox_selection_stopped",
			GTK_SIGNAL_FUNC (rendgtk_databox_selection_stopped),
			node);

    gtk_signal_connect (GTK_OBJECT (databox->widget),
			"gtk_databox_selection_canceled",
			GTK_SIGNAL_FUNC (rendgtk_databox_selection_canceled),
			node);

    enode_set_kv (node, "top-widget", databox->widget);
    enode_set_kv (node, "bottom-widget", databox->widget);

    databox->notify_changed = 1;	/* redraw at least once */
    event_source_funcs->prepare = rendgtk_graph_event_source_prepare;
    event_source_funcs->check = rendgtk_graph_event_source_check;
    event_source_funcs->dispatch = rendgtk_graph_event_source_dispatch;
    id = g_source_add (G_PRIORITY_DEFAULT, TRUE /* can recurse */ ,
		       event_source_funcs, (gpointer) & (databox->notify_changed)	/* source_data 
											 */
		       , (gpointer) databox->widget /* user_data */ ,
		       rendgtk_graph_event_source_destroy);
    enode_set_kv (node, "rendgtk-graph-source", GINT_TO_POINTER (id));
    enode_set_kv (node, "rendgtk-databox", databox);

    enode_attribs_sync (node);

    rendgtk_show_cond (node, databox->widget);
}


static gint
rendgtk_databox_graph_point_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *widget;
    DataPoint *point;
    GdkColor color;

    point = enode_get_kv (node, "rendgtk-databox-point");
    if (!point || !point->databox)
	return TRUE;

    widget = point->databox->widget;

    if (!widget)
	return (TRUE);

    if (ebuf_equal_str (attr, "x")) {
	point->x = erend_get_float (value);
	gtk_databox_set_x_y (GTK_DATABOX (widget), point->index, &(point->x),
			     &(point->y), 1 /* length */ );
	REDRAW_NOTIFY (point);
	return TRUE;
    }

    if (ebuf_equal_str (attr, "y")) {
	point->y = erend_get_float (value);
	gtk_databox_set_x_y (GTK_DATABOX (widget), point->index, &(point->x),
			     &(point->y), 1 /* length */ );
	REDRAW_NOTIFY (point);
	return TRUE;
    }

    if (ebuf_equal_str (attr, "size")) {
	gint size = erend_get_integer (value);
	gtk_databox_set_size (GTK_DATABOX (widget), point->index, size);
	REDRAW_NOTIFY (point);
	return TRUE;
    }

    if (ebuf_equal_str (attr, "width")) {
	gint width = erend_get_integer (value);
	gtk_databox_set_width (GTK_DATABOX (widget), point->index, width);
	REDRAW_NOTIFY (point);
	return TRUE;
    }

    if (ebuf_equal_str (attr, "height")) {
	gint height = erend_get_integer (value);
	gtk_databox_set_height (GTK_DATABOX (widget), point->index, height);
	REDRAW_NOTIFY (point);
	return TRUE;
    }

    if ((ebuf_equal_str (attr, "color")) || (ebuf_equal_str (attr, "colour"))) {
	/* point->size = atof (value->str); this is wrong here, isn't it ?
	 * (joze) */

	if (gdk_color_parse (value->str, &color)) {
	    gtk_databox_set_color (GTK_DATABOX (widget), point->index, color);
	}
	/* needed ? */
	/* databox_maybe_redraw (node, widget); */
	return (TRUE);
    }

    if (ebuf_equal_str (attr, "filled")) {
	gboolean filled = erend_value_is_true (value);
	gtk_databox_set_filled (GTK_DATABOX (widget), point->index, filled);
	REDRAW_NOTIFY (point);
	return (TRUE);
    }

    if (ebuf_equal_str (attr, "angle-start")) {
	gfloat angle = 0.0;
	angle = erend_get_float (value);
	angle *= (gfloat) 64.0;
	gtk_databox_set_angle (GTK_DATABOX (widget), point->index, angle, 1);
	REDRAW_NOTIFY (point);
	return TRUE;
    }

    if (ebuf_equal_str (attr, "angle-range")) {
	gfloat angle = 0.0;
	angle = erend_get_float (value);
	angle *= (gfloat) 64.0;
	gtk_databox_set_angle (GTK_DATABOX (widget), point->index, angle, 2);
	REDRAW_NOTIFY (point);
	return TRUE;
    }

    return (FALSE);
}

static void
rendgtk_databox_graph_point_render (ENode * node)
{
    DataPoint *point = g_new0 (DataPoint, 1);

    enode_set_kv (node, "rendgtk-databox-point", point);
    enode_set_kv (node, "rendgtk_databox_graph_point_render", "true");

    /* This is useless, as the point hasn't been made yet.. *
     * enode_attribs_sync (node); */
}

static void
rendgtk_databox_graph_point_destroy (ENode * node)
{
    int pos;
    GtkWidget *widget;
    DataPoint *point;

    if (!node->parent)
	return;

    widget = enode_get_kv (node->parent, "top-widget");
    if (!widget)
	return;

    pos = g_slist_index (node->parent->children, node);
    gtk_databox_data_remove (GTK_DATABOX (widget), pos);

    point = enode_get_kv (node, "rendgtk-databox-point");
    /* TODO: eventually remove the link from the (line) parent */
    if (point)
	g_free (point);
}

static void
rendgtk_databox_graph_point_parent (ENode * parent, ENode * child_node)
{
    GdkColor color;
    EBuf *value;
    GtkDataboxDataType graph_type = GTK_DATABOX_BARS;
    DataPoint *point = enode_get_kv (child_node, "rendgtk-databox-point");
    Databox *databox = enode_get_kv (parent, "rendgtk-databox");

    gboolean filled = TRUE;
    gint angle1 = 0;
    gint angle2 = 360;		/* 360 */
    gint size = 1;		/* default pointsize / linewidth */
    gint width = 5;
    gint height = 5;

    if (!databox) {
	/* check for parents different from graph (currently only `line') */
	DataLine *line = enode_get_kv (parent, "rendgtk-databox-line");
	if (line) {
	    databox = line->databox;
	    /* default to arcs, if this point is a child of line */
	    graph_type = GTK_DATABOX_ARCS;
	}
    }

    if (!databox) {
	/* should not occur */
	g_warning ("%s:%d unable to get databox pointer from parent node",
		   __FILE__, __LINE__);
	return;
    }

    if (!point) {
	/* should never occur */
	g_warning ("%s:%d unable to get point pointer from point node",
		   __FILE__, __LINE__);
	return;
    }

    point->databox = databox;	/* propagate databox information to point
				 * (the child) */

    /* g_warning("in rendgtk_databox_graph_point_parent"); */

    /* X, Y, size */
    value = enode_attrib (child_node, "x", NULL);
    if (ebuf_not_empty (value))
	point->x = erend_get_float (value);

    value = enode_attrib (child_node, "y", NULL);
    if (ebuf_not_empty (value))
	point->y = erend_get_float (value);

    value = enode_attrib (child_node, "size", NULL);
    if (ebuf_not_empty (value))
	size = erend_get_integer (value);

    if (size < 1)
	size = 1;


    /* Check color */
    value = enode_attrib (child_node, "color", NULL);
    if (ebuf_empty (value))
	value = enode_attrib (child_node, "colour", NULL);

    if (ebuf_not_empty (value))
	gdk_color_parse (value->str, &color);

    /* Check type */
    value = enode_attrib (child_node, "type", NULL);

    if (ebuf_not_empty (value)) {
	if (ebuf_equal_str (value, "bar"))
	    graph_type = GTK_DATABOX_BARS;
	else if (ebuf_equal_str (value, "point"))
	    graph_type = GTK_DATABOX_POINTS;
#if 0
	if (ebuf_equal_str (value, "line"))
	    graph_type = GTK_DATABOX_LINES;
#endif
	else if (ebuf_equal_str (value, "arc")) {


	    EBuf *val;

	    graph_type = GTK_DATABOX_ARCS;
	    angle1 =
		erend_get_integer (enode_attrib
				   (child_node, "angle-start", NULL));
	    val = enode_attrib (child_node, "angle-range", NULL);
	    if (ebuf_not_empty (val)) {
		angle2 = erend_get_integer (val);
	    }

	    width =
		erend_get_integer (enode_attrib (child_node, "width", NULL));
	    if (width < 1)
		width = 1;

	    value = enode_attrib (child_node, "height", NULL);
	    if (ebuf_not_empty (value)) {
		height = erend_get_integer (value);
	    } else {
		height = width;	/* height defaults to width */
	    }

	    value = enode_attrib (child_node, "filled", NULL);
	    if (ebuf_not_empty (value)) {
		filled = erend_value_is_true (value);
	    }

	    angle1 *= (gfloat) 64.0;
	    angle2 *= (gfloat) 64.0;

	    point->index =
		gtk_databox_data_add_x_y_arc (GTK_DATABOX (databox->widget), 1,
					      &point->x, &point->y, color,
					      graph_type, size, width, height,
					      filled, angle1, angle2);

	    EDEBUG (("databox-renderer",
		     "parented %s point\n\tx %f\n\ty %f\n\tsize %d\n\twidth %d\n\theight %d\n\tfilled %d\n\tangle-start %d\n\tangle-range %d\n",
		     GTK_DATABOX_BARS == graph_type ? "`bar'" :
		     GTK_DATABOX_POINTS == graph_type ? "`point'" :
		     GTK_DATABOX_LINES == graph_type ? "`line'" :
		     GTK_DATABOX_ARCS == graph_type ? "`arc'" :
		     "(unknown point type)",
		     point->x, point->y, size, width, height, filled, angle1,
		     angle2));

	    REDRAW_NOTIFY (point);
	    return;
	} else {
	    g_warning ("%s:%d unknown point type `%s'", __FILE__, __LINE__,
		       value->str);
	    return;
	}
    }

    /* default draw. */
    point->index = gtk_databox_data_add_x_y (GTK_DATABOX (databox->widget), 1,
					     &point->x, &point->y, color,
					     graph_type, size);
    REDRAW_NOTIFY (point);

    EDEBUG (("databox-renderer", "(rendgtk_databox_graph_point_parent)"));

    EDEBUG (("databox-renderer",
	     "parented %s point\n\tx %f\n\ty %f\n\tsize %d\n\twidth %d\n\theight %d\n\tfilled %d\n\tangle-start %d\n\tangle-range %d\n",
	     GTK_DATABOX_BARS == graph_type ? "`bar'" :
	     GTK_DATABOX_POINTS == graph_type ? "`point'" :
	     GTK_DATABOX_LINES == graph_type ? "`line'" :
	     GTK_DATABOX_ARCS == graph_type ? "`arc'" :
	     "(unknown point type)",
	     point->x, point->y, size, width, height, filled, angle1, angle2));
}

static void
rendgtk_databox_graph_line_parent (ENode * graph_node, ENode * child_node)
{
    GdkColor color;
    EBuf *value;
    DataLine *line = enode_get_kv (child_node, "rendgtk-databox-line");
    Databox *databox = enode_get_kv (graph_node, "rendgtk-databox");

    gint linewidth = 1;		/* default linewidth */

    EDEBUG (("databox-renderer", "(rendgtk_databox_graph_line_parent)"));

    if (!databox) {
	/* line must be an immediate child of graph */
	g_warning ("%s:%d unable to get databox pointer from graph node",
		   __FILE__, __LINE__);
	return;
    }

    if (!line) {
	/* should never occur */
	g_warning ("%s:%d unable to get line pointer from line node", __FILE__,
		   __LINE__);
	return;
    }

    line->databox = databox;	/* propagate databox information to line (the 
				 * child) */

    linewidth =
	erend_get_integer (enode_attrib (child_node, "linewidth", NULL));
    if (linewidth < 1)
	linewidth = 1;

    /* Check color */
    value = enode_attrib (child_node, "color", NULL);
    if (ebuf_empty (value))
	value = enode_attrib (child_node, "colour", NULL);

    if (!ebuf_not_empty (value) || !gdk_color_parse (value->str, &color)) {
	color.red = 0;
	color.green = 0;
	color.blue = 0;
    }

    /* add an empty line */
    line->index =
	gtk_databox_data_add_linked_line (GTK_DATABOX (databox->widget), color,
					  linewidth);

    EDEBUG (("databox-renderer", "parented line"));
}

/* dispatch between different child node types */
static void
rendgtk_databox_graph_item_parent (ENode * graph_node, ENode * child_node)
{
    /* We'll only handle point and line as children. * others will be
     * silently ignored */
    if (ebuf_equal_str (child_node->element, "graph-point")) {
	rendgtk_databox_graph_point_parent (graph_node, child_node);
    } else if (ebuf_equal_str (child_node->element, "graph-line")) {
	rendgtk_databox_graph_line_parent (graph_node, child_node);
    }
}

static void
rendgtk_databox_graph_line_render (ENode * node)
{
    DataLine *line = g_new0 (DataLine, 1);
    EDEBUG (("databox-renderer", "(rendgtk_databox_graph_line_render)"));
    enode_set_kv (node, "rendgtk-databox-line", line);
    enode_attribs_sync (node);
}

static void
rendgtk_databox_graph_line_point_parent (ENode * line_node, ENode * child_node)
{
    DataPoint *point;
    DataLine *line = enode_get_kv (line_node, "rendgtk-databox-line");
    EDEBUG (("databox-renderer", "(rendgtk_databox_graph_line_point_parent)"));
    if (!line) {
	/* should never occur */
	g_warning ("%s:%d unable to get line pointer from line node", __FILE__,
		   __LINE__);
	return;
    }
    /* dispatch between allowd child nodes of `line'. * currently this is
     * only `point' */
    if ((point = enode_get_kv (child_node, "rendgtk-databox-point"))) {
	rendgtk_databox_graph_point_parent (line_node, child_node);
	gtk_databox_add_link (GTK_DATABOX (point->databox->widget), line->index,
			      point->index);
    }
}

static void
rendgtk_databox_graph_line_destroy (ENode * node)
{
#if 0
    int pos;
    GtkWidget *widget;
#endif
    DataLine *line;
    EDEBUG (("databox-renderer", "(rendgtk_databox_graph_line_destroy)"));

    if (!node->parent)
	return;

#if 0
    pos = g_slist_index (node->parent->children, node);
    gtk_databox_data_remove (GTK_DATABOX (widget), pos);
#endif

    line = enode_get_kv (node, "rendgtk-databox-line");
    if (line)
	g_free (line);
}

static gint
rendgtk_databox_graph_line_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    EDEBUG (("databox-renderer", "(rendgtk_databox_graph_line_attr_set)"));
    return 0;
}

void
databox_renderer_register (void)
{
    Element *element;
    ElementAttr *e_attr;

    element = g_new0 (Element, 1);
    element->render_func = rendgtk_databox_graph_render;
    element->destroy_func = rendgtk_databox_graph_destroy;
    element->parent_func = rendgtk_databox_graph_item_parent;
    element->tag = "graph";
    element->description = "Create a new graph widget";
    element_register (element);
    rendgtk_widget_attr_register (element, GTK_TYPE_WIDGET);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "frozen";
    e_attr->description = "If set, no updates will ocurr on screen.";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "false,true";
    e_attr->set_attr_func = rendgtk_databox_graph_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "selectable";
    e_attr->description = "Allow/disallow selection in graph.";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "true,false";
    e_attr->set_attr_func = rendgtk_databox_selectable_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onselect";
    e_attr->description =
	"Function to call when a selection is made on the graph.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(graph_node, x1, y1, x2, y2)";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "rulers";
    e_attr->description = "Show/hide rulers.";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "true,false";
    e_attr->set_attr_func = rendgtk_databox_rulers_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "cross";
    e_attr->description = "Show/hide cross at zero position.";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "true,false";
    e_attr->set_attr_func = rendgtk_databox_cross_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "zoomable";
    e_attr->description = "Allow/disallow zooming of graph.";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "true,false";
    e_attr->set_attr_func = rendgtk_databox_zoomable_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "xmin";
    e_attr->description = "fixed lower bound of the graph's x-direction";
    e_attr->value_desc = "float";
    e_attr->set_attr_func = rendgtk_databox_graph_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "xmax";
    e_attr->description = "fixed upper bound of the graph's x-direction";
    e_attr->value_desc = "float";
    e_attr->set_attr_func = rendgtk_databox_graph_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "ymin";
    e_attr->description = "fixed lower bound of the graph's y-direction";
    e_attr->value_desc = "float";
    e_attr->set_attr_func = rendgtk_databox_graph_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "ymax";
    e_attr->description = "fixed upper bound of the graph's y-direction";
    e_attr->value_desc = "float";
    e_attr->set_attr_func = rendgtk_databox_graph_attr_set;
    element_register_attrib (element, e_attr);


    element = g_new0 (Element, 1);
    element->render_func = rendgtk_databox_graph_point_render;
    element->parent_func = NULL;
    element->destroy_func = rendgtk_databox_graph_point_destroy;
    element->tag = "graph-point";
    element->description = "Create a new point on a graph";
    element_register (element);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "x";
    e_attr->description = "The x location on the graph.";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = rendgtk_databox_graph_point_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "y";
    e_attr->description = "The y location on the graph.";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = rendgtk_databox_graph_point_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "color";
    e_attr->description = "The color of the point.";
    e_attr->value_desc = "string";
    e_attr->set_attr_func = rendgtk_databox_graph_point_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "filled";
    e_attr->description =
	"True, if the element should be filled with a solid color.";
    e_attr->value_desc = "boolean";
    e_attr->set_attr_func = rendgtk_databox_graph_point_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "size";
    e_attr->description = "The size/linewidth of the element.";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = rendgtk_databox_graph_point_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "width";
    e_attr->description = "The width of the element. (defaults to `size')";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = rendgtk_databox_graph_point_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "height";
    e_attr->description = "The height of the element. (defaults to `width')";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = rendgtk_databox_graph_point_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "type";
    e_attr->description = "The type of point to display.";
    e_attr->value_desc = "string";
    e_attr->possible_values = "point,bar,arc";	/* removed line */
    e_attr->set_attr_func = rendgtk_databox_graph_point_attr_set;
    element_register_attrib (element, e_attr);

    /* arc */
    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "angle-start";
    e_attr->description = "The angle where the arc begins (default: 0 [deg])";
    e_attr->value_desc = "float [deg]";
    e_attr->set_attr_func = rendgtk_databox_graph_point_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "angle-range";
    e_attr->description =
	"The angle which is covered by the arc (default: 360 [deg])";
    e_attr->value_desc = "float [deg]";
    e_attr->set_attr_func = rendgtk_databox_graph_point_attr_set;
    element_register_attrib (element, e_attr);


    /* LINE ELEMENTS */
    element = g_new0 (Element, 1);
    element->render_func = rendgtk_databox_graph_line_render;
    element->parent_func = rendgtk_databox_graph_line_point_parent;
    element->destroy_func = rendgtk_databox_graph_line_destroy;
    element->tag = "graph-line";
    element->description = "Create a new line on the graph.";
    element_register (element);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "color";
    e_attr->description = "The color of the point.";
    e_attr->value_desc = "string";
    e_attr->set_attr_func = rendgtk_databox_graph_line_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "linewidth";
    e_attr->description = "The linewidth of the line element.";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = rendgtk_databox_graph_line_attr_set;
    element_register_attrib (element, e_attr);

#if 0
    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "x";
    e_attr->description = "The x location on the graph.";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = rendgtk_databox_graph_line_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "y";
    e_attr->description = "The y location on the graph.";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = rendgtk_databox_graph_line_attr_set;
    element_register_attrib (element, e_attr);
#endif

}
