/*
 *  ---------
 * |.**> <**.|  CardContact
 * |*       *|  Software & System Consulting
 * |*       *|  Minden, Germany
 * |**> <**|  Copyright (c) 1999. All rights reserved
 *  --------- 
 *
 * See file LICENSE for details on licensing
 *
 * Abstract :       Defines tools/types for SDAP memory card handling
 *
 * Author :         Frank Thater (FTH)
 *
 * Last modified:   04/25/2000
 *
 *****************************************************************************/


#include <string.h>

#include "mc_sdap.h"



int MC_SDAPInit(struct eco5000_t *ctx)
     
{ 
    struct memorycard_t *pt;
    unsigned char output[3];
    int rc = OK;
    unsigned char buffer[17];
    
    ctx->CTModFunc = (CTModFunc_t) MC_SDAPProcess;
    
    pt = (struct memorycard_t *)malloc(sizeof(struct memorycard_t)); 
    
    ctx->Data.card = pt;
    
    pt->SelectedFile = NONE;
    pt->Path = 0x00;
    pt->PINVerified = FALSE;
    
    pt->CardID = I2C;
    
    /* ATR already found ?? */
    if (ctx->ATR[0] == 0xFF)  {
        
        output[0] = I2C_CHAR;
        output[1] = 0x00;
        
        /* First try to read 16 bytes from an i2c < 2048K */
        rc = ecoCommand(ctx, READ_16BYTE_I2C, 2, (unsigned char *)&output, 17, (unsigned char *)&buffer);
        
        if (rc < 0)
            return rc;
        
        /* ATR already found ?? */
        if (buffer[0] == 0xFF)  {
            
            rc = ecoCommand(ctx, SYNC_ATR, 0, NULL , 4, (unsigned char *)&buffer);
            
            if (rc < 0)
                return rc;
            
            output[0] = I2C_CHAR;
            output[1] = 0x00;
            output[2] = 0x00;
            
            /* Now try to read 16 bytes from an i2c > 2048K */
            rc = ecoCommand(ctx, READ_16BYTE_I2C_64K, 3, (unsigned char *)&output, 17, (unsigned char *)&buffer);
            if (rc < 0)       
                return rc;
            
            if ((buffer[0] >> 4) == 0x08)   {     /* valid ATR found ... */
                memcpy(ctx->ATR, &buffer, 0x04);
            }
            
            
        } else {

            memcpy(ctx->ATR, &buffer, 0x04);

        }
        
    }
    
    
    /* Set all necessary variables ... */
    
    rc = DetermineVariables(ctx);
    
    if (rc < 0)
        return rc;
    
    
    rc = SDAP_Read_Card(ctx, 0x00, -1);

    if (rc < 0)
        return rc;
    
    return OK;
    
}



int MC_SDAPProcess(struct eco5000_t *ctx,
                   unsigned int lc,
                   unsigned char *cmd,
                   unsigned int *lr,
                   unsigned char *rsp) {
    
    int rc = OK;
    struct memorycard_t *pt = ctx->Data.card;
    
#ifdef DEBUG
    printf("\nMC_I2CProcess called ...\n");
#endif
    
    
    /* Try calling the basic functions ... */
    rc = MemoryCardProcess(ctx, lc, cmd, lr, rsp);
    
    if (rc == OK)  /* command completed */
        return OK;
    
    if (rc < 0)    /* error processing command */
        return rc;
    
    
    /* "specialized" Interindustry Commands */
    switch(cmd[1]) {
        case 0xA4:
            /* not supported, so FILE NOT FOUND */
            pt->SelectedFile = NONE;
            rsp[0] = HIGH(FILE_NOT_FOUND);
            rsp[1] = LOW(FILE_NOT_FOUND);
            *lr = 2;     
            break;
            
        case 0xD6:
            rc = SDAP_Update_Binary(ctx, lc, cmd, lr, rsp);
            break;
            
        case 0x20:
        case 0x24:  /* pin verification */
            /* not supported by card, so ignore and respond "command not allowed" */
            *lr = 2;
            rsp[0] = HIGH(COMMAND_NOT_ALLOWED);
            rsp[1] = LOW(COMMAND_NOT_ALLOWED);
            break;
            
        default:
            *lr = 2;
            rsp[0] = HIGH(CLASS_NOT_SUPPORTED);
            rsp[1] = LOW(CLASS_NOT_SUPPORTED);
    }
    
    return rc;
    
}



/* 
 * SDAP_Update_Binary
 * Updates the main memory of SDAP memory cards
 *
 */

int SDAP_Update_Binary(struct eco5000_t *ctx,
                       unsigned int lc,
                       unsigned char *cmd,
                       unsigned int *lr,
                       unsigned char *rsp)

{
    struct memorycard_t *pt;
    unsigned char output[36];
    unsigned int  offset, length;
    unsigned int  addr, i;
    unsigned char *data,*po;
    int rc, len;
    int newFirmware = FALSE;

    pt = ctx->Data.card;

    if (pt->SelectedFile != MF) {       /* Generic command only for MF       */
        rsp[0] = 0x69;
        rsp[1] = 0x86;
        *lr = 2;
        return 0;
    }
        
    offset = cmd[2] << 8 | cmd[3];      /* Decode file offset from P1 and P2 */

    if (offset >= pt->Number_of_Data_Units) {  /* Offset out of range        */
        rsp[0] = HIGH(OUT_OF_RANGE);
        rsp[1] = LOW(OUT_OF_RANGE);
        *lr = 2;
        return 0;
    }
        
    if(strcmp(ctx->Firmware, ECO_V223) >= 0)  /* Check for new Firmware */
      newFirmware = TRUE;

                                /* Decode lc and data field          */
    rc = DecodeAPDU(lc, cmd, &length, &data, NULL);

    if ((rc != 3) && (rc != 6)) {       /* Must the case 3S or 3E            */
        rsp[0] = HIGH(COMMUNICATION_NOT_POSSIBLE);
        rsp[1] = LOW(COMMUNICATION_NOT_POSSIBLE);
        *lr = 2;
        return 0;
    }
   
    if (offset + length >= pt->Number_of_Data_Units) {
        rsp[0] = HIGH(WRONG_LENGTH);                  /* Wrong lc                          */
        rsp[1] = LOW(WRONG_LENGTH);
        *lr = 2;
        return 0;
    }

    po = data;
    addr = offset;
    i = length;

    while (i) {
        if (pt->Number_of_Data_Units > 2048) {
	  if(newFirmware == TRUE) {
	      len = i > 32 ? 32 : i;
	      output[0] = I2C_CHAR;
	      output[1] = (unsigned char)(addr >> 8);
	      output[2] = (unsigned char)(addr & 0xFF);
	      output[3] = len; 
	      memcpy(&output[4], po, len);
	      rc = ecoCommand(ctx, WRITE_BLOCK_I2C_64K, len + 4, output, 0, NULL);
	      addr += len;
	      po += len; 
	      i -= len;
	  }
	  else {
	      output[0] = I2C_CHAR;
	      output[1] = (unsigned char)(addr >> 8);
	      output[2] = (unsigned char)(addr & 0xFF);
	      output[3] = *po;
	      rc = ecoCommand(ctx, WRITE_BYTE_I2C_64K, 4, output, 0, NULL);
	      addr++;
	      po++;
	      i--;
	  }
	} else {
            output[0] = I2C_CHAR | ((unsigned char)(addr >> 7) & 0x0E);
            output[1] = (unsigned char)(addr & 0xFF);
            output[2] = *po;
            rc = ecoCommand(ctx, WRITE_BYTE_I2C, 3, output, 0, NULL);
	    addr++;
	    po++;
	    i--;
	}
            
        if (rc < 0) {
            rsp[0] = HIGH(COMMUNICATION_NOT_POSSIBLE);
            rsp[1] = LOW(COMMUNICATION_NOT_POSSIBLE);
            *lr = 2;
            return rc;
        }
            
      
    }

    rc = SDAP_Read_Card(ctx, offset, length);

    if (rc < 0)
        return rc;
                                        /* Verify update process             */
    if (memcmp(data, pt->Memory + offset, length)) {
        rsp[0] = HIGH(DATA_CORRUPTED);
        rsp[1] = LOW(DATA_CORRUPTED);
        *lr = 2;
        return 0;
    }   
    
    rsp[0] = HIGH(SMARTCARD_SUCCESS);
    rsp[1] = LOW(SMARTCARD_SUCCESS);
    *lr = 2;

    return 0;
}



/*
 * SDAP_Read_Card
 * copies the content of the card memory into the memory image 
 * for all SDAP cards
 *  
 *
 * length = -1 && offset = 0x00 -> Read whole card
 * length = 0xff && offset = 0x10 -> Read 256 bytes at offset 0x10
 *
 */

int SDAP_Read_Card(struct eco5000_t *ctx, unsigned int offset, int length)

{
    struct memorycard_t *pt;
    unsigned char output[4];  
    unsigned char buffer[257];
    int rc = OK;
    int len;
    int newFirmware = FALSE;

    pt = ctx->Data.card;
    
    if(strcmp(ctx->Firmware, ECO_V223) >= 0)  /* Check for new Firmware */
      newFirmware = TRUE;

    if (length == -1) {
        length = pt->Number_of_Data_Units;
        offset = 0;
    }

    length += offset & 0x0F;            /* Make sure we start at 16 byte     */
    offset &= 0xFFF0;                   /* page because of ECO Firmware bug  */

    while (length > 0) {
        if (pt->Number_of_Data_Units > 2048) {
	    if (newFirmware == TRUE)  {
		output[0] = I2C_CHAR;
		output[1] = (unsigned char)(offset >> 8);
		output[2] = (unsigned char)(offset & 0xFF);
		output[3] = length > 256 ? 00 : length;
		rc = ecoCommand(ctx, READ_BLOCK_I2C_64K, 4, output, length > 256 ? 257 : length + 1, buffer);               
		len = length > 256 ? 256 : length;
	    }  
	    else {
		output[0] = I2C_CHAR;
		output[1] = (unsigned char)(offset >> 8);
		output[2] = (unsigned char)(offset & 0xFF);
		rc = ecoCommand(ctx, READ_16BYTE_I2C_64K, 3, output, 17, buffer);
		len = length > 16 ? 16 : length;
	    }
	   
	    memcpy(pt->Memory + offset, &buffer, len);
	    offset += len;
	    length -= len;
	    
        } else {
            output[0] = I2C_CHAR | ((unsigned char)(offset >> 7) & 0x0E);
            output[1] = (unsigned char)(offset & 0xFF);
            rc = ecoCommand(ctx, READ_16BYTE_I2C, 2, output, 17, buffer);
	    
	    len =  length > 16 ? 16 : length;
	    memcpy(pt->Memory + offset, &buffer, len);
	    offset += len;  
	    length -= len;
        }
        
        if (rc < 0)
            return rc;	
    }
    
    return rc;

}







