/*
 *  ---------
 * |.**> <**.|  CardContact
 * |*       *|  Software & System Consulting
 * |*       *|  Minden, Germany
 * |**> <**|  Copyright (c) 1999. All rights reserved
 *  --------- 
 *
 * See file LICENSE for details on licensing
 *
 * Abstract :       Defines tools/types for 2 wire bus memory card handling
 *
 * Author :         Frank Thater (FTH)
 *
 * Last modified:   04/25/2000
 *
 *****************************************************************************/

#include <string.h>

#include "mc_2wbp.h"
#include "defines.h"


int MC_2WBPInit(struct eco5000_t *ctx)

{
    struct memorycard_t *pt;
    int rc;
    unsigned char cmd[3];
      
    ctx->CTModFunc = (CTModFunc_t) MC_2WBPProcess;

    pt = (struct memorycard_t *)malloc(sizeof(struct memorycard_t));

    ctx->Data.card = pt;
 
    pt->SelectedFile = NONE;
    pt->Path = 0x00;
    pt->PINVerified = FALSE;

    pt->CardID = SLE4442;

    rc = DetermineVariables(ctx);
    
    if (rc < 0)
        return rc;

    cmd[0] = READ_MAIN_MEMORY;
    cmd[1] = 0x00;
    cmd[2] = 0x00;
    
    rc = MC2WBP_Command(ctx, cmd, pt->Memory);

    if (rc < 0)
        return rc;

    return OK;

}



int MC_2WBPProcess(struct eco5000_t *ctx,
                   unsigned int lc,
                   unsigned char *cmd,
                   unsigned int *lr,
                   unsigned char *rsp)

{
    int rc = OK;
    struct memorycard_t *pt;


#ifdef DEBUG
    printf("\nMC_2WBPProcess called...\n");
#endif

    pt = ctx->Data.card;

    /* Try calling the basic functions ... */

    rc = MemoryCardProcess(ctx, lc, cmd, lr, rsp);

    if(rc == OK)  /* command completed */
        return OK;
    
    if(rc < 0)    /* error processing command */
        return rc;


    /* "specialized" interindustry Commands */
    switch(cmd[1]) {
        case 0xA4:
            rc = MC2WBP_Select_File(ctx, lc, cmd, lr, rsp);
            break;
        case 0xB0:
            rc = MC2WBP_Read_Binary(ctx, lc, cmd, lr, rsp);
            break;
        case 0xD6:
            rc = MC2WBP_Update_Binary(ctx, lc, cmd, lr, rsp);
            break;
        case 0x20:
            rc = MC2WBP_Verify(ctx, lc, cmd, lr, rsp);
            break;
        case 0x24:
            rc = MC2WBP_Change_Verification_Data(ctx, lc, cmd, lr, rsp);
            break;
        default:
            *lr = 2;
            rsp[0] = HIGH(CLASS_NOT_SUPPORTED);
            rsp[1] = LOW(CLASS_NOT_SUPPORTED);
    }
  
    return rc;
}



int MC2WBP_Select_File(struct eco5000_t *ctx,
                       unsigned int lc,
                       unsigned char *cmd,
                       unsigned int *lr,
                       unsigned char *rsp)

{
    struct memorycard_t *pt = ctx->Data.card;
    unsigned int temp;
  
    temp = (cmd[5] << 8) | cmd[6];
    
    rsp[0] = HIGH(SMARTCARD_SUCCESS);
    rsp[1] = LOW(SMARTCARD_SUCCESS);
    *lr = 2;      
    
    switch(temp) {
        case PROTECTION_MEMORY:
            pt->SelectedFile = PROTECTION_MEMORY;
            pt->Path = 0x00;
            break;
        default:
            pt->SelectedFile = NONE;
            rsp[0] = HIGH(FILE_NOT_FOUND);
            rsp[1] = LOW(FILE_NOT_FOUND);
            *lr = 2;
    }

    return OK;

}





int MC2WBP_Read_Binary(struct eco5000_t *ctx,
                       unsigned int lc,
                       unsigned char *cmd,
                       unsigned int *lr,
                       unsigned char *rsp)

{
    unsigned char command[3];
    unsigned char buffer[256];
    int rc;
 
    struct memorycard_t *pt = ctx->Data.card;

    unsigned int exspected_bytes = cmd[4];
    unsigned int recieved_bytes = 0;


    
    switch(pt->SelectedFile) {
        case PROTECTION_MEMORY:
            exspected_bytes = 0x04; 
            break;
        default:
	    rsp[0] = HIGH(FILE_NOT_FOUND);
	    rsp[1] = LOW(FILE_NOT_FOUND);
	    *lr = 2;
            return OK;
    }

    command[0] = READ_PROTECTION_MEMORY;
    command[1] = 0x00;
    command[2] = 0x00;     
    
    rc = MC2WBP_Command(ctx, command, buffer);
    
    if (rc < 0)
        return rc;
    
    recieved_bytes = 4;
    if (exspected_bytes == 0x00)
        exspected_bytes = recieved_bytes;
    memcpy(rsp, &buffer, exspected_bytes);
    rsp[exspected_bytes] = recieved_bytes < exspected_bytes ? 0x62 : 0x90;
    rsp[exspected_bytes+1] = recieved_bytes < exspected_bytes ? 0x82 : 0x00;
    *lr = recieved_bytes < exspected_bytes ? recieved_bytes+2 : exspected_bytes+2;
    
    return OK;
}



int MC2WBP_Update_Binary(struct eco5000_t *ctx,
                         unsigned int lc,
                         unsigned char *cmd,
                         unsigned int *lr,
                         unsigned char *rsp)

{
    struct memorycard_t *pt = ctx->Data.card;
    unsigned char command[3];
    unsigned char address = 0x00;
    unsigned char length = cmd[4];
    unsigned char counter = 0x00;
    unsigned int offset;
    int rc;

    offset = cmd[2] << 8 | cmd[3];

    switch (pt->SelectedFile) {
        case MF:
            address = pt->Path + offset;
            while ((counter < length) && (counter < pt->Number_of_Data_Units)) {
                command[0] = UPDATE_MAIN_MEMORY;
                command[1] = address;
                command[2] = cmd[5 + counter];

                rc = MC2WBP_Command(ctx, command, NULL);

                if (rc < 0)
                    return rc;

                counter++;
                address++;
            }

            command[0] = READ_MAIN_MEMORY;
            command[1] = 0x00;
            command[2] = 0x00;

            rc = MC2WBP_Command(ctx, command, pt->Memory);

            if (rc < 0)
                return rc;

            /* Verify update process */
            if (memcmp(&cmd[5], &pt->Memory[pt->Path + offset], length) != 0) {
	        rsp[0] = HIGH(W_NO_CARD_PRESENTED);
                rsp[1] = LOW(W_NO_CARD_PRESENTED);
                *lr = 2;
                return OK;
            }   
            break;
        case PROTECTION_MEMORY:
            address = pt->Path + offset;
            while (counter <= length) {
                command[0] = WRITE_PROTECTION_MEMORY;
                command[1] = address;
                command[2] = cmd[5 + counter];
            
                rc = MC2WBP_Command(ctx, command, NULL);
            
                if (rc < 0)
                    return rc;
            
                counter++;
                address++;
            }
            break;
        default:
	    rsp[0] = HIGH(FILE_NOT_FOUND);
            rsp[1] = LOW(FILE_NOT_FOUND);
            *lr = 2;
            return OK;
    }

    rsp[0] = HIGH(SMARTCARD_SUCCESS);
    rsp[1] = LOW(SMARTCARD_SUCCESS);
    *lr = 2;

    return OK;
}



/*
 * Verify the Programmable Security Code (PSC) of the SLE4442/PCF3042
 *
 * The error counter in a SLE4442 is implemented as a bitmap with 3
 * bits. The number of tries left equals the number of bits set to
 * 1 in the bitmap. Because there is not defined way of implementing
 * the manipulation of the bitmap, the following algorithm is used
 * to switch off one bit:
 *
 * No try left          0 0 0 (0)
 * One try left         1 0 0 (4), 0 0 1 (1), 0 1 0 (2)
 * Two tries left       1 1 0 (6), 1 0 1 (5), 0 1 1 (3)
 * Three tries left     1 1 1 (7)
 *
 * so decec[] translates 7->6->4->0, 5->1->0, 3->2->0
 *
 */

int MC2WBP_Verify(struct eco5000_t *ctx,
                  unsigned int lc,
                  unsigned char *cmd,
                  unsigned int *lr,
                  unsigned char *rsp)

{
    int rc;
    unsigned char pin[3];
    unsigned char command[3];
    unsigned char buffer[256];
    unsigned char ec;
    unsigned char decec[] = { 0, 0, 0, 2, 0, 1, 4, 6 };
    struct memorycard_t *pt = ctx->Data.card;

    memcpy(&pin, cmd + 5, 0x03); /* Three Bytes of command must be pin !*/
  
    *lr = 0;

    /* Get error counter */

    command[0] = READ_SECURITY_MEMORY;
    command[1] = 0;
    command[2] = 0;

    rc = MC2WBP_Command(ctx, command, buffer);

    if (rc < 0)
        return rc;

    ec = buffer[0] & 0x07;                /* 3 LSB are bit counter             */
  
    if (ec == 0) {                         /* Nothing left in error counter     */
        rsp[0] = HIGH(VERIFICATION_METHOD_BLOCK);
        rsp[1] = LOW(VERIFICATION_METHOD_BLOCK);
        *lr = 2;
        return OK;
    }
  
    ec = decec[ec];                       /* Switch one bit off                */

    command[0] = UPDATE_SECURITY_MEMORY;
    command[1] = 0x00;
    command[2] = ec;

    rc = MC2WBP_Command(ctx, command, NULL);

    if (rc < 0)
        return rc;

    command[0] = COMPARE_VERIFICATION_DATA;
    command[1] = 0x01;
    command[2] = pin[0];

    rc = MC2WBP_Command(ctx, command, NULL);

    if (rc < 0)
        return rc;

    command[0] = COMPARE_VERIFICATION_DATA;
    command[1] = 0x02;
    command[2] = pin[1];

    rc = MC2WBP_Command(ctx, command, NULL);

    if (rc < 0)
        return rc;

    command[0] = COMPARE_VERIFICATION_DATA;
    command[1] = 0x03;
    command[2] = pin[2];

    rc = MC2WBP_Command(ctx, command, NULL);

    if (rc < 0)
        return rc;

    command[0] = UPDATE_SECURITY_MEMORY;
    command[1] = 0x00;
    command[2] = 0x07;

    rc = MC2WBP_Command(ctx, command, NULL);

    if (rc < 0)
        return rc;

    command[0] = READ_SECURITY_MEMORY;
    command[1] = 0x0;
    command[2] = 0x0;

    rc = MC2WBP_Command(ctx, command, buffer);

    if (rc < 0)
        return rc;

    ec = buffer[0] & 0x07;

    if (ec == 0x00) {     /* Card blocked */
        rsp[0] = HIGH(VERIFICATION_METHOD_BLOCK);
        rsp[1] = LOW(VERIFICATION_METHOD_BLOCK);
        *lr = 2;
        return OK;
    }

    if (ec != 0x07) {    /* wrong pin */
        rsp[0] = HIGH(VERIFICATION_UNSUCCESSFUL);
        rsp[1] = (ec == 0x06) ||
            (ec == 0x05) ||
            (ec == 0x03) ? 0xC2 : 0xC1;
        *lr = 2;
        return OK;
    }

    pt->PINVerified = TRUE;

    rsp[0] = HIGH(SMARTCARD_SUCCESS);
    rsp[1] = LOW(SMARTCARD_SUCCESS);
    *lr = 2;

    return OK;
}



/*
 * Change the current Programmable Security Code with a new code.
 * If the code was not previously verified, do it here.
 *
 */

int MC2WBP_Change_Verification_Data(struct eco5000_t *ctx,
                                    unsigned int lc,
                                    unsigned char *cmd,
                                    unsigned int *lr,
                                    unsigned char *rsp)

{
    struct memorycard_t *pt = ctx->Data.card;
    unsigned char newpin[3];
    unsigned char command[3];
    int rc;
#ifdef DEBUG
    int i = 0;
#endif

    memcpy(&newpin, cmd + 8, 3);

    if (pt->PINVerified != TRUE) {
        MC2WBP_Verify(ctx, lc - 3, cmd, lr, rsp);

        if (rsp[0] != 0x90) {
            return OK;
        }
    }

#ifdef DEBUG
    for(i=0;i<3;i++)
        printf("\n%02x", newpin[i]);
#endif
    *lr = 0;

    command[0] = UPDATE_SECURITY_MEMORY;
    command[1] = 0x01;
    command[2] = newpin[0];

    rc = MC2WBP_Command(ctx, command, NULL);

    if (rc < 0)
        return rc;

    command[0] = UPDATE_SECURITY_MEMORY;
    command[1] = 0x02;
    command[2] = newpin[1];

    rc = MC2WBP_Command(ctx, command, NULL);

    if (rc < 0)
        return rc;

    command[0] = UPDATE_SECURITY_MEMORY;
    command[1] = 0x03;
    command[2] = newpin[2];

    rc = MC2WBP_Command(ctx, command, NULL);

    if (rc < 0)
        return rc;
  
    rsp[0] = HIGH(SMARTCARD_SUCCESS);
    rsp[1] = LOW(SMARTCARD_SUCCESS);
    
    * lr = 2;

    return OK;
}



/*
 * MC2WBP_Command
 *
 * Send a command to a 2 wire bus card and either
 * read the response (rsp != NULL) or clock the chip until
 * command completes
 *
 */

int MC2WBP_Command(struct eco5000_t *ctx,
                   unsigned char *cmd2wbp,
                   unsigned char *rsp)

{
    int rc;
    unsigned char ack;

    rc = ecoCommand(ctx, COMMAND_4442, 3, cmd2wbp, 0, NULL);

    if (rc != ACK_CVCC_ON) {
#ifdef DEBUG
        printf("[MC2WBP_Command] Sending command rc=%d [%02x]\n", rc, rc);
#endif
        return ERR_TRANS;
    }

    if (rsp)
        rc = ecoCommand(ctx, READ_4442, 0, NULL, 256, rsp);
    else
        rc = ecoCommand(ctx, CLOCK_4442, 0, NULL, 1, &ack);

    if (rc != ACK_CVCC_ON) {
#ifdef DEBUG
        printf("[MC2WBP_Command] Getting response rc=%d [%02x]\n", rc, rc);
#endif
        return ERR_TRANS;
    }

    return 0;
}









