import java.io.*;


/**
 * Generic image parser.
 *
 * @version $Id: ImageParser.java,v 1.10 2002/09/22 23:13:29 blsecres Exp $
 * @author Ben Secrest &lt;blsecres@users.sourceforge.net&gt;
 */
public class ImageParser {
    /** Determines if this parser will search for description */
    private boolean wantDescription;

    /** Determines if the parser will provide file type */
    private boolean wantFileType;

    /** Determines if the parser will provide parser information */
    private boolean wantParser;

    /** Search string for basic text comments */
    private byte[] findSegment = null;

    /** The number of bytes used to store a comments length */
    private int numLengthBytes;

    /** Set if the length bytes include themselves in the total length */
    private boolean sizeIncludesLength;

    /**
     * Size for input buffers.  Taken from /usr/include/stdio.h#BUFSIZ
     */
    private static final int inputBufferSize = 8196;

    /** The logging object for this module */
    private IGLog log;

    /** The default logging level for this module */
    protected final static int LOGLEVEL = 9;


    /**
     * Constructor for a generic image parser
     * @param logObj The object used for logging data
     * @param extract The set of desired fields to extract
     * @param searchBytes The set of bytes that denote the beginning of a
     *		comment
     * @param lengthSize The number of bytes used to store the comment length
     * @param sizeInLen Set to true if the bytes used to indicate the comment
     *		length are accounted for in the length
     */
    public ImageParser(byte[] searchBytes, int lengthSize, boolean sizeInLen) {
	log = null;
	wantDescription = wantFileType = wantParser = false;
	findSegment = searchBytes;
	numLengthBytes = lengthSize;
	sizeIncludesLength = sizeInLen;
    }


    /**
     * Set the desired attributes to extract
     * @param wanted A set of bits describing preferences
     */
    public void setWantedItems(IGKeySet wanted) {
	wantDescription = wanted.wants(IGKey.DESCRIPTION);
	wantFileType = wanted.wants(IGKey.FILE_TYPE);
	wantParser = wanted.wants(IGKey.PARSER);
    }


    /**
     * Set the logger to use with this parser
     * @param logObj The object to use for logging data
     */
    public void setLog(IGLog logObj) {
	log = logObj;
    }


    /**
     * Parse the given file using the required search bytes
     * @param file The file to open and structure to fill in
     * @throws IOException if an error occurs reading file
     * @throws FileNotFoundException if the file given to be parsed does not
     *	exist
     */
    public void parse(IGFile file) throws IOException, FileNotFoundException {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "ImageParser.parse(IGFile)");

	try {
	    parse(file, new FileInputStream(file.getLocation()));
	} catch (StreamResetException sre) {
	    throw new IOException("Stream Reset Exception shouldn't happen");
	}
    }


    /**
     * Parse the opened input stream
     * @param file The IGFile structure to fill in
     * @param stream The input data stream
     * @throws IOException if an error occurs reading data
     */
    public void parse(IGFile file, InputStream stream)
	    throws IOException, StreamResetException {
	if (log == null)
	    // FIXME
	    return;

	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "ImageParser.parse(IGFile, InputStream, "
		    + "String)");

	if (LOGLEVEL >= IGLog.FILE)
	    log.addResource(IGLog.FILE, "PROCESS_FILE",
		    new String[]{file.getLocation()});

	/*
	 * reset attribute strings
	 * if not searching for an item, assign it a value so the search can
	 * kick out when all items != null
	 */
	String description = (wantDescription ? null : "");

	boolean readLength = false, readText = false;
	byte[] buffer = new byte[inputBufferSize];
	int bytesRead, numTextBytes = 0, numSizeBytes = 0;
	int findPosition = 0;
	String curValue = "";

	if (LOGLEVEL >= IGLog.SECTION)
	    log.addResource(IGLog.SECTION, "FP_BEGIN_PARSE",
			    new String[]{log.getString("FP_IMAGE")});

	while (description == null && (bytesRead = stream.read(buffer)) != -1) {
	    for (int i = 0; description == null && i < bytesRead; i++) {
		if (readLength) {
		    numTextBytes = numTextBytes * (1 << (numSizeBytes * 4))
				   + buffer[i];
		    numSizeBytes--;

		    if (numSizeBytes == 0) {
			readLength = false;
			readText = true;
			if (sizeIncludesLength)
			    numTextBytes -= numLengthBytes;
		    }
		} else if (readText) {
		    if (numTextBytes > 0) {
			curValue += new Character((char) buffer[i]);
			numTextBytes--;
		    } else {
			description = curValue;
			readText = false;
			if (LOGLEVEL >= IGLog.PROGRESS)
			    log.addResource(IGLog.PROGRESS, "FP_FOUND_DESC",
				    new String[]{description});
		    }
		} else if (findSegment[findPosition] == buffer[i]) {
		    findPosition++;
		} else {
		    findPosition = 0;
		}

		if (findPosition == findSegment.length) {
		    numSizeBytes = numLengthBytes;
		    numTextBytes = 0;
		    readLength = true;
		    findPosition = 0;
		}
	    }
	}

	if (LOGLEVEL >= IGLog.SECTION)
	    log.addResource(IGLog.SECTION, "FP_FINISH_PARSE",
		    new String[]{log.getString("FP_IMAGE")});

	if (wantDescription)
	    file.put(IGKey.DESCRIPTION, description);
	// this is a hack, but...
	if (wantFileType)
	    file.put(IGKey.FILE_TYPE, (sizeIncludesLength ? new String("JPEG")
			: new String("GIF")));
	if (wantParser)
	    file.put(IGKey.PARSER, getClass().getName());

	stream.close();
    }
}
