package Language::INTERCAL::Optimiser;

# Oprimiser for CLC-INTERCAL

# This file is part of CLC-INTERCAL.

# Copyright (C) 1999 Claudio Calvelli <lunatic@assurdo.com>, all rights reserved

# WARNING - do not operate heavy machinery while using CLC-INTERCAL

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

use vars qw($VERSION);
$VERSION = '0.05';

use Language::INTERCAL::Opcodes;
use Language::INTERCAL::Runtime::Library;

sub _is_plainreg {
    $_[0][0] == E_REGISTER;
}

sub _is_const {
    $_[0][0] == E_CONSTANT;
}

sub first {
    my ($ptree, $filename, $line) = @_;
    # I've temporarily moved everything to the second optimiser until I
    # convince myself of what is safe to run before seeing the whole program
    $ptree;
}

sub second {
    my ($ptree, $filename, $line) = @_;
    my $quantum = exists $ptree->{'flags'}{'quantum'};
    my $ignore = exists $ptree->{'flags'}{'ignore'};
    my $postprocess = exists $ptree->{'flags'}{'postprocess'};
    my $overload = exists $ptree->{'flags'}{'overload'};
    my $qip = $quantum || $postprocess || $ignore;
    my $abstain = $ptree->{'abstain'};
    my $reinstate = $ptree->{'reinstate'};
    my $gabstain = $ptree->{'gabstain'};
    my $greinstate = $ptree->{'greinstate'};
    my $labels = $ptree->{'labels'};
    my $abstain_calculate = 0;

    $ptree->iterate(sub {
	my ($p, $fid, $bid, $sid, $stmt) = @_;
	if ($stmt->[3][0] == S_GABSTAIN || $stmt->[3][0] == S_REINSTATE) {
	    $abstain_calculate = 1 if grep {$_ == S_ASSIGN} @{$stmt->[3]};
	}
    });

    my %regcon = ();
    my $code;
    $code = sub {
	my ($p, $fid, $bid, $sid, $stmt) = @_;
	my $bc = $stmt->[3][0];
	if ($bc == S_ABSTAIN ||
	    $bc == S_COME ||
	    $bc == S_REINSTATE ||
	    $bc == S_FORGET ||
	    $bc == S_NEXT ||
	    $bc == S_RESUME)
	{
	    do_expression($stmt->[3], 1, \%regcon, $quantum);
	}
	if ($bc == S_WHILE_E) {
	    do_expression($stmt->[3], 2, \%regcon, $quantum);
	}
	if ($bc == S_WHILE_BC || $bc == S_WHILE_CB) {
	    %regcon = ();
	    &$code($p, -1, 0, 0, [1, 100, 0, $stmt->[3][1]]);
	    %regcon = ();
	    &$code($p, -1, 0, 0, [1, 100, 0, $stmt->[3][2]]);
	    %regcon = ();
	}
	if ($bc == S_WRITE) {
	    my $rname = $stmt->[3][1][1] . $stmt->[3][1][2];
	    delete $regcon{$rname};
	}
	if ($bc == S_READ) {
	    my $rname = $stmt->[3][1][1] . $stmt->[3][1][2];
	    $regcon{$rname}[3] = 0 if exists $regcon{$rname};
	}
	if ($bc == S_ASSIGN) {
	    my $i;
	    my $s = $stmt->[3];
	    for $i (2..$#{$s}) {
		do_expression($s, $i, \%regcon, $quantum);
	    }
	    # the next condition errs on the side of correctness, it
	    # would be OK to say "$s->[1] cannot be overloaded" instead
	    # of "the program does not contain overloading". Same for
	    # ignore. Perhaps building a list of registers which appear
	    # in overload/ignore... make a note for another release.
	    # we also need to check for ABSTAIN FROM CALCULATING
	    # (ABSTAIN FROM (label) does not matter because we never join
	    # together statements if one has a label)
	    if (! $qip && ! $overload && _is_plainreg($s->[1])) {
		my $rname = $s->[1][1] . $s->[1][2];
		if (ref $stmt->[2] || $stmt->[2] < 100 || $abstain_calculate) {
		    delete $regcon{$rname};
		} else {
		    if (exists $regcon{$rname} && $regcon{$rname}[3]) {
			_remove($ptree, $regcon{$rname}[1], $regcon{$rname}[2]);
		    }
		    if (@$s == 3 && _is_const($s->[2])) {
			my $v = $s->[2][1];
			$v = $s->[1][1] eq ':' ? ato32($v) : ato16($v);
			$regcon{$rname} = [$v, $fid, $stmt, 1];
		    } else {
			delete $regcon{$rname};
		    }
		}
	    }
	}
	if ($bc == S_RETRIEVE || $bc == S_STASH) {
	    my $i;
	    my $s = $stmt->[3];
	    for $i (1..$#{$s}) {
		if (_is_plainreg($s->[$i])) {
		    # this errs on the side of correctness - we could
		    # easily build a stack within $regcon{$rname} and
		    # push it at stash, pop at retrieve. Maybe in a
		    # future version.
		    my $rname = $s->[$i][1] . $s->[$i][2];
		    delete $regcon{$rname};
		}
	    }
	}
	if ($bc == S_LEARN) {
	    do_expression($stmt->[3], 2, \%regcon, $quantum);
	}
	if ($bc == S_STUDY) {
	    do_expression($stmt->[3], 1, \%regcon, $quantum);
	    do_expression($stmt->[3], 2, \%regcon, $quantum);
	    do_expression($stmt->[3], 3, \%regcon, $quantum);
	}
	if ($bc == S_ENROL) {
	    my $i;
	    for $i (2..$#{$stmt->[3]}) {
		do_expression($stmt->[3], $i, \%regcon, $quantum);
	    }
	}
    };
    $ptree->iterate($code, sub { %regcon = (); });

    my $stop = 0;
    my $first = 1;
    $ptree->iterate(sub {
	my ($p, $fid, $bid, $sid, $stmt) = @_;
	my $bc = $stmt->[3][0];

	# anything following a stop can be removed
	_remove($ptree, $fid, $stmt) if $stop;
	$stop = 1 if $stmt->[1] &&
		     (! ref $stmt->[2]) && $stmt->[2] == 100 &&
		     $bc == S_STOP &&
		     ! $postprocess;

	# if we are in the first block, we can execute any ABSTAINs etc
	if ($first && $bid == 1) {
	    if (($bc == S_ABSTAIN || $bc == S_REINSTATE) &&
		(! ref $stmt->[2]) && $stmt->[2] == 100 &&
		$stmt->[1] &&
		_is_const($stmt->[3][1]) && exists $labels->{$stmt->[3][1][1]})
	    {
		my $l = $labels->{$stmt->[3][1][1]};
		$ptree->{'files'}[$l->[0]][$l->[1]][$l->[2]][1] =
		    $bc == S_ABSTAIN ? 0 : 1;
		_remove($ptree, $fid, $stmt);
	    }

	    if (($bc == S_GABSTAIN || $bc == S_GREINSTATE) &&
		(! ref $stmt->[2]) && $stmt->[2] == 100 &&
		$stmt->[1])
	    {
		my $ab = $bc == S_GABSTAIN ? 0 : 1;
		my @s = @{$stmt->[3]};
		shift @s;
		$ptree->iterate(sub {
		    my ($p1, $fid1, $bid1, $sid1, $stmt1) = @_;
		    if (grep {$_ == $stmt1->[3][0]} @s) {
			$stmt1->[1] = $ab;
		    }
		});
		_remove($ptree, $fid, $stmt);
	    }

	    if ($bc == S_CONVERT || $bc == S_SWAP) {
		if ((! ref $stmt->[2]) && $stmt->[2] == 100 && $stmt->[1]) {
		    my $code1 = $stmt->[3][1][0];
		    my $code2 = $stmt->[3][2][0];

		    if ($code1 != $code2) {
			$ptree->iterate(sub {
			    my ($p1, $fid1, $bid1, $sid1, $stmt1) = @_;
			    if ($fid1 > $fid ||
				($fid1 == $fid && $bid1 > $bid) ||
				($fid1 == $fid &&
				 $bid1 == $bid &&
				 $sid1 > $sid))
			    {
				my $code = $stmt1->[3][0];
				if ($code == S_SWAP || $code == S_CONVERT) {
				    for (my $i = 1; $i < @{$stmt1->[3]}; $i++) {
					my $c = $stmt1->[3][$i][0];
					if ($c == $code1) {
					    $stmt1->[3][$i][0] = $code2;
					}
					if ($c == $code2 && $bc == S_SWAP) {
					    $stmt1->[3][$i][0] = $code1;
					}
				    }
				}
				if ($code == $code1) {
				    _convert($ptree, $fid1, $bid1, $sid1,
					     $stmt1, $code2);
				}
				if ($code == $code2 && $bc == S_SWAP) {
				    _convert($ptree, $fid1, $bid1, $sid1,
					     $stmt1, $code1);
				}
			    }
			});
		    }
		    _remove($ptree, $fid, $stmt);
		} else {
		    $first = 0;
		}
	    }
	}

	# a statement initially abstained from -- if there is no
	# way to reinstate it, away it goes...
	if (! $stmt->[1] && ($bc == S_STOP || ! $quantum && ! $postprocess)) {
	    my $ok = 1;
	    if ($bc != S_STOP) {
		if ($stmt->[0]) {
		    my $s;
		    for $s (@$reinstate) {
			if (_is_const($s->[0])) {
			    $ok = 0 if $stmt->[0] == atoi($s->[0][1]);
			} else {
			    $ok = 0;
			}
		    }
		}
		if ($ok) {
		    my $s;
		    for $s (@$greinstate) {
			my @s = @{$s->[0]};
			shift @s;
			$ok = 0 if grep {$_ == $bc} @s;
		    }
		}
	    }
	    _remove($ptree, $fid, $stmt) if $ok;
	}
    }, sub {
	$stop = 0;
    }, sub {
	my ($p, $fid, $bid, $sid, $stmt) = @_;
	$first = $first && $stop if $bid == 1;
    });
    $ptree;
}

sub do_expression {
    my ($stmt, $place, $regcon, $quantum) = @_;
    my $e = $stmt->[$place];
    my $ec = $e->[0];
    if ($ec == E_AND || $ec == E_OR || $ec == E_XOR) {
	do_expression($e, 1, $regcon, $quantum);
	if (_is_const($e->[1])) {
	    my $val = $e->[1][1];
	    $val = i_and($val) if $ec == E_AND;
	    $val = i_or($val) if $ec == E_OR;
	    $val = i_xor($val) if $ec == E_XOR;
	    $stmt->[$place] = [E_CONSTANT, $val];
	}
	return;
    }
    if ($ec == E_INTERLEAVE || $ec == E_SELECT || $ec == E_OVERLOAD_RANGE) {
	do_expression($e, 1, $regcon, $quantum);
	do_expression($e, 2, $regcon, $quantum);
	if (_is_const($e->[1]) && _is_const($e->[2])) {
	    my $val;
	    if ($ec == E_INTERLEAVE) {
		$val = i_interleave($e->[1][1], $e->[2][1]);
	    } elsif ($ec == E_SELECT) {
		$val = i_select($e->[1][1], $e->[2][1]);
	    }
	    $stmt->[$place] = [E_CONSTANT, $val];
	} elsif ($ec == E_SELECT &&
		 ($e->[1][0] == E_AND ||
		  $e->[1][0] == E_OR ||
		  $e->[1][0] == E_XOR) &&
		 $e->[1][1][0] == E_INTERLEAVE &&
		 $e->[2][0] == E_CONSTANT &&
		 atoi($e->[2][1]) == 0x55555555)
	{
	    # This is a "binary" logical operator in disguise
	    my $op = $e->[1][0] == E_AND ? E_BAND
					 : $e->[1][0] == E_OR ? E_BOR
							      : E_BXOR;
	    $stmt->[$place] = [$op, $e->[1][1][1], $e->[1][1][2]];
	}
	return;
    }
    if ($ec == E_SUBSCRIPT) {
	my $i;
	for $i (2..$#$e) {
	    do_expression($e, $i, $regcon, $quantum);
	}
	return;
    }
    if ($ec == E_OVERLOAD_REGISTER) {
	do_expression($e, 2, {}, $quantum);
	return;
    }
#    if ($ec == E_OWNER) {
#	do_expression($e, 2, $regcon, $quantum);
#	return;
#    }
    if ($ec == E_REGISTER && exists $regcon->{$e->[1] . $e->[2]}) {
	$stmt->[$place] = [E_CONSTANT, $regcon->{$e->[1] . $e->[2]}[0]];
    }
}

sub _remove {
    my ($ptree, $fid, $stmt) = @_;
    my $code = $stmt->[3][0];
    $stmt->[3][0] = S_NOP;
    my $id = '';
    $id = 'come_froms' if $code == S_COME;
    $id = 'abstain' if $code == S_ABSTAIN;
    $id = 'gabstain' if $code == S_GABSTAIN;
    $id = 'reinstate' if $code == S_REINSTATE;
    $id = 'greinstate' if $code == S_GREINSTATE;
    return $ptree if $id eq '';
    my $cid;
    my $c = $ptree->{$id};
    for ($cid = 0; $cid < @$c; $cid++) {
	if ($c->[$cid][1] == $fid && $c->[$cid][4] == $stmt->[4]) {
	    splice(@$c, $cid, 1);
	    last;
	}
    }
    $ptree;
}

sub _add {
    my ($ptree, $fid, $bid, $sid, $stmt, $code) = @_;
    $stmt->[3][0] = $code;
    my $id = '';
    $id = 'come_froms' if $code == S_COME;
    $id = 'abstain' if $code == S_ABSTAIN;
    $id = 'gabstain' if $code == S_GABSTAIN;
    $id = 'reinstate' if $code == S_REINSTATE;
    $id = 'greinstate' if $code == S_GREINSTATE;
    return $ptree if $id eq '';
    my $cid;
    my $c = $ptree->{$id};
    my $l = $stmt->[3][1];
    $l = $l->[1] if $l->[0] == E_CONSTANT;
    push @$c, [$stmt->[3][1], $fid, $bid, $sid, $stmt->[4]];
    $ptree;
}

sub _convert {
    my ($ptree, $fid, $bid, $sid, $stmt, $newcode) = @_;
    _remove($ptree, $fid, $stmt);
    _add($ptree, $fid, $bid, $sid, $stmt, $newcode);
}

1;
