;;;
;;; GLIMAGE-GLX.LISP
;;;
;;; Put up image using GL routines.
;;;
;;;
;;; TODO:
;;;
;;; Experiment with various visuals and colormaps.
;;;
;;; Note: This program shows how to select visual and setup window
;;; using XCreateSimpleWindow.  I'm not sure if it can do double buffer
;;; or not.  (See glspin-xlib.lisp for an example of double buffering with
;;; Xlib.)
;;;
;;; Richard Mann
;;; 30 November 1996
;;;

(require 'pnm)

(defvar *image* nil)

(defun redraw (display window)
 (format t "Redrawing~%")
 (gl:glClear gl:GL_COLOR_BUFFER_BIT)
 (gl:glRasterPos2i 0 0)
 ;; Put image up.  PGM is array of Gray bytes; PPM is array of RGB bytes.
 (if (pnm:pgm? *image*)
     (gl:glDrawPixels (pnm:pgm-width *image*) (pnm:pgm-height *image*) 
                      GL:GL_LUMINANCE GL:GL_UNSIGNED_BYTE 
                      (pnm:pgm-xels *image*))
     (gl:glDrawPixels (pnm:ppm-width *image*) (pnm:ppm-height *image*) 
                      GL:GL_RGB GL:GL_UNSIGNED_BYTE 
                      (pnm:ppm-xels *image*)))
 (gl:glxswapbuffers display window))

;;; Always keeps 2d geometry fullsize in window
;;; Does not seem to work for images?
(defun resize (width height)
 (format t "Resizing~%")
 (gl:glviewport 0 0 width height)
 (gl:glmatrixmode gl:GL_PROJECTION)
 (gl:glloadidentity)
 (gl:glortho 0d0 (coerce (pnm:pgm-width *image*) 'double-float)
             0d0 (coerce (pnm:pgm-height *image*) 'double-float) -1d0 1d0)
 (gl:glmatrixmode gl:GL_MODELVIEW))

;;;
;;; Event loop:
;;; 1. "gobble" redraw events to prevent lag/queue overflow.
;;; 2. would have to check other events if there were more than one window
;;;    active...
;;; It seems that a single Resize occurs only once resize done.
;;; For some reason, several Expose occur during resize.  (Need to gobble).
;;; Also, since redraw may be slow, need to gobble.
;;;
(defun event-loop (display)
 ;; Event loop
 (let ((debug t)
       (event (xlib:make-xevent))) ; pointer to struct
  (when debug (format t "Event-loop.~%"))
  (loop
   ;; Wait for next event
   (when debug (format t "Wait..."))
   (xlib:xnextevent display event)
   (let ((event-type (xlib:xanyevent-type event))
	 (window (xlib:xanyevent-window event)))
    (when debug (format t "event-type:~a~%" event-type))
    (cond
      ;; Return on buttonpress event.
      ((eq event-type xlib:buttonpress) (return))
      ;; Expose
      ((eq event-type xlib:expose)
       ;; Gobble all other expose events
       (loop
	(unless (> (xlib:xeventsqueued display xlib:queuedafterflush) 0)
	 (return))
	(xlib:xnextevent display event)
	(let ((event-type (xlib:xanyevent-type event)))
	 (unless (eq event-type xlib:expose)
	  (xlib:xputbackevent display event)
	  (return))
	 (when debug (format t "Gobble event-type:~a~%" event-type))))
       (redraw display window))
      ;; Resize
      ((eq event-type xlib:configurenotify)
       (resize (xlib:xconfigureevent-width event)
	       (xlib:xconfigureevent-height event))))))))


;;;
;;; This call binds GL to an existing X window.
;;; This is useful if other program made the window and you just want
;;; to bind to it.
;;; Of course, the image dithering will depend on the type (and colormap)
;;; of the current window...
;;;

(defun bind-gl-to-window (display screen window)
 (let ((debug t))
  (when debug (format t "BIND-GL-TO-WINDOW.~%"))
  ;;
  (when debug (format t "XGetWindowAttributes..."))
  (let* ((attr (xlib:make-xwindowattributes))
	 (foo (xlib:xgetwindowattributes display window attr)) ; for-effects
	 (class (xlib:xwindowattributes-class attr))
	 (depth (xlib:xwindowattributes-depth attr))
	 (visual (xlib:xwindowattributes-visual attr))
	 (visual-class (xlib:visual-class visual)))
    ;;
    (when debug
      (format t "screen:~a, " screen)
      (format t "class:~a, depth:~a, " class depth)
      (format t "visual-class:~a~%" visual-class))
    ;;
    (when debug (format t "XMatchVisualInfo..."))
    (let* ((visualinfo (xlib:make-xvisualinfo))
	   (num-visuals (xlib:xmatchvisualinfo display screen depth
					       visual-class visualinfo)))
      (unless (> num-visuals 0)
        (error "BIND-GL-TO-WINDOW: Could not get visual of class:~a, depth~a!"
	       visual-class depth))
      (when debug (format t "~a visuals found.~%" num-visuals))
      ;;
      (when debug (format t "glXCreateContext..."))
      (let ((glx-context (gl:glxcreatecontext display visualinfo
					      xlib:NULL gl:GL_TRUE)))
	(when debug (format t "~%glXMakeCurrent..."))
	(gl:glxmakecurrent display window glx-context))))
  ;;
  (when debug (format t "~%Done.~%"))))

;;;
;;; An example of how to create a simple window (that you can bind GL to
;;; later).
;;; This window will be of same visual class as the parent (default) window.
;;;

(defun create-gl-simple-window (display width height)
 (let* ((screen (xlib:xdefaultscreen display))
        (root (xlib:xrootwindow display screen))
        (black-pixel (xlib:xblackpixel display screen))
        (white-pixel (xlib:xwhitepixel display screen))
        (window (xlib:xcreatesimplewindow display root 0 0 width height
                                          1 white-pixel black-pixel)))
  ;; Enable events
  (xlib:xselectinput display window
		     (+ xlib:structurenotifymask
			xlib:exposuremask
			xlib:buttonpressmask))
  ;; Bind to GL
  (bind-gl-to-window display screen window)
  ;; Map window
  (xlib:xmapwindow display window)
  ;; Return window
  window))

;;;
;;; Open a GL window, either for color or grey image.  Window type selected
;;; depends on color depth:
;;;
;;; xlib:TrueColor for pgm and ppm on 16/24 bit display
;;; xlib:PseudoColor for ppm (color) on 8 bit display
;;; xlib:GrayScale for pgm (grey) on 8 bit display
;;; xlib:StaticGray for pgm and ppm images on 1 bit display (seems to dither)
;;;
;;; Returns window, or else fails (eg., if visual type is not supported
;;; on the display).
;;;

(defun create-gl-window (display color? width height)
 (let* ((debug t)
	(screen (xlib:xdefaultscreen display))
	(root (xlib:xrootwindow display screen))
	(depth (xlib:xdefaultdepthofscreen
		(xlib:xdefaultscreenofdisplay display))))
  ;;
  (when debug
   (format t "CREATE-GL-WINDOW~%")
   (format t "Default color depth:~a~%" depth))
  ;;
  (let* ((vinfo (xlib:make-xvisualinfo))
	 (visual-class
	  (cond ((> depth 8) xlib:TrueColor)
		((and color? (= depth 8)) xlib:PseudoColor)
		((and (not color?) (= depth 8)) xlib:GrayScale)
		(t xlib:StaticGray)))
	 (num-visuals (xlib:xmatchvisualinfo display screen depth
					     visual-class vinfo)))
   (unless (> num-visuals 0)
    (error "CREATE-GL-WINDOW: Could not get visual of class:~a, depth~a!"
	   visual-class depth))
   (when debug
    (format t "~a visuals found.~%" num-visuals)
    (format t "depth:~a~%" (xlib:xvisualinfo-depth vinfo))
    (format t "class:~a~%" (xlib:xvisualinfo-class vinfo))
    (format t "colormap_size:~a~%" (xlib:xvisualinfo-colormap_size vinfo))
    (format t "bits_per_rgb:~a~%" (xlib:xvisualinfo-bits_per_rgb vinfo)))
   ;;
   ;; Once we get visual, use XCreateWindow to make window...
   ;; This is more complicated than XCreateSimpleWindow.
   (let ((attr (xlib:make-xsetwindowattributes))
	 (window nil)) ; window to return
    (xlib:set-xsetwindowattributes-background_pixel! attr 0)
    (xlib:set-xsetwindowattributes-border_pixel! attr 0)
    (xlib:set-xsetwindowattributes-colormap!
     attr (xlib:xcreatecolormap display root (xlib:xvisualinfo-visual vinfo)
				xlib:allocnone))
    ;; Controls events that window reports
    (xlib:set-xsetwindowattributes-event_mask!
     attr (+ xlib:StructureNotifyMask xlib:ExposureMask
	     xlib:ButtonPressMask))
    (setq window (xlib:xcreatewindow display root 0 0 width height 0
				     (xlib:xvisualinfo-depth vinfo)
				     xlib:InputOutput
				     (xlib:xvisualinfo-visual vinfo)
				     (+ xlib:CWBackPixel xlib:CWBorderPixel
					xlib:CWColormap xlib:CWEventMask)
				     attr))
    (when debug (format t "XCreateWindow:~a~%" window))
    ;;
    ;; Bind to GL
    (let ((glx-context (gl:glxcreatecontext display vinfo
					    xlib:NULL gl:GL_TRUE)))
     (when debug (format t "glXCreateContext:~a~%" glx-context))
     (gl:glxmakecurrent display window glx-context))
    
    ;;
    ;; Map window and return
    (xlib:xmapwindow display window)
    (when debug (format t "XMapWindow.~%"))
    window))))

  
(defun main ()
 (format t "GLIMAGE.~%")
 ;;
 (format t "Input image name:")
 (let* ((pathname (read-line *standard-input*))
	(color? nil)
	(width nil)
	(height nil))
  (setq *image* (pnm:read-pnm pathname)))
 (cond ((pnm:ppm? *image*)
	(setq color? t)
	(setq width (pnm:ppm-width *image*))
	(setq height (pnm:ppm-height *image*))
	(pnm:ppm-flip! *image*))
       ((pnm:pgm? *image*)
	(setq color? nil)
	(setq width (pnm:pgm-width *image*))
	(setq height (pnm:pgm-height *image*))
	(pnm:pgm-flip! *image*)))
 ;;
 (let* ((display (xlib:xopendisplay ""))
	(window (create-gl-window display color? width height)))
  ;;
  (gl:glClearIndex 0.0)
  (gl:glShadeModel gl:GL_FLAT)
  (gl:glclearcolor 0.0 0.0 0.0 1.0)
  (event-loop display)))
