(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** OCamldoc generator handling command custom tags *)

open Odoc_info
module Naming = Odoc_html.Naming
open Odoc_info.Value
open Odoc_info.Module

let opt = Odoc_info.apply_opt

let tables_only = ref false

class gen () =
  object (self)
    inherit Odoc_html.html as html

    method generate_for_module pre post m =
      try
	Odoc_info.verbose ("Generate for module "^m.m_name);
	let (html_file, _) = Naming.html_files m.m_name in
	let chanout = open_out (Filename.concat !Odoc_args.target_dir html_file) in
	let pre_name = opt (fun m -> m.m_name) pre in
	let post_name = opt (fun m -> m.m_name) post in
	if not !tables_only then
	   output_string chanout 
	    ("<html>\n"^
	     (self#header 
		~nav: (Some (pre_name, post_name, m.m_name))
		~comments: (Module.module_comments m)
		(self#inner_title m.m_name)
	     )
	    )
	else
	  output_string chanout 
	    (self#header 
	       ~nav: None
	       ~comments: []
	       "");

	if not !tables_only then
	  (
	   output_string chanout  "<body>\n";
	   output_string chanout
	     ((self#navbar pre_name post_name m.m_name)^
	      "<center><h1>Plug-in"^
	      " "^m.m_name^
	      "</h1></center>\n"^
	      "<br>\n"^
	      (match m.m_info with
		None -> ""
	      | Some i -> 
		  match i.i_desc with
		    None -> ""
		  | Some desc -> self#html_of_text desc
	      )
	     );
	   output_string chanout "<hr width=\"100%\">\n"
	  );
	let vals = Odoc_info.Search.values [m] in
	let mets = Odoc_info.Search.methods [m] in
	let rec iter tag f_info = function
	    [] -> []
	  | v :: q ->
	      match f_info v with
		None -> iter tag f_info q
	      |	Some i ->
		  try
		    let t = List.assoc tag i.i_custom in
		    (t, i.i_desc) :: (iter tag f_info q)
		  with
		    Not_found ->
		      iter tag f_info q
	in
	let commands_pre =
	  (iter "command" (fun v -> v.val_info) vals) @
	  (iter "command" (fun m -> m.met_value.val_info) mets)
	in
	let commands = 
	  List.sort 
	    (fun (t1,_) (t2,_) -> 
	      compare (Odoc_info.string_of_text t1) (Odoc_info.string_of_text t2))
	    commands_pre
	in	    
	(
	 match commands with
	   [] ->
	     ()
	 | l ->
	     let cpt = ref 0 in
	     output_string chanout
	       "<h2>Commands</h2><table class=\"commands\"><tr class=\"table_header\"><td>Command</td><td>Description</td></tr>\n";
	     List.iter
	       (fun (t, d) ->
		 output_string chanout
		   (Printf.sprintf "<tr class=\"%s\"><td>%s</td><td>%s</td></tr>\n"
		      (if !cpt = 0 then "even" else "odd")
		      (self#html_of_text t)
		      (match d with
			None -> ""
		      |	Some desc -> self#html_of_text desc));
		 cpt := (!cpt + 1) mod 2
	       )
	       commands;
	     output_string chanout "</table>\n";
	);

	let editors_pre =
	  (iter "editor" (fun v -> v.val_info) vals) @
	  (iter "editor" (fun m -> m.met_value.val_info) mets)
	in
	let editors = 
	  List.sort 
	    (fun (t1,_) (t2,_) -> 
	      compare (Odoc_info.string_of_text t1) (Odoc_info.string_of_text t2))
	    editors_pre
	in	    
	(
	 match editors with
	   [] ->
	     ()
	 | l ->
	     let cpt = ref 0 in
	     output_string chanout
	       "<h2>Editors</h2><table class=\"commands\"><tr class=\"table_header\"><td>Editor</td><td>Description</td></tr>\n";
	     List.iter
	       (fun (t, d) ->
		 output_string chanout
		   (Printf.sprintf "<tr class=\"%s\"><td>%s</td><td>%s</td></tr>\n"
		      (if !cpt = 0 then "even" else "odd")
		      (self#html_of_text t)
		      (match d with
			None -> ""
		      |	Some desc -> self#html_of_text desc));
		 cpt := (!cpt + 1) mod 2
	       )
	       editors;
	     output_string chanout "</table>\n";
	);
	
	if not !tables_only then 
	  output_string chanout "</body></html>";
	close_out chanout
	
      with
	Sys_error s ->
	  prerr_endline s

    method generate_values_index _ = ()
    method generate_exceptions_index _ = ()
    method generate_types_index _ = ()
    method generate_attributes_index _ = ()
    method generate_methods_index _ = ()
    method generate_classes_index _ = ()
    method generate_class_types_index _ = ()
    method generate_modules_index _ = ()
    method generate_module_types_index _ = ()

    method generate_index l =
      list_values <- [];
      list_exceptions <-  [];
      list_types <- [];
      list_attributes <- [];
      list_methods <- [];
      list_classes <- [];
      list_class_types <- [];
      list_modules <- [];
      list_module_types <- [];
      if not !tables_only then html#generate_index l


    initializer
      default_style_options <-
	default_style_options @
	[ ".commands { border-width : thin ; border-style: solid } ";
	  ".table_header { background-color: #f0d0f0 ; font-weight: bold }" ;
	  ".odd { background-color : #a0d0f0 }" ;
	  ".even { background-color: #a0d0d0 }" ;
	] 

(*
    method generate (modules : Module.t_module list) =
      (* init the style *)
      self#init_style ;

      List.iter self#generate_module modules
*)
  end

let generator = ((new gen ()) :> Odoc_args.doc_generator)

let op_tables_only = ("-tables-only", Arg.Set tables_only, "Generate code for tables only, no index, no module description")

let _ = Odoc_args.add_option op_tables_only

let _ = Odoc_args.set_doc_generator (Some generator)
