/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_formbuttons.pike,v 1.67.2.3 2004/01/19 07:37:23 vida Exp $
 */
//
//! module: CAMAS: Form Button provider
//!  This module gives to CAMAS the correct buttons needed for the template.
//!  It choose for you the imagebutton, form button or even gbutton graphics
//!  button.<br />
//!  <b>This module is automatically selected if you select "CAMAS: Main
//!  module".</b>
//! inherits: module
//! inherits: caudiumlib
//! type: MODULE_PROVIDER
//! cvs_version: $Id: camas_formbuttons.pike,v 1.67.2.3 2004/01/19 07:37:23 vida Exp $
//
#include <module.h>
inherit "module";
inherit "caudiumlib";
#include <camas/utf.h>		// TO_UTF8
#include <camas/globals.h>

constant cvs_version   = "$Id: camas_formbuttons.pike,v 1.67.2.3 2004/01/19 07:37:23 vida Exp $";
constant module_type   = MODULE_PROVIDER;
constant module_name   = "CAMAS: Form Button provider";
constant module_doc    = "This module handle the different ways for CAMAS to draw buttons:"
                         "<ul><li>form-button</li>"
												 "<li>icon</li>"
                         "<li>gbutton</li>"
                         "<li>camas-mode-basic</li>"
                         "<li>camas-mode-advanced</li></ul><br />"
                         "Since CAMAS 1.2, layouts can now have button configuration built-in.<br />"
                         "The following settings will be then used as default for the current virtual server if:"
                         "<ul><li>The layout used is a CAMAS 1.0 layout</li>"
                         "<li>The layout used is a CAMAS 1.2 and has no built-in button configuration or a buggy one</li></ul>"
                         "Check the \"Status and debug info\" for your CAMAS Filesystem Layout copies to get more information.<p>"
                         "<b>This module is automatically selected if you select \"CAMAS: Main module\".</b>";
constant module_unique = 1;
constant thread_safe = 1;

// For default values in defvars
#define DEFAULTGBUTTON "<gbutton-url font='lucida sans unicode' bgcolor='lightblue'>$TEXT$</gbutton-url>"

#define DEFAULT_ICON_FOLDER	"icons"
#define DEFAULT_ICON_FORMAT	"png"

mapping buttonlist;  // Storage for camas-mode-basic button list
object  filewatch;   // Filewatcher for camas formbutton autoloading... TODO: use FileWatch.pmod?

/* Functions used to hide the defvar when not used */

//! method: hide_gbutton ()
//!  Hides the gbutton defvars if gbutton mode is not selected
int hide_gbutton ()
{
  if (QUERY(mode) == "gbutton")
    return 0;
  return 1;
}

//! method: hide_camasbasic ()
//!  Hides the camasbasic defvars if camas-mode-basic mode is not selected
int hide_camasbasic ()
{
  if (QUERY(mode) == "camas-mode-basic")
    return 0;
  return 1;
}

/* Module creation & configuration */

//! method: create()
//!  Constructor
void create()
{
#ifdef CAMAS_DEBUG
  // Debug
  defvar("debug",0,"Debug",TYPE_FLAG,"Debug the calls / errors into Caudium "
         "error log ?");
#endif
  // Mode setup
  defvar("mode","form-button","Formbutton mode",TYPE_STRING_LIST,
         "The mode used for providing image button.<br />"
         "<ul>"
         " <li><b>form-button</b> mode is the standard mode for those awfull forms buttons.</li>"
         " <li><b>gbutton</b> mode is allmost same as <i>form-button</i> but it use instead "
         "     gbutton module to make images</li>"
         " <li><b>camas-mode-basic</b> mode is the new mode with flat file with images to load "
         "     for every actions</li>"
         " <li><b>camas-mode-advanced</b> mode is the new mode with flat file with the complete "
         "     RXML code for every action</li>"
         "</ul>"
         "<b>Warning</b>: from <i>camas-mode-basic</i> and <i>camas-mode-advanced</i> don't forget "
         "to reload the module for file been loaded by this module.",
         ({ "form-button", "gbutton", "camas-mode-basic", "camas-mode-advanced" }));
  // Gbutton part
  //  Gbutton is simple, it does only a single type of button
  defvar("gsyntax",DEFAULTGBUTTON,"GButton syntax", TYPE_STRING,
         "GButton syntax used for autocreate buttons for all form actions. "
         "Use <b>$TEXT$</b> to set the place where button's text will be "
         "added."
         "Please see &lt;gbutton-url&gt; manual for more informations.",
         0,hide_gbutton);
  // CAMAS basic mode
  defvar("cbfile","modules/camas/formbuttons/camasbasic.txt","Camas Basic Mode file", TYPE_FILE,
         "The file where is defined the RXML code according the CAMAS's button "
         "definitions. (add there some description about this format).<br/>"
         "Path is relative to Caudium server.",
         0,hide_camasbasic);
}

//! method: check_variables (string var, mixed value)
//!  Checks the variables
//! arg: string var
//!  The var to check
//! arg: mixed value
//!  The value to check
string check_variable(string var, mixed value)
{
  switch(var)
  {
  case "cbfile":
    if (!Stdio.is_file(value) && (QUERY(mode) == "camas-mode-basic"))
      return "File doesn't exist or is not a plain file. Please check again.";
    break;
  }
  return 0;
}

/* Functions depending how the module is setup */

//! method: camas_iconbutton(object id, string img, string action, string value, void|mapping args)
//! Handle the icon buttons drawing when icon mode is selected
string camas_iconbutton(object id, string img, string action, string value, mapping args)
{
	args = args || ([ ]);

  object layout_manager = id->conf->get_provider("camas_layout_manager");

	string icon_path = layout_manager->query_location()+CAMAS.Tools.current_layout(id)+"/"+DEFAULT_ICON_FOLDER+"/";

	string icon_extension = args->format ? args->format : layout_manager->get_icon_extension(CAMAS.Tools.current_layout(id)) ? layout_manager->get_icon_extension(CAMAS.Tools.current_layout(id)) : DEFAULT_ICON_FORMAT;

	args->type = "image";
	args->alt = value;
	args->title = value;
	args->value = value;
	args->name = action;
	args->src = icon_path+img+"."+icon_extension;

	m_delete(args, "defaultformat");

	return CAMAS.Tools.make_tag("input", args);
}


// === GBUTTON MODE ===

//! method: gbutton_buttons(object id, string img, string action, string value, mapping args)
//!  Handle the buttons drawing when gbutton mode is selected
string gbutton_buttons(object id, string img, string action, string value, mapping args)
{
  object gbp     = id->conf->get_provider("gbutton");
  object clmangr = id->conf->get_provider("camas_layout_manager");
  string gsyntax = "";   // syntax for gbutton

  if(!objectp(gbp))
    return camasbasic_buttons(id, img, action, value, args);

  if(objectp(clmangr))
  {
    gsyntax = clmangr->get_gbutton_syntax(CAMAS.Tools.current_layout(id)) || "";
  }
  if(sizeof(gsyntax) == 0)
    gsyntax = QUERY(gsyntax);

  args -> type = "image";
  args -> name = action;
  args -> value= value;
  args -> alt  = value;

  // we use a cache here so that we don't parse the same thing ever and ever 
  args->src = cache_lookup("camas_gbutton", value + gsyntax);
  if(!args->src)
  {
    mapping gbp_args = ([ ]);
    string gbp_contents;
    string gbp_tag;
    string _gsyntax = 
        replace(gsyntax,({"$TEXT$" }), 
                   ({ value }) );
    // the lambda function is here only because we want to add the encoding 
    // arguments to every gbutton-url call
    CAMAS.Parse.parse_html(_gsyntax,
       ([ ]),
       ([ "gbutton-url": 
               lambda(string _tag, mapping _args, string _contents) 
	         { 
		   gbp_tag = _tag;
		   gbp_args = _args;
		   gbp_contents = Locale.Charset.encoder("utf-8")->feed(_contents)->drain();
		 }
         ]));
    gbp_args->encoding = "utf-8";
    if(gbp_contents)
	    args->src = gbp->tag_gbutton(gbp_tag, gbp_args, gbp_contents, id);
    cache_set("camas_gbutton", value + gsyntax, args->src, 3600);
  }
  return CAMAS.Tools.make_tag("input", args);
}

// === CAMAS BASIC ===

//! method: camasbasic_buttons (object id, string img, string action, string value, void|mapping args)
//!  Handle the drawing of the buttons when camas-mode-basic is selected
string camasbasic_buttons (object id, string img, string action, string value, void|mapping args)
{
  string out, temp;

  object clmangr = id->conf->get_provider("camas_layout_manager");
  if (objectp(clmangr))
  {
    temp = clmangr->get_basicbutton_list(CAMAS.Tools.current_layout(id))[lower_case(img)];
  }

  // Fallback to ugly form button if there was a problem with camas basic mode
  if (temp == 0)
    out = CAMAS.FormButtons.form_buttons(action, value, args || ([ ]));
  else
    out = replace ( temp, ({ "$TEXT$","$ACTION$","$ONCLICK$" }), ({ value, action, (string)args->onclick }) );

  return out;
}

// ==== CAMAS ADVANCED ===

//! method: camasadvanced_buttons(object id, string img, string action, string value, void|string arg)
//!  Handles the drawing of the button when camas-mode-advance is selected
string camasadvanced_buttons(object id, string img, string action, string value, void|string arg)
{
  string out = "";

  out += "camas mode advanced not implemented yet";

  return out;
}

//! method: load_basicfile (string file)
//!  Load the file for button definitions if camas-mode-basic is selected
//! arg: string file
//!  The path in the filesystem for the file to load
int load_basicfile(string file)
{
  if (QUERY(mode) != "camas-mode-basic")
    return -1;
  CDEBUG("(re)loading camas-mode-basic configuration file from disk.");
  buttonlist = CAMAS.FormButtons.Basic.File(file)->doindex();
  return 0;
}

//! method: start (int cnt, object conf)
//!  Function called when the module is started
void start(int cnt, object conf)
{
  module_dependencies(conf, ({ "gbutton" }));
  //if (QUERY(mode) == "camas-mode-basic") buttonlist = CAMAS.FormButtons.Basic.File(QUERY(cbfile))->doindex();
  if (QUERY(mode) == "camas-mode-basic") {
    load_basicfile(QUERY(cbfile));
    filewatch = FileWatch.Callout(QUERY(cbfile), 5, load_basicfile);
  }
  else buttonlist = 0;	// To help gc...
}

// Module provider functions
string query_provides()
{
  return("camas_formbuttons");
}

// Module provider methods

//! method: string button(object id, string img, string action, string value, void|string onClick)
//!  Return the RXML code needed to handle Formbuttons extension
//! returns:
//!  RXML code for handling fancy form buttons depending on
//!  how is configured the layout. 
//! arg: object id
//!  Caudium ID object
//! arg: string img
//!  The image needed to be loaded. Used also as a key on
//!  CAMAS Basic and extended mode.
//! arg: string action
//!  The form action to use with this button
//! arg: string value
//!  The text to add to the button coming from language file (wide-
//!  string compliant).
//! arg: void|mapping args
//!  Optional arguments to append to the button
//! void|string layoutmode
//!  
string button(object id, string img, string action, string value, void|mapping args, void|string layoutmode)
{
  string out = "";

  string mode = layoutmode || QUERY(mode);

  CDEBUG("Mode called: "+mode);

  switch (mode)
  {
  case "form-button":
    out = CAMAS.FormButtons.form_buttons(action, value, args || ([ ]));
    break;
		
  case "gbutton":
    out = gbutton_buttons(id, img, action, value, args || ([ ]));
    break;
		
  case "icon":
    out = camas_iconbutton(id, img, action, value, args || ([ ]));
    break;
		
  case "camas-mode-basic":
    out = camasbasic_buttons(id, img, action, value, args);
    break;
	
	
  case "camas-mode-advanced":
    out = camasadvanced_buttons(id, img, action, value, args);
    break;

  default:
    out = "<!-- Check 'CAMAS: Form Button provider -> Formbutton mode' , maybe you're still in deprecated pre-camas mode -->";
  }
  CDEBUG("Output sent to the parser : "+out);
  return out;
}

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: debug
//! Debug the calls / errors into Caudium error log ?
//!  type: TYPE_FLAG
//!  name: Debug
//
//! defvar: mode
//! The mode used for providing image button.<br /><ul> <li><b>form-button</b> mode is the standard mode for those awfull forms buttons.</li> <li><b>gbutton</b> mode is allmost same as <i>form-button</i> but it use instead      gbutton module to make images</li> <li><b>camas-mode-basic</b> mode is the new mode with flat file with images to load      for every actions</li> <li><b>camas-mode-advanced</b> mode is the new mode with flat file with the complete      RXML code for every action</li></ul><b>Warning</b>: from <i>camas-mode-basic</i> and <i>camas-mode-advanced</i> don't forget to reload the module for file been loaded by this module.
//!  type: TYPE_STRING_LIST
//!  name: Formbutton mode
//
//! defvar: gsyntax
//! GButton syntax used for autocreate buttons for all form actions. Use <b>$TEXT$</b> to set the place where button's text will be added. Use <b>$ENCODING$</b> to place the <em>encoding='encoding-name'</em> attribute corresponding to the current language charset (as read from the language file).Please see &lt;gbutton-url&gt; manual for more informations.
//!  type: TYPE_STRING
//!  name: GButton syntax
//
//! defvar: gbuttonutf8
//! When set, connect directly to gbutton using module provider method, so UTF8 text and encoding are not lost by the RXML parser.<br/>Note: You need Caudium 1.1 and Gbutton v1.16 or newer to enable this.
//!  type: TYPE_FLAG
//!  name: Text is UTF8
//
//! defvar: cbfile
//! The file where is defined the RXML code according the CAMAS's button definitions. (add there some description about this format).<br/>Path is relative to Caudium server.
//!  type: TYPE_FILE
//!  name: Camas Basic Mode file
//
