/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * Based on IMHO  Stefan Wallstrm and Bosse Lincoln.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: imapclient.pmod,v 1.145.2.10 2004/03/31 22:23:38 vida Exp $
 */
/*
 * CAMAS IMAP client
 *  
 *  Warning: This is the best from IMHO, may the force be with you.
 *  Be sure to understand the Tao before modifying this file.
 *
 */
#include <camas/globals.h>
#include <camas/parser.h>
#include <camas/msg.h>		// Language module macros
#include <module.h>
#include <pcre.h>
/*
  oliv3 FIXME: if !mboxencode -> save a filter fails (does not return
  back to the previous screen).

  oliv3 FIXME: if !mboxencode -> rename a folder fails
*/

//
// IMAP-Client
//

// headers and folders info cache
class Cache {
  mapping mcache = ([ ]);
  private int isshared = 0;
  private object sessobj;
  private object camas_main;
  private object _lock;
  private string mailbox;
  
  void init_cache()
  {
    if (!mcache)
    {
      IMAPCACHE_DEBUG("init_cache called for "+mailbox + "\n");
      mcache = ([ ]);
      mcache->mails = ({ });
      mcache->new_mails = ({ });
      mcache->uid_validity = -1;
      mcache->messages = mcache->recent = mcache->unseen = 0;
      mcache->size = 0;
      mcache->uidnext = 0;
      mcache->changed = 1;
    }
  }
  
  void create(string _mailbox, object _sessobj, object _camas_main)
  {
    sessobj = _sessobj;
    mailbox = _mailbox;
    camas_main = _camas_main;
    string sharedpath;
    int sharemailcache;
    if(!camas_main->isimapcachefeeder)
      sharemailcache = camas_main->QUERY(sharemailcache);
    else
      sharemailcache = camas_main->sharemailcache;
    if(sharemailcache && search(mailbox, sessobj->sharedpath) == 0)
      isshared = 1;
    if(isshared)
    {
      IMAPCACHE_DEBUG("Looking existing cache for "+mailbox+" in shared cache\n");
      mcache = cache_lookup("camas_mcache", mailbox);
    }
    else 
    {
      IMAPCACHE_DEBUG("Looking existing cache for "+mailbox+" in personnal cache\n");
      mcache = sessobj->cache[mailbox];
    }
    lock();
    mixed err = catch {
      init_cache();
    };
    unlock();
    if(err)
      report_error("error in imapclient: %s\n", describe_backtrace(err));
      IMAPCACHE_DEBUG("sizeof(cache->mails)="+sizeof(mcache->mails)+" uidnext="
        + mcache->uidnext+" uid_validity="+mcache->uid_validity
        + " messages=" + mcache->messages + "\n");
  }

  void lock()
  {
    if(isshared)
    {
      SHARED_LOCK();
    }
    else
    {
      LOCK_FEEDER();
    }
  }

  void unlock()
  {
    if(isshared)
    {
      SHARED_UNLOCK();
    }
    else
    {
      UNLOCK_FEEDER();
    }
  }
 
  void save()
  {
    if(mcache->changed)
    {
      lock();
      mixed err = catch {
        if(isshared)
        {
          IMAPCACHE_DEBUG("Saving "+mailbox+" data to cache shared cache\n"
              "sizeof(cache->mails)="+sizeof(mcache->mails)+" uidnext="
            + mcache->uidnext+" uid_validity="+mcache->uid_validity
            + " messages="+mcache->messages+"\n");
          cache_set("camas_mcache", mailbox, mcache, CACHE_TIMEOUT);
        }
        else if(sessobj)
        {
          IMAPCACHE_DEBUG("Saving "+mailbox+" data to cache personnal cache\n"
              "sizeof(cache->mails)="+sizeof(mcache->mails)+" uidnext="
            + mcache->uidnext+" uid_validity="+mcache->uid_validity
            + " messages="+mcache->messages+"\n");
          sessobj->cache[mailbox] = mcache;
        }
      };
      unlock();
      if(err)
        report_error("error in imapclient: %s\n", describe_backtrace(err));
    }
  }

  void remove()
  {
    lock();
    mixed err = catch {
      if(isshared)
      {
        IMAPCACHE_DEBUG("Removing "+ mailbox+" for shared cache\n");
        cache_remove("camas_mcache", mailbox);
      }
      else
      {
        IMAPCACHE_DEBUG("Removing "+ mailbox+" for personnal cache\n");
        m_delete(sessobj->cache, mailbox);
      }
    };
    unlock();
    if(err)
      report_error("error in imapclient: %s\n", describe_backtrace(err));
  }

  void destroy()
  {
    if(mcache->changed)
      save();
    mcache = 0;
    sessobj = 0;
  }
}

class imapclient {
  // variables
  object camas_main;
  object parser;
  object cache;

  int feature_extendedabook = 0;

  int loggedin = 0;
  int abort=0;
  int result_sent=0;
  object prefsmailobj;
  object ebookmailobj;
  object sessobj;
  object id;
  string data="";
  string imapwritedata = "";
  string line=0;
  string tmp_line="";
  string dsn_comm="";
  string session_id;
  int dsn=0;
  int parse_type;
  mixed parse_result=0;
  int tokens_to_get=0;
  array (mapping) commands;
  static mapping command=0;
  int command_result=0;
  int mode;
  mixed timeout_call_out=0;
  mixed imap_idle_call_out=0;
  object fd;
  int imap_seq=0;

  array folderstodelete=({ });
  int trycreate=0;
  int reconnects=0;
  array parserstate = 0;
  mixed cmd_result = 0;
  mixed cmd_tmp = 0;

  int command_seq=0;
  string foo,bar;
  array tmp_array;

  string selected_mailbox = 0;

  array (string) imaplinehist = ({ });

  string literal_string = 0;
  int literal_addcrlf = 0;
  int literal_continue = 0;
  // are we the imapclient of the imap cache feeder ?
  int isimapcachefeeder = 0;
  // the capabilities of the IMAP server
  multiset capabilities = (< >);
  // the roots paths to personal, public and shared boxes
  array roots = ({ });
  // the separtors for these mailboxes
  array separators = ({ });
  // a mutex lock
  private object _lock;
  // temporary mailboxes storage for LIST command
  private array mailboxes;
  // temporary mailpath list for LIST command
  private array mailpath;

  // methods

  void camas_log (string event, mapping data) {
    if (!camas_main)
      return;
    object logger = camas_main->my_configuration ()->get_provider ("camas_logger");
    if (objectp (logger) && logger->log)
      logger->log (event, data);
  }

  // Return data from an argument in a command structure
  // If arg is a string, it is retuned untouched
  // If arg is an array, the variable with name in arg[0] contains the
  // data to be returned. If the name is "-", the data is taken
  // from cmd_result instead (a kind of inter-command-pipe).
  inline mixed get_cmd_arg (mixed arg) {
    if (arg && sizeof (arg) > 0) {
      if (arrayp (arg) && (sizeof (arg) > 0) && intp (arg[0]) && (arg[0] == CAMAS.Const.CMD_VAR_POINTER)) {
        string var = arg[1];
        if (var == "-")
         return (cmd_result);
        else
          return (sessobj[var]);
      }
      else
        return (arg);
    }
    return 0;
  }

  inline void set_cmd_arg (mixed arg, mixed data) {
    if (arg && sizeof (arg) > 0) {
      if (arrayp (arg) && (sizeof (arg) > 0) && intp (arg[0]) && (arg[0] == CAMAS.Const.CMD_VAR_POINTER)) {
        string var = arg[1];
        if (var == "-")
          cmd_result = data;
        else
        {
          sessobj[var] = data;
        }
      }
      else
        report_warning ("imapclient: set_cmd_arg: Wrong argument name!");
    }
  }

  string fixmail (string mail) {
    // Fix mails which the MIME module cannot parse. This function is
    // called recursively for each mailpart

    int headend1 = search(mail, "\n\n");
    int headend2 = search(mail, "\r\n\r\n");
    int headend = (headend1<0? headend2 :
                   (headend2<0? headend1 :
                    (headend1<headend2? headend1 : headend2)));
    array (string) headers = (replace(mail[0..headend-1], ({"\r", "\t", "\n "}), ({ "", " ", " "})))/"\n";
    string newheaders = "", contenttype = "", boundary = "";

    foreach (headers, string h) {
      string key = "", data = "";
      if (sscanf (h, "%[!-9;-~]%*[ ]:%*[ ]%s", key, data) > 2) {
        if (lower_case (key) == "content-type") {
          //write ("content-type: data= " + data + "\n");
          string k1 = "", k2 = "", opts = "";
          sscanf (data, "%s/%s;%s", contenttype, k2, opts);
          foreach (opts/";", string opt) {
            string l1 = "", l2 = "";
            if (sscanf (opt, "%*[ ]%s=%s", l1, l2) > 1) {
              if (lower_case (l1) == "boundary") {
                boundary = l2;
                sscanf (boundary, "\"%s\"", boundary);
              }
            }
          }
        }
      }
    }

    foreach (headers, string h) {
      string key = "", data = "";
      if (sscanf (h, "%[!-9;-~]%*[ ]:%*[ ]%s", key, data) > 2)
      {
        string lc_key = lower_case (key);

        if (lc_key == "content-transfer-encoding" &&
            ((lower_case (contenttype) == "multipart") || (lower_case (data) == "8bits"))) {
          data = "8bit";
        }

        if (lc_key == "content-transfer-encoding" &&
            ((lower_case (contenttype) == "multipart") || (lower_case (data) == "7-bit"))) {
          data = "7bit";
        }

        if ((lc_key == "content-type" ||
             lc_key == "content-description" ||
             lc_key == "content-disposition") && sizeof (data) > 0)
          while (data[-1] == ' ' || data[-1] == '\t' || data[-1] == ';')
            data = data[0..sizeof(data)-2];

        if (lc_key == "content-disposition") {
          array foo = data / " ";
          for (int i = 1; i < sizeof (foo); i++)
            if (!has_value (foo[i], "="))
              foo[i] += "=fixed";
          data = foo * " ";
        }

        if (lc_key == "content-type") {
          /*
            yet another fix for those spam-bots:
            "content-type: foo/bar, baz"   instead of
            "content-type: foo/bar; baz"
          */

          /* RFC 2045, 5.1

               content := "Content-Type" ":" type "/" subtype
          *(";" parameter)
          ; Matching of media type and subtype
          ; is ALWAYS case-insensitive. */
          string type, subtype, parameter;
          if (sscanf (data, "%s/%s, %s", type, subtype, parameter) == 3) {
            //write ("bad content-type: " + data + "...\n");
            data = type + "/" + subtype + "; " + parameter;
          }

          /*
            and here we go again...

            MIME.Message failed: invalid parameter in Content-Type
            content-type: data= text/plain;X-Mailer: cls_mail
          */
          string junk_header, junk_data;
          if (sscanf (data, "%s/%s;%s:%s", type, subtype, junk_header, junk_data) == 4) {
            //write ("Header: " + type + "/" + subtype + ";\n");
            //write ("Junk: " + junk_header + ":" + junk_data + ";\n");
            newheaders += junk_header + ":" + junk_data + ";\n";
            data = type + "/" + subtype + ";";
          }
        }

        newheaders += key + ": " + data + "\n";
      }
    }

    string newmail = "";
    if (sizeof(boundary) > 0) {
      array (string) mailparts = mail[headend+2..] / ("--"+boundary);
      newmail += newheaders+"\n"+mailparts[0];
      for (int i = 1; i < sizeof(mailparts)-1; i++)
        newmail += "--"+boundary+"\r\n"+fixmail(mailparts[i]);
      newmail += "--"+boundary+"--\r\n\r\n";
    }
    else
      newmail = newheaders + "\n" + mail[headend+2..];
    return(newmail);
  }

  void imapwrite (string data) { // Write data to a nonblocking socket
    if(imap_idle_call_out)
      remove_call_out(imap_idle_call_out);
    if(sessobj)
      imap_idle_call_out=call_out(handle_imap_idle_timeout,sessobj->imapidlelogout);
    int res;
    if (sizeof(imapwritedata) > 0)
      imapwritedata += data;
    else {
      res = fd->write(data);
      if (res < sizeof(data))
        imapwritedata = data[res..];
    }
  }

  void init_parser (int pt) {
    mode = MODE_PARSE;
    parser->init ();
    line = 0;
    parse_type = pt;
  }

  string imap_tag () {
    return "camas" + sprintf ("%04d", imap_seq);
  }

  void imap_firsttag () {
    imap_seq = 0;
  }

  void imap_newtag () {
    imap_seq++;
  }

  void handle_error () {
    if(sessobj->status > 0)
      report_warning(sprintf("\nError in IMAP communication: Backtrace: %s"
            "commands=%O, command=%O", describe_backtrace(backtrace()), commands, command));
    if(command && command->error && sizeof(command->error) > 0) {
      sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
      if (command && command->erroraction)
        sessobj->dialogactions = ({ command->erroraction });
      else
        sessobj->dialogactions = ({ "actionfolderlist" });
      sessobj->dialogtext = command->error;
      if (sessobj->displayimaperrors)
        sessobj->dialogtext += "\n"+line;
      sessobj->status = DIALOGBOX;
      return;
    }

    if (command && (command->cmd == "low_login"))
    {
      // don't keep connection to imapd open
      // if login fails.
#if constant(thread_create)
        _lock = camas_main->global_lock->lock();
#endif
     mixed err = catch {  
        if(fd)
        {
          fd->set_close_callback (0);
          fd->set_read_callback (0);
          fd->set_write_callback (0);
          fd->close ();
          destruct(fd);
          fd = 0;
        }
      };
#if constant(thread_create)
      destruct(_lock);
#endif
      if(err)
        report_error("error in imapclient: %s\n", describe_backtrace(err));
      selected_mailbox = 0;
    }

    if (sessobj->status < 0)
    {
      if (command_result == -1)
      {
	      camas_log ("loginfailed", ([ "login" : sessobj->address ]));
        sessobj->status = LOGINFAILED;
      }
      else
        sessobj->status = LOGINFAILEDIMAP;
    }
    else
    {
      // don't try to output an error to client if we are called from the cache
      if(id && id->conf)
      {
        sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
        if (command && command->erroraction)
          sessobj->dialogactions = ({ command->erroraction });
        else
          sessobj->dialogactions = ({ "actionmailindex" });
        sessobj->dialogtext = MSG(M_IMAPERROR);
        camas_log ("imap_errors", ([ "login" : sessobj->address,
                               "command" : ((command) ? command->cmd : "unknown command"),
                               "line" : line,
                               "history" : imaplinehist * "\n" ]));
        if (sessobj->displayimaperrors)
          sessobj->dialogtext += "\n" + line;
      }
      report_warning("imapclient: IMAP error: imaplinehist: " + imaplinehist * "\n");
      sessobj->status = DIALOGBOX;
    }
  }

  void handle_timeout () {
    report_warning(sprintf("CAMAS: imapclient@"+__LINE__+": Timeout (%d seconds) happened\n", TIMEOUT1));
    handle_error ();
    command_result = -1;
    if (command && !command->stop)
      next_command ();
  }

  void timeout (int s) {
    if (timeout_call_out)
      remove_call_out (timeout_call_out);

    if (s)
      timeout_call_out = call_out (handle_timeout, s);
  }

  void handle_imap_idle_timeout () {
    // Close connection to IMAP server
#if constant(thread_create)
      _lock = camas_main->global_lock->lock();
#endif
    mixed err = catch {  
      fd->set_close_callback (0);
      fd->set_read_callback (0);
      fd->set_write_callback (0);
      fd->close ();
      destruct(fd);
      fd = 0;
    };
#if constant(thread_create)
    destruct(_lock);
#endif
    if(err)
      report_error("error in imapclient: %s\n", describe_backtrace(err));
    selected_mailbox = 0;
  }

  void send_result () {
    if (id && camas_main)
      id->send_result (camas_main->create_output (id));
  }

  inline void done () {
    remove_call_out (timeout_call_out);
    remove_call_out (imap_idle_call_out);
    if (!result_sent) {
      result_sent = 1;
      call_out (send_result, 0);
    }
  }

  void socket_close () {
    IMAP_DEBUG(sprintf( ("Socket closed with IMAP server on " + ctime (time ()))));
    remove_call_out (timeout_call_out);
    remove_call_out (imap_idle_call_out);
    timeout_call_out = 0;
    imap_idle_call_out = 0;
    fd = 0;
    selected_mailbox = 0;
    if (!abort) {
      if (reconnects > 0) // Did we already try once?
        // other end closed unexpectedly
        handle_error ();
      else {
        if (command != 0) {
          reconnects++; // Try to connect again
          commands = ({ command }) + commands;
          next_command ();
        }
      }
    }
    else {
      if (!result_sent) {
        result_sent = 1;
        call_out (send_result, 0);
      }
    }
  }

  void imap_data_written () {
    int res;
    if (sizeof (imapwritedata) > 0)
    {
      res = fd->write (imapwritedata);
      if (res < sizeof (imapwritedata))
        imapwritedata = imapwritedata[res..];
      else
        imapwritedata = "";
    }
  }

  void got_data (mixed from, string input) {
      timeout (0);

      int going = 1; // true when rows are available
      data += input;

      while (going && !abort) {
	switch (mode) {
	case MODE_LINE:
	  bar = "";
	  if (sscanf (data, "%s%[\r\n]%s", foo, bar, data) && sizeof (bar)) {
	    line = tmp_line + foo;
	    tmp_line = "";
	  }
	  else {
	    tmp_line += foo;
	    line = 0;
	    going = 0;
	  }
	  break;

	case MODE_PARSE:
	  parserstate = parser->parseimap (data, parse_type, tokens_to_get, parserstate);

	  data = parserstate[2];
	  if (parserstate[0]) { // Done!
	    parse_result = parserstate[1];
	    parserstate = 0;
	    mode = MODE_LINE;
	    handle_command ();
	  }
	  else
	    going = 0;
	  break;
	}

	if (going && (line && sizeof (line)))
	  handle_command ();
      }
  }

  string imapwrite_literal (string s, int addcrlf, int cont) {
    // addcrlf - if !0 : sends an additional \r\n after the string.
    //           usable when this string ends a command.
    // cont    - if !0 : calls handle_command after successfully
    //           sending the string
    //           NOTE! handle_command is always called if writing
    //           should fail.
    imapwrite ("{" + sizeof (s) + "}\r\n");
    literal_string = s;
    literal_addcrlf = addcrlf;
    literal_continue = cont;
  }


  //oliv3: start_ / handle_ functions

  // low_login
  void start_low_login () {
    loggedin = 0; // because we're NOT logged in :)
    timeout (TIMEOUT1);
  }

  void handle_low_login () {
    switch (command_seq) {
    case 0: // expect greeting
      IMAP_DEBUG("imap server responding ?\n");
      if (line && sscanf (line, "* OK%*s")) {
        //oliv3: keep this ?
        //sscanf (line, "* OK %s %*s", sessobj->realimapserver);
        //write ("IMAP server: " + sessobj->realimapserver + "\n");
        command_seq = 1;
        imap_newtag ();
        imapwrite (imap_tag () + " LOGIN " + IMAP_QSTRING (sessobj->login) + " " + IMAP_QSTRING (sessobj->passwd) + "\r\n");
        IMAP_DEBUG(sprintf ("imap login: auth token sent (%s, *****).\n", sessobj->login));
      }
      timeout (TIMEOUT1);
      break; // 0

    case 1: // wait for login result
      if (sscanf (line, imap_tag () + " %s", foo))
      {
        if (foo[0..1] == "OK")
        {
          command_result = 0;
          loggedin = 1; // now we're logged in :)
          start_command(CAMAS.IMAPTools.imap_cmd("low_capability"));
        }
        else
        {
          command_result = -1;
          next_command ();
        }
      }
      else
        timeout (TIMEOUT1);
      break; // 1
    }
  }

  // capability
  void start_low_capability()
  {
    IMAP_DEBUG("Getting capabilities of IMAP server\n");
    imapwrite (imap_tag() + " CAPABILITY\r\n");
    timeout(TIMEOUT1);
  }

  void handle_low_capability()
  {
    switch(command_seq)
    {
      case 0:
        if(sscanf(line, "* CAPABILITY %s", foo))
        {
          array cap = foo / " ";
          foreach(cap, string index)
            capabilities += (< lower_case(index) >);
          command_seq = 1;
        }
        else
        {
          command_result = -1;
          report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at capability\n");
          next_command ();
        }
        break; // 0
      case 1:
        if(sscanf(line, imap_tag() + " %s", foo))
          if(foo[0..1] == "OK")
          {
            IMAP_DEBUG(sprintf("Capabilities of IMAP server: %O\n", capabilities));
            command_result = 0;
          }
          else
          {
            command_result = -1;
            report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at capability\n");
          }
        next_command ();
        break;
    }
  }

  // namespace, RFC 2342, partial support
  void start_low_namespace()
  {
    if(capabilities["namespace"])
    {
      IMAP_DEBUG("Getting namespace\n");
      imapwrite (imap_tag() + " NAMESPACE\r\n");
      timeout(TIMEOUT1);
    }
    else
      next_command();
  }

  void handle_low_namespace()
  {
    switch(command_seq)
    {
      case 0:
        if(sscanf(line, "* NAMESPACE %s", foo))
        {
          mixed err = catch {
            array name_parts = Regexp("(\\(\\(.*\\)\\)|NIL) (\\(\\(.*\\)\\)|NIL) (\\(\\(.*\\)\\)|NIL)")->split(foo);
            foreach(name_parts, string namespace)
            {
              if(namespace != "NIL")
              {
                string s;
                if(sscanf(namespace, "((%s)", s))
                {
                  array arr = s / " ";
                  arr =  Array.map(arr, lambda(string value) 
                      { return replace(value, "\"", ""); });
                  roots += ({ arr[0] });
                  separators += ({ arr[1] });
                }
                else
                {
                  roots += ({ 0 });
                  separators += ({ 0 });
                }
              }
              else
              {
                roots += ({ 0 });
                separators += ({ 0 });
              } //if
            } // foreach
            command_seq = 1;
          }; // catch
          if(err)
          {
            command_result = -1;
            report_warning(sprintf("CAMAS: imapclient@"+__LINE__+": IMAP error at namespace, backtrace is %s\n",
                describe_backtrace(err)));
            next_command();
          }
        } // if * namespace
        else
        {
          command_result = -1;
          report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at namespace\n");
          next_command ();
        }
        break;
      case 1:
        if(sscanf(line, imap_tag() + " %s", foo))
          if(foo[0..1] == "OK")
          {
            IMAP_DEBUG(sprintf("Namespace of IMAP server: roots=%O, separators=%O\n", 
                  roots, separators));
            if(sizeof(roots) == 3)
            {
              if(stringp(roots[0]))
              {
                sessobj->mailpath = roots[0];
                sessobj->prefsbox = sessobj->mailpath + sessobj->baseprefsbox;
              }
              if(stringp(roots[2]))
               sessobj->sharedpath = roots[2];
            }
            command_result = 0;
          }
          else
          {
            command_result = -1;
            report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at capability\n");
          }
        next_command ();
        break;
    }
  }
  
  // low_select
  void start_low_select () {
    if (command->mailbox == 0)
      command->mailbox = sessobj->mailbox[MB_FOLDERNAME_IDX];
    IMAP_DEBUG("Select: " + command->mailbox + "\n");
    if (command->mailbox == selected_mailbox)
    {
      command_result = 0;
      next_command ();
    }
    else
    {
      if (sessobj->mboxencode)
        imapwrite (imap_tag () + " SELECT " + IMAP_QSTRING (command->mailbox) + "\r\n");
      else
      {
        imapwrite (imap_tag () + " SELECT ");
        imapwrite_literal (command->mailbox, 1, 0);
      }
      timeout (TIMEOUT1);
    }
  }

  void handle_low_select () {
    if (sscanf (line, imap_tag () + " %s", foo))
    {
      if (foo[0..1] == "OK") {
        command_result = 0;
        selected_mailbox = command->mailbox;
      }
      else
      {
        command_result = -1;
        report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at select %O\nfor mailbox %O\n", line, command->mailbox);
        selected_mailbox = 0;
      }
      next_command ();
    }
    else
      timeout (TIMEOUT1);
  }

  // status
  void start_status () {
    IMAP_DEBUG("status " + command->mailbox + ": ");
    cache = Cache(command->mailbox, sessobj, camas_main);

    if (sessobj->mboxencode)
      imapwrite (imap_tag () + " STATUS " + IMAP_QSTRING (command->mailbox)
                 +" (MESSAGES RECENT UNSEEN UIDNEXT UIDVALIDITY)\r\n");
    else
    {
      imapwrite (imap_tag () + " STATUS ");
      imapwrite_literal (command->mailbox, 0, 1);
      command_seq = 2;
    }
    timeout (TIMEOUT1);
  }

  void handle_status () {
#define CACHE(reason) IMAPCACHE_DEBUG("Cache changed: " + reason + "\n")
    mapping mcache = cache->mcache;

    switch (command_seq)
    {
    case 2: {
        // Continue STATUS command after literal
        imapwrite (" (MESSAGES RECENT UNSEEN UIDNEXT UIDVALIDITY)\r\n");
        command_seq = 0;
        timeout (TIMEOUT1);
      };
      break; // 2

    case 0:
      if (sscanf (line, imap_tag() + "NO %*s"))
      {
        // don't make it a fatal error so that we can fetch other folders
        report_warning("CAMAS: imapclient@"+__LINE__+": Can't STATUS on %O\n", 
            command->mailbox);
        command_result = 0;
        selected_mailbox = 0;
        next_command ();
      }
      else
      {
        if (sscanf (line, "* STATUS {%s", foo) == 1)
          command_seq = 5;
        else
          if ((sscanf (line, "* STATUS %*s (%s", foo) == 2) || (sscanf (line, "* STATUS \"%*s\" (%s", foo) == 2)) {
            data = "(" + foo + "\n" + data;
            init_parser (PT_MAPPING);
            command_seq = 1;
          }
        timeout (TIMEOUT1);
      }
      break; // 0

    case 5:
      if ((sscanf (line, "%*s (%s", foo) == 2) || (sscanf (line, "\"%*s\" (%s", foo) == 2)) {
        data = "(" + foo + "\n" + data;
        init_parser (PT_MAPPING);
        command_seq = 1;
      }
      timeout (TIMEOUT1);
      break; // 5

    case 1:
      if (!parse_result) {
        report_warning("CAMAS: imapclient@"+__LINE__+": Parser %O\n", line);
        command_result = -1;
        next_command ();
      }
      else
      {
        foreach (indices (parse_result), string i)
        parse_result[i] = (int)parse_result[i];

        cache->lock();
        mixed err = catch {
          if (!sessobj->foldersinfo[command->mailbox])
            sessobj->foldersinfo += ([ command->mailbox: ({ 0, 0, 0, MB_UNKNOWN_SIZE, 0 }) ]);
          
          if (mcache->uid_validity != parse_result->UIDVALIDITY) {
            CACHE ("uid validity: " + parse_result->UIDVALIDITY);
            CACHE ("previous validity: " + mcache->uid_validity);
            mcache->uid_validity = parse_result->UIDVALIDITY;
            mcache->mails = ({ });
            mcache->new_mails = ({ });
            mcache->messages = mcache->recent = mcache->unseen = 0;
            mcache->size = 0;
            mcache->uidnext = 0;
            mcache->changed = 1;
          }
         
          if (parse_result->RECENT) {
            CACHE("recent messages: " + parse_result->RECENT);
            mcache->changed = 1;
            mcache->recent = parse_result->RECENT;
            sessobj->foldersinfo[command->mailbox][MB_RECENT_IDX] = (int)parse_result->RECENT;
          }


          if (mcache->uidnext != parse_result->UIDNEXT) {
            CACHE("next uid: " + parse_result->UIDNEXT);
            CACHE("previous uid: " + mcache->uidnext);
            mcache->changed = 1;
            mcache->uidnext = parse_result->UIDNEXT;
            sessobj->foldersinfo[command->mailbox][MB_NEXTUID_IDX] =
              (int) parse_result->UIDNEXT;
          }

          if (parse_result->MESSAGES) {
            if (mcache->messages != parse_result->MESSAGES 
                || mcache->messages != sizeof(mcache->mails)) {
              CACHE("messages: " + parse_result->MESSAGES);
              mcache->changed = 1;
              mcache->messages = parse_result->MESSAGES;
              sessobj->foldersinfo[command->mailbox][MB_MESSAGES_IDX] =
                (int) parse_result->MESSAGES;
            }
          }
          else { // empty mailbox
            CACHE("empty mailbox");
            mcache->mails = ({ });
            mcache->new_mails = ({ });
            mcache->messages = mcache->recent = mcache->unseen = 0;
            mcache->size = 0;
            mcache->changed = 0;
            sessobj->foldersinfo[command->mailbox] = ({ 0, 0, 0, 0, 0 });
          }

          mcache->unseen = parse_result->UNSEEN;
          sessobj->foldersinfo[command->mailbox][MB_UNSEEN_IDX] =
            (int) parse_result->UNSEEN;

          if ((command->mailbox == "INBOX") && mcache->changed && !sessobj->filters_have_changed) {
            IMAP_DEBUG ("new mail has arrived. 'apply_mail_filters' scheduled.\n");
            sessobj->filters_have_changed = 1;
        }
        };
        cache->unlock();
        if(err)
          report_error("error in imapclient: %s\n", describe_backtrace(err));
        data = line + "\n" + data;
        parse_result = 0;
        command_seq = 3;
        timeout (TIMEOUT1);
      }
      break; // 1

    case 3:
      if (sscanf (line, imap_tag () + " %s", foo)) {
        if (foo[0..1] == "OK") {
          if (mcache->changed) {
            command_seq = 4;
            imap_newtag ();
            imapwrite (imap_tag () + " UID SEARCH ALL\r\n");
            timeout (TIMEOUT1);
          }
          else {
            IMAP_DEBUG ("mailbox has not changed.\n");
            //set_cmd_arg (command->output, mcache->mails);
            command_result = 0;
            next_command ();
          }
        }
        else {
          report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at status %O\n", line);
          command_result = -1;
          selected_mailbox = 0;
          next_command ();
        }
      }
      else
        timeout (TIMEOUT1);
      break; // 3

    case 4: { // SEARCH result
        if (sscanf (line, "* SEARCH %s", foo) == 1) {
          array new_uids = Array.map (foo / " ", lambda (string s) { return (int)s; });
          array old_uids = Array.map (mcache->mails, lambda (mapping m) { return m->imap->UID; });
          array new_mails = new_uids - old_uids;

          if (sizeof (new_mails)) {
            cache->lock();
            mixed err = catch {
              mcache->new_mails |= new_mails; };
            cache->unlock();
            if(err)
              report_error("error in imapclient: %s\n", describe_backtrace(err));
            IMAP_DEBUG (sprintf ("new mails: %d/%d\n", sizeof (new_mails), sizeof (mcache->new_mails)));
          }
          else {
            IMAP_DEBUG ("no new mails.\n");
          }
          new_mails = 0;

          array del_mails = old_uids - new_uids;
          if (sizeof (del_mails)) {
            IMAP_DEBUG(sprintf ("deleted mails: %d\n", sizeof (del_mails)));
            array (mapping) deleted_mails = ({ });
            foreach (mcache->mails, mapping m)
            if (has_value(del_mails, m->imap->UID)) {
              // write ("deleted UID: " + m->imap->UID + "\n");
              deleted_mails += ({ m });
              del_mails -= ({ m->imap->UID });
              if (!sizeof (del_mails))
                break;
            }
            cache->lock();
            mixed err = catch {
              mcache->mails -= deleted_mails;
            };
            cache->unlock();
            if(err)
              report_error("error in imapclient: %s\n", describe_backtrace(err));
          }
          else {
            IMAP_DEBUG ("no deleted mails.\n");
          }
          new_uids = old_uids = 0;
          cache->lock();
          mixed err = catch {
            mcache->changed = 0;
          };
          cache->unlock();
          if(err)
            report_error("error in imapclient: %s\n", describe_backtrace(err));
          cache->save();
        }
        else
          if (sscanf (line, imap_tag () + " %s", foo)) {
            command_result = (foo[0..1] == "OK") ? 0 : -1;
            if(command_result == -1)
              report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at search %O\n", line);
            next_command ();
          }
      }; break; // 4
    }
  }

  // low_list
  void start_low_list () {

    mailpath = ({ "", sessobj->mailpath, sessobj->sharedpath });
    mailboxes = ({ });
    // First we list the INBOX
    imapwrite (imap_tag () + " LIST \"" +mailpath[0] + "\" INBOX\r\n");
    timeout (TIMEOUT1);
  }

  void handle_low_list () {
 
    switch (command_seq) 
    {
    case 0: // expect "* LIST" or command result
      if (sscanf (line, "* LIST %s", foo))
      {
        data = foo + "\n" + data;
        init_parser (PT_LIST);
        command_seq = 1;
      }
      else
        if (sscanf (line, imap_tag () + " %s", foo))
        {
          if (foo[0..1] == "OK")
          {

	    if (sizeof(mailpath) > 1) 
	    {
	      // We have some other path to look at
	      mailpath = mailpath[1..];
	      IMAP_DEBUG ("Listing next path (" + mailpath[0] + ")...\n");
	      imapwrite (imap_tag () + " LIST \"" + mailpath[0] + "\" *\r\n");
	      timeout (TIMEOUT1);
	      command_seq = 0;
	      break;
	    }
	    else mailpath = 0; // for the gc

            command_result = 0;
            // there should always be an INBOX...
            if(Array.search_array(mailboxes,
                                  lambda(array a)
				  {
				    return (a[MB_FOLDERNAME_IDX] == "INBOX");
                                  }) == -1)
	      mailboxes += ({ ({ "INBOX","INBOX",
	  		         MB_NOINFERIORS,0,
				 ({"INBOX"})  }) });
	    
            // add folders that are missing in the hierarchy
            for (int i; i < sizeof (mailboxes); i++)
            {
              if ((Array.search_array (mailboxes,
                                       lambda (array mbox_a, array hierarchy)
                                     {
                                       if (sizeof (hierarchy) < 2)
                                           return 1;
                                         if (sizeof (hierarchy) - 1 != sizeof (mbox_a[MB_HIERARCHY_IDX]))
                                           return 0;
                                         int i;
                                         for (i = 0; i < sizeof (hierarchy) - 1; i++)
                                           if (hierarchy[i] != mbox_a[MB_HIERARCHY_IDX][i])
                                             return 0;
                                         return 1;
                                       }, mailboxes[i][MB_HIERARCHY_IDX])) == -1)
              {
                // add implicit folder
                int foo = sizeof (mailboxes[i][MB_SEPARATOR_IDX])+
                          sizeof (mailboxes[i][MB_HIERARCHY_IDX][-1]);
                string fname = mailboxes[i][MB_FOLDERNAME_IDX][0..sizeof(mailboxes[i][MB_FOLDERNAME_IDX])-1-foo];
                string dname = mailboxes[i][MB_DISPLAYNAME_IDX][0..sizeof(mailboxes[i][MB_DISPLAYNAME_IDX])-1-foo];
                mailboxes = mailboxes + ({ ({ fname, dname, MB_NOSELECT | MB_IMPLICIT_FOLDER,mailboxes[i][MB_SEPARATOR_IDX],mailboxes[i][MB_HIERARCHY_IDX][0..sizeof(mailboxes[i][MB_HIERARCHY_IDX])-2] }) });
              }
            }
            // sort mailboxes. INBOX should always be first.
            mailboxes = sort(mailboxes);
            for(int i = 0; i < sizeof(mailboxes); i++)
            {
              if(mailboxes[i][MB_FOLDERNAME_IDX] == "INBOX")
              {
                array tmp = mailboxes[0];
                mailboxes[0] = mailboxes[i];
                mailboxes[i] = tmp;
                continue;
              }
            }
      	    sessobj->mailboxes = mailboxes;
          }
          else
          {
            report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at list %O\n", line);
            command_result = -1;
          }
          next_command ();
        }
        else
          timeout(TIMEOUT1);
      break;

    case 1: // we should have a list of flags now
      if(!parse_result) {
        report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at list, parse error %O\n", line);
        command_result=-1;
        next_command();
      }
      else {
        tmp_array=parse_result;
        command_seq=2;
        tokens_to_get=2;
        init_parser(PT_TOKENS);
        timeout(TIMEOUT1);
      }
      break; // 1

    case 2:
      if(!parse_result) {
        report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at list, parse error %O\n", line);
        command_result=-1;
        next_command();
      }
      else {
        // ({ foldername, displayname, flags, separator, displayname/separator })
        string foldername = parse_result[1];
        if (upper_case (foldername) == "INBOX")
          foldername = "INBOX";
        string separator = parse_result[0];
        int flags = 0;
        string displayname = foldername;
	sscanf(foldername, mailpath[0] + "%s", displayname);
        //oliv3 RFCCHECK sscanf (displayname, separator + "%s", displayname);
        // if the mailbox name was given as a quoted string, we may want to decode it
        if (sessobj->mboxencode)
          displayname = CAMAS.Tools.decode_mboxname (displayname);
        if ((displayname != "") && 
	    (sessobj->showprefsbox || (foldername != sessobj->prefsbox)) &&
	    (!sessobj->hidemboxes || ((separator ? displayname / separator : ({ displayname }))[-1][0..sizeof (sessobj->hidemboxes) - 1] != sessobj->hidemboxes))) {
          array flag_a = Array.map (tmp_array, lambda (string s) { return lower_case (s); });
          if (has_value (flag_a, "\\noselect"))
            flags |= MB_NOSELECT;
          if (has_value (flag_a, "\\noinferiors") || has_value (flag_a, "\\hasnochildren"))
            flags |= MB_NOINFERIORS;

	  if(Array.search_array(mailboxes,
				lambda(array a)
                                {
                                  return (a[MB_FOLDERNAME_IDX] == foldername);
				}) == -1)
	    mailboxes += ({ ({ foldername,
			       displayname,
			       flags,
			       separator,
			       separator ? ((foldername / separator) - ({ "" })) : ({ foldername }) }) });
	  //write(sprintf("mailboxes: %O\n", mailboxes));
          //separator ? (displayname / separator) : ({ displayname }) }) });
        }
        timeout(TIMEOUT1);
        command_seq=0;
      }
      break; // 2
    }
  }

  // updatelist
  void start_updatelist () {
    // command->why = reason of the update
    // if create
    //	newmailbox
    // if rename
    //	oldfoldername
    //	newfoldername
    // if delete
    //  folders
    mailpath = 0;
    mailboxes = sessobj->mailboxes;

#define SHOULD_EXIST 0
#define SHOULDNT_EXIST 1
    switch (command->why) 
    {
    case "create": mailpath = ({ SHOULD_EXIST, command->newmailbox }); break;
    case "rename": mailpath = ({ SHOULDNT_EXIST, command->oldfoldername, 
				 SHOULD_EXIST, command->newfoldername }); break;
    case "delete": 
      mailpath = ({ });
      foreach (get_cmd_arg (command->folders), string mbox)
	mailpath += ({ SHOULDNT_EXIST, mbox }); 
      break;
    default:
      next_command();
      //do an error
    }
    IMAP_DEBUG("Checking "+mailpath[0]+" on "+mailpath[1]+"...\n");
    if (!(sizeof (mailpath) % 2 == 0)) {
      command_result=0;
      next_command ();
    }
    imapwrite (imap_tag () + " LIST \"\" " + mailpath[1] + "\r\n");
    timeout (TIMEOUT1);
  }

  void handle_updatelist () {
    switch (command_seq)
    {
    case 0: // Verify the presence of the mailbox
      if (sscanf (line, "* LIST %s", foo))
      {
	if (mailpath[0] == SHOULD_EXIST) 
	{
	  data = foo + "\n" + data;
	  init_parser (PT_LIST);
	  command_seq = 1;
	}
	else 
	{
	  // Bouh pas beau
	  mailpath[0] = SHOULD_EXIST; // don't delete it because something hadn't worked
	}
      }
      else
        if (sscanf (line, imap_tag () + " %s", foo))
        {
          if (foo[0..1] == "OK")
	  {
	    // we have to free mailpath[0] in sessobj->mailboxes
	    if (mailpath[0] == SHOULDNT_EXIST) 
	    {
	      foreach (mailboxes, array mbox_a)
	      {
		if (mbox_a[MB_FOLDERNAME_IDX] ==  mailpath[1])
		{
		  IMAP_DEBUG("Removing " + mailpath[1]+" from mailboxes\n");
		  mailboxes -= ({ mbox_a });
		  break;
		}
	      }
	    }

	    // check the next path
	    if (sizeof(mailpath)>2) 
	    {
	      mailpath = mailpath[2..];

	      IMAP_DEBUG("Checking "+mailpath[0]+" on "+mailpath[1]+"...\n");
	      imapwrite (imap_tag () + " LIST \"\" " + mailpath[1] + "\r\n");
	      command_seq=0;
	      timeout(TIMEOUT1);
	      break;
	    }
	    else 
	    {
	      IMAP_DEBUG("Update finished, saving new mailboxes\n");
	      sessobj->mailboxes = mailboxes;
	      mailpath = 0;
	      command_result=0;
	      next_command ();
	    }
	  }
	  else
          {
            report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at list %O\n", line);
            command_result = -1;
	    next_command();
          }
        }
        else
          timeout(TIMEOUT1);
      break;
      
    case 1: // Analyse LIST result
      // we should have a list of flags now
      if(!parse_result) {
        report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at list, parse error %O\n", line);
	command_result = -1;
	mailpath = 0;
	next_command();
      }
      else {
        tmp_array=parse_result;
        command_seq=2;
        tokens_to_get=2;
        init_parser(PT_TOKENS);
        timeout(TIMEOUT1);
      }
      break; // 3

    case 2: // Analyse LIST result
      if(!parse_result) {
        report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at list, parse error %O\n", line);
	command_result = -1;
	mailpath = 0;
	next_command();
      }
      else {
        // ({ foldername, displayname, flags, separator, displayname/separator })
        string foldername = parse_result[1];
        if (upper_case (foldername) == "INBOX")
          foldername = "INBOX";
        string separator = parse_result[0];
        int flags = 0;
        string displayname = foldername;

	object reg = Regexp("$("+command->mailpath+"|"+command->sharedpath+")?(.*)");
	array|int res = reg -> split(displayname);
	IMAP_DEBUG("Regsplit give: " +sprintf("%O\n",res));
	if (res != 0)
	  displayname= ((array)(reg -> split(displayname)))[1];
	reg = 0;

        //oliv3 RFCCHECK sscanf (displayname, separator + "%s", displayname);
        // if the mailbox name was given as a quoted string, we may want to decode it

        if (sessobj->mboxencode)
          displayname = CAMAS.Tools.decode_mboxname (displayname);

        if ((displayname != "") && (sessobj->showprefsbox || (foldername != sessobj->prefsbox))
            && (!sessobj->hidemboxes || ((separator ? displayname / separator : ({ displayname }))[-1][0..sizeof (sessobj->hidemboxes) - 1] != sessobj->hidemboxes))) {
          array flag_a = Array.map (tmp_array, lambda (string s) { return lower_case (s); });
          if (has_value (flag_a, "\\noselect"))
            flags |= MB_NOSELECT;
          if (has_value (flag_a, "\\noinferiors") || has_value (flag_a, "\\hasnochildren"))
            flags |= MB_NOINFERIORS;

	  if(Array.search_array(mailboxes,
				lambda(array a)
                                {
                                  return (a[MB_FOLDERNAME_IDX] == foldername);
				}) == -1) 
	  {
	    IMAP_DEBUG("Adding "+mailpath[1]+" from mailboxes\n");

	    mailboxes += ({ ({ foldername,
			       displayname,
			       flags,
			       separator,
			       separator ? ((foldername / separator) - ({ "" })) : ({ foldername }) }) });
	    //write(sprintf("mail	boxes: %O\n", mailboxes));
	    //separator ? (displayname / separator) : ({ displayname }) }) });
	  }
        }
	command_seq = 0;
	timeout(TIMEOUT1);
      }
      break;
    }
    
  }

  // low_logout
  void start_low_logout () {
#if constant(thread_create)
      _lock = camas_main->global_lock->lock();
#endif
    mixed err = catch {  
      fd->set_close_callback (0);
      imapwrite (imap_tag () + " LOGOUT\r\n");
      fd->set_read_callback (0);
      fd->set_write_callback (0);
      fd->close ();
      destruct(fd);
      fd = 0;
    };
#if constant(thread_create)
    destruct(_lock);
#endif
    if(err)
      report_error("error in imapclient: %s\n", describe_backtrace(err));
    remove_call_out (timeout_call_out);
    remove_call_out (imap_idle_call_out);
    timeout_call_out = 0;
    imap_idle_call_out = 0;
    selected_mailbox = 0;
    command_result = 0;
    abort = 1;
    sessobj->connected = 0;
    next_command ();
  }

  void handle_low_logout () {

  }

  // low_close
  void start_low_close () {
    imapwrite (imap_tag () + " CLOSE\r\n");
    timeout (TIMEOUT1);
  }

  void handle_low_close () {
    switch (command_seq) {
    case 0:
      if (sscanf (line, imap_tag () + " %s", foo)) {
        // whatever
        command_result = 0;
        selected_mailbox = 0;
        next_command ();
      }
      else
        timeout (TIMEOUT1);
      break; //0
    }
  }

  // get_headers
  void start_get_headers () {
    mapping mcache = cache->mcache;
    set_cmd_arg (command->output, mcache->mails);
    if(command->setsortcolumn)
      set_cmd_arg (command->setsortcolumn, "num");
   
    if (sizeof (mcache->new_mails)) {
      if (!sizeof (mcache->mails)) {
        imap_newtag ();
        IMAP_DEBUG ("getting all headers (" + selected_mailbox + ")...\n");
        set_cmd_arg (command->output, ({  }));
        imapwrite (imap_tag () + " FETCH 1:* (INTERNALDATE ENVELOPE UID FLAGS RFC822.SIZE BODY)\r\n");
        // should be IMAP4 RFC-equivalent (see ALL definition). but WU-IMAP doesn't know about this :(
        // imapwrite (imap_tag () + " FETCH 1:* (UID ALL)\r\n");
        sessobj->foldersinfo[selected_mailbox][MB_SIZE_IDX] = 0;
        cache->lock();
        mixed err = catch {
          mcache->new_mails = ({ });
        };
        cache->unlock();
        if(err)
          report_error("error in imapclient: %s\n", describe_backtrace(err));
        timeout (TIMEOUT1);
      }
      else
        if (sizeof (mcache->new_mails)) {
          imap_newtag ();
          IMAP_DEBUG ("getting some headers (" + selected_mailbox + ": " + sizeof (mcache->new_mails) + ")...\n");
          string fuids;
          cache->lock();
          mixed err = catch {
            mcache->new_mails = Array.map (mcache->new_mails, lambda (mixed m) { return (string)m; } );
            fuids = mcache->new_mails * ",";
          };
          cache->unlock();
          if(err)
            report_error("error in imapclient: %s\n", describe_backtrace(err));
          imapwrite (imap_tag () + " UID FETCH " + fuids + " (INTERNALDATE ENVELOPE UID FLAGS RFC822.SIZE BODY)\r\n");
          cache->lock();
          err = catch {
            mcache->new_mails = ({ });
          };
          cache->unlock();
          if(err)
            report_error("error in imapclient: %s\n", describe_backtrace(err));
          timeout (TIMEOUT1);
        }
    }
    else {
      IMAP_DEBUG("unchanged mailbox\n");
      command_result=0;
      next_command();
    }
    cache->save();
  }

  void handle_get_headers () {
    switch (command_seq) {
    case 0: // expect mailheaders or command result
      if (sscanf (line, "* %*d FETCH%s", foo) == 2) {
        data = foo + "\n" + data;
        init_parser (PT_MAPPING);
        command_seq = 1;
        timeout (TIMEOUT1);
      }
      else
        if (sscanf (line, imap_tag () + " %s", foo)) {
          if(foo[0..1] == "OK") { // done
            cache->lock();
            mixed err = catch {
              cache->mcache->mails = get_cmd_arg (command->output) || ([ ]);
              cache->mcache->new_mails = ({ });
              // this is just to force the saving into the cache
              cache->mcache->changed = 1;
            };
            cache->unlock();
            if(err)
              report_error("error in imapclient: %s\n", describe_backtrace(err));
            cache->save();
            cache->lock();
            err = catch {
              cache->mcache->changed = 0;
            };
            cache->unlock();
            if(err)
              report_error("error in imapclient: %s\n", describe_backtrace(err));
            IMAP_DEBUG ("setting mail numbers ...\n");
            int number = 0;
            foreach (cache->mcache->mails, mapping m)
              m->number = number++;
            command_result = 0;
            next_command ();
          }
          else {
            report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at fetch %O\n", line);
            command_result = -1;
            next_command ();
          }
        }
        else
          timeout (TIMEOUT1);
      break; //0

    case 1: // expect parsed headers
      if (!parse_result) {
        report_warning ("CAMAS : IMAP parse error!\n");
        command_result = -1;
        next_command ();
      }
      else {
        object _lock;
        data = line + "\n" + data;
        parse_result->UID = (int)parse_result->UID;
        parse_result["RFC822.SIZE"] = (int)parse_result["RFC822.SIZE"];
        // storing the mailbox in each message allow us to browse message 
        // from different mailboxes in the same mailindex screen 
        parse_result["MAILBOX"] = selected_mailbox;
        LOCK_FEEDER();
        mixed err = catch {
          sessobj->foldersinfo[selected_mailbox][MB_SIZE_IDX] += parse_result["RFC822.SIZE"];
          array tmp = get_cmd_arg (command->output) || ({ });
          set_cmd_arg (command->output, tmp + ({ ([ "imap": parse_result ]) }));
        };
        UNLOCK_FEEDER();
        if(err)
          report_error("error in imapclient: %s\n", describe_backtrace(err));
        else
        {
          //write ("updating folders info\n");
          sessobj->allfolderssize = 0;
          foreach (indices (sessobj->foldersinfo), string key)
          if (sizeof (sessobj->foldersinfo[key]) == 5) // it's a leaf folder
            sessobj->allfolderssize += sessobj->foldersinfo[key][MB_SIZE_IDX];
        }
  			command_seq = 0;
        timeout (TIMEOUT1);
      }
      break; // 1
    }
  }

  // apply_mail_filters
  void start_apply_mail_filters () {
    if (sessobj->mailbox[MB_FOLDERNAME_IDX] != "INBOX" || !sessobj->mails
        || !sizeof (sessobj->mails) || !strlen (sessobj->filterbook) || !sessobj->filters_have_changed) {
      if (!sessobj->filters_have_changed)
        IMAP_DEBUG("Filters have not changed or no new mail => nothing to do.\n");
      sessobj->filters_have_changed = 0;
      command_result = 0;
      next_command ();
    }
    else {
      if (sessobj->filters_have_changed)
      IMAP_DEBUG("Filters have changed or new mail has arrived => applying filters.\n");
      sessobj->filters_have_changed = 0;
      int mails = sizeof (sessobj->mails);
      array (mapping) deleted_mails = ({ });

      array (array (string)) filters = CAMAS.Filters.get_filters(id);
      if (sizeof (filters))
        for (int i = 0; i < mails; i++) {
          int match = 0;
          //write (sprintf ("mail #%d, date= %s\n", i, (string)sessobj["mails"][i]->imap->ENVELOPE[DATE_IDX]));
          foreach (filters, array filter) {
            switch (filter[0]) {
            case "From" :
              if (sessobj->mails[i]->imap->ENVELOPE[FROM_IDX])
                if (has_value (CAMAS.Tools.fix_header (sessobj->mails[i]->imap->ENVELOPE[FROM_IDX][0][0])+" "+CAMAS.Tools.fix_header (sessobj->mails[i]->imap->ENVELOPE[FROM_IDX][0][2])+"@"+CAMAS.Tools.fix_header (sessobj->mails[i]->imap->ENVELOPE[FROM_IDX][0][3]), filter[1])) {
                  match = 1;
                  //write ("matched on From\n");
                }
              break;

            case "Subject" :
              if (has_value (CAMAS.Tools.fix_header (sessobj->mails[i]->imap->ENVELOPE[SUBJECT_IDX]), filter[1])) {
                match= 1;
                //write ("matched on Subject\n");
              }
              break;

            case "Date" :
              if (has_value (CAMAS.Tools.fix_header (sessobj->mails[i]->imap->ENVELOPE[DATE_IDX]), filter[1])) {
                match= 1;
                //write ("matched on Date\n");
              }
              break;

            case "To" :
              if (sessobj->mails[i]->imap->ENVELOPE[TO_IDX]) {
                //write (sprintf ("size= %d\n", sizeof (sessobj["mails"][i]->imap->ENVELOPE[TO_IDX])));
                int nb_to = sizeof (sessobj->mails[i]->imap->ENVELOPE[TO_IDX]);
                for (int to = 0; to < nb_to; to++) {
                  string to_dest = CAMAS.Tools.fix_header (sessobj->mails[i]->imap->ENVELOPE[TO_IDX][to][0])+" "+CAMAS.Tools.fix_header(sessobj->mails[i]->imap->ENVELOPE[TO_IDX][to][2])+"@"+CAMAS.Tools.fix_header(sessobj->mails[i]->imap->ENVELOPE[TO_IDX][to][3]);
                  //write (sprintf ("checkTO: to= %d, dest= %s\n", to, to_dest));
                  if (has_value (to_dest, filter[1])) {
                    match = 1;
                    //write ("matched on To\n");
                    break;
                  }
                }
              }
              break;

            case "Cc" :
              if (sessobj->mails[i]->imap->ENVELOPE[CC_IDX]) {
                //write (sprintf ("size= %d\n", sizeof (sessobj["mails"][i]->imap->ENVELOPE[CC_IDX])));
                int nb_cc = sizeof (sessobj->mails[i]->imap->ENVELOPE[CC_IDX]);
                for (int cc = 0; cc < nb_cc; cc++) {
                  string cc_dest = CAMAS.Tools.fix_header (sessobj->mails[i]->imap->ENVELOPE[CC_IDX][cc][0])+" "+CAMAS.Tools.fix_header (sessobj->mails[i]->imap->ENVELOPE[CC_IDX][cc][2])+"@"+CAMAS.Tools.fix_header(sessobj->mails[i]->imap->ENVELOPE[CC_IDX][cc][3]);
                  //write (sprintf ("checkCC: cc= %d, dest= %s\n", cc, cc_dest));
                  if (has_value (cc_dest, filter[1])) {
                    match = 1;
                    //write ("matched on Cc\n");
                    break;
                  }
                }
              }
              break;
            default: break;
            }

            if (match) {
              // write ("match on #" + i + " => " + filter[2] + "\n");
              commands += ({ CAMAS.IMAPTools.imap_cmd ("copy",
                                       "uids", ({ sessobj->mails[i]->imap->UID }),
                                       "copytobox", filter[2],
                                       "updatembox", command->updatembox),
                             CAMAS.IMAPTools.imap_cmd ("delete",
                                       "uids", ({ sessobj->mails[i]->imap->UID }),
                                       "updatembox", command->updatembox,
                                       "noexpunge", 1) });
              deleted_mails += ({ sessobj->mails[i] });
              break;
            }
          }
        }

      if (sizeof (deleted_mails)) {
        commands += ({ CAMAS.IMAPTools.imap_cmd ("expunge") });
        sessobj->mails -= deleted_mails;
        deleted_mails = ({ });
        object lcache = Cache(selected_mailbox, sessobj, camas_main);
        lcache->lock();
        mixed err = catch {
          lcache->mcache->mails = sessobj->mails;
        };
        lcache->unlock();
        if(err)
          report_error("error in imapclient: %s\n", describe_backtrace(err));
        set_cmd_arg (command->updatembox, sessobj->mails);
      }

      command_result = 0;
      next_command ();
    }
  }

  void handle_apply_mail_filters () {
    command_result = 0;
    next_command ();
  }

  // get_prefs_mail
  void start_get_prefs_mail () {
    command->uid = -1;
    foreach (indices (sessobj->prefsmails), int i)
    {
      object mail = sessobj->prefsmails[i];
      if (mail->imap->ENVELOPE[SUBJECT_IDX] == "CAMAS Preferences")
        command->uid = sessobj->prefsmails[i]->imap->UID;
    }
    if ((command->uid == -1) && camas_main->QUERY (upgrading)) {
      foreach (indices (sessobj->prefsmails), int i)
      {
        object mail = sessobj->prefsmails[i];
        if (mail->imap->ENVELOPE[SUBJECT_IDX] == "IMHO Preferences")
          command->uid = sessobj->prefsmails[i]->imap->UID;
      }
    }
    if (command->uid >= 0)
    {
      command->abortfail = 0;
      command->subcommands = ({
                                CAMAS.IMAPTools.imap_cmd ("get_mail", "uid", command->uid,
                                          "mailbox",sessobj->prefsbox,
                                          "abortfail", 1, "output", CAMAS.IMAPTools.imap_cmd_var ("prefsmailobj"))
                              });
      commands = ({ command }) + commands;
    }
    command_result = 0;
    next_command ();
  }

  // get_ebook_mail
  void start_get_ebook_mail () {
    command->uid = -1;
    foreach (indices (sessobj->prefsmails), int i)
    {
      object mail = sessobj->prefsmails[i];
      if (mail->imap->ENVELOPE[SUBJECT_IDX] == HEADER_EBOOK_SUBJECT_VALUE)
        command->uid = sessobj->prefsmails[i]->imap->UID;
    }
    if (command->uid >= 0)
    {
      command->abortfail = 0;
      command->subcommands = ({
                                CAMAS.IMAPTools.imap_cmd ("get_mail", "uid", command->uid,
                                          "mailbox", sessobj->prefsbox,
                                          "abortfail", 1, "output", CAMAS.IMAPTools.imap_cmd_var ("ebookmailobj"))
                              });
      commands = ({ command }) + commands;
    }
    command_result = 0;
    next_command ();
  }

  // get_mail
  void start_get_mail () {
    imapwrite (imap_tag () + " UID FETCH " + command->uid + " RFC822\r\n");
    timeout (TIMEOUT1);
  }

  void handle_get_mail () {
    switch (command_seq) {
    case 0: // expect data or command result
      if (sscanf (line, "* %*d FETCH%s", foo) == 2) {
        data = foo + "\n" + data;
        init_parser (PT_MAPPING);
        command_seq = 1;
        timeout (TIMEOUT1);
      }
      else
        if (sscanf (line, imap_tag () + " %s", foo)) {
          if (foo[0..1] == "OK") {
            commands += ({ CAMAS.IMAPTools.imap_cmd ("add_flag", "uid", command->uid, "flag", "\\Seen") });
            command_result = 0;
            next_command ();
          }
          else {
            report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at fetch %O\n", line);
            command_result = -1;
            next_command ();
          }
        }
        else
          timeout (TIMEOUT1);
      break; //0

    case 1: // expect parsed mapping
      if (!parse_result) {
        report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at fetch, parse error %O\n", line);
        command_result = -1;
        next_command ();
      }
      else {
        data = line + "\n" + data;
        if (!zero_type (parse_result->RFC822)) {
          //write ("got RFC822\n");
          object mailobj = 0;
          mixed mime_error = catch {
                               mailobj = MIME.Message (parse_result->RFC822);
                               if (mailobj)
                                 string test = mailobj->getdata ();
                               };

        string mail = parse_result->RFC822;

        if (mime_error || !mailobj) {
            // Fix a number of problems that makes the MIME module fail.
            write ("MIME.Message failed: " + mime_error[0]);
            mail = fixmail (mail);
            mime_error = catch { mailobj = MIME.Message (mail); };
            if (mime_error)
              report_error ("MIME.Message failed: " + describe_backtrace(mime_error));
          }

          set_cmd_arg (command->output, mailobj);
          parse_result = 0;
        }
        else
          if (!zero_type (parse_result->FLAGS)) {
            //oliv3 FIXME: update the flags
            //write ("got FLAGS\n");
            /*
            write ("uid= " + command->uid + " ( " + typeof (command->uid) + " )\n");
            write ("sizeof (sessobj->mails)= " + sizeof (sessobj->mails) + "\n");
            write (sprintf ("FLAGS= %O\n", parse_result->FLAGS));
            */
            for (int i = 0; i < sizeof (sessobj->mails); i++) {
              //write ("UID= " + sessobj->mails[i]->imap->UID + " ( " + typeof (sessobj->mails[i]->imap->UID) + " )\n");
              if ((int)sessobj->mails[i]->imap->UID == (int)command->uid) {
                //write (sprintf ("old FLAGS= %O\n" + sessobj->mails[i]->imap->FLAGS));
                sessobj->mails[i]->imap->FLAGS |= parse_result->FLAGS;
                //write (sprintf ("new FLAGS= %O\n" + sessobj->mails[i]->imap->FLAGS));
                break;
              }
            }
          }
        command_seq = 0;
        timeout (TIMEOUT1);
      }

      break; // 1
    }
  }

  // delete
  void start_delete () {
    array mails_to_delete = get_cmd_arg (command->uids);
    if (mails_to_delete)
    {
      int nuids = sizeof (mails_to_delete);
      if (!nuids)
      {
        command_result = 0;
        next_command ();
      }
      else
      {
      	mails_to_delete = Array.map (mails_to_delete, lambda (mixed m) { return (string)m; } );
        string duids = mails_to_delete * ",";
        //write ("DELETING: " + duids + "\n");
        imapwrite (imap_tag () + " UID STORE " + duids + " +FLAGS.SILENT (\\Deleted)\r\n");
        timeout (TIMEOUT1);
      }
    }
    else
    {
      command_result = 0;
      next_command ();
    }
  }

  void handle_delete () {
    // expect store command result
    if (sscanf (line, imap_tag () + " %s", foo))
      if (foo[0..1] == "OK") {
        if (!command->noexpunge) {
          // all mails flagged. now expunge
          imap_newtag ();
          imapwrite (imap_tag () + " EXPUNGE\r\n");

          array mails = get_cmd_arg (command->updatembox);
          if (mails) 
	  {
            array delmails = ({ });
            array deletedmail = get_cmd_arg (command->uids);

            foreach (deletedmail, mixed uid)
            if (intp (uid)) // deleting a single mail
              delmails += Array.filter (mails,
                                        lambda (mapping m) {
                                        return (m->imap->UID == uid);
                                      });
            else { // deleting several mails
              int start, end;
              sscanf (uid, "%d:%d", start, end);
              delmails += Array.filter (mails,
                                        lambda (mapping m) {
                                          return ((m->imap->UID >= start) && (m->imap->UID <= end));
                                        });
            }

            mails -= delmails;
            object lcache = Cache(selected_mailbox, sessobj, camas_main);
            lcache->lock();
            mixed err = catch {
              lcache->mcache->mails = mails;
            };
            lcache->unlock();
            if(err)
              report_error("error in imapclient: %s\n", describe_backtrace(err));
            set_cmd_arg (command->updatembox, mails);
          }
        }

        command_result = 0;
        next_command ();
      }
      else {
        report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at delete %O\n", line);
        command_result = -1;
        next_command ();
      }
  }

  // move
  void start_move () {
    // a move is a copy + a delete
    command->subcommands = (command->copytobox) ?
                           ({ CAMAS.IMAPTools.imap_cmd ("copy",
							"uids", command->uids,
							"copytobox", command->copytobox,
							"updatembox", command->updatembox) })
                           : ({ CAMAS.IMAPTools.imap_cmd ("copy",
							  "uids", command->uids,
							  "updatembox", command->updatembox) });
    command->subcommands +=
      ({ CAMAS.IMAPTools.imap_cmd ("delete",
				   "uids", command->uids,
				   "updatembox", command->updatembox) });
    commands += ({ command });
    command_result = 0;
    next_command ();
  }

  // copy
  void start_copy () {
    array mails_to_copy = get_cmd_arg (command->uids);
    if(!arrayp(mails_to_copy)) 
      mails_to_copy = ({ });
    if (command->copytobox)
      sessobj->copytobox = get_cmd_arg (command->copytobox);
    IMAP_DEBUG("Copying to "+sessobj->copytobox+"\n");
    int nuids = sizeof (mails_to_copy);
    if (!nuids) {
      command_result = 0;
      next_command ();
    }
    else {
      mails_to_copy = Array.map (mails_to_copy, lambda (mixed m) { return (string)m; } );
      string cuids = mails_to_copy * ",";
      //write ("COPYING: " + cuids + "\n");
      if (sessobj->mboxencode)
        imapwrite (imap_tag () + " UID COPY " + cuids + " " + IMAP_QSTRING (sessobj->copytobox) + "\r\n");
      else {
        imapwrite (imap_tag () + " UID COPY " + cuids + " ");
        imapwrite_literal (sessobj->copytobox, 1, 0);
      }
      trycreate = 0;
      timeout (TIMEOUT1);
    }
  }

  void handle_copy () {
    switch (command_seq) {
    case 0: // check COPY command result
      if (sscanf (line, imap_tag () + " %s", foo)) {
        if (foo[0..1] == "OK") {
          command_result = 0;
          next_command ();
        }
        else {
          // a lot of IMAP server don't send TRYCREATE in cases where they must
          // according to the RFC so let's just try to create a mailbox
          // anyway
          trycreate = 1;
          if (sscanf (foo, "NO %s", foo) && (trycreate /* || has_value (foo, "[TRYCREATE]") */)) {
            imap_newtag ();
            if (sessobj->mboxencode)
              imapwrite (imap_tag () + " CREATE " + IMAP_QSTRING (sessobj->copytobox) + "\r\n");
            else {
              imapwrite (imap_tag () + " CREATE ");
              imapwrite_literal (sessobj->copytobox, 1, 0);
            }
            command_seq = 2;
            timeout (TIMEOUT1);
          }
          else {
            report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at copy %O mbox=%O\n", line,
                sessobj->copytobox);
            command_result = -1;
            next_command ();
          }
        }
      }
      else
        timeout (TIMEOUT1);
      break; //0

    case 1: // expect COPY command result
      if (sscanf (line, imap_tag () + " %s", foo)) {
        if (foo[0..1] == "OK") {
          command_result = 0;
          next_command ();
        }
        else {
          report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at copy %O mbox=%O\n", 
              line, sessobj->copytobox);
          command_result = -1;
          next_command ();
        }
      }
      else
        timeout (TIMEOUT1);
      break; //1

    case 2: // Retry copying after CREATE
      if (sscanf (line, imap_tag () + " %s", foo)) {
        if (foo[0..1] == "OK") {
          imap_newtag ();

          array mails_to_copy = get_cmd_arg (command->uids);

      	  mails_to_copy = Array.map (mails_to_copy, lambda (mixed m) { return (string)m; } );
          string cuids = mails_to_copy * ",";

          if (sessobj->mboxencode)
            imapwrite (imap_tag () + " UID COPY " + cuids + " " + IMAP_QSTRING (sessobj->copytobox) + "\r\n");
          else {
            imapwrite (imap_tag () + " UID COPY " + cuids + " ");
            imapwrite_literal (sessobj->copytobox, 1, 0);
          }

          command_seq = 1;
          timeout (TIMEOUT1);
        }
        else {
          report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at copy %O, mbox=%O\n",
              line, sessobj->copytobox);
          command_result = -1;
          next_command ();
        }
      }
      else
        timeout (TIMEOUT1);
      break; //2
    }
  }

  // low_create
  void start_low_create () {
    IMAP_DEBUG("Creating " + command->newmailbox + "\n");
    if (sessobj->mboxencode)
      imapwrite (imap_tag () + " CREATE " + IMAP_QSTRING (command->newmailbox) + "\r\n");
    else {
      imapwrite (imap_tag () + " CREATE ");
      imapwrite_literal (command->newmailbox, 1, 0);
    }
    timeout (TIMEOUT1);
  }

  void handle_low_create () {
    switch (command_seq) {
    case 0: // expect CREATE command result
      if (sscanf (line, imap_tag () + " %s", foo))
      {
        if (foo[0..1] == "OK")
        {
          // ok. folder created
          commands = ({ CAMAS.IMAPTools.imap_cmd ("cache_init", "mailbox", command->newmailbox) }) + commands;
          command_result = 0;
          next_command ();
        }
        else
        {
          command_result=-1;
          next_command();
        }
      }
      else
        timeout (TIMEOUT1);
      break; //0
    }
  }

  // create
  void start_create () {
    string newmbox;
    string error = command->error;
    if(command->addmailpath)
    {
      command->newmailbox = sessobj->mailpath + command->newmailbox;
    }
    if(command->createonlyifnotexists)
    {
      if(Array.search_array(mailboxes,
				lambda(array a)
                                {
                                  return (a[MB_FOLDERNAME_IDX] == command->newmailbox);
				}) != -1)
      {
        command_result = 0;
        next_command();
        return;
      }
    }
    newmbox = command->newmailbox; 
    if (error)
      commands = ({ CAMAS.IMAPTools.imap_cmd ("low_create", "newmailbox", newmbox, "error", error) }) + commands;
    else
      commands = ({ CAMAS.IMAPTools.imap_cmd ("low_create", "newmailbox", newmbox, "abortfail", 0) }) + commands;
    command_result = 0;
    next_command ();
  }

  // delete_folders
  void start_delete_folders () {
    folderstodelete = ({ });
    foreach (get_cmd_arg (command->folders), string mbox)
    folderstodelete+=({ mbox });

    if (!sizeof (folderstodelete)) {
      command_result=0;
      next_command ();
    }
    else {
      if (sessobj->mboxencode)
        imapwrite (imap_tag () + " DELETE " + IMAP_QSTRING (folderstodelete[0]) + "\r\n");
      else {
        imapwrite (imap_tag () + " DELETE ");
        imapwrite_literal (folderstodelete[0], 1, 0);
      }
      timeout (TIMEOUT1);
    }
  }

  void handle_delete_folders () {
    // expect DELETE command result
    if (sscanf (line, imap_tag () + " %s", foo)) {
      if (foo[0..1] == "OK") {
        CAMAS.Filters.delete_filter(id, 0, 0, folderstodelete[0]);
      	object features = camas_main->my_configuration ()->get_provider ("camas_features");
        if (features->QUERY (foldersinfo)) {
          sessobj->allfolderssize -= sessobj->foldersinfo[folderstodelete[0]][MB_SIZE_IDX];
          m_delete (sessobj->foldersinfo, folderstodelete[0]);
        }
        object lcache = Cache(folderstodelete[0], sessobj, camas_main);
        lcache->remove();

        folderstodelete=folderstodelete[1..];
        if(sizeof(folderstodelete)) {
          imap_newtag();
          if(sessobj->mboxencode)
            imapwrite(imap_tag()+" DELETE "+IMAP_QSTRING(folderstodelete[0])+"\r\n");
          else {
            imapwrite(imap_tag()+" DELETE ");
            imapwrite_literal(folderstodelete[0],1,0);
          }
          timeout(TIMEOUT1);
        } else {
          command_result=0;
          next_command();
        }
      }
      else {
        report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at delete folders %O\n", line);
        command_result=-1;
        next_command();
      }
    }
    else
      timeout (TIMEOUT1);
  }

  // low_append
  void start_low_append () {
    IMAP_DEBUG("Appending " + command->tobox + "\n");
    if (sessobj->mboxencode) {
      imapwrite (imap_tag () + " APPEND " + IMAP_QSTRING (command->tobox) + " ");
      imapwrite_literal (get_cmd_arg (command->data), 1 ,0);
      command_seq = 1;
    }
    else {
      imapwrite (imap_tag() + " APPEND ");
      imapwrite_literal (command->tobox, 0, 1);
    }
    trycreate = 0;
    timeout (TIMEOUT1);
  }

  void handle_low_append () {
    switch(command_seq) {

    case 0: // send appenddata
      if(line[0..0]!="+") {
        command_seq=1;
        handle_command();
      }
      else {
        imapwrite(" ");
        imapwrite_literal(get_cmd_arg(command->data),1,0);
        command_seq=1;
      }
      break; //0

    case 1: // expect APPEND command result
      if(sscanf(line,imap_tag()+" %s",foo)) {
        if(foo[0..1]=="OK") {
          command_result=0;
          next_command();
        }
        else {
          if(sscanf(foo,"NO %s",foo) && (trycreate /* || has_value (foo, "[TRYCREATE]") */)) {
            imap_newtag();
            if(sessobj->mboxencode)
              imapwrite(imap_tag()+" CREATE "+IMAP_QSTRING(command->tobox)+"\r\n");
            else {
              imapwrite(imap_tag()+" CREATE ");
              imapwrite_literal(command->tobox,1,0);
            }
            command_seq=2;
            timeout(TIMEOUT1);
          }
          else {
            report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at append %O\n", line);
            command_result=-1;
            next_command();
          }
        }
      }
      else {
        // a lot of IMAP server don't send TRYCREATE where they must
        // according to the RFC so let's just try to create a mailbox
        // anyway
        //if (search(line, "[TRYCREATE]")!=-1)
        trycreate = 1;
        timeout(TIMEOUT1);
      }
      break; //1

    case 2: // expect create results
      if(sscanf(line,imap_tag()+" %s",foo)) {
        if(foo[0..1]=="OK") {
          imap_newtag();
          if(sessobj->mboxencode) {
            imapwrite(imap_tag()+" APPEND "+IMAP_QSTRING(command->tobox)+" ");
            imapwrite_literal(get_cmd_arg(command->data),1,0);
            command_seq=4;
          }
          else {
            imapwrite(imap_tag()+" APPEND ");
            imapwrite_literal(command->tobox,0,1);
            command_seq=3;
          }
          timeout(TIMEOUT1);
        }
        else {
          report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at append %O\n", line);
          command_result=-1;
          next_command();
        }
      }
      else
        timeout(TIMEOUT1);
      break; //2

    case 3: // send appenddata
      if(line[0..0]!="+") {
        command_seq=4;
        handle_command();
      }
      else {
        imapwrite(" ");
        imapwrite_literal(get_cmd_arg(command->data),1,0);
        command_seq=1;
      }
      break; // 3

    case 4: // expect APPEND command result
      if(sscanf(line,imap_tag()+" %s",foo)) {
        if(foo[0..1]=="OK") {
          command_result=0;
          next_command();
        }
        else {
          report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at append %O\n", line);
          command_result=-1;
          next_command();
        }
      }
      else
        timeout(TIMEOUT1);
      break; // 4
    }
  }

  // add_flag
  void start_add_flag () {
    imapwrite (imap_tag () + " UID STORE " + command->uid + " +FLAGS.SILENT (" + command->flag + ")\r\n");
    timeout (TIMEOUT1);
  }

  void handle_add_flag () {
    if (sscanf (line, imap_tag () + " %s", foo)) {
      if (foo[0..1] == "OK") {
        command_result = 0;
        if (!command->stop)
          next_command ();
      }
      else {
        report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at add flag %O\n", line);
        command_result = -1;
        next_command ();
      }
    }
    else
      timeout (TIMEOUT1);
  }

  // expunge
  void start_expunge () {
    imapwrite (imap_tag () + " EXPUNGE\r\n");
    timeout (TIMEOUT1);
  }

  void handle_expunge () {
    if (sscanf (line, imap_tag () + " %s", foo)) {
      if (foo[0..1] == "OK")
        command_result = 0;
      else
      {
        report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at expunge %O\n", line);
        command_result = -1;
      }
      next_command ();
    }
    else
      timeout (TIMEOUT1);
  }

  // save_user_prefs
  void start_save_user_prefs () {
    IMAP_DEBUG(sprintf("prefsbox=%s\n", sessobj->prefsbox));
    if(sessobj->usersetup) {
      if (camas_main->QUERY (buggyimap))
      {
        IMAP_DEBUG ("buggyimap: creating " + sessobj->prefsbox + "\n");
        command->subcommands = ({ CAMAS.IMAPTools.imap_cmd ("create", "newmailbox", sessobj->prefsbox) });
      }
      else
        command->subcommands = ({ });

      //command->subcommands = ({ CAMAS.IMAPTools.imap_cmd ("create", "newmailbox", sessobj->prefsbox) });

      command->subcommands += ({
                                 CAMAS.IMAPTools.imap_cmd("status","mailbox",sessobj->prefsbox),
                                 CAMAS.IMAPTools.imap_cmd("get_headers",
                                          "mailbox",sessobj->prefsbox,
                                          "output",CAMAS.IMAPTools.imap_cmd_var("prefsmails"),
                                          "abortfail", 0),
                                 CAMAS.IMAPTools.imap_cmd("create_prefs_mail",
                                          "output",CAMAS.IMAPTools.imap_cmd_var("-")),
                                 CAMAS.IMAPTools.imap_cmd("low_append",
                                          "tobox",sessobj->prefsbox,
                                          "data",CAMAS.IMAPTools.imap_cmd_var("-")),
                                 CAMAS.IMAPTools.imap_cmd("delete",
                                          "mailbox",sessobj->prefsbox,
                                          "uids",CAMAS.IMAPTools.imap_cmd_var("prefuids"))
                               });

      if (feature_extendedabook) {
        command->subcommands += ({
                                   CAMAS.IMAPTools.imap_cmd("create_ebook_mail",
                                            "output", CAMAS.IMAPTools.imap_cmd_var("-")),
                                   CAMAS.IMAPTools.imap_cmd("low_append",
                                            "tobox",sessobj->prefsbox,
                                            "data",CAMAS.IMAPTools.imap_cmd_var("-")),
                                   CAMAS.IMAPTools.imap_cmd("delete",
                                            "mailbox",sessobj->prefsbox,
                                            "uids",CAMAS.IMAPTools.imap_cmd_var("ebookuids"))
                                 });
      }
    }
    commands = ({ command }) + commands;
    command_result = 0;
    next_command ();
  }

  // create_ebook_mail
  void start_create_ebook_mail () {
    if (sessobj->extendedabook)
    {
      ebookmailobj = sessobj->extendedabook->export_mime();
      sessobj->ebookuids=({ });
      if (sessobj->prefsmails) 
      {
        foreach(indices(sessobj->prefsmails),int i) 
	{
          object mail=sessobj->prefsmails[i];
          if (i < sizeof(sessobj->prefsmails) &&
              mail->imap->ENVELOPE[SUBJECT_IDX] == HEADER_EBOOK_SUBJECT_VALUE)
            sessobj->ebookuids += ({ mail->imap->UID });
        }
      }
      set_cmd_arg(command->output,(string)ebookmailobj);
    }
    command_result = 0;
    next_command();
  }

  // create_prefs_mail
  void start_create_prefs_mail () {
    object prefsmailobj = 0;
    prefsmailobj = MIME.Message("",
                          ([ "MIME-Version" : "1.0",
                             "Content-Type" : "text/plain; charset=UTF-8",
                             "User-Agent" : "CAMAS/"+camas_main->camas_version+" (Webmail for Caudium)" ]) );
    prefsmailobj->headers["to"]  = (sessobj)["address"];
    prefsmailobj->headers["from"] = (sessobj)["name"]+" <"+(sessobj)["address"]+">" ;
    prefsmailobj->headers["subject"] = "CAMAS Preferences";
    prefsmailobj->setencoding("8bit");
    mapping mapprefs = ([ ]);
    foreach (indices(camas_main->prefproperties), string prop) 
    {
      if (sessobj[prop] && sessobj["usersetup"+prop])
        mapprefs += ([ prop: (string) sessobj[prop] ]);
    }
    sessobj->prefuids=({ });
    foo = CAMAS.Tools.serialize_prefs(mapprefs) || "";
    prefsmailobj->setdata(foo);
    if (sessobj->prefsmails) 
    {
      foreach(indices(sessobj->prefsmails),int i) 
      {
        object mail=sessobj->prefsmails[i];
        if (i < sizeof(sessobj->prefsmails) &&
            mail->imap->ENVELOPE[SUBJECT_IDX] ==  "CAMAS Preferences")
          sessobj->prefuids += ({ mail->imap->UID });
      }
    }
    set_cmd_arg(command->output,(string)prefsmailobj);
    command_result=0;
    next_command();
  }

  // load user prefs
  void start_load_user_prefs () {
    sessobj->prefsmailobj=0;
    if(sessobj->usersetup) {
      command->subcommands = ({
                                CAMAS.IMAPTools.imap_cmd("status","mailbox",sessobj->prefsbox),
                                CAMAS.IMAPTools.imap_cmd("get_headers",
                                         "mailbox",sessobj->prefsbox,
                                         "output",CAMAS.IMAPTools.imap_cmd_var("prefsmails"),
                                         "abortfail",0),
                                CAMAS.IMAPTools.imap_cmd("get_prefs_mail",
                                         "mailbox",sessobj->prefsbox,
                                         "abortfail",0),
                                CAMAS.IMAPTools.imap_cmd("set_prefs")
                              });

      if (feature_extendedabook) {
        command->subcommands += ({
                                   CAMAS.IMAPTools.imap_cmd("status","mailbox",sessobj->prefsbox),
                                   CAMAS.IMAPTools.imap_cmd("get_headers",
                                            "mailbox",sessobj->prefsbox,
                                            "output",CAMAS.IMAPTools.imap_cmd_var("prefsmails"),
                                            "abortfail",0),
                                   CAMAS.IMAPTools.imap_cmd("get_ebook_mail",
                                            "mailbox", sessobj->prefsbox,
                                            "abortfail", 0),
                                   CAMAS.IMAPTools.imap_cmd("set_ebook")
                                 });
      }
      commands = ({ command }) + commands;
    }
    else
      commands=({ CAMAS.IMAPTools.imap_cmd("set_prefs") })+commands;
    command_result=0;
    next_command();
  }

  // set_ebook
  void start_set_ebook () {
    object features = camas_main->my_configuration ()->get_provider ("camas_features");
    if (objectp (features)) {
      if (sessobj->ebookmailobj)
        sessobj->extendedabook->import_mime (sessobj->ebookmailobj);
      sessobj->extendedabook->set_fields (features->QUERY (extendedabookfields)/ ",");
      command_result = 0;
    }
    else
    {
      report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at ebook %O\n", line);
      command_result = -1;
    }
    next_command ();
  }

  // set_prefs
  void start_set_prefs () {
    if (sessobj->prefsmailobj) {
      if (sessobj->prefsmailobj->body_parts)
        sessobj->prefsmailobj = sessobj->prefsmailobj->body_parts[0];
      mapping mapprefs = CAMAS.Tools.deserialize_prefs(sessobj->prefsmailobj->getdata());
      foreach(indices(mapprefs), string prefname)
      {
        if(camas_main->prefproperties[prefname] 
          && sessobj[prefname] && sessobj["usersetup"+prefname])
             sessobj[prefname] = mapprefs[prefname];
      }
      if (!sessobj->layout || (sessobj->layout == "0"))
        sessobj->layout = camas_main->QUERY (defaultlayout);
      if (sessobj->overridelayout)
        sessobj->layout = sessobj->overridelayout;
      sessobj->prefsloaded = 1;
    }

    command_result = 0;
    next_command ();
  }

  // check_mailboxes
  void start_check_mailboxes () {
    IMAP_DEBUG ("checkmailboxes: checking INBOX\n");
    commands = ({ CAMAS.IMAPTools.imap_cmd ("low_close"), CAMAS.IMAPTools.imap_cmd ("status", "mailbox", "INBOX") }) + commands;
    command_result = 0;
    next_command ();
  }

  // search
  void start_search () {
    if (command->searchtext && command->searchfield) {
      imapwrite(imap_tag()+" UID SEARCH CHARSET UTF-8 "+
                command->searchfield+" ");
      imapwrite_literal(string_to_utf8(command->searchtext),1,0);
      command_seq = 0;
      timeout(TIMEOUT1);
    }
    else {
      command_result = 0;
      next_command();
    }
  }

  void handle_search () {
    // expect SEARCH command result
    if(sscanf(line,imap_tag()+" %s",foo)) {
      if(foo[0..1]=="OK") {
        command_result=0;
        next_command();
      } else {
        report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at search %O\n", line);
        command_result=-1;
        next_command();
      }
    } else {
      timeout(TIMEOUT1);
      foo = "";
      //oliv3: FIXED ? in testing
      if (sscanf (line,"* SEARCH %s", foo)) {
        sessobj->nothingfound = 0;
        array uids = Array.map (foo / " ", lambda (string s) { return (int)s; });
        array mails = get_cmd_arg (command->headers);
        mails = Array.filter (mails, lambda (mapping mail, array uids)
                              { return has_value (uids, mail->imap->UID); }, uids);
        set_cmd_arg(command->headers, mails);
      }
      else
      {
        /* no mails found matching the query */
        sessobj->nothingfound = 1;
      }
    }
  }

  // scan_folders
  void start_scan_folders () {
    if (!sessobj->foldersinfo)
      sessobj->foldersinfo = ([ ]);
    
    foreach (sessobj->mailboxes, array mbox_a)
      if (!sessobj->foldersinfo[mbox_a[MB_FOLDERNAME_IDX]])
        sessobj->foldersinfo += ([ mbox_a[MB_FOLDERNAME_IDX]: ({ 0, 0, 0, MB_UNKNOWN_SIZE, 0 }) ]);
    sessobj->folderstoscan = ({ });
    sessobj->nodefoldersinfo = ([ ]);
    array new_commands = ({ });
    foreach (sessobj->mailboxes, array mbox_a) {
      // don't scan folders the admin put in camas main -> incoming mail -> donotreloadmailboxes
      // (only available in expert mode)
      if(!(command->reload 
          && camas_main->notreloadmailboxes
          && Array.search_array(camas_main->notreloadmailboxes, 
          lambda(string mbox) { return has_value(mbox_a[MB_FOLDERNAME_IDX], mbox); }) != -1))
      {
        //write ("scan_folders: %O ...", mbox_a);
        // RFC 2060  6.3.8
        // "If these levels of hierarchy are not also selectable mailboxes, they are returned with the \Noselect mailbox name attribute"
        // So, levels that don't accept emails should be those that have the \Noselect attribute,
        // not those with \HasNoChildren
        if (!(mbox_a[MB_FLAGS_IDX] & MB_NOSELECT)) {
          sessobj->folderstoscan += ({ mbox_a[MB_FOLDERNAME_IDX] });
          new_commands += ({ CAMAS.IMAPTools.imap_cmd ("start_scan") }) 
                      + ({ CAMAS.IMAPTools.imap_cmd ("end_scan") });
          //write("leaf folder\n");
        }
        // now, nodesfoldersinfo may not be an accurate name. noselectinfo should be better.
        else {
          sessobj->nodefoldersinfo += ([ mbox_a[MB_FOLDERNAME_IDX]: ({ 0, 0, 0, 0 }) ]);
          //write ("node folder\n");
        }
      }
    }
    commands = new_commands + commands;
    if(command->updatequota && capabilities->quota)
      foreach(sessobj->folderstoscan, string folder)
        commands = ({ CAMAS.IMAPTools.imap_cmd("getquotaroot", "mailbox", folder) }) + commands;
    command_result = 0;
    next_command ();
  }

  // start_scan
  void start_start_scan () {
    string mbox2scan = sessobj->folderstoscan[0];
    imap_newtag ();
    if (sessobj->mboxencode) {
      imapwrite(imap_tag()+" STATUS "+IMAP_QSTRING(mbox2scan)+" (MESSAGES RECENT UNSEEN UIDNEXT)\r\n");
    }
    else {
      imapwrite(imap_tag()+" STATUS ");
      imapwrite_literal(mbox2scan,0,1);
      command_seq = 1;
    }
    timeout (TIMEOUT1);
  }

  void handle_start_scan () {
    //oliv3 FIXME reorder the switches...
    switch (command_seq) {
    case 1: // Continue STATUS command after literal
      imapwrite(" (MESSAGES RECENT UNSEEN UIDNEXT)\r\n");
      command_seq = 0;
      timeout (TIMEOUT1);
      break;

    case 0:
      if (sscanf (line, imap_tag() + " NO %*s"))
      {
        // don't make it a fatal error so that we can check other folders
        report_warning("CAMAS: imapclient@"+__LINE__+": Can't STATUS on %O\n", 
            sessobj->folderstoscan[0]);
        command_result = 0;
        next_command();
      }
      else
      {
        if (sscanf (line, "* STATUS {%s", foo) == 1)
          command_seq = 6;
        else
          if ((sscanf (line, "* STATUS %*s (%s", foo) == 2) || (sscanf (line, "* STATUS \"%*s\" (%s", foo) == 2)) {
            data = "(" + foo + "\n" + data;
            init_parser (PT_MAPPING);
            command_seq = 2;
          }
        timeout (TIMEOUT1);
      }
      break; // 0

    case 6:
      if ((sscanf (line, "%*s (%s", foo) == 2) || (sscanf (line, "\"%*s\" (%s", foo) == 2)) {
        data = "(" + foo + "\n" + data;
        init_parser (PT_MAPPING);
        command_seq = 2;
      }
      timeout (TIMEOUT1);
      break; // 6

    case 2:
      if (!parse_result) {
        report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at scan %O\n", line);
        command_result = -1;
        next_command ();
      }
      else
      {
        sessobj->exists_old = sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_MESSAGES_IDX];
        sessobj->uidnext_old = sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_NEXTUID_IDX];
        sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_MESSAGES_IDX] = (int)parse_result->MESSAGES;
        if(sessobj->exists_old != sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_MESSAGES_IDX])
          IMAP_DEBUG (sessobj->folderstoscan[0]+": scheduled fetch due to MESSAGES change...\n");
        sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_RECENT_IDX] = (int)parse_result->RECENT;
        sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_UNSEEN_IDX] = (int)parse_result->UNSEEN;
        if(sessobj->uidnext_old != sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_UNSEEN_IDX])
          IMAP_DEBUG (sessobj->folderstoscan[0]+": scheduled fetch due to UIDNEXT change...\n");
        sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_NEXTUID_IDX] = (int)parse_result->UIDNEXT;
        command_seq = 3;
        timeout (TIMEOUT1);
      }
      break; // 2

    case 3: // do EXAMINE only if (non-empty mailbox)
      if (sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_MESSAGES_IDX]) {
        imap_newtag ();
        string mb = sessobj->folderstoscan[0];
        if (sessobj->mboxencode)
          imapwrite(imap_tag()+" EXAMINE "+IMAP_QSTRING(mb)+"\r\n");
        else {
          imapwrite(imap_tag()+" EXAMINE ");
          imapwrite_literal(mb,1,0);
        }
        command_seq = 4;
        timeout (TIMEOUT1);
      }
      else {
        command_result = 0;
        next_command ();
      }
      break; // 3

    case 4: // EXAMINE result
      if (sscanf (line, imap_tag () + " OK [READ-ONLY]%s", foo)) {
        if ((sessobj->uidnext_old != sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_NEXTUID_IDX]) ||
            (sessobj->exists_old != sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_MESSAGES_IDX])) {
          if (sessobj->uidnext_old != sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_NEXTUID_IDX])
            IMAP_DEBUG (sessobj->folderstoscan[0]+": scheduled fetch due to UIDNEXT change...\n");
          if (sessobj->exists_old != sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_MESSAGES_IDX])
            IMAP_DEBUG (sessobj->folderstoscan[0]+": scheduled fetch due to MESSAGES change...\n");
        }
        command_result = 0;
        next_command ();
      }
      break; // 4
    }
  }

  // end_scan
  void start_end_scan () {
    if (sizeof (sessobj->folderstoscan) > 1) {
      sessobj->folderstoscan = sessobj->folderstoscan[1..];
      command_result = 0;
      next_command ();
    }
    else { // done
      sessobj->folderstoscan = 0;

      if (sessobj->foldershavechanged) {
        //write ("updating folders info\n");
        sessobj->allfolderssize = 0;
        foreach (indices (sessobj->foldersinfo), string key)
        if (sizeof (sessobj->foldersinfo[key]) == 5) // it's a leaf folder
          sessobj->allfolderssize += sessobj->foldersinfo[key][MB_SIZE_IDX];

        foreach (sessobj->mailboxes, array mbox_a) {
          if (!(mbox_a[MB_FLAGS_IDX] & MB_NOSELECT))
            if (sizeof (mbox_a[MB_HIERARCHY_IDX]) > 1) {
              //write (mbox_a[MB_FOLDERNAME_IDX] + " is a sub-mailbox\n");
              for (int p_mbox = 0; p_mbox < (sizeof (mbox_a[MB_HIERARCHY_IDX]) - 1); p_mbox++) {
                //write ("parent mbox #" + p_mbox + " = ");
                string p_mbox_name = sessobj->mailpath + mbox_a[MB_HIERARCHY_IDX][0];
                for (int i = 1; i <= p_mbox; i++)
                  p_mbox_name += (mbox_a[MB_SEPARATOR_IDX] + mbox_a[MB_HIERARCHY_IDX][i]);
                for (int k = 0; k < 4; k++)
		{
		  // sanity check: check if foldersinfo can be indexed by mbox_a[MB_FOLDERNAME_IDX], if the mbox is shared this may not the case
		  // Since p_mbox_name has a sessobj->mailpath at its beginning, it can't work - Bertrand
   		    mixed err = catch { sessobj->nodefoldersinfo[p_mbox_name][k] += sessobj->foldersinfo[mbox_a[MB_FOLDERNAME_IDX]][k]; };

		}
              }
            }
        }

        foreach (indices (sessobj->nodefoldersinfo), string mbox)
        sessobj->foldersinfo[mbox] = sessobj->nodefoldersinfo[mbox];
      }
      //else
      //write ("folders info has not changed\n");

      sessobj->nodefoldersinfo = 0;
      sessobj->foldershavechanged = 0;
      sessobj->exists_old = 0;
      sessobj->uidnext_old = 0;
      commands = ({ CAMAS.IMAPTools.imap_cmd ("low_close") }) + commands;
      command_result=0;
      next_command ();
    }
  }

  // rename_folder
  void start_rename_folder () {
    if(!command->oldfoldername || !sizeof(command->oldfoldername) ||
        !command->newfoldername || !sizeof(command->newfoldername)) {
      report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at rename folder %O\n", line);
      command_result=-1;
      next_command();
    }
    else {
        IMAP_DEBUG(sprintf("About to rename folder : %s => %s\n",command->oldfoldername,command->newfoldername));
      imap_newtag ();
      if(sessobj->mboxencode)
        imapwrite(imap_tag()+" EXAMINE "+IMAP_QSTRING(command->newfoldername)+"\r\n");
      else {
        imapwrite(imap_tag()+" EXAMINE ");
        imapwrite_literal(command->newfoldername,1,0);
      }
      timeout(TIMEOUT1);
    }
  }

  void handle_rename_folder () {
    switch(command_seq) {
    case 0: // expect EXAMINE command result (newfoldername)
      if(sscanf(line,imap_tag()+" %s",foo)) {
        if(foo[0..1]=="OK") {
          // oops. mailbox does exist. skip.
          command->newfoldername = 0;
          report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at rename folder %O\n", line);
          command_result=-1;
          next_command();
        }
        else {
          imap_newtag();
          if(sessobj->mboxencode)
            imapwrite(imap_tag()+" EXAMINE "+IMAP_QSTRING(command->oldfoldername)+"\r\n");
          else {
            imapwrite(imap_tag()+" EXAMINE ");
            imapwrite_literal(command->oldfoldername,1,0);
          }
          command_seq=1;
          timeout(TIMEOUT1);
        }
      }
      else
        timeout(TIMEOUT1);
      break; //0

    case 1: // expect EXAMINE command result (oldfoldername)
      if(sscanf(line,imap_tag()+" %s",foo)) {
        if(foo[0..1]=="OK") {
          imap_newtag();
          imapwrite(imap_tag()+" CLOSE\r\n");
	  selected_mailbox = 0;
          command_seq=2;
          timeout(TIMEOUT1);
        }
        else
        {
          command_result=-1;
          next_command();
        }
      }
      else
        timeout(TIMEOUT1);
      break; //1

    case 2: // expect CLOSE command result
      if(sscanf(line, imap_tag()+ " %s", foo)) {
        if(foo[0..1]=="OK") {
	  imap_newtag();
	  // ok. folder closed
          string oldfoldername;
          string newfoldername;
          if(sessobj->mboxencode)
          {
            oldfoldername = IMAP_QSTRING(command->oldfoldername);
            newfoldername = IMAP_QSTRING(command->newfoldername);
          }
          else
          {
            oldfoldername = command->oldfoldername;
            newfoldername = command->newfoldername;
          }
          imapwrite(imap_tag() + " RENAME " + oldfoldername 
             + " " + newfoldername + "\r\n");
	  command_seq=3;
	  timeout(TIMEOUT1);
	}
        else
        {
          command_result=-1;
          next_command();
        }
      }
      else
        timeout(TIMEOUT1);
      break; //2 

    case 3: // expect RENAME command result
      if(sscanf(line,imap_tag()+" %s",foo)) {
        if(foo[0..1]=="OK") {
          // ok. folder renamed
      	  object features = camas_main->my_configuration ()->get_provider ("camas_features");
          if (features->QUERY (foldersinfo)) {
            sessobj->foldersinfo[command->newfoldername] = sessobj->foldersinfo[command->oldfoldername];
            m_delete (sessobj->foldersinfo, command->oldfoldername);
          }
          object oldcache = Cache(command->oldfoldername, sessobj, camas_main);
          object newcache = Cache(command->newfoldername, sessobj, camas_main);
          newcache->lock();
          mixed err = catch {
            newcache->mcache = oldcache->mcache;
          };
          newcache->unlock();
          if(err)
            report_error("error in imapclient: %s\n", describe_backtrace(err));
          newcache->save();
          oldcache->remove();

          CAMAS.Filters.replace_filter(id, 0, 0, command->oldfoldername, 0, 0, command->newfoldername);
          command_result = 0;
          next_command();
        } else {
          report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at rename folder %O\n", line);
          command_result=-1;
          next_command();
        }
      }
      else
        timeout(TIMEOUT1); 
      break; //3
    }
  }

  // cache_init
  void start_cache_init () {
    cache = Cache(command->mailbox, sessobj, camas_main);
    mapping mcache = cache->mcache;
    if (!sizeof(mcache->mails)) {
      IMAP_DEBUG("cache_init (" + command->mailbox + ")\n");

      object features = camas_main->my_configuration ()->get_provider ("camas_features");
      if (features->QUERY (foldersinfo))
        sessobj->foldersinfo += ([ command->mailbox: ({ 0, 0, 0, MB_UNKNOWN_SIZE, 0 }) ]);

      command_result = 0;
    }
    else {
      report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at cache init, "
          "initializing a know folder (" + command->mailbox + ")\n");
      command_result = -1;
    }
    next_command ();
  }

  // check
  void start_check () {
    if (selected_mailbox) {
      imap_newtag ();
      imapwrite (imap_tag () + " CHECK\r\n");
      timeout (TIMEOUT1);
    }
    else {
      command_result = 0;
      next_command ();
    }
  }

  void handle_check () {
    if (sscanf (line, imap_tag () + " %s", foo))
    {
      if (foo[0..1] == "OK")
        command_result = 0;
      else
      {
        report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at check %O\n", line);
        command_result = -1;
      }
      next_command ();
    }
    else
      timeout (TIMEOUT1);
  }

  // getquotaroot
  void start_getquotaroot () {
    if (command->mailbox == 0)
      command->mailbox = sessobj->mailbox[MB_FOLDERNAME_IDX];
    imap_newtag ();
    imapwrite (imap_tag () + " GETQUOTAROOT \"" + command->mailbox + "\"\r\n");
    timeout (TIMEOUT1);
  }
  
  void handle_getquotaroot () {
    /* Courier-IMAP:
     * g getquotaroot "INBOX"
     * * QUOTAROOT "INBOX" "ROOT"
     * * QUOTA "ROOT" (STORAGE 0 489 MESSAGE 0 500)
     * g OK GETQUOTAROOT Ok.
     * Cyrus-IMAP:
     * g GETQUOTAROOT INBOX
     * * QUOTAROOT INBOX
     * g OK Completed
     * g GETQUOTAROOT INBOX
     * QUOTAROOT INBOX user.david
     * QUOTA user.david (STORAGE 1 100000)
     * OK Completed
     */
    switch (command_seq) {
    case 0:
      string rootline, mailbox, rootname, ressources;
      mapping mailbox2rootname = ([ ]);
      if(sscanf(line,"* %*sQUOTAROOT %s", rootline) == 2)
      {
        // fool proof for mailboxes which contains space and are not 
        // surrounded by ""
        array arr_rootline = rootline / " ";
	// it's possible that the root doesn't exist, default to "ROOT"
	// in this case
	if(sizeof(arr_rootline) == 1)
	  arr_rootline += ({ "ROOT" });
        mailbox = arr_rootline[..sizeof(arr_rootline)-2] * " ";
        rootname = arr_rootline[sizeof(arr_rootline)-1];
        if(mailbox[0] == '\"' && mailbox[sizeof(mailbox)-1] == '\"') 
          mailbox = mailbox[1..sizeof(mailbox)-2];
        sessobj->mailbox2rootname[mailbox] = rootname;
      }
      if(sscanf(line,"* QUOTA %s (%s)", rootname, ressources)==2) {
        array arr_ressources = ressources / " ";
        // map ressource name to its current value and quota
        mapping map_ressources = ([ ]);
        string index; 
        for (int i = 0; i < sizeof(arr_ressources); i++) {
          if(!(i%3)) 
            index = arr_ressources[i]; 
          else
            map_ressources[index] += ({ arr_ressources[i] });
        }
        sessobj->quota[rootname] = map_ressources;
      	command_seq = 1;
      	timeout (TIMEOUT1);
      }
      if (sscanf (line, imap_tag () + " %s", foo)) {
      	if (foo[0..1] == "OK")
      	  command_result = 0;
      	else
        {
          report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at quota %O\n", line);
      	  command_result = -1;
        }
      	next_command ();
      }
      break;
    case 1:
      if (sscanf (line, imap_tag () + " %s", foo)) {
      	if (foo[0..1] == "OK")
      	  command_result = 0;
      	else
        {
      	  report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at quota %O\n", line);
          command_result = -1;
        }
      	next_command ();
      }
      else {
        report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at quota %O\n", line);
      	command_result = -1;
      	next_command ();
      }
      break;
    default:
      report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at quota %O\n", line);
      command_result = -1;
      next_command ();
      break;
    }
  }

  // ------------------------------------------------------------------

  void start_command (mapping com) {
    IMAP_DEBUG("start command: " + com->cmd + "\n");
    command = com;
    command_seq = 0;
    imap_newtag ();

    if (this_object () ["start_" + com->cmd])
      this_object () ["start_" + com->cmd] ();
    else
    {
      report_warning("CAMAS: Unimplemented command: start_" + com->cmd + "\n");
      command_result = -1;
      next_command ();
    }
  }


  void handle_command () {
    if (command) // && !((< "get_headers", "low_list", "status", "low_select" >) [command->cmd]))
    {
      IMAP_DEBUG ("CAMAS command: " + command->cmd + " (" + command_seq + ")\n");
      IMAP_DEBUG ("CAMAS line: " + line + "\n");
    }

    if (line && sizeof (line) > 0)
    {
      if (sizeof (imaplinehist) > 4)
    	imaplinehist -= ({ imaplinehist[4] });
      imaplinehist = ({ (sizeof(line) > 70) ? line[0..69] : line }) + imaplinehist;
    }

    if (literal_string) 
    {
      // see imapwrite_string above
      if (line && line[0..0] == "+") 
      {
      	imapwrite (literal_string + (literal_addcrlf ? "\r\n" : ""));
      	literal_string = 0;
    	if (literal_continue)
    	  handle_command ();
      }
      else 
      {
      	literal_string = 0;
      	handle_command ();
      }
      return;
    }

    if (command) 
    {
      if (this_object () ["handle_" + command->cmd])
        this_object () ["handle_" + command->cmd] ();
      else 
      {
        report_warning ("CAMAS: Unimplemented command: handle_" + command->cmd + "\n");
      	command_result = -1;
      	next_command ();
      }
    }
    else 
    {
      string response = "", code = "", text = "";
      if (sscanf (line, "* BYE%s", text) == 1) 
      {
      	// we'll have to reconnect :)
      	loggedin = 0;
      	IMAP_DEBUG("got BYE response:" + text + "\n");
      }
      else
      	if (sscanf (line, "* %s [%s] %s", response, code, text) > 0) 
	{
	  // Anything interesting ?
	  IMAP_DEBUG("Response= " + response + "\nCode= " + code + "\nText= " + text + "\n");
	}
    }
  }

  void next_command() {
    if(command)
    {
      if (command_result < 0) 
      {
        if (command->abortfail) 
	{
      	  if (command->container) 
	  {
      	    command = command->container; // Step back
      	    next_command();
      	    return;
      	  }
      	  abort=1;
      	  commands=({ });
      	  handle_error();
      	}
      	if (command->subcommands)
    	  command->subcommands = ({ });
      }
    }
    if(sizeof(commands)) 
    {
      if (!fd) 
      {
        IMAP_DEBUG(sprintf("next_command, connecting to %O:%d\n", 
			   sessobj->imapserver,sessobj->imapport));
      	fd=Stdio.File();
      	mixed err, cerr;
      	err = catch { 
      	  cerr = !(fd->connect(sessobj->imapserver,sessobj->imapport));
        };
        if (err != 0 || cerr != 0 || !fd) 
	{
          report_error("IMAPclient: Can't connect to %s:%d\n", sessobj->imapserver,sessobj->imapport);
      	  abort = 1; commands = ({ }); fd = 0;
      	  handle_error(); 
      	  done();
        }
        else
        {
	  start_command(CAMAS.IMAPTools.imap_cmd("low_login"));
	  fd->set_nonblocking(got_data,imap_data_written,socket_close);
        }
      } else 
      {
      	if (loggedin) 
	{
          IMAP_DEBUG("next_command, loggedin\n");
          mapping com;
      	  mapping container;
      	  // Find the next command. 
      	  com=commands[0];
      	  while (sizeof(commands) && com->subcommands) 
	  {
      	    com=commands[0];
      	    container = 0;
      	    while (com->subcommands && sizeof(com->subcommands)) 
	    {
      	      container = com;
    	      com = com->subcommands[0];
	    }
      	    if (com->subcommands) 
	    {
      	      if (container)
		container->subcommands = container->subcommands[1..];
      	      else
		commands=commands[1..];
      	    }
      	  }
          if (sizeof(commands)) 
	  {
            // Make sure the right box is selected
            if (!com->noselect &&
              ((com->mailbox  && com->mailbox != selected_mailbox) ||
              (!com->mailbox && !isimapcachefeeder &&
               sessobj->mailbox[MB_FOLDERNAME_IDX] != selected_mailbox))) 
	    {
	      if (com->tryselect) { // If we've already tried to select then
		// fail the command that needed the select
		if (container)
		  container->subcommands = container->subcommands[1..];
		else
		  commands=commands[1..];
		com->container = container;
		command = com;
		command_result = -1; // Muahaha
		report_warning("CAMAS: imapclient@"+__LINE__+": IMAP error at next_command: "
			       "can't select mailbox %O, selected_mailbox=%O, \n"
			       "sessobj->mailbox[MB_FOLDERNAME_IDX]=%O, imaplinehist=%O\n", 
			       com->mailbox, selected_mailbox, 
			       sessobj->mailbox[MB_FOLDERNAME_IDX], imaplinehist);
		next_command();
	      } 
	      else 
	      {
		if(!com->mailbox)
		  com->mailbox = sessobj->mailbox[MB_FOLDERNAME_IDX];
		com->tryselect = 1; // Try to select only once
		start_command(CAMAS.IMAPTools.imap_cmd("low_select", "mailbox", com->mailbox, 
						       "abortfail", 0));
	      }
	    } 
	    else 
	    {
	      if (container)
		container->subcommands = container->subcommands[1..];
	      else
		commands=commands[1..];
	      com->container = container;
	      start_command(com);
	    }
	  }
	  else
	  {
	    command=0;
	    done();
	  }
	}
      }
    }
    else 
    {
      command=0;
      done();
    }
  }

  void create (int _feature_extendedabook, object camas_main_in) {
    parser = CAMAS.parser.Parser ();
    feature_extendedabook = _feature_extendedabook;
    fd = 0;
    command = 0;
    camas_main = camas_main_in;
  }

  void destroy () {
    if (fd && loggedin) 
    {
#if constant(thread_create)
      _lock = camas_main->global_lock->lock();
#endif
      mixed err = catch(fd->set_close_callback (0));
#if constant(thread_create)
      destruct(_lock);
#endif
      if(err)
        report_error("error in imapclient: %s\n", describe_backtrace(err));
      imapwrite (imap_tag () + " LOGOUT\r\n");
#if constant(thread_create)
      _lock = camas_main->global_lock->lock();
#endif
      err = catch {
        fd->set_read_callback  (0);
        fd->set_write_callback (0);
        catch {
          fd->close (); 
          destruct(fd);
        };
        fd = 0;
      };
#if constant(thread_create)
      destruct(_lock);
#endif
      if(err)
        report_error("error in imapclient: %s\n", describe_backtrace(err));
      selected_mailbox = 0;
      IMAP_DEBUG("LOGOUT...\n");
    }
    
    remove_call_out (timeout_call_out);
    remove_call_out (imap_idle_call_out);
    timeout_call_out = 0;
    imap_idle_call_out = 0;
    sessobj = 0;
    camas_main = 0;
    
    if (fd) {
#if constant(thread_create)
      object _lock = camas_main->global_lock->lock();
#endif
      mixed err = catch {
        fd->set_close_callback (0);
        fd->set_read_callback  (0);
        fd->set_write_callback (0);
        catch {
  	      fd->close();
          destruct(fd);
        };
      };
#if constant(thread_create)
      destruct(_lock);
#endif     
      if(err)
        report_error("error in imapclient: %s\n", describe_backtrace(err));
    }
    destruct(this_object());
  }
  
  void imap_command (object id_in, object sessobj_in, array (mapping) commands_in) {
    cmd_result = 0;
    abort = 0;
    result_sent = 0;
    command_result = 0;
    reconnects = 0;
    id = id_in;
    sessobj = sessobj_in;
    mode = MODE_LINE;
    isimapcachefeeder = camas_main->isimapcachefeeder;
    if (id)
      id->do_not_disconnect = 1;
    if (commands && sizeof(commands) > 0)
      commands += commands_in;
    else
    {
      commands = commands_in;
      next_command ();
    }
  }
}

/*
 * If you visit a file that doesn't contain these lines at its end, please
 * cut and paste everything from here to that file.
 */

/*
 * Local Variables:
 * c-basic-offset: 2
 * End:
 *
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */
