/* javagrammar.y -- Parser for Java source language
 * Written by Charles Briscoe-Smith; refer to the file LEGAL for details.
 */

%{
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "types.h"
#include "parsetree.h"

#ifdef YYDEBUG
#define YYERROR_VERBOSE 1
#endif

extern void yyerror(const char *s);
extern int yylex(void);
extern void yyrestart(FILE *);
int input_line;
const char *input_file;
extern struct typedecl *bock_cu;
extern int num_errors;
%}

%union {
	s8			integer;
	char *			string;
	struct name *		name;
	struct type *		type;
	struct typedecl *	typedecl;
	struct memberdecl *	memberdecl;
	struct fielddecl *	fielddecl;
	struct vardecl *	vardecl;
	struct methoddecl *	methoddecl;
	struct methodsig *	methodsig;
	struct formparlist *	formparlist;
	struct constrdecl *	constrdecl;
	struct constinv *	constrinv;
	struct stmt *		stmt;
	struct catchclause *	catchclause;
	struct expr *		expr;
	struct exprlist *	exprlist;
}

/* Literals */
%token <string> IDENTIFIER STRINGLITERAL
%token <integer> INTEGERLITERAL LONGINTEGERLITERAL BOOLEANLITERAL
%token <integer> CHARACTERLITERAL
%token <string> FLOATLITERAL DOUBLELITERAL
%token NULLLITERAL

/* Non-terminals with semantic values */

/********** File structure **********/
%type <name> packagedeclarationopt
%type <name> importdeclarationsopt
%type <typedecl> typedeclarationsopt

/* Misc small stuff */
%type <name> name
%type <integer> modifiers
%type <integer> modifier
%type <type> type
%type <type> primitivetype
%type <type> referencetype
%type <type> arraytype
%type <name> namelist

/********** Classes **********/
%type <typedecl> classdeclaration
%type <name> superopt
%type <name> interfacesopt
%type <memberdecl> classbody
%type <memberdecl> classbodydeclarationsopt

/* Field declarations */
%type <fielddecl> fielddeclaration
%type <vardecl> variabledeclarators
%type <vardecl> variabledeclarator
%type <expr> variableinitializer
/* arrays */
%type <expr> arrayinitializer
%type <expr> variableinitializersopt
%type <expr> variableinitializersandcommas

/* Method declarations */
%type <methoddecl> methoddeclaration
%type <expr> nativebody
%type <methoddecl> methodheader
%type <methodsig> methoddeclarator
%type <formparlist> formalparameterlistopt
%type <formparlist> formalparameterlist
%type <formparlist> formalparameter
%type <name> throwsopt

/* static initializers */
%type <stmt> staticinitializer

/* constructor declarations */
%type <constrdecl> constructordeclaration
%type <methodsig> constructordeclarator
%type <constrdecl> constructorbody
%type <constrinv> explicitconstructorinvocation

/********** interfaces **********/
%type <typedecl> interfacedeclaration
%type <name> extendsinterfacesopt
%type <memberdecl> interfacebody
%type <memberdecl> interfacememberdeclarationsopt

/********** blocks and statements **********/
%type <stmt> block
%type <stmt> blockstatementsopt
%type <stmt> blockstatements
%type <stmt> blockstatement
%type <stmt> localvariabledeclarationstatement
%type <fielddecl> localvariabledeclaration
%type <stmt> statement
%type <stmt> statementnoshortif
%type <stmt> statementwithouttrailingsubstatement
%type <stmt> emptystatement
%type <stmt> labeledstatement
%type <stmt> labeledstatementnoshortif
%type <stmt> expressionstatement
%type <expr> statementexpression
%type <stmt> ifthenstatement
%type <stmt> ifthenelsestatement
%type <stmt> ifthenelsestatementnoshortif
%type <stmt> switchstatement
%type <stmt> switchblock
%type <stmt> switchblockstatementsopt
%type <stmt> switchblockstatements
%type <stmt> switchblockstatement
%type <stmt> switchlabel
%type <stmt> whilestatement
%type <stmt> whilestatementnoshortif
%type <stmt> dostatement
%type <stmt> forstatement
%type <stmt> forstatementnoshortif
%type <stmt> forinitopt
%type <stmt> forupdateopt
%type <stmt> statementexpressionlist
%type <stmt> breakstatement
%type <stmt> continuestatement
%type <stmt> returnstatement
%type <stmt> throwstatement
%type <stmt> synchronizedstatement
%type <stmt> trystatement
%type <catchclause> catches
%type <catchclause> catchclause
%type <stmt> finally

/********** expressions **********/
%type <expr> primary
%type <expr> primarynonewarray
%type <expr> literal
%type <expr> classinstancecreationexpression
%type <expr> argumentlistopt
%type <expr> arraycreationexpression
%type <expr> dimexprs
%type <expr> dimexpr
%type <integer> dimsopt
%type <integer> dims
%type <expr> fieldaccess
%type <expr> methodinvocation
%type <expr> arrayaccess
%type <expr> postfixexpression
%type <expr> postincrementexpression
%type <expr> postdecrementexpression
%type <expr> unaryexpression
%type <expr> preincrementexpression
%type <expr> predecrementexpression
%type <expr> unaryexpressionnotplusminus
%type <expr> castexpression
%type <expr> multiplicativeexpression
%type <expr> additiveexpression
%type <expr> shiftexpression
%type <expr> relationalexpression
%type <expr> equalityexpression
%type <expr> andexpression
%type <expr> exclusiveorexpression
%type <expr> inclusiveorexpression
%type <expr> conditionalandexpression
%type <expr> conditionalorexpression
%type <expr> conditionalexpression
%type <expr> assignmentexpression
%type <expr> assignment
%type <expr> lefthandside
%type <expr> expressionopt
%type <expr> expression
%type <expr> constantexpression

/* Keywords */
%token ABSTRACT BOOLEAN BREAK BYTE CASE CATCH CHAR CLASS CONST CONTINUE
%token DEFAULT DO DOUBLE ELSE EXTENDS FINAL FINALLY FLOAT FOR GOTO
%token IF IMPLEMENTS IMPORT INSTANCEOF INT INTERFACE LONG NATIVE NEW
%token PACKAGE PRIVATE PROTECTED PUBLIC RETURN SHORT STATIC SUPER SWITCH
%token SYNCHRONIZED THIS THROW THROWS TRANSIENT TRY VOID VOLATILE WHILE

/* Multicharacter operators */
%token PLUSPLUS MINUSMINUS LLTHAN GGTHAN GGGTHAN LTHANEQUAL GTHANEQUAL
%token EQUALEQUAL BANGEQUAL ANDAND PIPEPIPE STAREQUAL SLASHEQUAL
%token PERCENTEQUAL PLUSEQUAL MINUSEQUAL LLTHANEQUAL GGTHANEQUAL
%token GGGTHANEQUAL ANDEQUAL CARETEQUAL PIPEEQUAL

%%

/********** File structure **********/

compilationunit
	: packagedeclarationopt importdeclarationsopt typedeclarationsopt
	{	struct name *scan=$2, *single=NULL, *demand=NULL;
		struct typedecl *typescan;
		while (scan) {
			if (scan->data) {
				struct name *this=scan;
				scan=scan->rest;
				this->rest=demand;
				demand=this;
			} else {
				struct name *this=scan;
				scan=scan->rest;
				this->rest=single;
				single=this;
			}
		}
		typescan=$3;
		while (typescan) {
			typescan->typeimports=single;
			typescan->typesearchpath=demand;
			typescan->name->prefix=$1;
			typescan=typescan->rest;
		}
		bock_cu=$3;
	}
	;

packagedeclarationopt
	: /* empty */		{ $$ = NULL; }
	| PACKAGE name ';'	{ $$ = $2; }
	;

importdeclarationsopt
	: /* empty */		{ struct name *java=newname(0, "java", 0, 0);
				  struct name *lang=newname(java, "lang", 1, 0);
				  $$ = lang;
				}
	| importdeclarationsopt IMPORT name ';'
				{ $3->rest = $1; $3->data = 0; $$ = $3; }
	| importdeclarationsopt IMPORT name '.' '*' ';'
				{ $3->rest = $1; $3->data = 1; $$ = $3; }
	;

typedeclarationsopt
	: /* empty */		{ $$ = NULL; }
	| typedeclarationsopt classdeclaration
				{ $2->rest=$1; $$=$2; }
	| typedeclarationsopt interfacedeclaration
				{ $2->rest=$1; $$=$2; }
	| typedeclarationsopt ';'
	;

/* Misc small stuff */

name	: IDENTIFIER		{ $$=newname(NULL, strdup($1), 0, 0); }
	| name '.' IDENTIFIER	{ $$=newname($1, strdup($3), 0, 0); }
	;

modifiers
	: modifier
	| modifiers modifier	{ $$=$1|$2; }
	;

modifier: PUBLIC		{ $$=ACC_public; }
	| PROTECTED		{ $$=ACC_protected; }
	| PRIVATE		{ $$=ACC_private; }
	| STATIC		{ $$=ACC_static; }
	| ABSTRACT		{ $$=ACC_abstract; }
	| FINAL			{ $$=ACC_final; }
	| NATIVE		{ $$=ACC_native; }
	| SYNCHRONIZED		{ $$=ACC_synchronized; }
	| TRANSIENT		{ $$=ACC_transient; }
	| VOLATILE		{ $$=ACC_volatile; }
	;

type	: primitivetype
	| referencetype
	;

primitivetype
	: BYTE			{ $$ = &ByteType; }
	| SHORT			{ $$ = &ShortType; }
	| INT			{ $$ = &IntType; }
	| LONG			{ $$ = &LongType; }
	| CHAR			{ $$ = &CharType; }
	| FLOAT			{ $$ = &FloatType; }
	| DOUBLE		{ $$ = &DoubleType; }
	| BOOLEAN		{ $$ = &BooleanType; }
	;

referencetype
	: name			{ $$ = referencetype($1); }
	| arraytype
	;

arraytype
	: primitivetype dims	{ $$ = arrayoftypedepth($1, $2); }
	| name dims		{ $$ = arrayoftypedepth(referencetype($1), $2); }
	;

namelist: name			{ $$ = $1; }
	| namelist ',' name	{ $3->rest = $1; $$ = $3; }
	;

/********** Classes **********/

classdeclaration
	: modifiers CLASS IDENTIFIER superopt interfacesopt classbody
		{ $$ = newtypedecl($1, newname(NULL, strdup($3), 0, 0),
		                   0, 0, $4, $5, 0, $6, NULL); }
	| CLASS IDENTIFIER superopt interfacesopt classbody
		{ $$ = newtypedecl(0, newname(NULL, strdup($2), 0, 0),
		                   0, 0, $3, $4, 0, $5, NULL); }
	;

superopt: /* empty */	{ static struct name java={0, "java", 0, 0};
			  static struct name lang={&java, "lang", 0, 0};
			  static struct name Object={&lang, "Object", 0, 0};
			  $$ = &Object;
			}
	| EXTENDS name		{ $$ = $2; }
	;

interfacesopt
	: /* empty */		{ $$ = NULL; }
	| IMPLEMENTS namelist	{ $$ = $2; }
	;

classbody
	: '{' classbodydeclarationsopt '}'
				{ $$ = $2; }
	;

classbodydeclarationsopt
	: /* empty */		{ $$ = NULL; }
	| classbodydeclarationsopt fielddeclaration
		{ $$ = newmemberdecl(Field, (union member) $2, $1); }
	| classbodydeclarationsopt methoddeclaration
		{ $$ = newmemberdecl(Method, (union member) $2, $1); }
	| classbodydeclarationsopt staticinitializer
		{ $$ = newmemberdecl(StaticInit, (union member) $2, $1); }
	| classbodydeclarationsopt block
		{ $$ = newmemberdecl(InstanceInit, (union member) $2,$1); }
	| classbodydeclarationsopt constructordeclaration
		{ $$ = newmemberdecl(Constructor, (union member) $2, $1); }
	| classbodydeclarationsopt classdeclaration
		{ num_errors++; yyerror("Nested classes not yet supported.");
		}
	| classbodydeclarationsopt interfacedeclaration
		{ num_errors++; yyerror("Nested classes not yet supported.");
		}
	;

/* Field declarations */

fielddeclaration
	: modifiers type variabledeclarators ';'
				{ $$ = newfielddecl($1, $2, $3); }
	| type variabledeclarators ';'
				{ $$ = newfielddecl(0, $1, $2); }
	;

variabledeclarators
	: variabledeclarator
	| variabledeclarators ',' variabledeclarator
				{ $3->rest = $1; $$ = $3; }
	;

variabledeclarator
	: IDENTIFIER dimsopt	{ $$ = newvardecl($1, $2, NULL, NULL); }
	| IDENTIFIER dimsopt '=' variableinitializer
				{ $$ = newvardecl($1, $2, $4, NULL); }
	;

variableinitializer
	: expression
	| arrayinitializer	{ $$ = newexpr(Arrayinit,$1,0,0,0,0,0,0,0); }
	;

/* Arrays */

arrayinitializer
	: '{' variableinitializersopt '}'
				{ $$ = $2; }
	| '{' ',' '}'		{ $$ = NULL; }
	;

variableinitializersopt
	: variableinitializersandcommas
	| variableinitializersandcommas variableinitializer
			{ $2->rest = $1; $$ = $2; }
	;

variableinitializersandcommas
	: /* empty */
			{ $$ = NULL; }
	| variableinitializersandcommas variableinitializer ','
			{ $2->rest = $1; $$ = $2; }
	;

/* Method declarations */

methoddeclaration
	: methodheader block	{ $$ = $1; $$->body = $2; }
	| methodheader ';'	{ $$ = $1; $$->body = NULL; }
	| methodheader nativebody ';'
				{ $$ = $1;
				  $$->body=newstmt(NativeStmt,0,$2,0,0,0,0,0,0);
				}
	;

nativebody
	: STRINGLITERAL		{ $$ = newexpr(0,0,0,0,0,0,0,$1,0); }
	| nativebody STRINGLITERAL
				{ $$ = newexpr(0,0,0,0,0,0,0,$2,$1); }
	;

methodheader
	: modifiers type methoddeclarator dimsopt throwsopt
		{ $$ = newmethoddecl($1, $2, $3, $5, NULL, NULL); }
	| type methoddeclarator dimsopt throwsopt
		{ $$ = newmethoddecl(0, $1, $2, $4, NULL, NULL); }
	| modifiers VOID methoddeclarator throwsopt
		{ $$ = newmethoddecl($1, &VoidType, $3, $4, NULL, NULL); }
	| VOID methoddeclarator throwsopt
		{ $$ = newmethoddecl(0, &VoidType, $2, $3, NULL, NULL); }
	;

methoddeclarator
	: IDENTIFIER '(' formalparameterlistopt ')'
			{ $$ = newmethodsig($1, $3); }
	;

formalparameterlistopt
	: /* empty */		{ $$ = 0; }
	| formalparameterlist
	;

formalparameterlist
	: formalparameter
	| formalparameterlist ',' formalparameter
				{ $$ = $3; $$->rest = $1; }
	;

formalparameter
	: type IDENTIFIER dimsopt
		{ $$ = newformparlist(arrayoftypedepth($1, $3), $2, NULL); }
	| FINAL type IDENTIFIER dimsopt
		{ $$ = newformparlist(arrayoftypedepth($2, $4), $3, NULL); }
	;

throwsopt
	: /* empty */		{ $$ = 0; }
	| THROWS namelist	{ $$ = $2; }
	;

/* static initializers */

staticinitializer
	: STATIC block		{ $$ = $2; }
	;

/* constructor declarations */

constructordeclaration
	: modifiers constructordeclarator throwsopt constructorbody
		{ $$ = $4; $$->modifiers = $1; $$->sig = $2; $$->throws = $3; }
	| constructordeclarator throwsopt constructorbody
		{ $$ = $3; $$->modifiers = 0; $$->sig = $1; $$->throws = $2; }
	;

constructordeclarator
	: IDENTIFIER '(' formalparameterlistopt ')'
				{ $$ = newmethodsig($1, $3); }
	;

constructorbody
	: '{' explicitconstructorinvocation blockstatements '}'
			{ $$ = newconstrdecl(0, NULL, NULL, $2, $3); }
	| '{' explicitconstructorinvocation '}'
			{ $$ = newconstrdecl(0, NULL, NULL, $2, NULL); }
	| '{' blockstatements '}'
			{ $$ = newconstrdecl(0, NULL, NULL, NULL, $2); }
	| '{' '}'	{ $$ = newconstrdecl(0, NULL, NULL, NULL, NULL); }
	;

explicitconstructorinvocation
	: THIS '(' argumentlistopt ')' ';'
				{ $$ = newconstinv(0, $3); }
	| SUPER '(' argumentlistopt ')' ';'
				{ $$ = newconstinv(1, $3); }
	| primary '.' SUPER '(' argumentlistopt ')' ';'
		{ num_errors++; yyerror("Inner classes not yet supported.");
		}
	;

/********** interfaces **********/

interfacedeclaration
	: modifiers INTERFACE IDENTIFIER extendsinterfacesopt interfacebody
		{ $$ = newtypedecl($1|ACC_interface|ACC_abstract,
		                   newname(NULL, $3, 0, 0),
		                   0, 0, NULL, $4, 0, $5, NULL); }
	| INTERFACE IDENTIFIER extendsinterfacesopt interfacebody
		{ $$ = newtypedecl(ACC_interface|ACC_abstract,
		                   newname(NULL, $2, 0, 0),
		                   0, 0, NULL, $3, 0, $4, NULL); }
	;

extendsinterfacesopt
	: /* empty */		{ $$ = 0; }
	| EXTENDS namelist	{ $$ = $2; }
	;

interfacebody
	: '{' interfacememberdeclarationsopt '}'
				{ $$ = $2; }
	;

interfacememberdeclarationsopt
	: /* empty */		{ $$ = 0; }
	| interfacememberdeclarationsopt fielddeclaration
		{ $2->modifiers |= ACC_public|ACC_static|ACC_final;
		  $$ = newmemberdecl(Field, (union member) $2, $1);
		}
	| interfacememberdeclarationsopt methodheader ';'
		{ $2->modifiers |= ACC_public|ACC_abstract;
		  $$ = newmemberdecl(Method, (union member) $2, $1);
		}
	| interfacememberdeclarationsopt classdeclaration
		{ num_errors++; yyerror("Nested classes not yet supported.");
		}
	| interfacememberdeclarationsopt interfacedeclaration
		{ num_errors++; yyerror("Nested classes not yet supported.");
		}
	;

/********** Blocks and statements **********/

block	: '{' blockstatementsopt '}'	{ $$ = $2; }
	;

blockstatementsopt
	: /* empty */		{ $$ = NULL; }
	| blockstatements
	;

blockstatements
	: blockstatement
	| blockstatements blockstatement
				{ struct stmt *scan = $1;
				  while (scan->rest) { scan=scan->rest; }
				  $$ = $1; scan->rest = $2;
				}
	;

blockstatement
	: localvariabledeclarationstatement
	| statement
	| classdeclaration
		{ num_errors++;
		  yyerror("Block-local classes noy yet supported");
		}
	;

localvariabledeclarationstatement
	: localvariabledeclaration ';'
		{ $$ = newstmt(Localvar, $1, 0, 0, 0, 0, 0, 0, 0); }
	;

localvariabledeclaration
	: type variabledeclarators
				{ $$ = newfielddecl(0, $1, $2); }
	| FINAL type variabledeclarators
				{ $$ = newfielddecl(0, $2, $3); }
	;

statement
	: statementwithouttrailingsubstatement
	| labeledstatement
	| ifthenstatement
	| ifthenelsestatement
	| whilestatement
	| forstatement
	;

statementnoshortif
	: statementwithouttrailingsubstatement
	| labeledstatementnoshortif
	| ifthenelsestatementnoshortif
	| whilestatementnoshortif
	| forstatementnoshortif
	;

statementwithouttrailingsubstatement
	: block		{ $$ = newstmt(Block, 0, 0, $1, 0, 0, 0, 0, 0); }
	| emptystatement
	| expressionstatement
	| switchstatement
	| dostatement
	| breakstatement
	| continuestatement
	| returnstatement
	| synchronizedstatement
	| throwstatement
	| trystatement
	;

emptystatement
	: ';'		{ $$ = newstmt(Empty, 0, 0, 0, 0, 0, 0, 0, 0); }
	;

labeledstatement
	: IDENTIFIER ':' statement
			{ yyerror("Ignoring statement label"); $$ = $3; }
	;

labeledstatementnoshortif
	: IDENTIFIER ':' statementnoshortif
			{ yyerror("Ignoring statement label"); $$ = $3; }
	;

expressionstatement
	: statementexpression ';'
			{ $$ = newstmt(Exprstmt, 0, $1, 0, 0, 0, 0, 0, 0); }
	;

statementexpression
	: assignment
	| preincrementexpression
	| predecrementexpression
	| postincrementexpression
	| postdecrementexpression
	| methodinvocation
	| classinstancecreationexpression
	;

ifthenstatement
	: IF '(' expression ')' statement
			{ $$ = newstmt(If, 0, $3, $5, 0, 0, 0, 0, 0); }
	;

ifthenelsestatement
	: IF '(' expression ')' statementnoshortif ELSE statement
			{ $$ = newstmt(If, 0, $3, $5, $7, 0, 0, 0, 0); }
	;

ifthenelsestatementnoshortif
	: IF '(' expression ')' statementnoshortif ELSE statementnoshortif
			{ $$ = newstmt(If, 0, $3, $5, $7, 0, 0, 0, 0); }
	;

switchstatement
	: SWITCH '(' expression ')' switchblock
			{ $$ = newstmt(Switch, 0, $3, $5, 0, 0, 0, 0, 0); }
	;

switchblock
	: '{' switchblockstatementsopt '}'
				{ $$ = $2; }
	;

switchblockstatementsopt
	: /* empty */		{ $$ = 0; }
	| switchblockstatements
	;

switchblockstatements
	: switchlabel
	| switchblockstatements switchblockstatement
				{ struct stmt *scan = $1;
				  while (scan->rest) {
				    if (scan->tag==Default && $2->tag==Case) {
				      num_errors++;
				      yyerror("`case' after `default' "
				              "not yet implemented");
				    }
				    scan=scan->rest;
				  }
				  $$ = $1; scan->rest = $2;
				}
	;

switchblockstatement
	: blockstatement
	| switchlabel
	;

switchlabel
	: CASE constantexpression ':'
			{ $$ = newstmt(Case, 0, $2, 0, 0, 0, 0, 0, 0); }
	| DEFAULT ':'
			{ $$ = newstmt(Default, 0, 0, 0, 0, 0, 0, 0, 0); }
	;

whilestatement
	: WHILE '(' expression ')' statement
			{ $$ = newstmt(While, 0, $3, $5, 0, 0, 0, 0, 0); }
	;

whilestatementnoshortif
	: WHILE '(' expression ')' statementnoshortif
			{ $$ = newstmt(While, 0, $3, $5, 0, 0, 0, 0, 0); }
	;

dostatement
	: DO statement WHILE '(' expression ')' ';'
			{ $$ = newstmt(Do, 0, $5, $2, 0, 0, 0, 0, 0); }
	;

forstatement
	: FOR '(' forinitopt ';' expressionopt ';' forupdateopt ')' statement
			{ $$ = newstmt(For, 0, $5, $3, $7, $9, 0, 0, 0); }
	;

forstatementnoshortif
	: FOR '(' forinitopt ';' expressionopt ';' forupdateopt ')'
							statementnoshortif
			{ $$ = newstmt(For, 0, $5, $3, $7, $9, 0, 0, 0); }
	;

forinitopt
	: /* empty */		{ $$ = 0; }
	| statementexpressionlist
	| localvariabledeclaration
			{ $$ = newstmt(Localvar, $1, 0, 0, 0, 0, 0, 0, 0); }
	;

forupdateopt
	: /* empty */		{ $$ = 0; }
	| statementexpressionlist
	;

statementexpressionlist
	: statementexpression
			{ $$ = newstmt(Exprstmt, 0, $1, 0, 0, 0, 0, 0, 0); }
	| statementexpressionlist ',' statementexpression
			{ struct stmt *scan = $1;
			  while (scan->rest) { scan=scan->rest; }
			  $$ = $1;
			  scan->rest = newstmt(Exprstmt,0,$3,0,0,0,0,0,0);
			}
	;

breakstatement
	: BREAK IDENTIFIER ';'
			{ num_errors++; yyerror("Ignoring break label");
			  $$ = newstmt(Break, 0, 0, 0, 0, 0, 0, 0, 0);
			}
	| BREAK ';'
			{ $$ = newstmt(Break, 0, 0, 0, 0, 0, 0, 0, 0); }
	;

continuestatement
	: CONTINUE IDENTIFIER ';'
			{ num_errors++; yyerror("Ignoring continue label");
			  $$ = newstmt(Cont, 0, 0, 0, 0, 0, 0, 0, 0);
			}
	| CONTINUE ';'
			{ $$ = newstmt(Cont, 0, 0, 0, 0, 0, 0, 0, 0); }
	;

returnstatement
	: RETURN expressionopt ';'
			{ $$ = newstmt(Return, 0, $2, 0, 0, 0, 0, 0, 0); }
	;

throwstatement
	: THROW expression ';'
			{ $$ = newstmt(Throw, 0, $2, 0, 0, 0, 0, 0, 0); }
	;

synchronizedstatement
	: SYNCHRONIZED '(' expression ')' block
			{ yyerror("Ignoring synchronization"); $$ = $5; }
	;

trystatement
	: TRY block catches
			{ $$ = newstmt(Try, 0, 0, $2, 0, 0, 0, $3, 0); }
	| TRY block catches finally
			{ $$ = newstmt(Try, 0, 0, $2, $4, 0, 0, $3, 0); }
	| TRY block finally
			{ $$ = newstmt(Try, 0, 0, $2, $3, 0, 0, 0, 0); }
	;

catches	: catchclause
	| catches catchclause	{ $$ = $2; $$->rest = $1; }
	;

catchclause
	: CATCH '(' formalparameter ')' block
				{ $$ = newcatchclause($3, $5, 0); }
	;

finally	: FINALLY block		{ $$ = $2; }
	;

/* S15: expressions */

primary	: primarynonewarray
	| arraycreationexpression
	;

primarynonewarray
	: literal
	| primitivetype '.' CLASS
				{ num_errors++;
				  yyerror("Class literals not yet supported.");
				}
	| primitivetype dims '.' CLASS
				{ num_errors++;
				  yyerror("Class literals not yet supported.");
				}
	| name '.' CLASS
				{ num_errors++;
				  yyerror("Class literals not yet supported.");
				}
	| name dims '.' CLASS
				{ num_errors++;
				  yyerror("Class literals not yet supported.");
				}
	| VOID '.' CLASS	{ num_errors++;
				  yyerror("Class literals not yet supported.");
				}
	| THIS			{ $$ = newexpr(This,0,0,0,0,0,0,0,0); }
	| name '.' THIS		{ num_errors++;
				  yyerror("Qualified `this' not yet supported.");
				}
	| '(' expression ')'	{ $$ = $2; }
	| classinstancecreationexpression
	| fieldaccess
	| methodinvocation
	| arrayaccess
	;

literal : INTEGERLITERAL	{ $$ = newexpr(Intlit,0,0,0,0,0,$1,0,0); }
	| LONGINTEGERLITERAL	{ $$ = newexpr(Longlit,0,0,0,0,0,$1,0,0); }
	| FLOATLITERAL		{ $$ = newexpr(Floatlit,0,0,0,0,0,0,$1,0); }
	| DOUBLELITERAL		{ $$ = newexpr(Doublit,0,0,0,0,0,0,$1,0); }
	| BOOLEANLITERAL	{ $$ = newexpr(Boollit,0,0,0,0,0,$1,0,0); }
	| CHARACTERLITERAL	{ $$ = newexpr(Charlit,0,0,0,0,0,$1,0,0); }
	| STRINGLITERAL		{ $$ = newexpr(Stringlit,0,0,0,0,0,0,$1,0); }
	| NULLLITERAL		{ $$ = newexpr(Nulllit,0,0,0,0,0,0,0,0); }
	;

classinstancecreationexpression
	: NEW name '(' argumentlistopt ')'
				{ $$ = newexpr(New,0,$4,0,0,$2,0,0,0); }
	| NEW name '(' argumentlistopt ')' classbody
				{ num_errors++;
				  yyerror("Anonymous classes not yet supported.");
				}
	| primary '.' NEW IDENTIFIER '(' argumentlistopt ')'
				{ num_errors++;
				  yyerror("Inner classes not yet supported.");
				}
	| primary '.' NEW IDENTIFIER '(' argumentlistopt ')' classbody
				{ num_errors++;
				  yyerror("Inner classes not yet supported.");
				}
	;

argumentlistopt
	: /* empty */
				{ $$ = 0; }
	| expression
	| argumentlistopt ',' expression
				{ $3->rest=$1; $$ = $3; }
	;

arraycreationexpression
	: NEW primitivetype dimexprs dimsopt
			{ $$ = newexpr(New,$3,0,0,$2,0,$4,0,0); }
	| NEW name dimexprs dimsopt
			{ $$ = newexpr(New,$3,0,0,0,$2,$4,0,0); }
	| NEW primitivetype dims arrayinitializer
			{ $$=newexpr(Arrayinit,$4,0,0,arrayoftypedepth($2,$3),
			             0,0,0,0);
			}
	| NEW name dims arrayinitializer
			{ $$=newexpr(Arrayinit,$4,0,0,
			             arrayoftypedepth(referencetype($2),$3),
			             0,0,0,0);
			}
	;

dimexprs: dimexpr
	| dimexprs dimexpr	{ $2->rest = $1; $$ = $2;
				  num_errors++;
				  yyerror("Can't instantiate >1 dimention at once!"); }
	;

dimexpr	: '[' expression ']'
			{ $$ = $2; }
	;

dimsopt	: /* empty */
			{ $$ = 0; }
	| dims
	;

dims	: '[' ']'		{ $$ = 1; }
	| dims '[' ']'		{ $$ = $1+1; }
	;

fieldaccess
	: primary '.' IDENTIFIER
			{ $$ = newexpr(Primfield,$1,0,0,0,0,0,$3,0); }
	| SUPER '.' IDENTIFIER
			{ $$ = newexpr(Super,0,0,0,0,0,1,$3,0); }
	;

methodinvocation
	: name '(' argumentlistopt ')'
			{ $$ = newexpr(Meth,newexpr(Name,0,0,0,0,$1->prefix,
			           0,0,0),$3,0,0,0,0,$1->id,0); }
	| primary '.' IDENTIFIER '(' argumentlistopt ')'
			{ $$ = newexpr(Meth,$1,$5,0,0,0,0,$3,0); }
	| SUPER '.' IDENTIFIER '(' argumentlistopt ')'
			{ $$ = newexpr(Super,0,$5,0,0,0,0,$3,0); }
	;

arrayaccess
	: name '[' expression ']'
			{ $$ = newexpr(Array,newexpr(Name,0,0,0,0,$1,0,0,0),
			               $3,0,0,0,0,0,0); }
	| primarynonewarray '[' expression ']'
			{ $$ = newexpr(Array, $1, $3, 0, 0, 0, 0, 0, 0); }
	;

postfixexpression
	: primary
	| name			{ $$ = newexpr(Name,0,0,0,0,$1,0,0,0); }
	| postincrementexpression
	| postdecrementexpression
	;

postincrementexpression
	: postfixexpression PLUSPLUS
			{ $$ = newexpr(Postinc, $1, 0, 0, 0, 0, 0, 0, 0); }
	;

postdecrementexpression
	: postfixexpression MINUSMINUS
			{ $$ = newexpr(Postdec, $1, 0, 0, 0, 0, 0, 0, 0); }
	;

unaryexpression
	: preincrementexpression
	| predecrementexpression
	| '+' unaryexpression
			{ $$ = newexpr(Plus, $2, 0, 0, 0, 0, 0, 0, 0); }
	| '-' unaryexpression
			{ $$ = newexpr(Minus, $2, 0, 0, 0, 0, 0, 0, 0); }
	| unaryexpressionnotplusminus
	;

preincrementexpression
	: PLUSPLUS unaryexpression
			{ $$ = newexpr(Preinc, $2, 0, 0, 0, 0, 0, 0, 0); }
	;

predecrementexpression
	: MINUSMINUS unaryexpression
			{ $$ = newexpr(Predec, $2, 0, 0, 0, 0, 0, 0, 0); }
	;

unaryexpressionnotplusminus
	: postfixexpression
	| '~' unaryexpression
			{ $$ = newexpr(Comp, $2, 0, 0, 0, 0, 0, 0, 0); }
	| '!' unaryexpression
			{ $$ = newexpr(Not, $2, 0, 0, 0, 0, 0, 0, 0); }
	| castexpression
	;

castexpression
	: '(' primitivetype dimsopt ')' unaryexpression
			{ $$ = newexpr(Cast,$5,0,0,newtype($3,$2->prim,0),
			               0,0,0,0); }
	| '(' expression ')' unaryexpressionnotplusminus
			{ $$ = newexpr(Cast,$4,0,0,newtype(0,Ref,$2->name),
			               0,0,0,0); }
	| '(' name dims ')' unaryexpressionnotplusminus
			{ $$ = newexpr(Cast,$5,0,0,newtype($3,Ref,$2),
			               0,0,0,0); }
	;

multiplicativeexpression
	: unaryexpression
	| multiplicativeexpression '*' unaryexpression
			{ $$ = newexpr(Mult, $1, $3, 0, 0, 0, 0, 0, 0); }
	| multiplicativeexpression '/' unaryexpression
			{ $$ = newexpr(Div, $1, $3, 0, 0, 0, 0, 0, 0); }
	| multiplicativeexpression '%' unaryexpression
			{ $$ = newexpr(Rem, $1, $3, 0, 0, 0, 0, 0, 0); }
	;

additiveexpression
	: multiplicativeexpression
	| additiveexpression '+' multiplicativeexpression
			{ $$ = newexpr(Add, $1, $3, 0, 0, 0, 0, 0, 0); }
	| additiveexpression '-' multiplicativeexpression
			{ $$ = newexpr(Sub, $1, $3, 0, 0, 0, 0, 0, 0); }
	;

shiftexpression
	: additiveexpression
	| shiftexpression LLTHAN additiveexpression
			{ $$ = newexpr(Lshift, $1, $3, 0, 0, 0, 0, 0, 0); }
	| shiftexpression GGTHAN additiveexpression
			{ $$ = newexpr(Rshift, $1, $3, 0, 0, 0, 0, 0, 0); }
	| shiftexpression GGGTHAN additiveexpression
			{ $$ = newexpr(Urshift, $1, $3, 0, 0, 0, 0, 0, 0); }
	;

relationalexpression
	: shiftexpression
	| relationalexpression '<' shiftexpression
			{ $$ = newexpr(Lt, $1, $3, 0, 0, 0, 0, 0, 0); }
	| relationalexpression '>' shiftexpression
			{ $$ = newexpr(Gt, $1, $3, 0, 0, 0, 0, 0, 0); }
	| relationalexpression LTHANEQUAL shiftexpression
			{ $$ = newexpr(Leq, $1, $3, 0, 0, 0, 0, 0, 0); }
	| relationalexpression GTHANEQUAL shiftexpression
			{ $$ = newexpr(Geq, $1, $3, 0, 0, 0, 0, 0, 0); }
	| relationalexpression INSTANCEOF referencetype
			{ $$ = newexpr(Inst, $1, 0, 0, $3, 0, 0, 0, 0); }
	;

equalityexpression
	: relationalexpression
	| equalityexpression EQUALEQUAL relationalexpression
			{ $$ = newexpr(Eq, $1, $3, 0, 0, 0, 0, 0, 0); }
	| equalityexpression BANGEQUAL relationalexpression
			{ $$ = newexpr(Neq, $1, $3, 0, 0, 0, 0, 0, 0); }
	;

andexpression
	: equalityexpression
	| andexpression '&' equalityexpression
			{ $$ = newexpr(And, $1, $3, 0, 0, 0, 0, 0, 0); }
	;

exclusiveorexpression
	: andexpression
	| exclusiveorexpression '^' andexpression
			{ $$ = newexpr(Xor, $1, $3, 0, 0, 0, 0, 0, 0); }
	;

inclusiveorexpression
	: exclusiveorexpression
	| inclusiveorexpression '|' exclusiveorexpression
			{ $$ = newexpr(Or, $1, $3, 0, 0, 0, 0, 0, 0); }
	;

conditionalandexpression
	: inclusiveorexpression
	| conditionalandexpression ANDAND inclusiveorexpression
			{ $$ = newexpr(Condand, $1, $3, 0, 0, 0, 0, 0, 0); }
	;

conditionalorexpression
	: conditionalandexpression
	| conditionalorexpression PIPEPIPE conditionalandexpression
			{ $$ = newexpr(Condor, $1, $3, 0, 0, 0, 0, 0, 0); }
	;

conditionalexpression
	: conditionalorexpression
	| conditionalorexpression '?' expression ':' conditionalexpression
			{ $$ = newexpr(Cond, $1, $3, $5, 0, 0, 0, 0, 0); }
	;

assignmentexpression
	: conditionalexpression
	| assignment
	;

assignment
	: lefthandside '=' assignmentexpression
			{ $$ = newexpr(Assign, $1, $3, 0, 0, 0, 0, 0, 0); }
	| lefthandside STAREQUAL assignmentexpression
			{ $$ = newexpr(Multass, $1, $3, 0, 0, 0, 0, 0, 0); }
	| lefthandside SLASHEQUAL assignmentexpression
			{ $$ = newexpr(Divass, $1, $3, 0, 0, 0, 0, 0, 0); }
	| lefthandside PERCENTEQUAL assignmentexpression
			{ $$ = newexpr(Remass, $1, $3, 0, 0, 0, 0, 0, 0); }
	| lefthandside PLUSEQUAL assignmentexpression
			{ $$ = newexpr(Addass, $1, $3, 0, 0, 0, 0, 0, 0); }
	| lefthandside MINUSEQUAL assignmentexpression
			{ $$ = newexpr(Subass, $1, $3, 0, 0, 0, 0, 0, 0); }
	| lefthandside LLTHANEQUAL assignmentexpression
			{ $$ = newexpr(Lshiftass, $1, $3, 0, 0, 0, 0, 0, 0); }
	| lefthandside GGTHANEQUAL assignmentexpression
			{ $$ = newexpr(Rshiftass, $1, $3, 0, 0, 0, 0, 0, 0); }
	| lefthandside GGGTHANEQUAL assignmentexpression
			{ $$ = newexpr(Urshiftass, $1, $3, 0, 0, 0, 0, 0, 0); }
	| lefthandside ANDEQUAL assignmentexpression
			{ $$ = newexpr(Andass, $1, $3, 0, 0, 0, 0, 0, 0); }
	| lefthandside CARETEQUAL assignmentexpression
			{ $$ = newexpr(Xorass, $1, $3, 0, 0, 0, 0, 0, 0); }
	| lefthandside PIPEEQUAL assignmentexpression
			{ $$ = newexpr(Orass, $1, $3, 0, 0, 0, 0, 0, 0); }
	;

lefthandside
	: name			{ $$ = newexpr(Name,0,0,0,0,$1,0,0,0); }
	| fieldaccess
	| arrayaccess
	;

expressionopt
	: /* empty */		{ $$ = 0; }
	| expression
	;

expression
	: assignmentexpression
	;

constantexpression
	: expression
	;

%%

void
yyerror(const char *s)
{
	fprintf(stderr, "%s: %d: %s\n", input_file, input_line, s);
}
